/*
 * File Name: main.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <gtk/gtk.h>
#include <signal.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "i18n.h"
#include "log.h"
#include "about_info.h"
#include "device_settings.h"
#include "ipc.h"
#include "profile_settings.h"
#include "power_settings.h"
#include "recent_settings.h"
#include "settings.h"
#include "settings_utils.h"
#include "sd_settings.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef enum
{
    SETTING_UPLEVEL,
    SETTING_DEVICE,
    SETTING_PROFILE,
    SETTING_POWER,
    SETTING_RECENT,
    SETTING_SD,
    SETTING_ABOUT,
    SETTING_COUNT
} SettingType;

typedef struct
{
    char*      caption;
    GdkPixbuf* pixbuf;
} SettingItem;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget*    g_main_window      = NULL;
static GtkWidget*    g_child_window     = NULL;
static GtkWidget*    g_num_items        = NULL;
static GtkWidget*    g_prev_page_arrow  = NULL;
static GtkWidget*    g_prev_page_filler = NULL;
static GtkWidget*    g_next_page_arrow  = NULL;
static GtkWidget*    g_next_page_filler = NULL;
static GtkWidget*    g_iconview         = NULL;
static GtkWidget*    g_label_settings   = NULL;
static GtkListStore* g_icon_store       = NULL;
static gboolean      g_has_prev_page    = FALSE;
static gboolean      g_has_next_page    = FALSE;
static int           g_items_per_page   = 0;
static int           g_current_offset   = 0;
static SettingItem   g_setting_item[SETTING_COUNT];

FILE *fplog = NULL;

//============================================================================
// Local Function Definitions
//============================================================================

static void       load_setting_items        (void);
static void       unload_setting_items      (void);
static void       update_setting_item_text  (void);
static void       load_items_in_model       (void);

static void       set_cursor                (gint row, gint column);
static gboolean   on_idle_set_cursor        (gpointer data);

static GtkWidget* create_iconview           (GtkWidget* parent);
static GtkWidget* create_screen_layout      (GtkWidget* parent);

static void       on_model_changed          (void);

static void       on_dialog_response        (GtkDialog *widget, gint rc, gpointer data);

static void       on_iconview_item_activated(GtkIconView        *iconview,
                                             GtkTreePath        *path,
                                             gpointer           user_data);

static void       on_iconview_size_allocate (GtkWidget          *widget,
                                             GtkAllocation      *allocation,
                                             gpointer           user_data);

static void       on_iconview_navigate_cursor(erGtkIconView     *er_iconview,
                                             erGtkIconViewKeyPress keycode,
                                             gpointer           user_data);

static void       on_num_items_size_allocate(GtkWidget          *widget,
                                             GtkAllocation      *allocation,
                                             gpointer           user_data);
                                         
static gboolean   on_prev_page_button_press (GtkWidget          *widget, 
                                             GdkEventButton     *event, 
                                             gpointer           user_data);
                                             
static gboolean   on_next_page_button_press (GtkWidget          *widget, 
                                             GdkEventButton     *event, 
                                             gpointer           user_data);
                                             
static void       on_sigterm                (int signo);
static gboolean   on_startup_complete       (gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

int main(int argc, char* argv[])
{
    GtkWidget*       widget      = NULL;
    struct sigaction on_term;

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

	// The log file have been changed to allow logging to be captured on the DR
	fplog = NULL;
	LOGINIT("/media/mmcblk0p1/settings.txt");
	
    // Initialize domain for translations
    textdomain(GETTEXT_PACKAGE);
    
    // Initialize GTK
    gtk_init(&argc, &argv);

    // Initialize GConf library
    gconf_initialize();

    // Initialize ipc
    if (ipc_set_services() == FALSE)
    {
        return -1;
    }

    // Load setting items, this is our real model.
    load_setting_items();

    // Parse resource script.
    gtk_rc_parse(DATADIR"/"PACKAGE_NAME".rc");

    g_main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_maximize(GTK_WINDOW(g_main_window));
    gtk_container_set_border_width(GTK_CONTAINER(g_main_window), 0);
    gtk_window_set_modal(GTK_WINDOW(g_main_window), TRUE);
  
    widget = create_screen_layout(g_main_window);
    update_main_window_text();
    gtk_container_add(GTK_CONTAINER(g_main_window), widget);
    
    // Show all widget
    gtk_widget_show_all(g_main_window);

    // Setup popup menu.
    menu_init();

    // Tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);

    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    return 0;
}


void update_main_window_text()
{
    // Update label "Settings"
    gtk_label_set_label(GTK_LABEL(g_label_settings), _("Settings"));

    // Update description of the setting items.
    update_setting_item_text();

    // Refresh model.
    load_items_in_model();

    // Update widgets since model has been changed.
    on_model_changed();

    // Set cursor to (0,0)
    set_cursor(0, 0);
}


// terminate application
void main_quit()
{
    WARNPRINTF("entry");
    
    ipc_remove_menu(SETTINGS_MENU);

    // Unload setting items.
    unload_setting_items();

    // Finalize ipc services.
    ipc_unset_services();

    // Clean up GConf library.
    gconf_finalize();
    
    if (g_child_window)
    {
        gtk_widget_destroy(g_child_window);
        g_child_window = NULL;
    }    
    
    if (g_main_window)
    {
        gtk_widget_destroy(g_main_window);
        g_main_window = NULL;
    }    
    
    if (gtk_main_level() > 0)
    {
        WARNPRINTF("quit main loop");
        gtk_main_quit();
    }
    else
    {
        WARNPRINTF("no main loop to quit, exit directly");
        _exit(0);
    }
}


//============================================================================
// Local Functions Implementation
//============================================================================

static void update_setting_item_text()
{
    g_setting_item[SETTING_UPLEVEL].caption = _("Up a Level");
    g_setting_item[SETTING_DEVICE].caption  = _("Device");
    g_setting_item[SETTING_PROFILE].caption = _("Personal Preferences");
    g_setting_item[SETTING_POWER].caption   = _("Power Management");
    g_setting_item[SETTING_RECENT].caption  = _("Recent Files");
    g_setting_item[SETTING_SD].caption      = _("SD Card Properties");
    g_setting_item[SETTING_ABOUT].caption   = _("About");
}


static void load_setting_items()
{
    // Load pixmap for "Up a level" icon.
    g_setting_item[SETTING_UPLEVEL].pixbuf = gdk_pixbuf_new_from_file(DATADIR"/icon-up-level.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_UPLEVEL].pixbuf != NULL);

    // Load pixmap for "Device" icon.
    g_setting_item[SETTING_DEVICE].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-device-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_DEVICE].pixbuf != NULL);

    // Load pixmap for "Personal Preferences" icon.
    g_setting_item[SETTING_PROFILE].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-profile-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_PROFILE].pixbuf != NULL);
    
    // Load pixmap for "Power Preferences" icon.
    g_setting_item[SETTING_POWER].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-power-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_POWER].pixbuf != NULL);

    // Load pixmap for "Recent Files" icon.
    g_setting_item[SETTING_RECENT].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-recent-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_RECENT].pixbuf != NULL);

    // Load pixmap for "SD Card Properties" icon.
    g_setting_item[SETTING_SD].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-sd-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_SD].pixbuf != NULL);

    // Load pixmap for "About" icon.
    g_setting_item[SETTING_ABOUT].pixbuf  = gdk_pixbuf_new_from_file(DATADIR"/icon-about-settings.png", NULL);
    g_return_if_fail(g_setting_item[SETTING_ABOUT].pixbuf != NULL);

    update_setting_item_text();
}


static void unload_setting_items()
{
    int i = 0;
    for (; i<SETTING_COUNT; i++)
    {
        g_object_unref(g_setting_item[i].pixbuf);
    }
}


static void load_items_in_model()
{
    int         i;
    int         items_to_be_loaded;
    GtkTreeIter iter;

    // Clear current items.
    gtk_list_store_clear(g_icon_store);
    items_to_be_loaded = MIN(SETTING_COUNT - g_current_offset, g_items_per_page);

    for (i=0; i<items_to_be_loaded; i++)
    {
        gtk_list_store_append(g_icon_store, &iter);
        gtk_list_store_set(g_icon_store,
            &iter,
            0,
            g_setting_item[g_current_offset+i].caption,
            1,
            g_setting_item[g_current_offset+i].pixbuf,
            -1);
    }

    g_has_prev_page = (g_current_offset != 0);
    g_has_next_page = (g_current_offset + g_items_per_page < SETTING_COUNT);
}


// on row-activated in icon view
static void on_iconview_item_activated(GtkIconView *iconview,
                                       GtkTreePath *path,
                                       gpointer    user_data)
{
    gint* indices = NULL;
    GtkWidget* top_level_window = (GtkWidget *)user_data;

    // Check the selected one.
    indices = gtk_tree_path_get_indices(path);
    if (indices == 0)
    {
        return;
    }
    
    ipc_menu_disable();

    *indices += g_current_offset;
    switch (*indices)
    {
    case SETTING_UPLEVEL:
        main_quit();
        return;
        break;
    case SETTING_DEVICE:
        load_device_settings();
        g_child_window = create_device_settings_window(top_level_window);
        break;
    case SETTING_PROFILE:
        load_profile_settings();
        g_child_window = create_profile_settings_window(top_level_window);
        break;
    case SETTING_POWER:
        load_power_settings();
        g_child_window = create_power_settings_window(top_level_window);
        break;
    case SETTING_RECENT:
        load_recent_settings();
        g_child_window = create_recent_settings_window(top_level_window);
        break;
    case SETTING_SD:
        g_child_window = create_sd_settings_window(top_level_window);
        break;
    case SETTING_ABOUT:
        load_about_info();
        g_child_window = create_about_info_window(top_level_window);
        break;
    default:
        break;
    }

    g_signal_connect(g_child_window, "response", G_CALLBACK (on_dialog_response), (void*)*indices);
    gtk_widget_show(g_child_window);
}


static void on_dialog_response(GtkDialog *widget, gint rc, gpointer data)
{
    gint type = (gint) data;

    if (rc != GTK_RESPONSE_ACCEPT)
    {
        if (type == SETTING_DEVICE)
        {
            // Need to rollback language settings if user clicked "Cancel" in device settings dialog.
            rollback_language_settings();
        }
        goto End;
    }

    switch (type)
    {
    case SETTING_DEVICE:
        // Save device settings.
        save_device_settings();
        break;
    case SETTING_PROFILE:
        // Save profile settings.
        save_profile_settings();
        break;
    case SETTING_POWER:
        // Save power settings.
        save_power_settings();
        break;
    case SETTING_RECENT:
        // Save recent file settings.
        save_recent_settings();
        break;
    case SETTING_SD:
        // Nothing to do.
        break;
    case SETTING_ABOUT:
        // Nothing to do.
        break;
    default:
        break;
    }

End:
    if (g_child_window)
    {
        gtk_widget_destroy(g_child_window);
        g_child_window = NULL;
    }
            
    ipc_menu_enable();
}


// on size-allocate for numboer-of-items displayed in title
static void on_num_items_size_allocate ( GtkWidget     *widget,
                                         GtkAllocation *allocation,
                                         gpointer      user_data )
{
    gint            width;
    GtkRequisition  requisition;

    LOGPRINTF( "entry: x y [%d %d] width [%d] height [%d] user_data [%s]",
               allocation->x,
               allocation->y,
               allocation->width,
               allocation->height,
               (const char*) user_data );

    // adjust next-/prev-page bars
    //
    //   calculate width for right filler
    gtk_widget_size_request(g_prev_page_arrow, &requisition);
    width = allocation->width + (TITLE_SPACING / 2) - (requisition.width / 2);
    if (width < 0)
    {
        width = 0;
    }
    //   adjust next-/prev-page bar
    gtk_widget_set_size_request(g_next_page_filler, width, -1);
    gtk_widget_set_size_request(g_prev_page_filler, width, -1);
}


static gboolean on_idle_set_cursor( gpointer data )
{
    gulong  row_col = (gulong)data;
    gint    row     = (row_col >> 8) & 0xFF;
    gint    column  =  row_col       & 0x00FF;

    LOGPRINTF("entry");

    g_assert(g_iconview);
    ergtk_icon_view_set_cursor( ERGTK_ICON_VIEW(g_iconview), row, column );

    return FALSE;  // don't call me again
}


// set cursor on specified item
static void set_cursor ( gint row, gint column )
{
    gulong row_col = ((row & 0xFF) << 8) | (column & 0xFF);

    g_idle_add( on_idle_set_cursor, (gpointer)row_col);
}


static void on_model_changed()
{
    char *cp;
    int  n_items_current_page = 0;

    // Update widgets when model changed.
    if (g_has_prev_page)
    {
        gtk_widget_show(g_prev_page_arrow);
    }
    else
    {
        gtk_widget_hide(g_prev_page_arrow);
    }

    if (g_has_next_page)
    {
        gtk_widget_show(g_next_page_arrow);
    }
    else
    {
        gtk_widget_hide(g_next_page_arrow);
    }

    // Update g_num_items.
    n_items_current_page = MIN(g_items_per_page, SETTING_COUNT - g_current_offset);
    
    if (g_current_offset == 0)
    {
        // Don't count item "Up a Level"
        n_items_current_page--;
    }

    /* TRANSLATORS: %1$d is replaced by the first visible item number
                    %2$d is replaced by the last visible item number
                    %3$d is replaced by the total number of items
       Example: '1 - 9 of 40' */
    cp = g_strdup_printf( _("%1$d - %2$d of %3$d"), 
		g_current_offset + 1,
        g_current_offset + n_items_current_page,
        SETTING_COUNT - 1);

    gtk_label_set_text(GTK_LABEL(g_num_items), cp);
    g_free(cp);
}


// previous-page bar clicked
static gboolean on_prev_page_button_press (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    int  row    = 0;
    int  column = 0;

    LOGPRINTF("entry");
    if (event->type == GDK_BUTTON_PRESS)
    {
        g_current_offset = MAX(0, g_current_offset-g_items_per_page);

        // Save cursor location.
        ergtk_icon_view_get_cursor( ERGTK_ICON_VIEW(g_iconview), &row, &column );

        // Refresh model.
        load_items_in_model();

        // Update widgets since model has been changed.
        on_model_changed();

        // Restore cursor location.
        set_cursor(row, column);
    }

    return FALSE;
}


// next-page bar clicked
static gboolean on_next_page_button_press (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    int  row    = 0;
    int  column = 0;

    LOGPRINTF("entry");
    if (event->type == GDK_BUTTON_PRESS)
    {
        g_current_offset += g_items_per_page;

        // Save cursor location.
        ergtk_icon_view_get_cursor( ERGTK_ICON_VIEW(g_iconview), &row, &column );

        // Refresh model.
        load_items_in_model();

        // Update widgets since model has been changed.
        on_model_changed();

        // Restore cursor location.
        set_cursor(row, column);
    }

    return FALSE;
}


// on size-allocate for icon view
static void on_iconview_size_allocate ( GtkWidget     *widget,
                                        GtkAllocation *allocation,
                                        gpointer      user_data   )
{
    int num_columns = 0;
    int num_rows    = 0;

    LOGPRINTF("entry");
    ergtk_icon_view_get_view_size(ERGTK_ICON_VIEW(g_iconview), &num_columns, &num_rows, NULL);

    if (g_items_per_page == num_rows * num_columns)
    {
        return;
    }

    g_items_per_page = num_rows * num_columns;

    // Since the view size changed, reload the model again.
    load_items_in_model();

    // Update widgets since model has been changed.
    on_model_changed();

    // Set cursor to (0,0)
    set_cursor(0, 0);
}


// on navigate-cursor outside icon view
static void on_iconview_navigate_cursor ( erGtkIconView         *er_iconview,
                                          erGtkIconViewKeyPress keycode,
                                          gpointer              user_data )
{
    // old iconview details
    gint        old_row;
    gint        old_col;
    gint        num_rows;
    gint        num_cols;
    gint        num_items;

    // new iconview details
    gboolean    prev_page = FALSE;      // goto previous page
    gboolean    next_page = FALSE;      // goto next page
    gint        new_row = 0;
    gint        new_col = 0;


    LOGPRINTF("entry: keycode [%d]", keycode);

    // get iconview details
    ergtk_icon_view_get_cursor(er_iconview, &old_row, &old_col);
    ergtk_icon_view_get_view_size(er_iconview, &num_rows, &num_cols, &num_items);

    // determine new cursor position
    switch (keycode)
    {
        case ERGTK_ICON_VIEW_PRESS_SHORT_UP:
            // previous page, last row, same column
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_DOWN:
            // next page, first row, same column
            next_page = TRUE;
            new_row   = 0;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_LEFT:
            // previous page, last item
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = num_cols - 1;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_RIGHT:
            // next page, first item
            next_page = TRUE;
            new_row   = 0;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_UP:
            // previous page, same position
            prev_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_DOWN:
            // next page, same position
            next_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_LEFT:
            // previous page, start of last row
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_RIGHT:
            // next page, end of first row
            next_page = TRUE;
            new_row   = 0;
            new_col   = num_cols - 1;
            break;

        default:
            ERRORPRINTF("illegal erGtkIconViewKeyPress [%d]", keycode);
            ;  // ignore
    }

    // move page as needed, set new cursor position
    if ( prev_page  &&  g_has_prev_page )
    {
        g_current_offset = MAX(0, g_current_offset-g_items_per_page);

        load_items_in_model();

        set_cursor(new_row, new_col);
    }
    else if ( next_page  &&  g_has_next_page )
    {
        g_current_offset += g_items_per_page;

        load_items_in_model();

        set_cursor(new_row, new_col);
    }
}


// create icon view
static GtkWidget* create_iconview (GtkWidget* parent)
{
    GtkWidget   *widget = NULL;   // return value
    GtkIconView *view   = NULL;

    LOGPRINTF("entry");

    g_icon_store = gtk_list_store_new(2, G_TYPE_STRING, GDK_TYPE_PIXBUF);

    widget = ergtk_icon_view_new_with_model( GTK_TREE_MODEL(g_icon_store) );
    g_signal_connect(G_OBJECT(widget), "size-allocate",   G_CALLBACK(on_iconview_size_allocate),   NULL);
    g_signal_connect(G_OBJECT(widget), "item-activated",  G_CALLBACK(on_iconview_item_activated),  parent);
    g_signal_connect(G_OBJECT(widget), "navigate-cursor", G_CALLBACK(on_iconview_navigate_cursor), NULL);
    gtk_widget_set_name(widget, "iconview-irex-settings");
    view = GTK_ICON_VIEW(widget);
    ergtk_icon_view_set_focus_mode(ERGTK_ICON_VIEW(widget), TRUE, FALSE);    // no focus out
    gtk_icon_view_set_spacing(view, 8);    // pixels between icon and text

    // add columns
    //   filename
    gtk_icon_view_set_text_column(view, 0);
    //   thumbnail
    gtk_icon_view_set_pixbuf_column(view, 1);

    return widget;
}


// create screen widgets, borrowed from ctb.
static GtkWidget* create_screen_layout (GtkWidget* parent)
{
    GtkWidget   *alignment = NULL;
    GtkWidget   *widget    = NULL;
    GtkBox      *vbox      = NULL;
    GtkBox      *hbox      = NULL;
    GtkLabel    *label     = NULL;
    GtkMisc     *misc      = NULL;

    LOGPRINTF("entry");

    // object hierarchy:
    //     alignment
    //       |
    widget = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               WINDOW_TOP_PADDING,
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
    alignment = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    vbox = GTK_BOX(widget);
    //             |
    //             |-- hbox
    //             |     |
    widget = gtk_hbox_new(FALSE, TITLE_SPACING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, TITLE_V_PADDING);
    hbox = GTK_BOX(widget);
    //             |     |
    //             |     |-- label
    //             |     |
    widget = gtk_label_new(_("Settings"));
    gtk_widget_set_name(widget, "irex-settings-pathbar");
    gtk_misc_set_alignment(GTK_MISC(widget), 0.0, 0.0);
    gtk_box_pack_start(hbox, widget, TRUE, TRUE, 0);
    g_label_settings = widget;
    //             |     |
    //             |     |-- g_num_items (label)
    //             |
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-settings-num-items");
    g_signal_connect(G_OBJECT(widget), "size-allocate", G_CALLBACK(on_num_items_size_allocate), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    misc = GTK_MISC(widget);
    gtk_misc_set_alignment(misc, 1.0, (gfloat)0.85);
    label = GTK_LABEL(widget);
    gtk_label_set_line_wrap(label, FALSE);
    gtk_label_set_ellipsize(label, PANGO_ELLIPSIZE_NONE);
    g_num_items = widget;
    //             |
    //             |-- hbox
    //             |     |
    //             |     |-- g_prev_page_filler (filler right)
    //             |     |
    //             |     |-- g_prev_page_arrow
    //             |     |
    //             |     |-- event_box (filler left)
    //             |
    widget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, PREVNEXT_HEIGHT);
    hbox = GTK_BOX(widget);
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-prev-page-filler");
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    g_prev_page_filler = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-prev-page-arrow");
    g_signal_connect(G_OBJECT(widget), "button-press-event", G_CALLBACK(on_prev_page_button_press), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, PREVNEXT_ARROW_WIDTH, -1);
    GTK_WIDGET_UNSET_FLAGS(widget, GTK_CAN_FOCUS);
    // note: keep previous-page-arrow hidden
    g_prev_page_arrow = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-prev-page-filler");
    gtk_box_pack_end(hbox, widget, TRUE, TRUE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    //             |
    //             |
    //             |-- g_iconview
    //             |
    widget = create_iconview(parent);
    gtk_box_pack_start(vbox, widget, TRUE, TRUE, FILEGRID_V_PADDING);
    g_iconview = widget;
    //             |
    //             |-- hbox
    //             |     |
    //             |     |-- g_next_page_filler (filler right)
    //             |     |
    //             |     |-- g_next_page_arrow
    //             |     |
    //             |     |-- event_box (filler left)
    //             |
    widget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, PREVNEXT_HEIGHT);
    hbox = GTK_BOX(widget);
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-next-page-filler");
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    g_next_page_filler = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-next-page-arrow");
    g_signal_connect(G_OBJECT(widget), "button-press-event", G_CALLBACK(on_next_page_button_press), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, PREVNEXT_ARROW_WIDTH, -1);
    GTK_WIDGET_UNSET_FLAGS(widget, GTK_CAN_FOCUS);
    // note: keep next-page-arrow hidden
    g_next_page_arrow = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-settings-next-page-filler");
    gtk_box_pack_end(hbox, widget, TRUE, TRUE, 0);
    gtk_widget_set_size_request(widget, -1, -1);

    return alignment;
}


static gboolean on_startup_complete (gpointer data)
{
    LOGPRINTF("entry");

    ipc_sys_startup_complete();

    return FALSE; // remove timer source
}


static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}
