/*
 * File Name: device_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <time.h>
#include <locale.h>

// ereader include files, between < >

// local include files, between " "
#include "i18n.h"
#include "log.h"
#include "ipc.h"
#include "settings.h"
#include "settings_utils.h"
#include "time_selection_dialog.h"
#include "device_settings.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef enum
{
    Rotation_Clockwise,
    Rotation_Anti_Clockwise
} RotationDirection;

typedef struct
{
    int               cur_language;
    time_t            time;
    RotationDirection rotation;
} DeviceSettings;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

#define MAX_LANGUAGES                 16
#define LANGUAGE_BUTTONS_PER_LINE      2
#define DATE_STRING_LEN               64
#define DATETIME_TABLE_SPACING        10


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static DeviceSettings g_orig_device_settings;
static DeviceSettings g_cur_device_settings;

static GtkWidget* g_main_dialog          = NULL;
static GtkWidget* g_language_radios[MAX_LANGUAGES] = {NULL};
static GtkWidget* g_rotation_radios[2]   = {NULL, NULL};
static GtkWidget* g_time_button          = NULL;
static GtkWidget* g_date_button          = NULL;
static GtkWidget* g_save_button          = NULL;
static GtkWidget* g_cancel_button        = NULL;

static GtkWidget* g_label_device         = NULL;
static GtkWidget* g_label_language       = NULL;
static GtkWidget* g_label_language_info  = NULL;
static GtkWidget* g_label_datetime       = NULL;
static GtkWidget* g_label_time           = NULL;
static GtkWidget* g_label_time_desc      = NULL;
static GtkWidget* g_label_date           = NULL;
static GtkWidget* g_label_rotation       = NULL;
static GtkWidget* g_label_rotation_info  = NULL;
static GtkWidget* g_label_clockwise      = NULL;
static GtkWidget* g_label_anti_clockwise = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_language_widgets       (GtkBox* parent);
static GtkWidget* create_datetime_widgets       (GtkBox* parent,
                                                 GtkDialog* parent_dialog);
static GtkWidget* create_rotation_widgets       (GtkBox* parent);
static void       init_widgets_with_settings    ();
static void       update_labels_and_buttons     ();

static GSList*    get_available_locales         ();
static GSList*    get_available_languages       ();
static void       free_string_list              (GSList* lang_list);

static void       on_language_selection_changed (GtkWidget *widget, gpointer data);
static void       on_rotation_selection_changed (GtkWidget *widget, gpointer data);
static void       on_time_entry_clicked         (GtkWidget *widget, gpointer data);
static void       on_date_entry_clicked         (GtkWidget *widget, gpointer data);
static void       on_day_selected               (GtkWidget *widget, gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--language section vbox
//          |--separator
//          |--Time and date section vbox
//          |--separator
//          |--Rotation direction section vbox
//          |--separator
GtkWidget* create_device_settings_window(GtkWidget* parent)
{
    GtkWidget* alignment = NULL;
    GtkWidget* vbox      = NULL;
    GtkWidget* filler    = NULL;
    GtkWidget* separator = NULL;

    bzero(g_language_radios, MAX_LANGUAGES);
    bzero(g_rotation_radios, 2);
    
    GtkWidget* top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        NULL);

    // Create "Cancel" button and add it to action area.
    g_cancel_button = gtk_button_new();
    gtk_dialog_add_action_widget(GTK_DIALOG(top_window), g_cancel_button, GTK_RESPONSE_REJECT);

    // Create "Save & Close" button and add it to action area.
    g_save_button = gtk_button_new();
    gtk_dialog_add_action_widget(GTK_DIALOG(top_window), g_save_button, GTK_RESPONSE_ACCEPT);

    set_popup_window_style(GTK_WINDOW(top_window));

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("Device").
    g_label_device = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_device, "irex-settings-pathbar");
    gtk_widget_set_size_request(g_label_device, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_device), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), g_label_device, FALSE, FALSE, 0);

    // The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The Language section.
    create_language_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The datetime section.
    create_datetime_widgets(GTK_BOX(vbox), GTK_DIALOG(top_window));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The rotation section.
    create_rotation_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // Update widgets with current settings.
    init_widgets_with_settings();

    gtk_widget_show_all(top_window);
    g_main_dialog = top_window;
    return top_window;
}


void load_device_settings()
{
    GSList*     available_locales = NULL;
    GSList*     p                 = NULL;
    const char* cur_locale        = NULL;
    const char* rotation          = NULL;
    int         index             = 0;

    // Get current locale string.
    cur_locale = get_value_string(GCONF_CURRENT_LOCALE);

    // Get locale list.
    available_locales = get_available_locales();
    p = available_locales;
    while (p)
    {
        if (strcmp(p->data, cur_locale) == 0)
        {
            break;
        }

        index++;
        p = p->next;
    }

    g_orig_device_settings.cur_language = index;
    g_orig_device_settings.time = time(NULL);

    // Get rotation direction.
    rotation = get_value_string(GCONF_ROTATION_DIRECTION);
    if (strcmp(rotation, "clockwise") == 0)
    {
        g_orig_device_settings.rotation = Rotation_Clockwise;
    }
    else
    {
        g_orig_device_settings.rotation = Rotation_Anti_Clockwise;
    }

    g_cur_device_settings = g_orig_device_settings;
}


void rollback_language_settings()
{
    GSList* available_locales = NULL;
    const char* cur_locale    = NULL;

    if (g_cur_device_settings.cur_language == g_orig_device_settings.cur_language)
    {
        return;
    }

    g_cur_device_settings.cur_language = g_orig_device_settings.cur_language;

    // Set locale back to original.
    available_locales = get_available_locales();
    cur_locale = g_slist_nth_data(available_locales, g_orig_device_settings.cur_language);
    setlocale(LC_ALL, cur_locale);
    free_string_list(available_locales);
}


void save_device_settings()
{
    GSList*  available_locales = NULL;
    gpointer cur_locale        = NULL;

    // Save language settings if necessary.
    if (g_cur_device_settings.cur_language != g_orig_device_settings.cur_language)
    {
        // Set current locale.
        available_locales = get_available_locales();
        cur_locale = g_slist_nth_data(available_locales, g_cur_device_settings.cur_language);
        set_value_string(GCONF_CURRENT_LOCALE, cur_locale);
        free_string_list(available_locales);
    }

    // Set time if necessary.
    if (g_cur_device_settings.time != g_orig_device_settings.time)
    {
        set_date_time(g_cur_device_settings.time);
    }

    // Set rotation direction if necessary.
    if (g_cur_device_settings.rotation != g_orig_device_settings.rotation)
    {
        set_value_string(GCONF_ROTATION_DIRECTION,
            g_cur_device_settings.rotation == Rotation_Clockwise ? "clockwise" : "anticlockwise");
    }

    LOGPRINTF("Saving device settings, done.");
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Language")
//    |--label ("Select your preferred language")
//    |--table (available languages)
static GtkWidget* create_language_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* table          = NULL;
    GSList*    lang_list      = NULL;
    GSList*    p              = NULL;
    int        rows           = 0;
    int        n_languages    = 0;
    int        i              = 0;

    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Language".
    g_label_language = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_language, "irex-section-title");
    gtk_widget_set_size_request(g_label_language, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_language), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_label_language, FALSE, FALSE, 0);

    // The label "Select your preferred language"
    g_label_language_info = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_language_info, "irex-normal-text");
    gtk_widget_set_size_request(g_label_language_info, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_language_info), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_label_language_info, FALSE, FALSE, 0);

    // The radio buttons.
    lang_list = get_available_languages();
    n_languages = g_slist_length(lang_list);

    rows = n_languages / LANGUAGE_BUTTONS_PER_LINE;
    if (n_languages % LANGUAGE_BUTTONS_PER_LINE != 0)
    {
        rows++;
    }

    p = lang_list;
    table = gtk_table_new(rows, LANGUAGE_BUTTONS_PER_LINE, TRUE);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), table, FALSE, FALSE, 0);

    for (i=0; i<n_languages; i++, p=p->next)
    {
        if (i == 0)
        {
            g_language_radios[i] = gtk_radio_button_new_with_label(NULL, p->data);
        }
        else
        {
            g_language_radios[i] =
                gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_language_radios[0]), p->data);
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_language_radios[i]),
            "toggled",
            G_CALLBACK(on_language_selection_changed),
            (gpointer)i);
        gtk_table_attach_defaults(GTK_TABLE(table), g_language_radios[i], i%2, i%2+1, i/2, i/2+1);
    }

    free_string_list(lang_list);
    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Time and Date")
//    |--hbox
//       |--vbox
//          |--label ("Time")
//          |--label ("Date")
//       |--vbox
//          |--entry ("Time")
//          |--entry ("Date")
//       |--vbox
//          |--Alignment
//             |--label (in 24 hour format)
static GtkWidget* create_datetime_widgets(GtkBox* parent, GtkDialog* parent_dialog)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* table          = NULL;

    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Time and Date".
    g_label_datetime = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_datetime, "irex-section-title");
    gtk_widget_set_size_request(g_label_datetime, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_datetime), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_label_datetime, FALSE, FALSE, 0);

    table = gtk_table_new(2, 3, FALSE);
    gtk_table_set_col_spacings(GTK_TABLE(table), DATETIME_TABLE_SPACING);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), table, FALSE, FALSE, 0);

    // The label "Time".
    g_label_time = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_time, "irex-normal-text");
    gtk_widget_set_size_request(g_label_time, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_time), 0.0, 0.0);
    gtk_table_attach(GTK_TABLE(table), g_label_time, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

    // The Time entry, actually they are buttons.
    g_time_button = gtk_button_new();
    gtk_widget_set_size_request(g_time_button, -1, -1);
    g_signal_connect(G_OBJECT(g_time_button),
        "clicked",
        G_CALLBACK(on_time_entry_clicked),
        parent_dialog);
    gtk_table_attach(GTK_TABLE(table), g_time_button, 1, 2, 0, 1, GTK_FILL, GTK_SHRINK, 0, 0);

    // The label "(in 24 hour format)".
    g_label_time_desc = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_time_desc, "irex-normal-text");
    gtk_widget_set_size_request(g_label_time_desc, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_time_desc), 0.0, 0.0);
    gtk_table_attach(GTK_TABLE(table), g_label_time_desc, 2, 3, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

    // The label "Date".
    g_label_date = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_date, "irex-normal-text");
    gtk_widget_set_size_request(g_label_date, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_date), 0.0, 0.0);
    gtk_table_attach(GTK_TABLE(table), g_label_date, 0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

    // The Date entry, actually they are buttons.
    g_date_button = gtk_button_new();
    gtk_widget_set_size_request(g_date_button, -1, -1);
    g_signal_connect(G_OBJECT(g_date_button),
        "clicked",
        G_CALLBACK(on_date_entry_clicked),
        parent_dialog);
    gtk_table_attach(GTK_TABLE(table), g_date_button, 1, 2, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Rotation Direction")
//    |--label ("Choose the way to hold the device when it is rotated")
//    |--hbox
//       |--nested_hbox
//          |--radio
//          |--GtkWidget
//          |--label ("Clockwise")
//       |--nested_hbox
//          |--radio
//          |--GtkWidget
//          |--label ("Anti-clockwise")
static GtkWidget* create_rotation_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* hbox           = NULL;
    GtkWidget* nested_hbox    = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* image          = NULL;
    int        i              = 0;

    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Rotation Direction".
    g_label_rotation = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_rotation, "irex-section-title");
    gtk_widget_set_size_request(g_label_rotation, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_rotation), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_label_rotation, FALSE, FALSE, 0);

    // The label "Choose the way..."
    g_label_rotation_info = gtk_label_new(NULL);
    gtk_widget_set_name(g_label_rotation_info, "irex-normal-text");
    gtk_widget_set_size_request(g_label_rotation_info, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_label_rotation_info), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_label_rotation_info, FALSE, FALSE, 0);

    // The hbox.
    hbox = gtk_hbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), hbox, FALSE, FALSE, 0);

    for (i=0; i<2; i++)
    {
        // Radio -- bitmap -- label
        label = gtk_label_new(NULL);
        if (i == 0)
        {
            g_rotation_radios[i] = gtk_radio_button_new(NULL);
            image = gtk_image_new_from_file(DATADIR"/rotation-clockwise.png");
            g_label_clockwise = label;
        }
        else
        {
            g_rotation_radios[i] = gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(g_rotation_radios[0]));
            image = gtk_image_new_from_file(DATADIR"/rotation-anticlockwise.png");
            g_label_anti_clockwise = label;
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_rotation_radios[i]),
            "toggled",
            G_CALLBACK(on_rotation_selection_changed),
            (gpointer)i);

        // Set label style.
        gtk_widget_set_name(label, "irex-normal-text");
        gtk_widget_set_size_request(label, -1, -1);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);

        gtk_box_pack_start(GTK_BOX(hbox), g_rotation_radios[i], FALSE, FALSE, 0);

        // The nested hbox.
        nested_hbox = gtk_hbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(g_rotation_radios[i]), nested_hbox);

        // Add image and label to nested hbox.
        gtk_box_pack_start(GTK_BOX(nested_hbox), image, FALSE, FALSE, 0);
        gtk_box_pack_start(GTK_BOX(nested_hbox), label, FALSE, FALSE, 0);
    }

    return top_level_vbox;
}


static void update_labels_and_buttons()
{
    char       buf[DATE_STRING_LEN];
    struct tm* local_time = localtime(&g_cur_device_settings.time);

    // Language section.
    gtk_label_set_label(GTK_LABEL(g_label_device), _("Device"));
    gtk_label_set_label(GTK_LABEL(g_label_language), _("Language"));
    gtk_label_set_label(GTK_LABEL(g_label_language_info), _("Select your preferred language."));

    // Date and time section.
    gtk_label_set_label(GTK_LABEL(g_label_datetime), _("Time and Date"));
    gtk_label_set_label(GTK_LABEL(g_label_time), _("Time"));
    gtk_label_set_label(GTK_LABEL(g_label_time_desc), _("(in 24 hour format)"));
    gtk_label_set_label(GTK_LABEL(g_label_date), _("Date"));

    /* TRANSLATORS: This is a date and time format specifier, according to the strftime(3) man page.
       The corresponding string should be something like 30 Jul 2008. */
    strftime(buf, DATE_STRING_LEN, _("%d %b %Y"), local_time);  // xgettext:no-c-format
    gtk_button_set_label(GTK_BUTTON(g_date_button), buf);

    // Rotation section.
    gtk_label_set_label(GTK_LABEL(g_label_rotation), _("Rotation Direction"));
    gtk_label_set_label(GTK_LABEL(g_label_rotation_info),
        _("Choose the way to hold the device when it is rotated."));
    gtk_label_set_label(GTK_LABEL(g_label_clockwise), _("Clockwise"));
    gtk_label_set_label(GTK_LABEL(g_label_anti_clockwise), _("Anti-clockwise"));

    // "Save & Close" button and "Cancel" button.
    gtk_button_set_label(GTK_BUTTON(g_save_button), _("Save & Close"));
    gtk_button_set_label(GTK_BUTTON(g_cancel_button), _("Cancel"));
}


static GSList* get_available_locales()
{
    return get_value_string_list(GCONF_LOCALE_LIST);
}


static GSList* get_available_languages()
{
    return get_value_string_list(GCONF_LANGUAGE_LIST);
}


static void free_string_list(GSList* lang_list)
{
    GSList* p = lang_list;

    while (p)
    {
        g_free(p->data);
        p = p->next;
    }

    g_slist_free(lang_list);
}


static void on_language_selection_changed(GtkWidget *widget, gpointer data)
{
    GSList* available_locales = NULL;
    const char* cur_locale    = NULL;
    int index                 = (int)data;

    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    if (is_active == FALSE || index == g_cur_device_settings.cur_language)
    {
        return;
    }

    g_cur_device_settings.cur_language = index;

    // Step 1: Call setlocale.
    available_locales = get_available_locales();
    cur_locale = g_slist_nth_data(available_locales, g_cur_device_settings.cur_language);
    setlocale(LC_ALL, cur_locale);
    free_string_list(available_locales);

    // Step 2: Update labels and buttons with new locale.
    update_labels_and_buttons();

    // Step 3: Re-position the dialog to the center of main window.
    gtk_window_set_position(GTK_WINDOW(g_main_dialog), GTK_WIN_POS_CENTER_ON_PARENT);
}


static void on_rotation_selection_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_device_settings.rotation = index;
    }
}


static void on_time_entry_clicked(GtkWidget *widget, gpointer data)
{
    char time_str[8]  = {0};
    GtkWindow* parent = GTK_WINDOW(data);
    int result        = 0;
    int hh            = 0;
    int mm            = 0;

    // Need to popup a dialog with numeric buttons.
    GtkWidget* dialog = create_time_selection_dialog(parent);
    result = gtk_dialog_run(GTK_DIALOG(dialog));

    if (result == GTK_RESPONSE_ACCEPT)
    {
        // User set the time, update widget first.
        hh = get_hh();
        mm = get_mm();
        sprintf(time_str, "%02d:%02d", hh, mm);
        gtk_button_set_label(GTK_BUTTON(g_time_button), time_str);

        // Change the current time settings.
        g_cur_device_settings.time = (g_cur_device_settings.time) / 86400 * 86400 + hh*3600 + mm*60;
    }

    gtk_widget_destroy(dialog);
}


// Widget Hierarchy
// |--Top Level vbox
//    |--GtkCalendar
static void on_date_entry_clicked(GtkWidget *widget, gpointer data)
{
    // Need to popup a dialog with GtkCalendar widget.
    GtkWidget* dialog   = NULL;
    GtkWidget* calendar = NULL;
    guint      year     = 0;
    guint      month    = 0;
    guint      day      = 0;
    int        result   = 0;
    GtkWindow* parent   = GTK_WINDOW(data);
    char       buf[DATE_STRING_LEN] = {0};
    struct tm  tm_sel, *tm_cur = NULL;

    tm_cur = localtime(&g_cur_device_settings.time);
    memcpy(&tm_sel, tm_cur, sizeof(struct tm));

    dialog = gtk_dialog_new_with_buttons("",
        parent,
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        NULL);
    set_popup_window_style(GTK_WINDOW(dialog));

    // Calendar widget.
    calendar = gtk_calendar_new();
    gtk_calendar_select_month(GTK_CALENDAR(calendar), tm_sel.tm_mon, tm_sel.tm_year + 1900);
    gtk_calendar_select_day(GTK_CALENDAR(calendar), tm_sel.tm_mday);
    g_signal_connect(G_OBJECT(calendar), "day-selected", G_CALLBACK(on_day_selected), dialog);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), calendar);

    gtk_widget_show_all(dialog);
    result = gtk_dialog_run(GTK_DIALOG(dialog));

    if (result != GTK_RESPONSE_ACCEPT)
    {
        return;
    }

    // Get the selected date.
    gtk_calendar_get_date(GTK_CALENDAR(calendar), &year, &month, &day);
    gtk_widget_destroy(dialog);

    // User set the date, update widget first.
    tm_sel.tm_year = year - 1900;
    tm_sel.tm_mon  = month;
    tm_sel.tm_mday = day;

    /* TRANSLATORS: This is a date and time format specifier, according to the strftime(3) man page.
       The corresponding string should be something like 30 Jul 2008. */
    strftime(buf, DATE_STRING_LEN, _("%d %b %Y"), &tm_sel);  // xgettext:no-c-format
    gtk_button_set_label(GTK_BUTTON(g_date_button), buf);

    // Change the current time settings.
    g_cur_device_settings.time = mktime(&tm_sel);
}


static void on_day_selected(GtkWidget *widget, gpointer data)
{
    GtkDialog* dialog = GTK_DIALOG(data);
    gtk_dialog_response(dialog, GTK_RESPONSE_ACCEPT);
}


static void init_widgets_with_settings()
{
    char       buf[DATE_STRING_LEN];
    struct tm* local_time = localtime(&g_cur_device_settings.time);

    // Update labels with current locale.
    update_labels_and_buttons();

    // Set current chosen language option.
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_language_radios[g_cur_device_settings.cur_language]),
                                 TRUE);

    // Set current time.
    sprintf(buf, "%02d:%02d", local_time->tm_hour, local_time->tm_min);
    gtk_button_set_label(GTK_BUTTON(g_time_button), buf);

    // Set current roration option.
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_rotation_radios[g_cur_device_settings.rotation]),
                                 TRUE);
}
