/*
 * File Name: power_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <string.h>
#include <stdlib.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "recent_settings.h"
#include "settings.h"
#include "settings_utils.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// CAREFULL: These values need to match with the values in ctb!
static const gchar      *REGVAL_RECENTFILES_OFF   = "off";
static const gchar      *REGVAL_RECENTFILES_STORE = "store";
static const gchar      *REGVAL_RECENTFILES_START = "start";
static const gchar      *REGVAL_RECENTFILES_AUTO  = "auto";

// Do not store recent files
#define RECENTFILES_OFF   0
// Only store
#define RECENTFILES_STORE 1
// After power on, start in recent files folder
#define RECENTFILES_START 2
// After power on, start the last read document
#define RECENTFILES_AUTO  3

typedef struct
{
    int mode;
} RecentSettings;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static RecentSettings g_orig_recent_settings;
static RecentSettings g_cur_recent_settings;

static GtkWidget* g_enable_recentfile_check       = NULL;
static GtkWidget* g_show_recentfile_check         = NULL;
static GtkWidget* g_start_recentfile_check        = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_enable_widgets         (GtkBox* parent);
static GtkWidget* create_startup_widgets        (GtkBox* parent);

static void on_enable_changed                   (GtkWidget *widget, gpointer data);
static void on_show_changed                (GtkWidget *widget, gpointer data);
static void on_start_changed                  (GtkWidget *widget, gpointer data);
static void init_widgets_with_settings          (void);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--Enable section vbox
//          |--separator
//          |--Startup section vbox
//          |--separator
GtkWidget* create_recent_settings_window(GtkWidget* parent)
{
    GtkWidget* top_window = NULL;
    GtkWidget* alignment  = NULL;
    GtkWidget* vbox       = NULL;
    GtkWidget* label      = NULL;
    GtkWidget* filler     = NULL;
    GtkWidget* separator  = NULL;

    top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        _("Cancel"),
        GTK_RESPONSE_REJECT,
        _("Save & Close"),
        GTK_RESPONSE_ACCEPT,
        NULL);
    set_popup_window_style(GTK_WINDOW(top_window));

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("Recent Files Settings").
    label = gtk_label_new(_("Recent Files Settings"));
    gtk_widget_set_name(label, "irex-settings-pathbar");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	// Version number
    label = gtk_label_new("Version 0.3e");
    gtk_widget_set_name(label, "irex-settings-num-items");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	// The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The enable settings section.
    create_enable_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The startup settings section.
    create_startup_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // Update widget with current settings.
    init_widgets_with_settings();
    gtk_widget_show_all(top_window);
    return top_window;
}


void load_recent_settings()
{
	const char* mode_str;
	
	mode_str = get_value_string(GCONF_RECENTFILES_MODE);
	
	LOGPRINTF("entry:%s", mode_str);
	
    g_orig_recent_settings.mode = RECENTFILES_OFF;
	if (strcmp(mode_str, REGVAL_RECENTFILES_STORE) == 0)
	{
		g_orig_recent_settings.mode = RECENTFILES_STORE;
	}
	else if (strcmp(mode_str, REGVAL_RECENTFILES_START) == 0)
	{
		g_orig_recent_settings.mode = RECENTFILES_START;
	}
	else if (strcmp(mode_str, REGVAL_RECENTFILES_AUTO) == 0)
	{
		g_orig_recent_settings.mode = RECENTFILES_AUTO;
	}
	// else it is off
	
	LOGPRINTF("mode:%d", g_orig_recent_settings.mode);
	
    g_cur_recent_settings = g_orig_recent_settings;
}


void save_recent_settings()
{
	g_cur_recent_settings.mode = RECENTFILES_OFF;
	// get status of check boxes
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_enable_recentfile_check)))
	{
		g_cur_recent_settings.mode = RECENTFILES_STORE;
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_show_recentfile_check)))
		{
			g_cur_recent_settings.mode = RECENTFILES_START;
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_start_recentfile_check)))
			{
				g_cur_recent_settings.mode = RECENTFILES_AUTO;
			}
		}
	}
    if (g_cur_recent_settings.mode != g_orig_recent_settings.mode)
    {
		const char *mode_str = REGVAL_RECENTFILES_OFF;
		if (g_cur_recent_settings.mode == RECENTFILES_STORE)
		{
			mode_str = REGVAL_RECENTFILES_STORE;
		}
		else if (g_cur_recent_settings.mode == RECENTFILES_START)
		{
			mode_str = REGVAL_RECENTFILES_START;
		}
		else if (g_cur_recent_settings.mode == RECENTFILES_AUTO)
		{
			mode_str = REGVAL_RECENTFILES_AUTO;
		}

		set_value_string(GCONF_RECENTFILES_MODE, mode_str);
    }

    LOGPRINTF("Saving recent settings, done.");
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Recent Files Options")
//    |--vbox
//       |--check
static GtkWidget* create_enable_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Automatic Shutdown".
    label = gtk_label_new(_("Recent Files Options"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);
    
    // The vbox containing check button.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    g_enable_recentfile_check = gtk_check_button_new_with_label(_("Enable the storage of recently used files"));
    g_signal_connect(G_OBJECT(g_enable_recentfile_check),
        "toggled",
        G_CALLBACK(on_enable_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_enable_recentfile_check, FALSE, FALSE, 0);

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Startup Options")
//    |--vbox
//       |--check
//       |--check
static GtkWidget* create_startup_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);
    
    // The label "Power Saving".
    label = gtk_label_new(_("Startup Options"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The vbox containing the checkbox.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);
    
    g_show_recentfile_check = gtk_check_button_new_with_label(_("Show Recent Files folder after Power-up"));
    g_signal_connect(G_OBJECT(g_show_recentfile_check),
        "toggled",
        G_CALLBACK(on_show_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_show_recentfile_check, FALSE, FALSE, 0);

    g_start_recentfile_check = gtk_check_button_new_with_label(_("Start with last read document after Power-up"));
    g_signal_connect(G_OBJECT(g_start_recentfile_check),
        "toggled",
        G_CALLBACK(on_start_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_start_recentfile_check, FALSE, FALSE, 0);

	return top_level_vbox;
}

static void on_enable_changed(GtkWidget *widget, gpointer data)
{
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	LOGPRINTF("enter:%d", is_active);
	// these options only make sense if recent files is enabled
	gtk_widget_set_sensitive(g_show_recentfile_check, is_active);
	// Not yet implemented, keep disabled
	gtk_widget_set_sensitive(g_start_recentfile_check, FALSE);
}


static void on_show_changed(GtkWidget *widget, gpointer data)
{
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	LOGPRINTF("enter:%d", is_active);
	// if not shown, disable start
	if (!is_active)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_start_recentfile_check), FALSE);
	}
}

static void on_start_changed(GtkWidget *widget, gpointer data)
{
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	LOGPRINTF("enter:%d", is_active);
	// if start, the make show active
	if (is_active)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_show_recentfile_check), TRUE);
	}
}

static void init_widgets_with_settings()
{
	gboolean is_active;
	
	is_active = (g_cur_recent_settings.mode >= RECENTFILES_STORE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_enable_recentfile_check), is_active);
	if (!is_active)
	{
		gtk_widget_set_sensitive(g_show_recentfile_check, FALSE);
		gtk_widget_set_sensitive(g_start_recentfile_check, FALSE);
	}
	is_active = (g_cur_recent_settings.mode >= RECENTFILES_START);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_show_recentfile_check), is_active);
	is_active = (g_cur_recent_settings.mode >= RECENTFILES_AUTO);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_start_recentfile_check), is_active);
}

