/*
 * File Name: profile_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <string.h>
#include <stdlib.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "profile_settings.h"
#include "settings.h"
#include "settings_utils.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct
{
    gboolean lights_on;
    gboolean sound_on;
    gboolean lock_left_sensor;
    gboolean lock_middle_sensor;
    gboolean lock_right_sensor;
    gboolean stylus_on;
} ProfileSettings;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static ProfileSettings g_orig_profile_settings;
static ProfileSettings g_cur_profile_settings;

static GtkWidget* g_light_radios[2]             = {NULL, NULL};
static GtkWidget* g_sound_radios[2]             = {NULL, NULL};
static GtkWidget* g_lock_left_sensor_check      = NULL;
static GtkWidget* g_lock_middle_sensor_check    = NULL;
static GtkWidget* g_lock_right_sensor_check     = NULL;
static GtkWidget* g_sensor_lock_image           = NULL;
static GtkWidget* g_stylus_radios[2]            = {NULL, NULL};


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_lights_widgets         (GtkBox* parent);
static GtkWidget* create_sound_widgets          (GtkBox* parent);
static GtkWidget* create_sensor_widgets         (GtkBox* parent);
static GtkWidget* create_stylus_widgets         (GtkBox* parent);

static void on_use_light_changed                (GtkWidget *widget, gpointer data);
static void on_use_sound_changed                (GtkWidget *widget, gpointer data);
static void on_lock_left_changed                (GtkWidget *widget, gpointer data);
static void on_lock_middle_changed              (GtkWidget *widget, gpointer data);
static void on_lock_right_changed               (GtkWidget *widget, gpointer data);
static void on_use_stylus_changed               (GtkWidget *widget, gpointer data);
static void on_calibrate_stylus_clicked         (GtkWidget *widget, gpointer data);

static void update_sensor_lock_image            (gboolean lock_left,
                                                 gboolean lock_middle,
                                                 gboolean lock_right);
static void init_widgets_with_settings          (void);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--Lights section vbox
//          |--separator
//          |--Sound section vbox
//          |--separator
//          |--Sensor lock section vbox
//          |--separator
//          |--Stylus section vbox
//          |--separator
GtkWidget* create_profile_settings_window(GtkWidget* parent)
{
    GtkWidget* top_window = NULL;
    GtkWidget* alignment  = NULL;
    GtkWidget* vbox       = NULL;
    GtkWidget* label      = NULL;
    GtkWidget* filler     = NULL;
    GtkWidget* separator  = NULL;
    
    bzero(g_light_radios, 2);
    bzero(g_sound_radios, 2);
    bzero(g_stylus_radios, 2);
    
    top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        _("Cancel"),
        GTK_RESPONSE_REJECT,
        _("Save & Close"),
        GTK_RESPONSE_ACCEPT,
        NULL);
    set_popup_window_style(GTK_WINDOW(top_window));

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("Personal Preferences").
    label = gtk_label_new(_("Personal Preferences"));
    gtk_widget_set_name(label, "irex-settings-pathbar");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    // The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The lights settings section.
    create_lights_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The sound settings section.
    create_sound_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    if (get_device_type() >= IREX_DR1000S)
    {
        // The sensor settings section.
        create_sensor_widgets(GTK_BOX(vbox));

        // Add separator here.
        separator = create_separator_widgets();
        gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

        // The stylus settings section.
        create_stylus_widgets(GTK_BOX(vbox));

        // Add separator here.
        separator = create_separator_widgets();
        gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);
    }

    // Update widget with current settings.
    init_widgets_with_settings();
    gtk_widget_show_all(top_window);
    return top_window;
}


void load_profile_settings()
{
    g_orig_profile_settings.lights_on           = get_value_bool(GCONF_USE_LIGHT);
    g_orig_profile_settings.sound_on            = get_value_bool(GCONF_USE_SOUND);
    g_orig_profile_settings.lock_left_sensor    = get_value_bool(GCONF_LOCK_LEFT_SENSOR);
    g_orig_profile_settings.lock_middle_sensor  = get_value_bool(GCONF_LOCK_MIDDLE_SENSOR);
    g_orig_profile_settings.lock_right_sensor   = get_value_bool(GCONF_LOCK_RIGHT_SENSOR);
    g_orig_profile_settings.stylus_on           = get_value_bool(GCONF_USE_STYLUS);

    g_cur_profile_settings = g_orig_profile_settings;
}

void save_profile_settings()
{
    if (g_cur_profile_settings.lights_on != g_orig_profile_settings.lights_on)
    {
        set_value_bool(GCONF_USE_LIGHT, g_cur_profile_settings.lights_on);
    }

    if (g_cur_profile_settings.sound_on != g_orig_profile_settings.sound_on)
    {
        set_value_bool(GCONF_USE_SOUND, g_cur_profile_settings.sound_on);
    }

    if (g_cur_profile_settings.lock_left_sensor != g_orig_profile_settings.lock_left_sensor)
    {
        set_value_bool(GCONF_LOCK_LEFT_SENSOR, g_cur_profile_settings.lock_left_sensor);
    }

    if (g_cur_profile_settings.lock_middle_sensor != g_orig_profile_settings.lock_middle_sensor)
    {
        set_value_bool(GCONF_LOCK_MIDDLE_SENSOR, g_cur_profile_settings.lock_middle_sensor);
    }

    if (g_cur_profile_settings.lock_right_sensor != g_orig_profile_settings.lock_right_sensor)
    {
        set_value_bool(GCONF_LOCK_RIGHT_SENSOR, g_cur_profile_settings.lock_right_sensor);
    }

    if (g_cur_profile_settings.stylus_on != g_orig_profile_settings.stylus_on)
    {
        set_value_bool(GCONF_USE_STYLUS, g_cur_profile_settings.stylus_on);
    }

    LOGPRINTF("Saving profile settings, done.");
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Lights")
//    |--label ("This setting controls the green indication lights next to the sensor panels.")
//    |--vbox
//       |--radio
//       |--radio
static GtkWidget* create_lights_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Lights".
    label = gtk_label_new(_("Lights"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "This setting controls..."
    label = gtk_label_new(_("This setting controls the green indication lights next to the sensor panels."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i=0; i<2; i++)
    {
        if (i == 0)
        {
            // The radio button "On".
            g_light_radios[i] = gtk_radio_button_new_with_label(NULL, _("On"));
        }
        else
        {
            // The radio button "Off".
            g_light_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_light_radios[0]), _("Off"));
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_light_radios[i]),
            "toggled",
            G_CALLBACK(on_use_light_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_light_radios[i], FALSE, FALSE, 0);
    }

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Sound")
//    |--label ("You can turn the beep on or off with this setting.")
//    |--vbox
//       |--radio
//       |--radio
static GtkWidget* create_sound_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Sound".
    label = gtk_label_new(_("Sound"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "You can turn..."
    label = gtk_label_new(_("You can turn the beep on or off with this setting."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i=0; i<2; i++)
    {
        if (i == 0)
        {
            // The radio button "On".
            g_sound_radios[i] = gtk_radio_button_new_with_label(NULL, _("On"));
        }
        else
        {
            // The radio button "Off".
            g_sound_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_sound_radios[0]), _("Off"));
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_sound_radios[i]),
            "toggled",
            G_CALLBACK(on_use_sound_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_sound_radios[i], FALSE, FALSE, 0);
    }

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Sensor Lock")
//    |--label ("Customize your own sensor lock by choosing one or all of the lock options.")
//    |--hbox
//       |--vbox
//          |--check button
//          |--check button
//          |--check button
//       |--image
static GtkWidget* create_sensor_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* hbox           = NULL;
    GtkWidget* vbox           = NULL;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Sensor Lock".
    label = gtk_label_new(_("Sensor Lock"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "Customize your own..."
    label = gtk_label_new(_("Customize your own sensor lock by choosing one or all of the lock options."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The hbox containing check buttons and image.
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), hbox, FALSE, FALSE, 0);

    // The vbox containing check buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    // The fist check button "Lock Left Sensors".
    g_lock_left_sensor_check = gtk_check_button_new_with_label(_("Lock Left Sensors"));
    g_signal_connect(G_OBJECT(g_lock_left_sensor_check),
        "toggled",
        G_CALLBACK(on_lock_left_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_lock_left_sensor_check, FALSE, FALSE, 0);

    // The second check button "Lock Middle Sensors".
    g_lock_middle_sensor_check = gtk_check_button_new_with_label(_("Lock Middle Sensors"));
    g_signal_connect(G_OBJECT(g_lock_middle_sensor_check),
        "toggled",
        G_CALLBACK(on_lock_middle_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_lock_middle_sensor_check, FALSE, FALSE, 0);

    // The third check button "Lock Right Sensors".
    g_lock_right_sensor_check = gtk_check_button_new_with_label(_("Lock Right Sensors"));
    g_signal_connect(G_OBJECT(g_lock_right_sensor_check),
        "toggled",
        G_CALLBACK(on_lock_right_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_lock_right_sensor_check, FALSE, FALSE, 0);

    // The sensor lock image.
    g_sensor_lock_image = gtk_image_new();
    gtk_box_pack_start(GTK_BOX(hbox), g_sensor_lock_image, TRUE, FALSE, 0);
    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Stylus")
//    |--label ("To save power or to operate the device without a stylus, set the stylus to off.")
//    |--hbox
//       |--vbox
//          |--radio
//          |--radio
//       |--alignment
//          |--button "Calibrate Stylus"
static GtkWidget* create_stylus_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* hbox           = NULL;
    GtkWidget* vbox           = NULL;
    GtkWidget* alignment      = NULL;
    GtkWidget* button         = NULL;
    int        i              = 0;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Stylus".
    label = gtk_label_new(_("Stylus"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "To save power..."
    label = gtk_label_new(_("To save power or to operate the device without a stylus, set the stylus to off."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), hbox, FALSE, FALSE, 0);

    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    for (i=0; i<2; i++)
    {
        if (i == 0)
        {
            // The radio button "On".
            g_stylus_radios[i] = gtk_radio_button_new_with_label(NULL, _("On"));
        }
        else
        {
            // The radio button "Off".
            g_stylus_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_stylus_radios[0]), _("Off"));
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_stylus_radios[i]),
            "toggled",
            G_CALLBACK(on_use_stylus_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_stylus_radios[i], FALSE, FALSE, 0);
    }

    // The alignment containing calibration button.
    alignment = gtk_alignment_new(0.5, 1.0, 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), alignment, FALSE, FALSE, 0);

    // The button "Calibrate Stylus"
    button = gtk_button_new();
    g_signal_connect(G_OBJECT(button),
        "clicked",
        G_CALLBACK(on_calibrate_stylus_clicked),
        NULL);
    gtk_container_add(GTK_CONTAINER(alignment), button);

    alignment = gtk_alignment_new(0.5, 0.5, 0, 0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              0,
                              0,
                              LABEL_PADDING_WITHIN_BUTTON,
                              LABEL_PADDING_WITHIN_BUTTON);
    gtk_container_add(GTK_CONTAINER(button), alignment);

    label = gtk_label_new(_("Calibrate Stylus"));
    gtk_container_add(GTK_CONTAINER(alignment), label);

    return top_level_vbox;
}


static void on_use_light_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_profile_settings.lights_on = (index == 0) ? TRUE : FALSE;
    }
}


static void on_use_sound_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_profile_settings.sound_on = (index == 0) ? TRUE : FALSE;
    }
}


static void on_lock_left_changed(GtkWidget *widget, gpointer data)
{
    g_cur_profile_settings.lock_left_sensor = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    // Update sensor lock image.
    update_sensor_lock_image(g_cur_profile_settings.lock_left_sensor,
        g_cur_profile_settings.lock_middle_sensor,
        g_cur_profile_settings.lock_right_sensor);
}


static void on_lock_middle_changed(GtkWidget *widget, gpointer data)
{
    g_cur_profile_settings.lock_middle_sensor = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    // Update sensor lock image.
    update_sensor_lock_image(g_cur_profile_settings.lock_left_sensor,
        g_cur_profile_settings.lock_middle_sensor,
        g_cur_profile_settings.lock_right_sensor);
}


static void on_lock_right_changed(GtkWidget *widget, gpointer data)
{
    g_cur_profile_settings.lock_right_sensor = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    // Update sensor lock image.
    update_sensor_lock_image(g_cur_profile_settings.lock_left_sensor,
        g_cur_profile_settings.lock_middle_sensor,
        g_cur_profile_settings.lock_right_sensor);
}


static void on_use_stylus_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_profile_settings.stylus_on = (index == 0) ? TRUE : FALSE;
    }
}


static void on_calibrate_stylus_clicked(GtkWidget *widget, gpointer data)
{
    ipc_menu_popup("block");
    if (!get_value_bool(GCONF_USE_STYLUS))
    {
        ipc_set_stylus("enable");
    }

    // Call stylus calibration program.
    system("/usr/bin/gtktscal");

    if (!get_value_bool(GCONF_USE_STYLUS))
    {
        ipc_set_stylus("disable");
    }
    ipc_menu_popup("unblock");
}


static void update_sensor_lock_image(gboolean lock_left,
                                     gboolean lock_middle,
                                     gboolean lock_right)
{
    gchar* image_file = g_strdup_printf(DATADIR"/sensor-lock%1d%1d%1d.png",
        lock_left ? 1 : 0,
        lock_middle ? 1 : 0,
        lock_right ? 1 : 0);

    gtk_image_set_from_file(GTK_IMAGE(g_sensor_lock_image), image_file);
    g_free(image_file);
}


static void init_widgets_with_settings()
{
    int to_be_activated = 0;

    to_be_activated = g_cur_profile_settings.lights_on == TRUE ? 0 : 1;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_light_radios[to_be_activated]),
                                 TRUE);

    to_be_activated = g_cur_profile_settings.sound_on == TRUE ? 0 : 1;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_sound_radios[to_be_activated]),
                                 TRUE);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_lock_left_sensor_check),
        g_cur_profile_settings.lock_left_sensor);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_lock_middle_sensor_check),
        g_cur_profile_settings.lock_middle_sensor);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_lock_right_sensor_check),
        g_cur_profile_settings.lock_right_sensor);

    // Update sensor lock image.
    update_sensor_lock_image(g_cur_profile_settings.lock_left_sensor,
        g_cur_profile_settings.lock_middle_sensor,
        g_cur_profile_settings.lock_right_sensor);

    to_be_activated = g_cur_profile_settings.stylus_on == TRUE ? 0 : 1;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_stylus_radios[to_be_activated]),
                                 TRUE);
}
