/*
 * File Name: shortcut.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <glib.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

// local include files, between " "
#include "ctb_log.h"
#include "ctb_actions.h"
#include "filetypes.h"
#include "i18n.h"
#include "main.h"
#include "shortcut.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const char   *ILLEGAL_FILEPATH_CHARS = "*?\\:\"<>|";

static const gchar  *FILE_SHORTCUT   = "[Desktop Entry]\n"
                                        "Version=1.0\n"
                                        "Type=Link\n"
                                        "URL=file:%s\n"
                                        "Name=%s\n";

static const gchar  *FOLDER_SHORTCUT = "[Desktop Entry]\n"
                                       "Version=1.0\n"
                                       "Type=Directory\n"
                                       "Path=%s\n"
                                       "Name=%s\n";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------


//============================================================================
// Local Function Definitions
//============================================================================

static int          parse_file_url              ( const gchar      *directory,
                                                  const gchar      *url,
                                                        shortcut_t *shortcut );
                                                
static int          parse_folder_path           ( const gchar      *directory,
                                                  const gchar      *target_path,
                                                        shortcut_t *shortcut    );

static gchar*       path_fix_invalid_characters ( const gchar       *path );
static gboolean     path_has_invalid_characters ( const gchar       *target_path );

static shortcut_t*  shortcut_new                ( void );


//============================================================================
// Functions Implementation
//============================================================================

// activate: parse shortcut, then launch viewer, goto directory, ...
int activate_shortcut ( const GString           *directory,
                        const filelist_entry_t  *fileinfo  )
{
    g_assert(directory && directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename && fileinfo->filename->str);

    int                 ret = ER_OK;     // return code
    gchar               *cp;
    gchar               *error_msg       = NULL;
    shortcut_t          *shortcut        = NULL;
    filelist_entry_t    *fileinfo_target = NULL;
    GString             *dir_target      = NULL;
    GString             *cmd_line        = NULL;

    const gchar         *filename        = fileinfo->filename->str;

    LOGPRINTF( "entry: dir [%s] file [%s]", directory->str, filename );

    if ( fileinfo->is_template )
    {
        ERRORPRINTF("shortcut cannot be a template");
        return ER_INVALID_PARAMETER;
    }

    // get shortcut details
    ret = parse_shortcut_file(directory->str, filename, &shortcut);
    if (shortcut->type == SHORTCUT_TO_FOLDER)
    {
        ret = shortcut_propagate_name_to_folder(shortcut);
        if (   ret == ER_OK
            && shortcut->details.folder.directory == NULL )
        {
            ret = ER_NOT_FOUND;
        }
    }
    if (ret != ER_OK)
    {
        // show error message
        switch (ret)
        {
            case ER_NOT_FOUND:
                error_msg = g_strdup_printf( _("The location this shortcut points to no longer exists.\n"
                                               "The original file or folder could have been changed or moved.") );
                break;
            case ER_INVALID_DATA:
                error_msg = g_strdup_printf( _("'%s' cannot be opened; the file is corrupt."),
                                             filename );
                break;
            default:
                error_msg = g_strdup_printf( _("'%s' cannot be opened."),
                                             filename );
        }
        run_error_dialog( error_msg );
    }

    // create fileinfo from shortcut, then launch target
    if (ret == ER_OK)
    {
        fileinfo_target = filelist_entry_new();

        g_assert(shortcut);
        switch (shortcut->type)
        {
            case SHORTCUT_TO_FILE:
                cp = shortcut->details.file.filename;
                fileinfo_target->filename         = g_string_new( cp );
                fileinfo_target->filename_display = g_string_new( shortcut->name );
                fileinfo_target->filetype         = g_string_new( g_extension_pointer(cp) );
				fileinfo_target->directory_path   = g_string_new( shortcut->details.file.directory);
                fileinfo_target->is_directory     = FALSE;
                fileinfo_target->is_template      = FALSE;
                dir_target = g_string_new(shortcut->details.file.directory);
                ret = activate_item( dir_target, fileinfo_target );
                break;

            case SHORTCUT_TO_FOLDER:
                cp = shortcut->details.folder.directory;
                fileinfo_target->filename         = g_string_new( cp );
                fileinfo_target->filename_display = g_string_new( "" );
                fileinfo_target->filetype         = g_string_new( "" );
                fileinfo_target->is_directory     = TRUE;
                fileinfo_target->is_template      = FALSE;
                dir_target = g_string_new("");
                ret = activate_item( dir_target, fileinfo_target );
                break;

            case SHORTCUT_TO_APPLICATION:
                cmd_line = g_string_new(shortcut->details.application.command_line);
                ret = activate_shortcut_to_application( directory, fileinfo, cmd_line );
                break;

            default:
                // not implemented
                ret = ER_FAIL;
        }
    }

    // clean up
    if (cmd_line  ) { g_string_free(cmd_line  , TRUE); }
    if (dir_target) { g_string_free(dir_target, TRUE); }
    filelist_entry_free(fileinfo_target);
    shortcut_free(shortcut);
    g_free(error_msg);

    return ret;
}


// rename target folder
int shortcut_propagate_name_to_folder ( shortcut_t *shortcut )
{
    int         ret = ER_OK;    // return value
    int         rc;
    gboolean    ok;
    gchar       *cp;
    gchar       *buf = NULL;
    gsize       buf_len;
    gchar       *shortcut_dir    = NULL;
    gchar       *shortcut_name   = NULL;
    gchar       *target_name     = NULL;
    const gchar *target_dir      = NULL;
    gchar       *target_realpath = NULL;
    gchar       *target_path     = NULL;
    GError      *err             = NULL;

    LOGPRINTF("entry");
    g_assert(shortcut && shortcut->type == SHORTCUT_TO_FOLDER);
    g_assert(shortcut->details.folder.path);
    LOGPRINTF( "path [%s] directory [%s] file [%s]",
               shortcut->details.folder.path,
               shortcut->details.folder.directory,
               shortcut->details.folder.filename   );

    // function void if propagate-name not requested
    //            or if nested shortcut
    if (   shortcut->details.folder.do_propagate_name == FALSE
        || shortcut->details.folder.filename          != NULL  )
    {
        return ER_OK;
    }

    // check whether target must be renamed
    target_dir = shortcut->details.folder.directory;
    if (target_dir)
    {
        target_name = g_path_get_basename( target_dir );
    }
    else
    {
        target_name = g_path_get_basename( shortcut->details.folder.path );
    }
    shortcut_name = path_fix_invalid_characters( shortcut->name );

    if ( strcmp(target_name, shortcut_name) != 0 )
    {
        // target name differs from shortcut name: must rename target
        LOGPRINTF("rename target [%s] to [%s]", target_name, shortcut_name);

        if (target_dir)
        {
            // old target exists: rename it
            cp = g_path_get_dirname( target_dir );
            target_realpath = g_strdup_printf( "%s/%s", cp, shortcut_name );
            g_free(cp);

            if ( g_file_test(target_realpath, G_FILE_TEST_EXISTS) )
            {
                // new target exists
                if ( g_file_test(target_realpath, G_FILE_TEST_IS_DIR) )
                {
                    // new target exists as directory: ok
                    WARNPRINTF("new target dir exists [%s]", target_realpath);
                }
                else
                {
                    // new target exists but is not a directory: cannot rename
                    WARNPRINTF("new target exists but not a directory [%s]", target_realpath);
                    ret = ER_FAIL;
                }
            }
            else
            {
                // new target not present: rename old target
                LOGPRINTF("renaming [%s] to [%s]", shortcut->details.folder.directory, target_realpath);
                rc = rename(shortcut->details.folder.directory, target_realpath);
                if (rc != 0)
                {
                    ERRNOPRINTF("cannot rename [%s] to [%s]", shortcut->details.folder.directory, target_realpath);
                    ret = ER_FAIL;
                }
                else
                {
                    LOGPRINTF("folder renamed [%s] -> [%s]", shortcut->details.folder.directory, target_realpath);
                }
            }
        }

        // modify shortcut file for renamed target
        if (ret == ER_OK)
        {
            cp = g_path_get_dirname( shortcut->details.folder.path );
            g_free(target_path);
            target_path = g_strdup_printf( "%s/%s", cp, shortcut_name );
            g_free(cp);
            g_key_file_set_string( shortcut->keyfile,
                                   G_KEY_FILE_DESKTOP_GROUP,
                                   G_KEY_FILE_DESKTOP_KEY_PATH,
                                   target_path );
            buf = g_key_file_to_data( shortcut->keyfile, &buf_len, &err );
            if (buf == NULL)
            {
                ERRORPRINTF("cannot get data from keyfile [%p] - [%s]", shortcut->keyfile, err->message);
                g_clear_error(&err);
                ret= ER_FAIL;
            }
            else
            {
                // write shortcut file
                ok = g_file_set_contents( shortcut->filepath, buf, buf_len, &err );
                if ( !ok )
                {
                    ERRORPRINTF("cannot save shortcut [%s] - [%s]", shortcut->filepath, err->message);
                    g_clear_error(&err);
                    ret = ER_WRITE_ERROR;
                }
                else
                {
                    // shortcut file updated: update in-memory data
                    LOGPRINTF("shortcut file updated [%s] path [%s] [%s]", shortcut->filepath, target_path, target_realpath);
                    //   update path
                    g_free(shortcut->details.folder.path);
                    shortcut->details.folder.path = target_path;
                    target_path = NULL;
                    //   update target directory
                    if (target_dir)
                    {
                        g_free(shortcut->details.folder.directory);
                        shortcut->details.folder.directory = target_realpath;
                        target_realpath = NULL;
                    }
                    else
                    {
                        // old target directory not present, check if new target directory present
                        shortcut_dir = g_path_get_dirname(shortcut->filepath);
                        ret = parse_folder_path(shortcut_dir, shortcut->details.folder.path, shortcut);
                    }
                }
            }
        }
    }

    // create target if not present
    if (ret == ER_OK)
    {
        ret = shortcut_create_target_folder(shortcut);
    }

    // clean up
    g_free(target_path    );
    g_free(target_realpath);
    g_free(target_name    );
    g_free(shortcut_name  );
    g_free(shortcut_dir   );

    LOGPRINTF("leave: ret [%d]", ret);
    return ret;
}


// create target folder
int shortcut_create_target_folder ( shortcut_t *shortcut )
{
    int         ret = ER_OK;    // return value
    int         rc;
    gboolean    ok;
    gchar       *cp;
    gchar       *shortcut_dir = NULL;
    const gchar *target_dir   = NULL;
    gchar       *target_path  = NULL;

    LOGPRINTF("entry");
    g_assert(shortcut && shortcut->type == SHORTCUT_TO_FOLDER);
    g_assert(shortcut->details.folder.path);
    LOGPRINTF( "path [%s] directory [%s] file [%s]",
               shortcut->details.folder.path,
               shortcut->details.folder.directory,
               shortcut->details.folder.filename   );

    // function void if nested shortcut
    if ( shortcut->details.folder.filename != NULL )
    {
        return ER_OK;
    }

    // create target if not present
    if ( shortcut->details.folder.directory == NULL )
    {
        // determine path for target folder
        target_dir = shortcut->details.folder.path;
        if ( target_dir[0] == '/' )
        {
            target_path = g_strdup(target_dir);
        }
        else
        {
            cp = g_path_get_dirname(shortcut->filepath);
            target_path = g_strdup_printf("%s/%s", cp, target_dir);
            g_free(cp);
        }

        // create target folder
        rc = mkdir(target_path, 0755);
        if (rc != 0)
        {
            ERRNOPRINTF("cannot create directory [%s]", target_path);
        }
        else
        {
            WARNPRINTF("directory created [%s]", target_path);

            // undo if target location not allowed
            ok = fileview_is_directory_allowed(target_path);
            if ( ok )
            {
                // target's parent directory must be allowed too
                cp = g_path_get_dirname(target_path);
                ok = fileview_is_directory_allowed(cp);
                g_free(cp);
            }
            if ( ok )
            {
                shortcut_dir = g_path_get_dirname(shortcut->filepath);
                ret = parse_folder_path(shortcut_dir, shortcut->details.folder.path, shortcut);
            }
            else
            {
                ERRORPRINTF("shortcut [%s] target_dir not allowed [%s]", shortcut->filepath, target_path);
                (void) rmdir(target_path);
            }
        }
    }

    // clean up
    g_free(target_path );
    g_free(shortcut_dir);

    LOGPRINTF("leave: ret [%d]", ret);
    return ret;
}


// create a shortcut file
int create_shortcut_file ( const gchar   *target_dir,
                           const gchar   *target_file,
                           const gchar   *target_display,
                           const gchar   *shortcut_dir,
                                 GString *shortcut_file  )
{
    int             ret = ER_OK;     // return code
    int             sequence = 1;    // sequence number for shortcut file
    int             rc;
    gboolean        done;
    shortcut_type_t shortcut_type    = SHORTCUT_EMPTY;
    gchar           *cp1;
    gchar           *cp2;
    gchar           *target_path     = NULL;
    gchar           *temp_uri        = NULL;
    const gchar     *shortcut_ext    = "";
    GString         *file            = g_string_new("");;
    GString         *shortcut_path   = g_string_new("");;
    GString         *shortcut_target = g_string_new("");
    FILE            *fp              = NULL;

    LOGPRINTF("entry: target [%s] [%s] shortcut_dir [%s]", target_dir, target_file, shortcut_dir);
    g_assert( target_dir            && *target_dir   == '/'    );
    g_assert( target_file           && *target_file            );
    g_assert( target_display        && *target_display         );
    g_assert( shortcut_dir          && *shortcut_dir == '/'    );
    g_assert( shortcut_file == NULL || shortcut_file->len == 0 );

    // check target/shortcut directories allowed
    if (   fileview_is_directory_allowed(target_dir  ) == FALSE
        || fileview_is_directory_allowed(shortcut_dir) == FALSE )
    {
        ret = ER_FORBIDDEN;
    }

    // determine shortcut type
    if (ret == ER_OK)
    {
        target_path = g_strdup_printf("%s/%s", target_dir, target_file);
        if ( g_file_test(target_path, G_FILE_TEST_IS_REGULAR) )
        {
            shortcut_type = SHORTCUT_TO_FILE;
            shortcut_ext  = FILE_EXT_SHORTCUT;
        }
        else if ( g_file_test(target_path, G_FILE_TEST_IS_DIR) )
        {
            shortcut_type = SHORTCUT_TO_FOLDER;
            shortcut_ext  = FILE_EXT_SHORTCUT_TO_DIR;
        }
        else
        {
            ret = ER_NOT_FOUND;
        }
    }

    // determine shortcut filename
    if (ret == ER_OK)
    {
        done = FALSE;
        while ( !done )
        {
            g_string_printf( file,
                             "%s_%03d.%s",
                              target_file,
                                 sequence,
                                      shortcut_ext );
            g_string_printf( shortcut_path,
                             "%s/%s",
                              shortcut_dir,
                                 file->str );

            if ( g_file_test(shortcut_path->str, G_FILE_TEST_EXISTS) )
            {
                WARNPRINTF("filename [%s] already present", shortcut_path->str);
                sequence++;
            }
            else
            {
                LOGPRINTF("shortcut_path [%s]", shortcut_path->str);
                done = TRUE;
            }
        }
    }

    // determine relative path shortcut -> target
    if (ret == ER_OK)
    {
        // skip identical pre-fix
        cp1 = target_path;
        cp2 = shortcut_path->str;
        while ( *cp1 != '\0'  &&  *cp1 == *cp2 )
        {
            cp1++;
            cp2++;
        }
        while ( *(cp1 - 1) != '/' )
        {
            cp1--;
            cp2--;
        }
        // count parent directories to traverse
        done = FALSE;
        while ( !done )
        {
            while ( *cp2 == '/' )
            {
                cp2++;
            }
            cp2 = strchr(cp2, '/');
            if (cp2)
            {
                g_string_append(shortcut_target, "../");
            }
            else
            {
                done = TRUE;
            }
        }
        // append remaining part of shortcut target
        g_string_append(shortcut_target, cp1);
    }

    // create shortcut file
    if (ret == ER_OK)
    {
        fp = fopen(shortcut_path->str, "w");
        if (fp == NULL)
        {
            ERRNOPRINTF("cannot open file [%s]", shortcut_path->str);
            ret = ER_OPEN_ERROR;
        }
        else
        {
            // write shortcut details
            switch (shortcut_type)
            {
                case SHORTCUT_TO_FILE:
                    temp_uri = g_uri_escape_string(shortcut_target->str, G_URI_RESERVED_CHARS_ALLOWED_IN_PATH, TRUE);
                    rc = fprintf( fp,
                                  FILE_SHORTCUT,
                                  temp_uri,                 // URL=
                                  target_display        );  // Name=
                    break;

                case SHORTCUT_TO_FOLDER:
                    rc = fprintf( fp,
                                  FOLDER_SHORTCUT,
                                  shortcut_target->str,     // Path=
                                  target_display        );  // Name=
                    break;

                default:
                    g_assert_not_reached();
            }
            if (rc <= 0)
            {
                ERRNOPRINTF("cannot write file [%s]", shortcut_path->str);
                ret = ER_WRITE_ERROR;
            }

            // close file
            rc = fclose(fp);
            if (rc != 0)
            {
                ERRNOPRINTF("cannot close file [%s]", shortcut_path->str);
                ret = ER_WRITE_ERROR;
            }
        }
    }

    // set output parameters
    if (ret == ER_OK)
    {
        if (shortcut_file)
        {
            g_string_assign(shortcut_file, file->str);
        }
    }

    // clean up
    if (shortcut_target) { g_string_free(shortcut_target, TRUE); }
    if (shortcut_path  ) { g_string_free(shortcut_path,   TRUE); }
    if (file           ) { g_string_free(file,            TRUE); }
    g_free(target_path);
    g_free(temp_uri   );

    return ret;
}


// get shortcut details
int parse_shortcut_file ( const gchar      *directory,
                          const gchar      *filename,
                                shortcut_t **p_shortcut )
{
    g_assert(directory  && *directory );
    g_assert(filename   && *filename  );
    g_assert(p_shortcut && *p_shortcut == NULL);

    int         ret        = ER_OK;     // return code
    shortcut_t  *shortcut  = NULL;      // output parameter
    shortcut_t  *shortcut_nested = NULL;
    gboolean    b;
    gboolean    ok;
    const gchar *ext;
    gchar       *cp;
    gchar       *value    = NULL;
    gchar       *filepath = NULL;
    const gchar *file_ext = NULL;
    GError      *err      = NULL;
    GKeyFile    *keyfile  = NULL;
    const gchar *dir      = NULL;
    const gchar *file     = NULL;

    static int  recurse_level = 0;      // counts recursive calling levels

    LOGPRINTF("entry: level [%d] dir [%s] file [%s]", recurse_level, directory, filename);

    if (recurse_level >= MAX_SHORTCUT_NESTING)
    {
        ERRORPRINTF("recurse level [%d] reached max [%d]", recurse_level, MAX_SHORTCUT_NESTING);
        return ER_FAIL;
    }
    recurse_level++;

    filepath = g_strdup_printf("%s/%s", directory, filename);
    file_ext = g_extension_pointer(filepath);

    // check shortcut file exists
    ok = g_file_test(filepath, G_FILE_TEST_IS_REGULAR);
    if ( !ok )
    {
        ERRORPRINTF("shortcut file not present [%s]", filepath);
        ret = ER_NOT_FOUND;
    }

    //   allocate shortcut structure
    if (ret == ER_OK)
    {
        shortcut = shortcut_new();
    }

    // open shortcut file
    if (ret == ER_OK)
    {
        keyfile = g_key_file_new();
        g_assert(keyfile);
        ok = g_key_file_load_from_file( keyfile,
                                        filepath,
                                        G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS,
                                        &err );
        if ( !ok )
        {
            ERRORPRINTF("cannot load keyfile [%s] - [%s]", filepath, err ? err->message : "" );
            g_clear_error(&err);
            ret = ER_INVALID_DATA;
        }
        else
        {
            shortcut->keyfile = keyfile;
        }
    }

    // get generic keys
    //   get name
    if (ret == ER_OK)
    {
        value = g_key_file_get_locale_string( keyfile,
                                              G_KEY_FILE_DESKTOP_GROUP,
                                              G_KEY_FILE_DESKTOP_KEY_NAME,
                                              NULL,
                                              &err );
        if (value == NULL)
        {
            // name not found: error
            ERRORPRINTF("file [%s] no 'Name' key - [%s]", filepath, err->message);
            g_clear_error( &err );
            ret = ER_INVALID_DATA;
        }
        else
        {
            // name found: store in shortcut
            LOGPRINTF("  Name [%s]", value);
            shortcut->name = value;
            value = NULL;
        }
    }
    //   get version
    if (ret == ER_OK)
    {
        value = g_key_file_get_string( keyfile,
                                       G_KEY_FILE_DESKTOP_GROUP,
                                       G_KEY_FILE_DESKTOP_KEY_VERSION,
                                       &err );
        if (value == NULL)
        {
            // version not present: ok
            LOGPRINTF("file [%s] no 'Version' key - [%s]", filepath, err->message);
            g_clear_error( &err );
        }
        else
        {
            // version present: check its value
            LOGPRINTF("  Version [%s]", value);
            if ( strcmp(value, "1.0") != 0 )
            {
                ERRORPRINTF("file [%s] illegal version [%s]", filepath, value);
                ret = ER_INVALID_DATA;
            }

            // clean up
            g_free(value);
            value = NULL;
        }
    }
    //   get type
    if (ret == ER_OK)
    {
        value = g_key_file_get_string( keyfile,
                                       G_KEY_FILE_DESKTOP_GROUP,
                                       G_KEY_FILE_DESKTOP_KEY_TYPE,
                                       &err );
        if (value == NULL)
        {
            // type not found: error
            ERRORPRINTF("file [%s] no 'Type' key - [%s]", filepath, err->message);
            g_clear_error( &err );
            ret = ER_INVALID_DATA;
        }
        else
        {
            // type found: store in shortcut
            ext = "";
            LOGPRINTF("  Type [%s]", value);
            if ( strcmp(value, G_KEY_FILE_DESKTOP_TYPE_LINK) == 0 )
            {
                shortcut->type = SHORTCUT_TO_FILE;
                ext            = FILE_EXT_SHORTCUT;
            }
            else if ( strcmp(value, G_KEY_FILE_DESKTOP_TYPE_DIRECTORY) == 0 )
            {
                shortcut->type = SHORTCUT_TO_FOLDER;
                ext            = FILE_EXT_SHORTCUT_TO_DIR;
            }
            else if ( strcmp(value, G_KEY_FILE_DESKTOP_TYPE_APPLICATION) == 0 )
            {
                shortcut->type = SHORTCUT_TO_APPLICATION;
                ext            = FILE_EXT_SHORTCUT;
            }
            else
            {
                ERRORPRINTF("file [%s] invalid 'Type' [%s]", filepath, value);
                ret = ER_INVALID_DATA;
            }

            // check file extension matches with shortcut type
            if (ret == ER_OK)
            {
                if ( strcmp(ext, g_extension_pointer(filename)) != 0 )
                {
                    ERRORPRINTF("file [%s] 'Type' [%s] invalid for this file extension", filepath, value);
                    ret = ER_INVALID_DATA;
                }
            }

            // clean up
            g_free(value);
            value = NULL;
        }
    }

    // get type-dependent keys
    if (ret == ER_OK)
    {
        switch (shortcut->type)
        {
            case SHORTCUT_TO_FILE:
                // get url
                value = g_key_file_get_string( keyfile,
                                               G_KEY_FILE_DESKTOP_GROUP,
                                               G_KEY_FILE_DESKTOP_KEY_URL,
                                               &err );
                if (value == NULL)
                {
                    ERRORPRINTF("file [%s] no 'URL' key - [%s]", filepath, err->message);
                    g_clear_error( &err );
                    ret = ER_INVALID_DATA;
                }
                else
                {
                    // url found: convert to path, store in shortcut
                    LOGPRINTF("  URL [%s]", value);

                    // get scheme from url
                    // note: scheme might indicate url points to a web location
                    if ( g_str_has_prefix(value, "http:") )
                    {
                        // web location: remember details and leave
                        shortcut->type = SHORTCUT_TO_WEB_LOCATION;
                        shortcut->details.web.url = value;
                        value = NULL;
                        ret = ER_INVALID_DATA;  //TODO: implement web location
                        break;  // exit switch
                    }
                    else if ( g_str_has_prefix(value, "file:") )
                    {
                        // local file location: get real path
                        ret = parse_file_url( directory, value, shortcut );
                    }
                    else
                    {
                        // scheme not implemented
                        ERRORPRINTF("file [%s] invalid url scheme [%s]", filepath, value);
                        ret = ER_INVALID_DATA;
                    }

                    // clean up
                    g_free(value);
                    value = NULL;
                }

                // handle nested shortcuts
                if (ret == ER_OK)
                {
                    ext = g_extension_pointer(shortcut->details.file.filename);
                    if ( is_shortcut_file_extension(ext) )
                    {
                        dir  = shortcut->details.file.directory;
                        file = shortcut->details.file.filename;
                        ret  = parse_shortcut_file( dir, file, &shortcut_nested );
                        if (   (   ret == ER_OK
                                || ret == ER_NOT_FOUND )
                            && shortcut_nested
                            && shortcut_nested->type == SHORTCUT_TO_FILE )
                        {
                            // nested shortcut to file: copy relevant details from target
                            g_free(shortcut->details.file.directory);
                            g_free(shortcut->details.file.filename );
                            shortcut->details.file.directory = shortcut_nested->details.file.directory;
                            shortcut->details.file.filename  = shortcut_nested->details.file.filename;
                            shortcut_nested->details.file.directory = NULL;
                            shortcut_nested->details.file.filename  = NULL;
                        }
                        else
                        {
                            // invalid shortcut nesting
                            ERRORPRINTF( "invalid shortcut nesting [%s][%s] --> [%s][%s]",
                                         directory, filename, dir, file );
                        }
                    }
                }
                LOGPRINTF( "FILE dir [%s] file [%s]",
                           shortcut->details.file.directory,
                           shortcut->details.file.filename  );
                break;

            case SHORTCUT_TO_FOLDER:
                // get path
                value = g_key_file_get_string( keyfile,
                                               G_KEY_FILE_DESKTOP_GROUP,
                                               G_KEY_FILE_DESKTOP_KEY_PATH,
                                               &err );
                if (value == NULL)
                {
                    ERRORPRINTF("file [%s] no 'Path' key - [%s]", filepath, err->message);
                    g_clear_error( &err );
                    ret = ER_INVALID_DATA;
                }
                else
                {
                    // key found: check on invalid characters
                    if ( path_has_invalid_characters(value) )
                    {
                        ERRORPRINTF("file [%s] Path= [%s] has invalid character(s)", filepath, value);
                        ret = ER_INVALID_DATA;
                    }
                    else
                    {
                        // path found: convert to realpath, store in shortcut
                        LOGPRINTF("  Path [%s]", value);
                        ret = parse_folder_path( directory, value, shortcut );
                        shortcut->details.folder.path = value;
                        value = NULL;
                    }
                }

                // get do-propagate-name (optional)
                if (ret == ER_OK)
                {
                    b = g_key_file_get_boolean( keyfile,
                                                G_KEY_FILE_DESKTOP_GROUP,
                                                KEY_FILE_DESKTOP_KEY_IREX_PROPAGATE_NAME,
                                                &err );
                    if (err)
                    {
                        // key not found or key value not boolean
                        if (err->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND)
                        {
                            // key not found: ok
                            LOGPRINTF("file [%s] no 'DoPropagate' key", filepath);
                        }
                        else
                        {
                            // key value not boolean
                            ERRORPRINTF("file [%s] invalid 'DoPropagate' - [%s]", filepath, err->message);
                            ret = ER_INVALID_DATA;
                        }
                        g_clear_error(&err);
                    }
                    else
                    {
                        // key found and valid: store in shortcut
                        LOGPRINTF("  DoPropagate [%d]", b);
                        shortcut->details.folder.do_propagate_name = b;
                    }
                }

                // handle nested shortcuts
                if (   ret == ER_OK
                    && shortcut->details.folder.filename )
                {
                    ext = g_extension_pointer(shortcut->details.folder.filename);
                    if ( is_shortcut_file_extension(ext) )
                    {
                        dir  = shortcut->details.folder.directory;
                        file = shortcut->details.folder.filename;
                        ret  = parse_shortcut_file( dir, file, &shortcut_nested );
                        if (   (   ret == ER_OK
                                || ret == ER_NOT_FOUND )
                            && shortcut_nested
                            && shortcut_nested->type == SHORTCUT_TO_FOLDER )
                        {
                            // nested shortcut to folder: copy relevant details from target
                            g_free(shortcut->details.folder.directory);
                            g_free(shortcut->details.folder.filename );
                            shortcut->details.folder.directory = shortcut_nested->details.folder.directory;
                            shortcut->details.folder.filename  = shortcut_nested->details.folder.filename;
                            shortcut_nested->details.folder.directory = NULL;
                            shortcut_nested->details.folder.filename  = NULL;
                        }
                        else
                        {
                            // invalid shortcut nesting
                            ERRORPRINTF( "invalid shortcut nesting [%s][%s] --> [%s][%s]",
                                         directory, filename, dir, file );
                        }
                    }
                }
                LOGPRINTF( "FOLDER dir [%s] file [%s]",
                           shortcut->details.folder.directory,
                           shortcut->details.folder.filename  );
                break;

            // note: no nesting of shortcut to application
            case SHORTCUT_TO_APPLICATION:
                // get exec
                value = g_key_file_get_string( keyfile,
                                               G_KEY_FILE_DESKTOP_GROUP,
                                               G_KEY_FILE_DESKTOP_KEY_EXEC,
                                               &err );
                if (value == NULL)
                {
                    ERRORPRINTF("file [%s] no 'Exec' key - [%s]", filepath, err->message);
                    g_clear_error( &err );
                    ret = ER_INVALID_DATA;
                }
                else
                {
                    // key found: store in shortcut
                    LOGPRINTF("  Exec [%s]", value);
                    if (   strncmp(value, "./",  2) == 0
                        || strncmp(value, "../", 3) == 0 )
                    {
                        // relative path: prepend shortcut directory
                        // shell quote to preserve spaces in directory
                        gchar *directory_quoted = g_shell_quote(directory);
                        cp = g_strdup_printf("%s/%s", directory_quoted, value);
                        g_free(directory_quoted);
                        g_free(value);
                        value = cp;
                        LOGPRINTF("  Exec [%s]", value);
                    }
                    shortcut->details.application.command_line = value;
                    value = NULL;
                }

                // get path (optional)
                if (ret == ER_OK)
                {
                    value = g_key_file_get_string( keyfile,
                                                   G_KEY_FILE_DESKTOP_GROUP,
                                                   G_KEY_FILE_DESKTOP_KEY_PATH,
                                                   &err );
                    if (value == NULL)
                    {
                        // no 'Path' key: ok
                        LOGPRINTF("file [%s] no 'Path' key", filepath);
                        g_clear_error( &err );
                    }
                    else
                    { 
                        // key found: check on invalid characters
                        if ( path_has_invalid_characters(value) )
                        {
                            ERRORPRINTF("file [%s] Path= [%s] has invalid characters", filepath, value);
                            ret = ER_INVALID_DATA;
                        }
                        else
                        {
                            // path ok: store in shortcut
                            LOGPRINTF("  Path [%s]", value);
                            shortcut->details.application.work_dir = value;
                            value = NULL;
                        }
                    }
                }
                break;

            default:
                // cannot happen with ret == ER_OK
                g_assert_not_reached();
        }
    }

    // report whatever we found, even when shortcut file corrupt
    *p_shortcut = shortcut;
    if (shortcut)
    {
        shortcut->filepath = filepath;
        filepath = NULL;
    }

    // clean up
    g_free(filepath);
    g_free(value);
    shortcut_free(shortcut_nested);

    recurse_level--;
    LOGPRINTF("leave: ret [%d] level [%d]", ret, recurse_level);
    return ret;
}


// parse URL with file: scheme
//   in:     directory - directory in which shortcut file is located
//           url       - URL of target file, syntax:
//                         scheme:[//authority]path[?query][#fragment]
//   out:    shortcut->details.file.filename
//           shortcut->details.file.directory
//   return: ER_OK           - ok
//           ER_NOT_FOUND    - target location not found
//           ER_INVALID_DATA - invalid url
//           ER_FORBIDDEN    - target directory not allowed
static int parse_file_url ( const gchar      *directory,
                            const gchar      *url,
                                  shortcut_t *shortcut )
{
    int         ret = ER_OK;            // return code
    gboolean    ok;
    char        *cp;
    char        path[PATH_MAX];
    char        *url_path   = NULL;     // path component from url
    char        *fs_path    = NULL;     // path on filesystem
    char        *target_dir = NULL;

    LOGPRINTF("entry: dir [%s] url [%s]", directory, url);
    g_assert(directory && *directory);
    g_assert(url       && *url      );
    g_assert(shortcut  && shortcut->type == SHORTCUT_TO_FILE);

    // strip scheme:
    cp = strchr(url, ':');
    if (cp == NULL)
    {
        url_path = g_strdup(url);
    }
    else if ( g_str_has_prefix(url, "file:") )
    {
        url_path = g_strdup(cp + 1);
    }
    else
    {
        ret = ER_INVALID_DATA;
    }

    // strip //authority
    if (   url_path[0] == '/'
        && url_path[1] == '/' )
    {
        if ( url_path[2] == '/' )
        {
            // authority is empty: ok
            strcpy(url_path, url_path + 2);
        }
        else
        {
            // autohority filled in: not implemented
            ret = ER_INVALID_DATA;
        }
    }

    // strip ?query
    cp = strchr(url_path, '?');
    if (cp)
    {
        *cp = '\0';
    }

    // strip #fragment
    cp = strchr(url_path, '#');
    if (cp)
    {
        *cp = '\0';
    }

    // unescape %<hex><hex> strings, without failing on any character
    cp = g_uri_unescape_string(url_path, NULL);
    if (cp)
    {
        g_free(url_path);
        url_path = cp;
    }

    // check no illegal characters in path
    if ( path_has_invalid_characters(url_path) )
    {
        ERRORPRINTF("invalid character(s) in url_path [%s]", url_path);
        ret = ER_INVALID_DATA;
    }

    // detemine path on filesystem
    if (ret == ER_OK)
    {
        if (url_path[0] == '/')
        {
            fs_path  = url_path;
            url_path = NULL;
        }
        else
        {
            fs_path = g_strdup_printf("%s/%s", directory, url_path);
        }
    }

    // get real path
    if (ret == ER_OK)
    {
        cp = realpath(fs_path, path);
        if (cp != path)
        {
            ERRNOPRINTF("cannot determine realpath of [%s]", fs_path);
            ret = ER_NOT_FOUND;
        }
    }

    // check target is a file
    if (ret == ER_OK)
    {
        ok = g_file_test(path, G_FILE_TEST_IS_REGULAR);
        if ( !ok )
        {
            ERRORPRINTF("not a regular file [%s]", path);
            ret = ER_NOT_FOUND;
        }
    }

    // check if realpath allowed
    if (ret == ER_OK)
    {
        target_dir  = g_path_get_dirname( path );
        ok = fileview_is_directory_allowed( target_dir );
        if ( !ok )
        {
            ret = ER_FORBIDDEN;
        }
    }

    // report our findings
    if (ret == ER_OK)
    {
        shortcut->details.file.filename  = g_path_get_basename( path );
        shortcut->details.file.directory = target_dir;
        target_dir = NULL;
        LOGPRINTF("file [%s] dir [%s]", shortcut->details.file.filename, shortcut->details.file.directory);
    }

    // clean up
    g_free(target_dir);
    g_free(fs_path   );
    g_free(url_path  );

    return ret;
}


// parse path for directory shortcut
//   in:     directory   - directory in which shortcut file is located
//           target_path - path of target directory/file, absolute or relative
//   out:    shortcut->details.folder.directory   realpath of target directory,
//                                                or NULL when target not present on filesystem
//           shortcut->details.folder.filename    if target_path points to a file
//   return: ER_OK           - ok, also when target not present
//           ER_INVALID_DATA - invalid path
//           ER_FORBIDDEN    - target directory not allowed
static int parse_folder_path ( const gchar      *directory,
                               const gchar      *target_path,
                                     shortcut_t *shortcut    )
{
    int         ret = ER_OK;            // return code
    gboolean    ok;
    gboolean    target_is_file      = FALSE;;
    gboolean    target_is_directory = FALSE;
    char        *cp;
    char        path[PATH_MAX] = "";
    char        *url_path = NULL;       // path component from url
    char        *fs_path  = NULL;       // path on filesystem
    gchar       *dir      = NULL;
    gchar       *file     = NULL;

    LOGPRINTF("entry: dir [%s] path [%s]", directory, target_path);
    g_assert(directory   && *directory  );
    g_assert(target_path && *target_path);
    g_assert(shortcut    && shortcut->type == SHORTCUT_TO_FOLDER);
    g_assert(shortcut->details.folder.directory == NULL);
    g_assert(shortcut->details.folder.filename  == NULL);

    // check no illegal characters in path
    if ( path_has_invalid_characters(target_path) )
    {
        ERRORPRINTF("invalid character in target_path [%s]", target_path);
        ret = ER_INVALID_DATA;
    }

    // detemine path on filesystem
    if (ret == ER_OK)
    {
        if (target_path[0] == '/')
        {
            fs_path  = g_strdup(target_path);
        }
        else
        {
            fs_path = g_strdup_printf("%s/%s", directory, target_path);
        }
        LOGPRINTF("target fs_path [%s]", fs_path);
    }

    // get real path, if target present
    if (ret == ER_OK)
    {
        cp = realpath(fs_path, path);
        if (cp == path)
        {
            LOGPRINTF("target realpath [%s]", path);

            // get directory name (and filename if applicable)
            target_is_directory = g_file_test(path, G_FILE_TEST_IS_DIR);
            if (target_is_directory)
            {
                dir = g_strdup(path);
            }
            else
            {
                target_is_file = g_file_test(path, G_FILE_TEST_IS_REGULAR);
                if (target_is_file)
                {
                    dir  = g_path_get_dirname(path);
                    file = g_path_get_basename(path);
                }
            }

            // check if realpath allowed
            if ( dir )
            {
                ok = fileview_is_directory_allowed( dir );
                if ( !ok )
                {
                    ret = ER_FORBIDDEN;
                }
            }
        }
    }

    // report our findings
    if (ret == ER_OK)
    {
        LOGPRINTF("dir [%s] file [%s]", dir, file);
        shortcut->details.folder.directory = dir;
        shortcut->details.folder.filename  = file;
        dir  = NULL;
        file = NULL;
    }

    // clean up
    g_free( file     );
    g_free( dir      );
    g_free( fs_path  );
    g_free( url_path );

    return ret;
}


// check for invalid charactes in path
static gboolean path_has_invalid_characters ( const gchar *path )
{
    gboolean    found = FALSE;  // return value
    const gchar *illegal;
    gchar       *cp;

    LOGPRINTF("entry: path [%s]", path);
    g_assert(path && *path);

    for ( illegal = ILLEGAL_FILEPATH_CHARS ; *illegal ; illegal++ )
    {
        cp = strchr(path, *illegal);
        if (cp)
        {
            found = TRUE;
        }
    }

    LOGPRINTF("leave: found [%d]", found);
    return found;
}


// fix invalid charactes in path
static gchar* path_fix_invalid_characters ( const gchar *path )
{
    gchar       *new_path = NULL;     // return value
    const gchar *illegal;
    gchar       *cp;

    LOGPRINTF("entry: path [%s]", path);
    g_assert(path && *path);

    new_path = g_strdup(path);
    for ( illegal = ILLEGAL_FILEPATH_CHARS ; *illegal ; illegal++ )
    {
        cp = strchr(new_path, *illegal);
        if (cp)
        {
            *cp = '-';
        }
    }

    LOGPRINTF("leave: fixed_path [%s]", new_path);
    return new_path;
}


//----------------------------------------------------------------------------
// Functions for shortcut_t data type
//----------------------------------------------------------------------------

// allocate shortcut description
static shortcut_t* shortcut_new ( void )
{
    shortcut_t  *thiz;

    LOGPRINTF("entry");

    thiz = g_new0( shortcut_t, 1 );
    g_assert(thiz);

    thiz->type = SHORTCUT_EMPTY;

    return thiz;
}


// free shortcut description
void shortcut_free_impl ( shortcut_t* thiz )
{
    LOGPRINTF("entry: thiz [%p]", thiz );

    // function void on NULL pointer
    if (thiz == NULL)
    {
        return;
    }

    // free shortcut elements
    g_free(thiz->filepath);
    if (thiz->keyfile)
    {
        g_key_file_free(thiz->keyfile);
    }
    g_free(thiz->name);
    switch (thiz->type)
    {
        case SHORTCUT_EMPTY:
            ; //ignore
            break;

        case SHORTCUT_TO_FILE:
            g_free(thiz->details.file.directory);
            g_free(thiz->details.file.filename );
            break;

        case SHORTCUT_TO_WEB_LOCATION:
            g_free(thiz->details.web.url);
            break;

        case SHORTCUT_TO_FOLDER:
            g_free(thiz->details.folder.path);
            g_free(thiz->details.folder.directory);
            g_free(thiz->details.folder.filename );
            break;

        case SHORTCUT_TO_APPLICATION:
            g_free(thiz->details.application.command_line);
            break;

        default:
            ERRORPRINTF("illegal shortcut type [%d]", thiz->type);
            ; //ignore
    }


    // free shortcut struct
    g_free(thiz);
}
