#ifndef __STORAGE_H__
#define __STORAGE_H__

/**
 * File Name  : storage.h
 *
 * Description: Content browser functions handling storage media (e.g. SD card)
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <glib.h>

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================


/**---------------------------------------------------------------------------
 *
 * Name :  storage_init
 *
 * @brief  Initialise administration of storage media
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_init ( void );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_set_text
 *
 * @brief  Update screen texts, usually after language change
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_set_text ( void );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_quit
 *
 * @brief  Prepare for power down
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_quit ( void );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_get_media
 *
 * @brief  Get a list of currently mounted storage media
 *
 * @param  --
 *
 * @return GPtrArray holding gchar* do mount points of storage media
 *
 *--------------------------------------------------------------------------*/
const GPtrArray* storage_get_media ( void );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_get_shortcut_filename
 *
 * @brief  Get filename of desktop shortcut for specified storage media
 *
 * @param  [in]  mountpoint - location where this media has been mounted
 *
 * @return filename of the desktop shortcut for the specified media
 *         Caller must release this string using g_string_free();
 *
 *--------------------------------------------------------------------------*/
GString *storage_get_shortcut_filename( const gchar *mountpoint );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_get_desktop_dir
 *
 * @brief  Get path of the iRex desktop directory on the specified device
 *
 * @param  [in]  location - a valid path on the filesystem
 *
 * @return absolute path of the iRex desktop directory for the specified path
 *         or NULL when path not present or not allowed
 *         Caller must release this string using g_string_free();
 *
 *--------------------------------------------------------------------------*/
GString* storage_get_desktop_dir ( const gchar *location );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_get_recentfiles_dir
 *
 * @brief  Get path of the recent files directory on the specified device
 *
 * @param  [in]  location - a valid path on the filesystem
 *
 * @return absolute path of the recent files directory for the specified path
 *         or NULL when path not present or not allowed
 *         Caller must release this string using g_string_free();
 *
 *--------------------------------------------------------------------------*/
GString* storage_get_recentfiles_dir ( const gchar *location );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_media_mounted
 *
 * @brief  Report that a storage media has been mounted
 *
 * @param  [in]  mountpoint - location where this media has been mounted
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_media_mounted ( const gchar *mountpoint );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_media_prepare_unmount
 *
 * @brief  Report that a storage media will be unmounted
 *
 * @param  [in]  mountpoint - location where this media was mounted
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_media_prepare_unmount ( const gchar *mountpoint );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_media_unmounted
 *
 * @brief  Report that a storage media has been unmounted
 *
 * @param  [in]  mountpoint - location where this media was mounted
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_media_unmounted ( const gchar *mountpoint );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_media_prepare
 *
 * @brief  Prepare storage media for iOn use
 *
 * @param  [in]  mountpoint - location where this media is mounted
 *
 * @return ER_OK or error
 *
 *--------------------------------------------------------------------------*/
int storage_media_prepare ( const gchar *mountpoint );


/**---------------------------------------------------------------------------
 *
 * Name :  storage_refresh_symlinks
 *
 * @brief  Refresh symlinks for desktop shortcuts
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void storage_refresh_symlinks ( void );



G_END_DECLS

#endif // __STORAGE_H__
