#ifndef __SHORTCUT_H__
#define __SHORTCUT_H__

/**
 * File Name  : shortcut.h
 *
 * Description: Content browser functions handling shortcut files (xx.desktop)
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <glib.h>

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 

#define KEY_FILE_DESKTOP_KEY_IREX_PROPAGATE_NAME        "X-iRex-DoPropagateName"

#define MAX_SHORTCUT_NESTING            5       // maximum nesting level for shortcuts,
                                                // i.e. maximum number of shortcuts in a chain


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// types of shortcuts
typedef enum
        {
            SHORTCUT_EMPTY = 0,
            SHORTCUT_TO_FILE,
            SHORTCUT_TO_WEB_LOCATION,
            SHORTCUT_TO_FOLDER,
            SHORTCUT_TO_APPLICATION,
            NUM_SHORTCUTS
        }                       shortcut_type_t;

// details from shortcut file
typedef struct
        {
            // raw key-value data
            gchar               *filepath;
            GKeyFile            *keyfile;

            // interpreted fields for shortcut
            shortcut_type_t     type;
            gchar               *name;
            union
            {
                struct
                {
                    gchar       *directory;             // target directory
                    gchar       *filename;              // target filename
                }               file;
                struct
                {
                    gchar       *url;                   // target URL
                }               web;
                struct
                {
                    gchar       *path;                  // 'Path=' value
                    gchar       *directory;             // target directory
                    gchar       *filename;              // target filename (shortcut file)
                    gboolean    do_propagate_name;      // rename target directory to 'Name=' value
                }               folder;
                struct
                {
                    gchar       *command_line;
                    gchar       *work_dir;
                }               application;
            }                   details;
        }                       shortcut_t;

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================

/**---------------------------------------------------------------------------
 *
 * Name :  activate_shortcut
 *
 * @brief  activate: parse shortcut, then launch viewer, goto directory, ...
 *
 * @param  [in]  directory - directory in which the shortcut file (xx.desktop) is located
 * @param  [in]  fileinfo - details of the shortcut file
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int activate_shortcut ( const GString           *directory,
                        const filelist_entry_t  *fileinfo  );


/**---------------------------------------------------------------------------
 *
 * Name :  create_shortcut_file
 *
 * @brief  create a shortcut file
 *
 * @param  [in]  target_dir     - folder in which the target is located
 *                                specified as an absolute path
 * @param  [in]  target_file    - file or folder to which the shortcut must point
 * @param  [in]  target_display - name displayed for target
 * @param  [in]  shortcut_dir   - directory in which the shortcut file (xx.desktop) must be created
 *                                specified as an absolute path
 * @param  [out] shortcut_file  - filename of the shortcut file created in shortcut_dir,
 *                                or ignored when caller passes a NULL pointer here.
 *                                Note: caller must allocate and release this GString.
 *
 * @return ER_OK           - ok
 *         ER_NOT_FOUND    - target not found
 *         ER_FORBIDDEN    - directory or target directory not allowed
 *         or other error code
 *
 *--------------------------------------------------------------------------*/
int create_shortcut_file ( const gchar   *target_dir,
                           const gchar   *target_file,
                           const gchar   *target_display,
                           const gchar   *shortcut_dir,
                                 GString *shortcut_file  );


/**---------------------------------------------------------------------------
 *
 * Name :  parse_shortcut_file
 *
 * @brief  get details from shortcut file
 *         parses shortcut files recursively when applicable
 *
 * @param  [in]  directory  - directory in which the shortcut file (xx.desktop) is located
 * @param  [in]  filename   - filename of the shortcut file
 * @param  [out] p_shortcut - structure with details of the requested shortcut,
 *                            caller must release this with shortcut_free()
 *
 * @return ER_OK           - ok
 *         ER_NOT_FOUND    - target location not found
 *         ER_INVALID_DATA - contents of shortcut file is corrupt
 *         ER_FORBIDDEN    - target directory not allowed
 *         or other error code
 *
 *--------------------------------------------------------------------------*/
int parse_shortcut_file ( const gchar      *directory,
                          const gchar      *filename,
                                shortcut_t **p_shortcut );


/**---------------------------------------------------------------------------
 *
 * Name :  shortcut_propagate_name_to_folder
 *
 * @brief  rename target folder of a shortcut to directory
 *         create target folder when not present
 *
 * @param  [in]  shortcut - details of the shortcut file, generated by parse_shortcut_file()
 *
 * @return ER_OK or error
 *
 *--------------------------------------------------------------------------*/
int shortcut_propagate_name_to_folder ( shortcut_t  *shortcut );


/**---------------------------------------------------------------------------
 *
 * Name :  shortcut_create_target_folder
 *
 * @brief  create target folder of a shortcut to directory, if not present
 *
 * @param  [in]  shortcut - details of the shortcut file, generated by parse_shortcut_file()
 *
 * @return ER_OK or error
 *
 *--------------------------------------------------------------------------*/
int shortcut_create_target_folder ( shortcut_t  *shortcut );


/**---------------------------------------------------------------------------
 *
 * Name :  shortcut_free
 *
 * @brief  free the memory allocated to a shortcut_t structure
 *         and free the memory allocated to elements it contains
 *         and clear the pointer passed as argument
 *
 * @param  [in]  directory - directory in which the shortcut file (xx.desktop) is located
 * @param  [in]  fileinfo - details of the shortcut file
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
#define shortcut_free(thiz)             \
        {                               \
            shortcut_free_impl(thiz);   \
            (thiz) = NULL;              \
        }
void shortcut_free_impl ( shortcut_t* thiz );


G_END_DECLS

#endif // __SHORTCUT_H__
