#ifndef __FILEVIEW_H__
#define __FILEVIEW_H__

/**
 * File Name  : fileview.h
 *
 * Description: Content browser screens to display files
 *              for both portrait and landscape mode
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <gtk/gtk.h>

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// the different views of content browser
// note: corresponds to gtk_notebook page as used in fileview.c
typedef enum
        {
            CTB_ICONVIEW = 0,
            CTB_LISTVIEW,
            CTB_AUTOSELECT,
            N_CTB_VIEWS
        } ctb_viewtypes_t;

// sorting order in content browser views
typedef enum
        {
            CTB_SORT_BY_NAME = 0,
            CTB_SORT_BY_TYPE,
            CTB_SORT_BY_SIZE,
            CTB_SORT_BY_DATE,
            CTB_SORT_BY_AUTHOR,
            N_CTB_SORT_ORDER
        } ctb_sort_order_t;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

// special directory locations
#define DIR_IREX_PRIVATE            "System"                        // irex-private folder on SD card
                                    
#define DIR_DESKTOP                 "Desktop"                       // sub-folder name
#define DIR_DESKTOP_TEMPLATE        SYSCONFDIR "/desktop"           // template desktop, usually on /etc
#define DIR_DESKTOP_TEMPLATE_CARD   SYSCONFDIR "/desktop-card"      // template desktop for sd card
#define DIR_DESKTOP_INTERNAL        "/var/tmp/" DIR_DESKTOP         // desktop on internal filesystem
#define DIR_DESKTOP_TMP             "/var/tmp/" DIR_DESKTOP "/work" // work-directory for desktop
#define DIR_DESKTOP_SYMLINK         "/var/tmp/" DIR_DESKTOP "/shortcuts"
                                                                    // symlinks to target directories of desktop shortcuts
                                    
                                                                    // sub-folder to store pc application in
#define DIR_PC_APP                  DIR_IREX_PRIVATE "/pc_application" 
#define DIR_PC_APP_TEMPLATE         SYSCONFDIR "/pc_application" 

#define DIR_STORAGE_MNT             "/media"                        // mountpoint for removable storage devices
#define DIR_LIBRARY                 DIR_STORAGE_MNT "/mmcblk0p1"    // mountpoint for internal SD card
#define DIR_LIBRARY_DESKTOP         DIR_LIBRARY "/" DIR_IREX_PRIVATE "/" DIR_DESKTOP
                                                                    // desktop directory on internal SD card

#define DIR_RECENTFILES             "Recent"                        // sub-folder name
//#define DIR_LIBRARY_RECENTFILES     DIR_LIBRARY "/" DIR_IREX_PRIVATE "/" DIR_RECENTFILES
// the System folder seems to be write protected, so use folder in root of SD card
#define DIR_LIBRARY_RECENTFILES     DIR_LIBRARY "/" DIR_RECENTFILES
                                                                    // recent files directory on internal SD card

//============================================================================
// Public Functions
//============================================================================


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_init
 *
 * @brief  Initialisation
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_init (void);


GString *fileview_get_current_dir(void);

/**---------------------------------------------------------------------------
 *
 * Name :  fileview_set_text
 *
 * @brief  Update screen texts, using current language setting
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_set_text (void);


 /**---------------------------------------------------------------------------
  *
  * Name :  fileview_set_num_items
  *
  * @brief  Update screen item for number of items: "x - y of z"
  *
  * @param  first - first item displayed (x)
  * @param  last - last item displayed (y)
  * @param  total - total number of items (z)
  *
  * @return --
  *
  *--------------------------------------------------------------------------*/
void fileview_set_num_items (const int first, const int last, const int total);


/**---------------------------------------------------------------------------
 *
 * Name :  create_fileview
 *
 * @brief  Create fileview screens and data
 *
 * @param  none
 *
 * @return fileview widget
 *
 *--------------------------------------------------------------------------*/
GtkWidget* create_fileview (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_set_view_type
 *
 * @brief  Select which view to use for displaying the files
 *
 * @param  [in]  view - requested view type
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_set_view_type (const ctb_viewtypes_t view);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_set_sort_order
 *
 * @brief  Set sorting order for items in view
 *
 * @param  [in]  sort_order - requested sorting order
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_set_sort_order (const ctb_sort_order_t sort_order);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_show_dir
 *
 * @brief  Show content of the specified directory
 *
 * @param  [in]  dir - directory to be displayed
 * @param  [in]  cursor_item - file-/directory-name to position the cursor on,
 *                             or NULL to position cursor on the first item
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_show_dir (const gchar *dir, const gchar *cursor_item);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_show_desktop
 *
 * @brief  Show content of the desktop
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_show_desktop (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_desktop_changed
 *
 * @brief  Notification that desktop content has changed,
 *         so a screen refresh is required when desktop is currently shown
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_desktop_changed (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_media_mounted
 *
 * @brief  Notification that a storage device has been mounted
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_media_mounted (const gchar *mountpoint);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_media_unmounted
 *
 * @brief  Notification that a storage device will be (has been) unmounted
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_media_unmounted (const gchar *mountpoint);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_is_directory_allowed
 *
 * @brief  Check whether user access is allowed to the specified directory
 *
 * @param  [in]  dir - directory to be verified
 *
 * @return TRUE  - user access allowed to specified directory
 *         FALSE - user access forbidden
 *
 *--------------------------------------------------------------------------*/
gboolean fileview_is_directory_allowed (const gchar *dir);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_refresh
 *
 * @brief  Refresh the current screen data, keep cursor at same item if possible
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_refresh (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_set_cursor_at_filename
 *
 * @brief  Position the cursor at the specified file, if present
 *
 * @param  [in]  filename - filename of the requested file, as read from filesystem
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_set_cursor_at_filename (const gchar *filename);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_show_prev_button
 *
 * @brief  Show or hide "previous" button
 *
 * @param  [in]  do_show - show button
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_show_prev_button (const gboolean do_show);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_show_next_button
 *
 * @brief  Show or hide "next" button
 *
 * @param  [in]  do_show - show button
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_show_next_button (const gboolean do_show);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_open_items
 *
 * @brief  Open the currently selected item(s)
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_open_items (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_delete_items
 *
 * @brief  Delete the currently selected item(s)
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_delete_items (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_create_shortcut
 *
 * @brief  Create desktop shortcut(s) for the currently selected item(s)
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_create_shortcut (void);


/**---------------------------------------------------------------------------
 *
 * Name :  fileview_grab_focus
 *
 * @brief  Grab focus for current view
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void fileview_grab_focus( void );


G_END_DECLS

#endif // __FILEVIEW_H__
