#ifndef __FILEMODEL_H__
#define __FILEMODEL_H__

/**
 * File Name  : filemodel.h
 *
 * Description: Content browser data for file model
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <gtk/gtk.h>

#include <libermetadb/ermetadb.h>

#include "fileview.h"

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// columns in file model
typedef enum
        {
            MODCOL_FILENAME = 0,
            MODCOL_FILENAME_DISPLAY,
            MODCOL_FILETYPE,
            MODCOL_FILETYPE_DISPLAY,
            MODCOL_DIRECTORY_PATH,
            MODCOL_IS_DIRECTORY,
            MODCOL_IS_TEMPLATE,
            MODCOL_FILESIZE,
            MODCOL_FILEDATE,
            MODCOL_AUTHOR,
            MODCOL_THUMBNAIL,
            N_FILEMODEL_COLUMNS
        } filemodel_column_t;

#define FILEMODEL_COLUMN_TYPES                                 \
            G_TYPE_STRING,      /* MODCOL_FILENAME         */  \
            G_TYPE_STRING,      /* MODCOL_FILENAME_DISPLAY */  \
            G_TYPE_STRING,      /* MODCOL_FILETYPE         */  \
            G_TYPE_STRING,      /* MODCOL_FILETYPE_DISPLAY */  \
            G_TYPE_STRING,      /* MODCOL_DIRECTORY_PATH   */  \
            G_TYPE_BOOLEAN,     /* MODCOL_IS_DIRECTORY     */  \
            G_TYPE_BOOLEAN,     /* MODCOL_IS_TEMPLATE      */  \
            G_TYPE_STRING,      /* MODCOL_FILESIZE         */  \
            G_TYPE_STRING,      /* MODCOL_FILEDATE         */  \
            G_TYPE_STRING,      /* MODCOL_AUTHOR           */  \
            GDK_TYPE_PIXBUF,    /* MODCOL_THUMBNAIL        */  \
            NULL  /* end of list */

// thumbnail size
typedef enum
        {
            MODTHUMB_MINI = 0,
            MODTHUMB_SMALL,
            MODTHUMB_MEDIUM,
            MODTHUMB_LARGE,
            N_FILEMODEL_THUMB_SIZES
        } filemodel_thumbsize_t;

// entry in filelist delivered by filemodel_get_filenames_xx()
typedef struct
        {
            GString     *filename;
            GString     *filename_display;
            GString     *filetype;
            GString     *directory_path;
            gboolean    is_directory;
            gboolean    is_template;
        } filelist_entry_t;

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_quit
 *
 * @brief  Prepare for power down
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_quit (void);


/**---------------------------------------------------------------------------
 *
 * Name :  get_filemodel
 *
 * @brief  Get a pointer to the file model
 *
 * @param  none
 *
 * @return GtkTreeModel interface on the filestore object
 *
 *--------------------------------------------------------------------------*/
GtkTreeModel* get_filemodel (void);


/**---------------------------------------------------------------------------
 *
 * Name :  get_database
 *
 * @brief  Get the currently open metadata database
 *
 * @param  none
 *
 * @return Currently open metadata database,
 *         or NULL if no database open
 *
 *--------------------------------------------------------------------------*/
erMetadb* get_database (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_set_sortorder
 *
 * @brief  Set sorting order for items in view, same order must be used in model
 *
 * @param  [in]  sort_order - the database column to sort on
 * @param  [in]  is_ascending - sorting ascending or descending
 *
 * @return TRUE  if sort order has changed, i.e. requested order different from current (old) order
 *         FALSE of sort order not changed, i.e. requested order same as current (old) order
 *
 *--------------------------------------------------------------------------*/
gboolean filemodel_set_sortorder (const ctb_sort_order_t sort_order, const gboolean is_ascending);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_set_viewsize
 *
 * @brief  Set number of items in view, same number must be loaded in model
 *
 * @param  [in]  n_items - number of items to be loaded in model (> 0)
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_set_viewsize (const int n_items);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_set_thumbsize
 *
 * @brief  Set thumbnail to be loaded in model
 *
 * @param  [in]  thumb_size - thumbnail size to be loaded in model
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_set_thumbsize (const filemodel_thumbsize_t thumb_size);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_page_previous
 *
 * @brief  Prepare data for previous page to be displayed, if any
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_page_previous (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_page_next
 *
 * @brief  Prepare data for next page to be displayed, if any
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_page_next (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_get_offset
 *
 * @brief  Get offset of first item displayed (0 for first page)
 *
 * @param  [in] item - index of requested item in current page, value 0 ..
 *
 * @return offset of requested item from,
 *         value 0 .. specified as offset from first item on first page
 *         or -1 on error
 *
 *--------------------------------------------------------------------------*/
gint filemodel_get_offset (const guint item);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_scroll_to_offset
 *
 * @brief  Prepare data for page holding the item with the specified offset (index),
 *         or prepare data for last page when specified offset not present
 *
 * @param  [in] offset - offset of item to be displayed,
 *                       value 0 .. specified as offset from first item on first page
 *
 * @return index of requested item in current page, value 0 ..
 *         or -1 on error
 *
 *--------------------------------------------------------------------------*/
gint filemodel_scroll_to_offset (const guint offset);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_scroll_to_filename
 *
 * @brief  Prepare data for page holding the item with the specified filename,
 *         or prepare data for first page when specified filename not present
 *
 * @param  [in] filename - filename of item to be displayed
 *                         or NULL to display the first item
 *
 * @return index of requested item in current page, value 0 ..
 *         or -1 on error
 *
 *--------------------------------------------------------------------------*/
gint filemodel_scroll_to_filename (const gchar *filename);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_chdir
 *
 * @brief  Load the file model with details from the specified directory
 *
 * @param  [in]  dir - directory to be displayed
 *
 * @return ER_OK or error
 *
 *--------------------------------------------------------------------------*/
int filemodel_chdir (const gchar *dir, const gchar *parent_dir);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_chdir_desktop
 *
 * @brief  Load the file model with details from the desktop directory
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_chdir_desktop (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_item_deleted
 *
 * @brief  Callback to report to fileview module that one item has been deleted
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_item_deleted (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_get_filenames_from_treepath_list
 *
 * @brief  Return the list of filenames for the selected items
 *
 * @param  [in]  selected items - specifies the selected items
 *                                as a GList holding GtkTreePath entries
 *
 * @return filenames as a list of filelist_entry_t items
 *
 *--------------------------------------------------------------------------*/
GSList* filemodel_get_filenames_from_treepath_list (const GList *selected_items);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_get_filenames_all
 *
 * @brief  Return the list of filenames for all non-hidden files or directories
 *
 * @param  [in]  selected items - specifies the selected items
 *                                as a GList holding GtkTreePath entries
 *
 * @return filenames as a list of filelist_entry_t items
 *
 *--------------------------------------------------------------------------*/
GSList* filemodel_get_filenames_all (void);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_filelist_free
 *
 * @brief  Release a list of filenames returned by filemodel_get_filenames_xx()
 *
 * @param  [in] filelist - file list as returned by filemodel_get_filenames_xx()
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_filelist_free ( GSList *filelist );


/**---------------------------------------------------------------------------
 *
 * Name :  filelist_entry_new
 *
 * @brief  Allocate a new filelist entry
 *
 * @param  --
 *
 * @return a newly created filelist_entry_t
 *
 *--------------------------------------------------------------------------*/
filelist_entry_t* filelist_entry_new ( void );


/**---------------------------------------------------------------------------
 *
 * Name :  filelist_entry_free
 *
 * @brief  Release the memory allocated to the specified filelist_entry_t
 *
 * @param  thiz - the filelist_entry_t to be released
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filelist_entry_free ( filelist_entry_t *thiz );


/**---------------------------------------------------------------------------
 *
 * Name :  filelist_entry_copy
 *
 * @brief  Create a new filelist_entry_t as a copy of the specified one
 *
 * @param  src - the filelist_entry_t to be copied
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
filelist_entry_t* filelist_entry_copy ( const filelist_entry_t *src );


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_get_thumbnail
 *
 * @brief  Retrieve thumbnail from database for specified file in the current directory
 *
 * @param  [in]  file_info - a filelist_entry delivered by one of the filemodel functions
 * @param  [in]  thumbnail_size - thumbnail size to be extracted from database
 *
 * @return The requested thumbnail, formatted as the content of a .png file.
 *         Caller must free this array with g_byte_array_free();
 *
 *--------------------------------------------------------------------------*/
GByteArray* filemodel_get_thumbnail( const filelist_entry_t      *file_info,
                                     const filemodel_thumbsize_t thumbnail_size );


G_END_DECLS

#endif // __FILEMODEL_H__
