/*
 * File Name: main.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <glib.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

// local include files, between " "
#include "ctb_log.h"
#include "filemodel.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "storage.h"
#include "recent.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";

static const char  *EMULATOR_HOSTNAME = "qemuarm";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

FILE *fplog = NULL;

//============================================================================
// Local Function Definitions
//============================================================================


//============================================================================
// Functions Implementation
//============================================================================

// print usage text and quit
static void usage (const char *argv_0)
{
    static const char *usage_text = 
                        "\n"
                        "usage: %s [options]\n"
                        "\n"
                        "options:\n"
                        "    --help\n"
                        "        Print help text and quit\n";

    printf(usage_text, argv_0);

    exit(1);
}

// get command-line options
static void parse_arguments (int argc, char **argv)
{
    int  i;

    // parse contentLister options
    for (i = 1 ; i < argc ; i++)
    {
        if (strcmp(argv[i], "--help") == 0)
        {
            usage(argv[0]);
        }
    }
}


// set locale (language)
void main_set_locale (const char *locale)
{
    WARNPRINTF("entry: locale [%s]", locale);

    const char *old_locale = g_getenv("LANG");
    
    if (!old_locale || (strcmp(old_locale, locale) != 0))
    {
        LOGPRINTF("Locale has changed to %s", locale);

        // set locale in environment 
        g_setenv("LANG", locale, TRUE);
        setlocale(LC_ALL, "");
        
        // update text and refresh
        menu_set_text();
        storage_set_text();
        fileview_set_text();
        storage_refresh_symlinks();
    }
}


// terminate the main loop
static void on_destroy (GtkWidget * widget, gpointer data)
{
    WARNPRINTF("entry");

    gtk_main_quit();
}


// terminate application
void main_quit (void)
{
    WARNPRINTF("entry");

    gtk_main_quit();
}


// signal handlers
static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    gdk_threads_enter();
    gtk_main_quit();
    gdk_threads_leave();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


static gboolean on_startup_complete (gpointer data)
{
    ipc_sys_startup_complete();
	// MH also refresh screen, there is a refresh problem when Home is not used
	// 
	fileview_refresh();
	
    return FALSE; // remove timer source
}


// main function
int main (int argc, char *argv[])
{
    int              rc;
    char             hostname[100];
    GtkWidget        *window;
    GtkWidget        *widget;
    struct sigaction on_term;
    GString          *str = NULL;

	// The log file have been changed to allow logging to be captured on the DR
	// the first line gives logging on stderr, the 2nd line logs in a file.
	// please select the one you need.
	fplog = NULL;
	if ( g_file_test("/media/mmcblk0p1/ctb1.txt", G_FILE_TEST_EXISTS) )
	{
		LOGINIT("/media/mmcblk0p1/ctb2.txt");
	}
	else
	{
		LOGINIT("/media/mmcblk0p1/ctb1.txt");
	}
	
	// parse command-line arguments
    parse_arguments(argc, argv);

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

    // init domain for translations
    textdomain(GETTEXT_PACKAGE);
    
    // init gtk, list the default rc files
    gtk_init(&argc, &argv);
    gchar** files = gtk_rc_get_default_files();
    while( *files )
    {
        WARNPRINTF("gtk_rc_get_default_files [%s]", *files);
        files++;
    }

    // init CTB modules
    ipc_set_services();
    menu_init();
    fileview_init();
    storage_init();
	recent_init();

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // create the top level window 
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), PACKAGE " " VERSION);
    gtk_window_maximize(GTK_WINDOW(window));
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    gtk_window_set_modal(GTK_WINDOW(window), TRUE);
    g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(on_destroy), NULL);
    g_main_window = window;

    // add screen details
    widget = create_fileview();
    gtk_container_add(GTK_CONTAINER(window), widget);

    // set view details
    // fileview_show_desktop();
	//MH show the recent opened files
	recent_start();
    str = NULL;

    // debug: mount "internal SD" when running on emulator
    rc = gethostname(hostname, sizeof(hostname));
    if (   rc == 0
        && strcmp(hostname, EMULATOR_HOSTNAME) == 0 )
    {
        storage_media_mounted(DIR_LIBRARY);
    }

    // make sure that everything, window and label, are visible 
    gtk_widget_show(window);

    // tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);

    // run the main loop
    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    gdk_threads_leave();

    // clean up CTB modules
    storage_quit();
    filemodel_quit();

    return 0;
}


// run error dialog
void run_error_dialog (const gchar *msg)
{
    GtkWidget   *dialog = NULL;

    ERRORPRINTF("entry: msg [%s]", msg);

    dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     msg );
    gtk_window_set_deletable( GTK_WINDOW(dialog), FALSE );
    gtk_dialog_run( GTK_DIALOG(dialog) );
    gtk_widget_destroy( dialog );
}
