/*
 * File Name: fileview.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <sys/wait.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>
#include <liberutils/display_utils.h>

// local include files, between " "
#include "ctb_log.h"
#include "ctb_actions.h"
#include "fileview.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "storage.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// actions when item clicked
typedef int (*ctb_prepost_callback) ( const GString             *directory,
                                      const GSList              *filelist  );
typedef int (*ctb_item_callback)    ( const GString             *directory,
                                      const filelist_entry_t    *file_info );
//
typedef struct
        {
            const char           *name;                  // name of this action
            gboolean             is_multi_select;        // multi-select allowed for this action
            ctb_prepost_callback pre_callback;           // call as start of action, parm is directory
            ctb_item_callback    item_callback;          // call for each item, parm is an item in the directory
            ctb_prepost_callback post_callback;          // call as end of action, parm is directory
        }                        ctb_action_t;

// path element for screen title
typedef struct
        {
            GString     *title;
            GString     *path;
            gulong      handler_id;
        }               path_element_t;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// registry key "view type"
static const gchar      *REGKEY_VIEWTYPE      = "/apps/er/sys/ctb/viewtype";
static const gchar      *REGVAL_VIEWTYPE_ICON = "iconview";
static const gchar      *REGVAL_VIEWTYPE_LIST = "listview";
static const gchar      *REGVAL_VIEWTYPE_AUTO = "auto-select";

// screen layout
static const int        WINDOW_TOP_PADDING          =  46;
static const int        WINDOW_BOTTOM_PADDING       =  10;
static const int        WINDOW_H_PADDING            =  12;
static const int        PREVNEXT_HEIGHT             =  20;
static const int        PREVNEXT_ARROW_WIDTH        = 140;
static const int        TITLE_V_PADDING             =  10;  // vertical padding around title
static const int        TITLE_SPACING               =  20;  // space between path and number of items in title
static const int        TITLE_PATH_SPACING          =  10;  // space between path elements in title
static const int        TITLE_PATH_SEPARATOR_WIDTH  =  16;  // size of separator bitmap in pathbar
static const int        TITLE_PATH_SEPARATOR_HEIGHT =  16;  //
static const int        FILEGRID_V_PADDING          =  10;  // vertical padding around the file grid
static const int        LISTVIEW_COL_SPACING        =  20;  // space between columns in listview
static const int        LISTVIEW_ICON_WIDTH         =  30;
static const int        LISTVIEW_ICON_HEIGHT        =  30;

// disk space needed
static const gint64     SD_CARD_FREE_MIN = 1 * 1024 * 1024; // minimum free space needed on SD card


// Directory black-and-white list, a black-list and white-list combined into one table.
// Each entry specifies the black/white status of a directory and its children.
// Access to a directory is allowed only when it explicitly white-listed and not black-listed.
// Rationale: only the white-listed directories are user accessible,
//            but within this area we might want to impose some black-listed sub-areas
// Examples:
//   this allows wild-card functionality like /media is black but /media/* is white:
//     dir = "/media",             dir_is_white = FALSE, child_is_white = TRUE
//   additionally /media/rootfs may be black-listed entirely:
//     dir = "/media/rootfs",      dir_is_white = FALSE, child_is_white = FALSE
//   sub-folders on desktop are allowed:
//     dir = "/home/root/desktop", dir_is_white = TRUE,  child_is_white = TRUE
//   sub-folders in recent documents are forbidden:
//     dir = "/home/root/recent",  dir_is_white = TRUE,  child_is_white = FALSE
// directories must be in whitelist and not in blacklist
//   whitelist
typedef struct
        {
            const char      *dir;            // directory name
            const gboolean  dir_is_white;    // directory is white-listed so children too
            const gboolean  child_is_white;  // child directories are white-listed
            const char      *title;          // screen title
        } dir_bw_list_t;
//
static const dir_bw_list_t  DIR_BW_LIST[] =
                            {
                                { DIR_STORAGE_MNT     , FALSE, TRUE,      NULL               },
                                { DIR_DESKTOP_INTERNAL, TRUE,  TRUE,      NULL               },
                                { DIR_LIBRARY,          TRUE,  TRUE,  N_("Documents"       ) },
                                { "/home/root/recent",  TRUE,  FALSE, N_("Recent Documents") },
                                { "/tmp/search_result", TRUE,  FALSE, N_("Search Result"   ) },
                                {  NULL  }  // end of list
                            };

// actions to be executed on items in view/model
static const ctb_action_t       CTB_ACTIONS[] =
                                {
                                    // name                 multi   pre              item           post
                                    {  CTB_ACTION_ACTIVATE, FALSE,  NULL,            activate_item, NULL             },
                                    {  CTB_ACTION_DELETE,   TRUE,   pre_delete_item, delete_item,   post_delete_item },
                                    {  CTB_ACTION_SHORTCUT, TRUE,   NULL,            shortcut_item, NULL             },
                                    {  NULL }  // end of list
                                };


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// screen elements
static GtkWidget        *g_pathbar              = NULL;   // screen title: path main widget (hbox)
static gulong           g_pathbar_handler_id    = 0;      // handler id on "size-allocate" callback
static GtkWidget        *g_num_items            = NULL;   // screen title: number of items
static GtkWidget        *g_prev_page_arrow      = NULL;   // title bar for "previous page" 
static GtkWidget        *g_prev_page_filler     = NULL;   //
static gboolean         g_has_prev_page         = FALSE;  // button "previous page" is enabled
static GtkWidget        *g_next_page_arrow      = NULL;   // title bar for "next page" 
static GtkWidget        *g_next_page_filler     = NULL;   //
static gboolean         g_has_next_page         = FALSE;  // button "next page" is enabled
static GtkNotebook      *g_notebook             = NULL;   // notebook holding the different views
static GtkWidget        *g_iconview             = NULL;   // iconview widget
static GtkWidget        *g_listview             = NULL;   // listview widget
static gboolean         g_is_user_selected_view = FALSE;  // user has selected the current view

// directory navigation
static GString          *g_current_dir          = NULL;   // directory currently displayed
static gboolean         g_current_is_desktop    = FALSE;  // desktop directory currently displayed
static gboolean         g_do_select_all         = FALSE;  // select all files and directories in current dir


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget*       create_screen_layout          ( void );
static GtkWidget*       create_iconview               ( void );
static GtkWidget*       create_listview               ( void );
static GtkWidget*       create_pathbar_separator      ( void );
static int              do_action_on_selected_items   ( const char            *action_name );
static void             free_path_elements            ( GArray                *path_arr );
static void             get_cursor                    ( gint                  *row,
                                                        gint                  *column );
static GtkTreePath*     get_cursor_tree_path          ( void );
static gchar*           get_filename_at_cursor        ( void );
static ctb_viewtypes_t  get_viewtype_from_registry    ( void );
static gboolean         is_directory_allowed          ( const gchar           *dir,
                                                              GString         **real_dir,
                                                              GArray          **path_elements );
static void             listview_set_text             ( void );
static void             save_viewtype_to_registry     ( ctb_viewtypes_t       view );
static void             set_cursor                    ( gint                  row,
                                                        gint                  column );
static void             set_cursor_at_item            ( const gint            offset );
static void             treepath_list_free            ( GList                 **list );
static void             update_pathbar                ( GArray                *pathbar_data );
static void             update_view_size              ( void );

// signal handlers
static void             on_iconview_item_activated    ( GtkIconView           *iconview,
                                                        GtkTreePath           *path,
                                                        gpointer              user_data );
static void             on_iconview_navigate_cursor   ( erGtkIconView         *er_iconview,
                                                        erGtkIconViewKeyPress keycode,
                                                        gpointer              user_data );
static void             on_iconview_size_allocate     ( GtkWidget             *widget,
                                                        GtkAllocation         *allocation,
                                                        gpointer              user_data   );
static gboolean         on_idle_set_cursor            ( gpointer              data );
static gboolean         on_idle_set_cursor_at_filename( gpointer data );
static gboolean         on_idle_set_cursor_at_item    ( gpointer              data );
static void             on_listview_navigate_cursor   ( erGtkListView         *er_listview,
                                                        erGtkListViewKeyPress keycode,
                                                        gpointer              user_data );
static void             on_listview_row_activated     ( GtkTreeView           *view,
                                                        GtkTreePath           *path,
                                                        GtkTreeViewColumn     *column,
                                                        gpointer              user_data );
static void             on_listview_size_allocate     ( GtkWidget             *widget,
                                                        GtkAllocation         *allocation,
                                                        gpointer              user_data   );
static gboolean         on_next_page_button_press     ( GtkWidget             *widget,
                                                        GdkEventButton        *event,
                                                        gpointer              user_data );
static void             on_num_items_size_allocate    ( GtkWidget             *widget,
                                                        GtkAllocation         *allocation,
                                                        gpointer              user_data   );
static gboolean         on_pathbar_button_press_event ( GtkWidget             *widget,
                                                        GdkEventButton        *event,
                                                        gpointer              user_data );
static void             on_path_size_allocate         ( GtkWidget             *widget,
                                                        GtkAllocation         *allocation,
                                                        gpointer              user_data   );
static gboolean         on_prev_page_button_press     ( GtkWidget             *widget,
                                                        GdkEventButton        *event,
                                                        gpointer              user_data );

static gboolean         update_display                ( gpointer data );
static void             idle_update_display           ( gint type );


//============================================================================
// Functions Implementation
//============================================================================

// first-time initialisation
void fileview_init (void)
{
    int         argc = 0;
    const char  *argv[10];
    gint        stat;
    gboolean    ok;
    int         rc;

    LOGPRINTF("entry");

    // enforce function is called only once
    static gboolean firsttime = TRUE;
    g_assert(firsttime);
    firsttime = FALSE;

    // remove desktop directory
    argc = 0;
    argv[argc++] = "rm";
    argv[argc++] = "-rf";
    argv[argc++] = DIR_DESKTOP_INTERNAL;
    argv[argc++] = DIR_DESKTOP_TMP;
    argv[argc++] = NULL;
    g_assert(argc < sizeof(argv)/sizeof(argv[0]));
    ok = g_spawn_sync( NULL,                   // working directory: inherit
                       (char**)argv,           // child's argument vector
                       NULL,                   // environment: inherit
                       G_SPAWN_SEARCH_PATH,    // flags
                       NULL,                   // child_setup: none
                       NULL,                   // child setup data: none
                       NULL,                   // stdout: not interested
                       NULL,                   // stderr: not interested
                       &stat,                  // exit status
                       NULL                );  // error return

    // copy template desktop
    argc = 0;
    argv[argc++] = "cp";
    argv[argc++] = "-rp";
    argv[argc++] = DIR_DESKTOP_TEMPLATE;
    argv[argc++] = DIR_DESKTOP_INTERNAL;
    argv[argc++] = NULL;
    g_assert(argc < sizeof(argv)/sizeof(argv[0]));
    ok = g_spawn_sync( NULL,                   // working directory: inherit
                       (char**)argv,           // child's argument vector
                       NULL,                   // environment: inherit
                       G_SPAWN_SEARCH_PATH,    // flags
                       NULL,                   // child_setup: none
                       NULL,                   // child setup data: none
                       NULL,                   // stdout: not interested
                       NULL,                   // stderr: not interested
                       &stat,                  // exit status
                       NULL                );  // error return
    if ( ok && WIFEXITED(stat) )
    {
        if ( WEXITSTATUS(stat) != 0 )
        {
            ERRNOPRINTF("cannot copy desktop [%s] to [%s]", DIR_DESKTOP_TEMPLATE, DIR_DESKTOP_INTERNAL);
        }
    }
    else
    {
        ERRORPRINTF("g_spawn_sync failed for copy template desktop");
    }

    // create empty desktop work directory
    rc = g_mkdir_with_parents(DIR_DESKTOP_TMP,      0755);
    if ( rc != 0 )
    {
        ERRNOPRINTF("cannot create directory [%s]", DIR_DESKTOP_TMP);
    }
}


// create fileview widgets
GtkWidget* create_fileview (void)
{
    GtkWidget           *background = NULL;     // return value
    GtkWidget           *widget;
    ctb_viewtypes_t     view;

    LOGPRINTF("entry");

    // enforce function is called only once
    static gboolean firsttime = TRUE;
    g_assert(firsttime);
    firsttime = FALSE;

    // create screen layout
    widget = create_screen_layout();
    background = widget;

    // create iconview or listview
    view = get_viewtype_from_registry();
    fileview_set_view_type( view );
    g_is_user_selected_view = FALSE;

    return background;
}

GString *fileview_get_current_dir(void)
{
	return g_current_dir;
}

// update screen texts
void fileview_set_text()
{
    gchar   *cursor_item = NULL;

    LOGPRINTF("entry");

    // update listview column headers
    listview_set_text();

    // refresh screen
    fileview_refresh();

    // clean up
    g_free(cursor_item);
}


// grab focus for current page
void fileview_grab_focus( void )
{
    ctb_viewtypes_t     current_view;
    
    LOGPRINTF("entry");
    
    g_assert(g_notebook);

    current_view = gtk_notebook_get_current_page(g_notebook);
    switch (current_view)
    {
        case CTB_ICONVIEW:
            gtk_widget_grab_focus(GTK_WIDGET(g_iconview));
            break;
        case CTB_LISTVIEW:
            gtk_widget_grab_focus(GTK_WIDGET(g_listview));
            break;
        default:
            break;
    }
    
    idle_update_display(DM_HINT_NONE);
}


// select view
void fileview_set_view_type (const ctb_viewtypes_t view)
{
    GtkWidget   *widget;
    GtkWidget   *event_box;
    gchar       *cursor_item = NULL;

    LOGPRINTF("entry: view [%d] notebook_page [%d]", view, gtk_notebook_get_current_page(g_notebook));
    g_assert(g_notebook);
    g_assert( view >= 0  &&  view < N_CTB_VIEWS );

    // remember user has selected the current view
    g_is_user_selected_view = TRUE;

    // function void if request is current view
    if ( view == gtk_notebook_get_current_page(g_notebook) )
    {
        LOGPRINTF("view [%d] same as current view", view);
        return;
    }

    // get current item at cursor position
    cursor_item = get_filename_at_cursor();

    // select notebook page for requested view
    gtk_notebook_set_current_page(g_notebook, view);

    // create widgets for notebook page when needed
    switch (view)
    {
        case CTB_ICONVIEW:
            filemodel_set_thumbsize(MODTHUMB_MEDIUM);
            if (g_iconview == NULL)
            {
                // create icon view
                event_box = gtk_notebook_get_nth_page(g_notebook, view);
                if (event_box)
                {
                    widget = create_iconview();
                    if (widget)
                    {
                        gtk_container_add(GTK_CONTAINER(event_box), widget);
                        gtk_widget_show(widget);
                        g_iconview = widget;

                        // delete other views
                        if (g_listview)
                        {
                            gtk_widget_destroy(g_listview);
                            g_listview = NULL;
                        }
                    }
                }
            }
            break;

        case CTB_LISTVIEW:
            filemodel_set_thumbsize(MODTHUMB_MINI);
            if (g_listview == NULL)
            {
                // create list view
                event_box = gtk_notebook_get_nth_page(g_notebook, view);
                if (event_box)
                {
                    widget = create_listview();
                    if (widget)
                    {
                        gtk_container_add(GTK_CONTAINER(event_box), widget);
                        gtk_widget_show(widget);
                        g_listview = widget;
                        listview_set_text();

                        // delete other views
                        if (g_iconview)
                        {
                            gtk_widget_destroy(g_iconview);
                            g_iconview = NULL;
                        }
                    }
                }
            }
            break;

        default:
            ;  // ignore: handled by assert at function entry
    }

    // reposition cursor to the same item as before
    fileview_set_cursor_at_filename( cursor_item );

    // update popup menu
    menu_select_view_type( view );
    
    // clean up
    g_free(cursor_item);
}


// set sorting order
void fileview_set_sort_order (const ctb_sort_order_t sort_order)
{
    gboolean    order_has_changed;
    gchar       *cursor_item = NULL;

    // ascending/descending per sort order
    const int   is_sort_ascending[N_CTB_SORT_ORDER] =
                {
                    TRUE,       // CTB_SORT_BY_NAME
                    TRUE,       // CTB_SORT_BY_TYPE
                    FALSE,      // CTB_SORT_BY_SIZE
                    FALSE,      // CTB_SORT_BY_DATE
                    TRUE        // CTB_SORT_BY_AUTHOR
                };

    LOGPRINTF("entry: sort_order [%d] notebook_page [%d]", sort_order, gtk_notebook_get_current_page(g_notebook));
    g_assert(g_notebook);
    g_assert(sort_order >= 0  &&  sort_order < N_CTB_SORT_ORDER);

    // get current item at cursor position
    cursor_item = get_filename_at_cursor();

    // set requested sort order
    order_has_changed = filemodel_set_sortorder( sort_order, is_sort_ascending[sort_order] );

    // reposition cursor to the same item as before
    if (order_has_changed)
    {
        fileview_set_cursor_at_filename( cursor_item );
    }
    
    // clean up
    g_free(cursor_item);
}


// goto specified directory
void fileview_show_dir (const gchar *dir, const gchar *cursor_item)
{
    int             ret = ER_OK;    // return value
    int             rc;
    gint64          free_bytes;
    gboolean        ok;
    gboolean        is_desktop    = FALSE;
    gchar           *dir_title    = NULL;
    gchar           *parent_dir   = NULL;
    GString         *real_dir     = NULL;
    GString         *error_msg    = NULL;
    GArray          *pathbar_data = NULL;
    path_element_t  *path_el      = NULL;
    struct statfs   stat;

    LOGPRINTF("entry: dir [%s] cursor_item [%s]", dir, cursor_item);
    g_assert(dir && *dir);

    // check directory allowed
    ok = is_directory_allowed(dir, &real_dir, &pathbar_data);
    LOGPRINTF( "ok [%d] real_dir [%s]", ok, real_dir ? real_dir->str : NULL);
    if ( !ok )
    {
        ERRORPRINTF("directory not accessible [%s]", dir);
        ret = ER_FORBIDDEN;
    }
    else
    {
        // check if desktop
        if ( strcmp(dir, DIR_DESKTOP_INTERNAL) == 0 )
        {
            is_desktop = TRUE;
        }
    }

    // set as current directory
    if (ret == ER_OK)
    {
        rc = chdir(real_dir->str);
        if (rc != 0)
        {
            ERRNOPRINTF("cannot change to real_dir [%s]", real_dir->str);
            ret = ER_NOT_FOUND;
        }
    }

    // update local directory administration
    if (ret == ER_OK)
    {
        // remember new directory
        if (g_current_dir)
        {
            g_string_assign(g_current_dir, real_dir->str);
        }
        else
        {
            g_current_dir = g_string_new(real_dir->str);
        }
        g_current_is_desktop = is_desktop;

        // adjust popup menu
        if ( !is_desktop )
        {
            menu_set_current_is_desktop(FALSE);
            menu_set_fixed_sort_order(FALSE);
        }

        // show new directory
        //   update the model
        if (is_desktop)
        {
            filemodel_chdir_desktop();
        }
        else
        {
            parent_dir = g_strdup_printf("%s/..", g_current_dir->str);
            if ( is_directory_allowed(parent_dir, NULL, NULL) )
            {
                ret = filemodel_chdir(g_current_dir->str, "..");
            }
            else
            {
                ret = filemodel_chdir(g_current_dir->str, DIR_DESKTOP_INTERNAL);
            }
        }
        if (ret != ER_OK)
        {
            ERRORPRINTF("cannot update filemodel, error [%d]", ret);
        }
        else
        {
            update_pathbar(pathbar_data);
            pathbar_data = NULL;

            // set cursor on first item
            set_cursor(0, 0);

            // adjust popup menu
            if ( is_desktop )
            {
                menu_set_fixed_sort_order(TRUE);
                menu_set_current_is_desktop(TRUE);
            }

            // set cursor on requested item
            if ( cursor_item )
            {
                fileview_set_cursor_at_filename(cursor_item);
            }

            WARNPRINTF("current directory changed to [%s]", real_dir->str);
        }
    }

    // error handling
    if (ret != ER_OK)
    {
        // get directory name as shown in pathbar
        if (pathbar_data)
        {
            path_el = &g_array_index( pathbar_data,
                                      path_element_t,
                                      pathbar_data->len - 1 );
            if (path_el->title)
            {
                dir_title = g_strdup(path_el->title->str);
            }
        }
        else
        {
            dir_title = g_path_get_basename( real_dir ? real_dir->str : dir );
        }

        // show error message
        error_msg = g_string_new("");
        if (ret == ER_READONLY)
        {
            g_string_printf( error_msg,
                             _("'%s' cannot be opened; the memory card may be locked.\n"
                               "Try unlocking the card and try again."),
                             dir_title );
        }
        else
        {
            free_bytes = SD_CARD_FREE_MIN;
            if (real_dir)
            {
                rc = statfs(real_dir->str, &stat);
                if (rc == 0)
                {
                    free_bytes = stat.f_bfree * stat.f_bsize;
                }
            }

            if (free_bytes < SD_CARD_FREE_MIN)
            {
                g_string_printf( error_msg,
                                 _("'%s' cannot be opened.\n"
                                   "The memory card is (almost) full and there is not "
                                   "sufficient memory to open the folder."),
                                 dir_title );
            }
            else
            {
                g_string_printf( error_msg,
                                 _("'%s' cannot be opened."),
                                 dir_title );
            }
        }
        run_error_dialog( error_msg->str );

        // back to desktop
        fileview_show_desktop();
    }

    // clean up
    if (error_msg   ) { g_string_free(error_msg, TRUE);   }
    if (pathbar_data) { free_path_elements(pathbar_data); }
    if (real_dir    ) { g_string_free(real_dir, TRUE);    }
    g_free(parent_dir);
    g_free(dir_title);

    //return ret;
}


// display number of items in screen title
void fileview_set_num_items ( const int first, const int last, const int total )
{
    gchar       *cp;

    LOGPRINTF("entry: first [%d] last [%d] total [%d]", first, last, total);

    if (total > 0)
    {
        /* TRANSLATORS: %1$d is replaced by the first visible item number
                        %2$d is replaced by the last visible item number
                        %3$d is replaced by the total number of items
           Example: '1 - 9 of 40' */
        cp = g_strdup_printf( _("%1$d - %2$d of %3$d"), first, last, total );
        gtk_label_set_text(GTK_LABEL(g_num_items), cp);
        g_free(cp);
    }
    else
    {
        gtk_label_set_text(GTK_LABEL(g_num_items), "");
    }
}


// goto desktop
void fileview_show_desktop (void)
{
    LOGPRINTF("entry");

    fileview_show_dir(DIR_DESKTOP_INTERNAL, NULL);
}


// storage device has been mounted
void fileview_media_mounted (const gchar *mountpoint)
{
    ctb_viewtypes_t view;
    GString         *filename = NULL;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    // refresh desktop if needed
    if (g_current_is_desktop)
    {
        // get filename for shortcut file
        filename = storage_get_shortcut_filename(mountpoint);
        g_assert(filename && filename->str);

        // update screen, position cursor at newly mounted device
        fileview_show_dir(g_current_dir->str, filename->str);
    }

    // set viewtype when mounting internal SD card
    if ( strcmp(mountpoint, DIR_LIBRARY) == 0 )
    {
        view = get_viewtype_from_registry();
        if (view != CTB_AUTOSELECT)
        {
            fileview_set_view_type( view );
            g_is_user_selected_view = FALSE;
        }
    }

    if (filename) { g_string_free(filename, TRUE); }
}


// storage device will be (has been) unmounted
void fileview_media_unmounted (const gchar *mountpoint)
{
    int             n;
    ctb_viewtypes_t view;
    gchar           *cursor_item = NULL;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);
    g_return_if_fail(g_current_dir && g_current_dir->str);

    // save to registry on unmount internal SD card
    if ( strcmp(mountpoint, DIR_LIBRARY) == 0 )
    {
        // save viewtype to registry
        if (g_is_user_selected_view)
        {
            view = gtk_notebook_get_current_page(g_notebook);
            save_viewtype_to_registry(view);
            g_is_user_selected_view = FALSE;
        }
    }

    // goto desktop when we are on the unmounted device
    n = strlen(mountpoint);
    if (   strncmp(g_current_dir->str, mountpoint, n) == 0
        && (   g_current_dir->str[n] == '/'
            || g_current_dir->str[n] == '\0' ) )
    {
        fileview_show_desktop();
    }
    else if (g_current_is_desktop)
    {
        // desktop currently displayed: refresh screen
        fileview_refresh();
    }

    // clean up
    g_free(cursor_item);
}


// get cursor location
static void get_cursor ( gint *row, gint *column )
{
    int     current_view;

    LOGPRINTF("entry");
    g_assert(g_notebook);

    // select first item and set cursor
    current_view = gtk_notebook_get_current_page(g_notebook);
    switch (current_view)
    {
        case CTB_ICONVIEW:
            g_assert(g_iconview);
            ergtk_icon_view_get_cursor( ERGTK_ICON_VIEW(g_iconview), row, column );
            break;

        case CTB_LISTVIEW:
            g_assert(g_listview);
            ergtk_list_view_get_cursor( ERGTK_LIST_VIEW(g_listview), row);
            column = 0;
            break;

        default:
            g_assert_not_reached();
    }
}


// set cursor on specified item
static void set_cursor ( gint row, gint column )
{
    gulong row_col = ((row & 0xFF) << 8) | (column & 0xFF);

    g_idle_add( on_idle_set_cursor, (gpointer)row_col);
}

static gboolean on_idle_set_cursor( gpointer data )
{
    gulong  row_col = (gulong)data;
    gint    row    = (row_col >> 8) & 0xFF;
    gint    column =  row_col       & 0x00FF;
    int     current_view;
	//static gboolean first_time = TRUE;

    LOGPRINTF("entry");
    g_assert(g_notebook);
	
	// As a hack also do a refresh of caption the first time this function is called
	// MH-> This does not seem to help ...
	//if (first_time == TRUE)
	//{
	//	first_time = FALSE;
	//	gtk_widget_queue_draw(g_pathbar);
	//}

    // select first item and set cursor
    current_view = gtk_notebook_get_current_page(g_notebook);
    switch (current_view)
    {
        case CTB_ICONVIEW:
            g_assert(g_iconview);
            ergtk_icon_view_set_cursor( ERGTK_ICON_VIEW(g_iconview), row, column );
            break;

        case CTB_LISTVIEW:
            g_assert(g_listview);
            ergtk_list_view_set_cursor( ERGTK_LIST_VIEW(g_listview), row);
            break;

        default:
            g_assert_not_reached();
    }

    return FALSE;  // don't call me again
}


// show or hide "previous" button
void fileview_show_prev_button (const gboolean do_show)
{
    LOGPRINTF("entry: do_show [%d]", do_show);

    g_has_prev_page = do_show;
    if (do_show)
    {
        gtk_widget_show(g_prev_page_arrow);
    }
    else
    {
        gtk_widget_hide(g_prev_page_arrow);
    }
}


// show or hide "next" button
void fileview_show_next_button (const gboolean do_show)
{
    LOGPRINTF("entry: do_show [%d]", do_show);

    g_has_next_page = do_show;
    if (do_show)
    {
        gtk_widget_show(g_next_page_arrow);
    }
    else
    {
        gtk_widget_hide(g_next_page_arrow);
    }
}


// create screen widgets
static GtkWidget* create_screen_layout (void)
{
    GtkWidget   *background = NULL;  // return value
    GtkWidget   *widget;
    GtkBox      *vbox;
    GtkBox      *hbox;
    GtkBox      *path_hbox;
    GtkLabel    *label;
    GtkMisc     *misc;
    GtkNotebook *notebook;
    int         i;


    LOGPRINTF("entry");

    // object hierarchy:
    //     background (alignment)
    //       |
    widget = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               WINDOW_TOP_PADDING,
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
    gtk_widget_show(widget);
    background = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
    //             |
    //             |-- hbox
    //             |     |
    widget = gtk_hbox_new(FALSE, TITLE_SPACING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, TITLE_V_PADDING);
    gtk_widget_show(widget);
    hbox = GTK_BOX(widget);
    //             |     |
    //             |     |-- g_pathbar
    //             |     |     |-- ... to be filled in later
    //             |     |
    widget = gtk_hbox_new(FALSE, 0);
    g_pathbar_handler_id = g_signal_connect_after(G_OBJECT(widget), "size-allocate", G_CALLBACK(on_path_size_allocate), NULL);
    gtk_box_pack_start(hbox, widget, TRUE, TRUE, 0);
    gtk_widget_show(widget);
    path_hbox = GTK_BOX(widget);
    g_pathbar = widget;
    //             |     |
    //             |     |-- g_num_items (label)
    //             |
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-ctb-num-items");
    g_signal_connect(G_OBJECT(widget), "size-allocate", G_CALLBACK(on_num_items_size_allocate), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    misc = GTK_MISC(widget);
    gtk_misc_set_alignment(misc, 1.0, 0.85);
    label = GTK_LABEL(widget);
    gtk_label_set_line_wrap(label, FALSE);
    gtk_label_set_ellipsize(label, PANGO_ELLIPSIZE_NONE);
    gtk_widget_show(widget);
    g_num_items = widget;
    //             |
    //             |-- hbox
    //             |     |
    //             |     |-- g_prev_page_filler (filler right)
    //             |     |
    //             |     |-- g_prev_page_arrow
    //             |     |
    //             |     |-- event_box (filler left)
    //             |
    widget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, PREVNEXT_HEIGHT);
    gtk_widget_show(widget);
    hbox = GTK_BOX(widget);
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-prev-page-filler");
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    gtk_widget_show(widget);
    g_prev_page_filler = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-prev-page-arrow");
    g_signal_connect(G_OBJECT(widget), "button-press-event", G_CALLBACK(on_prev_page_button_press), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, PREVNEXT_ARROW_WIDTH, -1);
    GTK_WIDGET_UNSET_FLAGS(widget, GTK_CAN_FOCUS);
    // note: keep previous-page-arrow hidden
    g_prev_page_arrow = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-prev-page-filler");
    gtk_box_pack_end(hbox, widget, TRUE, TRUE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    gtk_widget_show(widget);
    //             |
    //             |
    //             |-- g_notebook
    //             |     |
    widget = gtk_notebook_new();
    gtk_box_pack_start(vbox, widget, TRUE, TRUE, FILEGRID_V_PADDING);
    notebook = GTK_NOTEBOOK(widget);
    gtk_notebook_set_show_tabs(notebook, FALSE);
    gtk_notebook_set_show_border(notebook, FALSE);
    gtk_widget_show(widget);
    g_notebook = notebook;
    //             |     |
    //             |     |-- event_box (n pages)
    //             |
    for (i = 0 ; i < N_CTB_VIEWS ; i++)
    {
        widget = gtk_event_box_new();
        gtk_notebook_append_page(notebook, widget, NULL);
        gtk_widget_show(widget);
    }
    //             |
    //             |-- hbox
    //             |     |
    //             |     |-- g_next_page_filler (filler right)
    //             |     |
    //             |     |-- g_next_page_arrow
    //             |     |
    //             |     |-- event_box (filler left)
    //             |
    widget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, PREVNEXT_HEIGHT);
    gtk_widget_show(widget);
    hbox = GTK_BOX(widget);
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-next-page-filler");
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    gtk_widget_show(widget);
    g_next_page_filler = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-next-page-arrow");
    g_signal_connect(G_OBJECT(widget), "button-press-event", G_CALLBACK(on_next_page_button_press), NULL);
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    gtk_widget_set_size_request(widget, PREVNEXT_ARROW_WIDTH, -1);
    GTK_WIDGET_UNSET_FLAGS(widget, GTK_CAN_FOCUS);
    // note: keep next-page-arrow hidden
    g_next_page_arrow = widget;
    //             |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-next-page-filler");
    gtk_box_pack_end(hbox, widget, TRUE, TRUE, 0);
    gtk_widget_set_size_request(widget, -1, -1);
    gtk_widget_show(widget);

    return background;
}


// display current path in screen title
// [in]  pathbar_data - path elements to be displayed,
//                      or NULL to clear the path in screen title
// note: function takes ownership of pathbar_data and releases it when needed
static void update_pathbar ( GArray *pathbar_data )
{
    int             index;
    gulong          handler_id;
    GtkWidget       *widget;
    GtkWidget       *event_box;
    GtkWidget       *label;
    GList           *pathbar_widgets = NULL;
    GList           *list;
    path_element_t  *path_el    = NULL;
    int             path_el_num = 0;

    static GArray   *g_pathbar_data = NULL;   // screen title: path as an array of path_element_t


    LOGPRINTF( "entry: pathbar_data [%p] len [%d]",
               pathbar_data,
               pathbar_data ? pathbar_data->len : 0 );
    g_assert(g_pathbar);

    // remove signal handlers for pathbar elements
    if (g_pathbar_data)
    {
        path_el     = (path_element_t*) g_pathbar_data->data;
        path_el_num =                   g_pathbar_data->len;
    }
    pathbar_widgets = gtk_container_get_children( GTK_CONTAINER(g_pathbar) );
    for ( list = pathbar_widgets, index = 0 ;
          list ;
          list = list->next, index++ )
    {
        widget = GTK_WIDGET(list->data);

        if (   index % 2 == 0
            && index / 2  < path_el_num )
        {
            // GtkLabel holding a path element: disconnect handler
            handler_id = path_el->handler_id;
            if (handler_id > 0)
            {
                g_signal_handler_disconnect(widget, handler_id);
                path_el->handler_id = 0;
            }
            path_el++;
        }
    }
    
    // replace pathbar data
    free_path_elements(g_pathbar_data);
    g_pathbar_data = pathbar_data;

    // update content of pathbar widgets and set new signal handlers
    if (pathbar_data)
    {
        path_el     = (path_element_t*) pathbar_data->data;
        path_el_num =                   pathbar_data->len;
    }
    //   update existing widgets
    for ( list = pathbar_widgets, index = 0 ;
          list ;
          list = list->next, index++ )
    {
        widget = GTK_WIDGET(list->data);
        LOGPRINTF("index [%d] path_el_num [%d] widget [%p]", index, path_el_num, widget);

        if ( (index + 1) / 2  < path_el_num )
        {
            // hide widget for now, on_size_allocate will do the rest
            gtk_widget_hide(widget);

            // pathbar widget needed in new path
            if (index % 2 == 0)
            {
                // GtkLabel holding a path element
                //   update text
                g_assert(path_el->title  &&  path_el->title->len > 0);
                gtk_widget_set_size_request(widget, -1, -1);
                label = gtk_bin_get_child( GTK_BIN(widget) );
                gtk_label_set_text( GTK_LABEL(label), path_el->title->str );
                gtk_label_set_ellipsize(GTK_LABEL(label), PANGO_ELLIPSIZE_NONE);
                //   update signal handler
                handler_id = g_signal_connect( widget,
                                               "button-press-event",
                                               G_CALLBACK(on_pathbar_button_press_event),
                                               path_el->path                             );
                path_el->handler_id = handler_id;
                //   next path element
                path_el++;
            }
            else
            {
                // separator widget: ignore
            }
        }
        else
        {
            // this pathbar widget not needed: destroy it
            // apparently the old pathbar was longer than the new one
            gtk_widget_destroy(widget);
        }
    }
    //   add new pathbar widgets as needed
    LOGPRINTF("index [%d] path_el_num [%d]", index, path_el_num);
    while ( (index + 1) / 2  < path_el_num )
    {
        LOGPRINTF("index [%d] path_el_num [%d]", index, path_el_num);
        if (index % 2 == 0)
        {

            // add GtkLabel holding a path element
            //   add event-box with label
            //   keep widget hidden for now, on_size_allocate will do the rest
            g_assert(path_el->title  &&  path_el->title->len > 0);
            event_box = gtk_event_box_new();
            label     = gtk_label_new( path_el->title->str );
            gtk_widget_set_name(label, "irex-ctb-pathbar");
            gtk_container_add( GTK_CONTAINER(event_box), label );
            gtk_box_pack_start( GTK_BOX(g_pathbar), event_box, FALSE, FALSE, 0);
            gtk_label_set_ellipsize(GTK_LABEL(label), PANGO_ELLIPSIZE_NONE); //MH
            gtk_widget_show(label);
            index++;
            //   set signal handler
            handler_id = g_signal_connect( event_box,
                                           "button-press-event",
                                           G_CALLBACK(on_pathbar_button_press_event),
                                           path_el->path                             );
            path_el->handler_id = handler_id;
            path_el++;
        }
        else
        {
            // add separator widget
            //   keep widget hidden for now, on_size_allocate will do the rest
            widget = create_pathbar_separator();
            gtk_box_pack_start( GTK_BOX(g_pathbar), widget, FALSE, FALSE, 0);
            index++;
        }
    }
}


// create separator widget for pathbar in screen title
static GtkWidget* create_pathbar_separator ( void )
{
    GtkWidget   *separator = NULL;  // return value
    GtkWidget   *widget = NULL;

    LOGPRINTF("entry");

    widget = gtk_alignment_new(0.5, 0.65, 0.0, 0.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(widget), 0, 0, TITLE_PATH_SPACING + 1, TITLE_PATH_SPACING - 1);
    separator = widget;

    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-ctb-pathbar-separator");
    gtk_widget_set_size_request(widget, TITLE_PATH_SEPARATOR_WIDTH, TITLE_PATH_SEPARATOR_HEIGHT);
    gtk_container_add(GTK_CONTAINER(separator), widget);
    gtk_widget_show(widget);

    return separator;
}


// pathbar button pressed: goto this directory
static gboolean on_pathbar_button_press_event ( GtkWidget      *widget,
                                                GdkEventButton *event,
                                                gpointer       user_data )
{
    GString     *path = (GString*)user_data;

    LOGPRINTF("entry: path [%s]", path ? path->str : NULL);
    g_assert(path && path->len > 0);

    if (event->type == GDK_BUTTON_PRESS)
    {
        fileview_show_dir(path->str, NULL);
    }

    return FALSE;  // continue with other signal handlers
}


// on size-allocate for path displayed in title
// Note: don't touch code this unless you have plenty of time to test it thoroughly
//       it has been a major headache to get this working so far, let's better not touch this anymnore
static void on_path_size_allocate ( GtkWidget     *pathbar,
                                    GtkAllocation *allocation,
                                    gpointer      user_data   )
{
    gint        width;
    gint        spacing;
    int         index;
    gpointer    gp;
    gchar       *cp;
    GList       *children = NULL;
    gint        n_children = 0;
    GList       *list;
    GtkWidget   *child;
    GtkWidget   *widget;
    GtkLabel    *label;
    GtkWidget   *item2  = NULL;         // 2nd pathbar item
    gboolean    item2_reduced = FALSE;  // 2nd pathbar item has been reduced
    GtkRequisition requisition;

    LOGPRINTF( "entry: x y [%d %d] width [%d] height [%d] user_data [%s]",
               allocation->x,
               allocation->y,
               allocation->width,
               allocation->height,
               (const char*) user_data );

	// Hack to get initial caption correct
	if (allocation->width < 600)
	{
		allocation->width = 600;
		LOGPRINTF("Changed allocation width");
	}
	
    spacing = gtk_box_get_spacing( GTK_BOX(pathbar) );

    // show children according to available space
    // Note: assume first item + separator + "..." always fits
    //
    //   get child widgets
    children   = gtk_container_get_children( GTK_CONTAINER(pathbar) );
    n_children = g_list_length(children);
    //
    //   show first pathbar item
    width = 0;
    index = 0;
    gp = g_list_nth_data(children, index);
    if (gp)
    {
        // get width
        child = GTK_WIDGET(gp);
        gtk_widget_size_request(child, &requisition);
        width = requisition.width;
        LOGPRINTF("index [%d] width [%d] total [%d]", index, requisition.width, width);

        // show widget
        gtk_widget_show(child);
    }
    //
    //   show first separator
    index++;
    gp = g_list_nth_data(children, index);
    if (gp)
    {
        // get width
        child = GTK_WIDGET(gp);
        gtk_widget_size_request(child, &requisition);
        width += spacing + requisition.width;
        LOGPRINTF("index [%d] width [%d] total [%d]", index, requisition.width, width);

        // show widget
        gtk_widget_show(child);
    }
    //
    //   show second pathbar item
    index++;
    gp = g_list_nth_data(children, 2);
    if (gp)
    {
        // get child widget
        child = GTK_WIDGET(gp);
        item2 = child;
        // get label inside child widget
        widget = gtk_bin_get_child( GTK_BIN(item2) );
        label  = GTK_LABEL(widget);

        // get current size
        gtk_widget_size_request(child, &requisition);
        width += spacing + requisition.width;
        LOGPRINTF("index [%d] width [%d] total [%d]", index, requisition.width, width);

        // reduce item when original does not fit
        if (width > allocation->width)
        {
            // item does not fit: allocate what's available
            //   show ... for the part that does not fit
            gtk_label_set_ellipsize(label, PANGO_ELLIPSIZE_END);
            //   adjust for old width
            width -= requisition.width;
            if (index + 1 == n_children)
            {
                // last item: reduce to available space
                gtk_widget_set_size_request(item2, allocation->width - width, -1);
                LOGPRINTF("item2 size request [%d]", allocation->width - width);
            }
            //   get new width
            gtk_widget_size_request(item2, &requisition);
            width += requisition.width;
            LOGPRINTF("index [%d] width [%d] total [%d]", index, requisition.width, width);

            // remember what we've done
            item2_reduced = TRUE;
        }

        // show item
        gtk_widget_show(child);
    }
    //   show further (separator + pathbar item) couples as far as these fit
    //   show only the last ones when all of them does not fit
    for ( index  = n_children - 1, list = g_list_last(children) ;
          index >= 3 ;
          index--,                 list = list->prev              )
    {
        // get child width
        child = GTK_WIDGET(list->data);
        gtk_widget_size_request(child, &requisition);
        width += spacing + requisition.width;
        LOGPRINTF("index [%d] width [%d] total [%d]", index, requisition.width, width);

        // reduce second pathbar item when needed
        if (   width > allocation->width
            && !item2_reduced            )
        {
            // adjust for old width
            gtk_widget_size_request(item2, &requisition);
            width -= requisition.width;
            // reduce item: set auto width + ellipsize
            gtk_widget_set_size_request(item2, -1, -1);
            LOGPRINTF("item2 size request [%d]", -1);
            widget = gtk_bin_get_child( GTK_BIN(item2) );
            label  = GTK_LABEL(widget);
            cp = "...";
            if ( strcmp( gtk_label_get_text(label), cp ) != 0 )
            {
                gtk_label_set_text(label, cp);
            }
            // get new width
            gtk_widget_size_request(item2, &requisition);
            width += requisition.width;
            // remember what we've done
            item2_reduced = TRUE;
            LOGPRINTF("index [%d] item2 width [%d] total [%d]", index, requisition.width, width);
        }

        // show or hide couple (separator + pathbar item) when it fits
        if (index % 2 != 0)
        {
            // separator: show or hide the couple (separator + pathbar item)
            widget = GTK_WIDGET(list->next->data);
            if (width <= allocation->width)
            {
                // fits: show them
                gtk_widget_show(child);
                gtk_widget_show(widget);
                LOGPRINTF("index [%d] show", index);
            }
            else if (index == n_children - 2)
            {
                // last pathbar item too large: reduce to available space
                //   adjust for old width
                gtk_widget_size_request(widget, &requisition);
                width -= requisition.width;
                //   reduce item: set available width + ellipsize
                gtk_widget_set_size_request(widget, allocation->width - width, -1);
                LOGPRINTF("item2 size request [%d]", allocation->width - width);
                label = GTK_LABEL( gtk_bin_get_child( GTK_BIN(widget) ));
                gtk_label_set_ellipsize(label, PANGO_ELLIPSIZE_END);
                //   get new width
                gtk_widget_size_request(widget, &requisition);
                width += requisition.width;
            }
            else
            {
                // too large: hide
                gtk_widget_hide(child);
                gtk_widget_hide(widget);
                LOGPRINTF("index [%d] hide", index);
            }
        }
    }
}


// on size-allocate for numboer-of-items displayed in title
static void on_num_items_size_allocate ( GtkWidget     *widget,
                                         GtkAllocation *allocation,
                                         gpointer      user_data   )
{
    gint            width;
    GtkRequisition  requisition;

    LOGPRINTF( "entry: x y [%d %d] width [%d] height [%d] user_data [%s]",
               allocation->x,
               allocation->y,
               allocation->width,
               allocation->height,
               (const char*) user_data );

    // adjust next-/prev-page bars
    //
    //   calculate width for right filler
    gtk_widget_size_request(g_prev_page_arrow, &requisition);
    width = allocation->width + (TITLE_SPACING / 2) - (requisition.width / 2);
    if (width < 0)
    {
        width = 0;
    }
    //   adjust next-/prev-page bar
    gtk_widget_set_size_request(g_next_page_filler, width, -1);
    gtk_widget_set_size_request(g_prev_page_filler, width, -1);
}


// check whether directory is allowed
gboolean fileview_is_directory_allowed (const gchar *dir)
{
    gboolean            ok = TRUE;  // return value

    LOGPRINTF("entry: dir [%s]", dir);
    g_assert(dir  && *dir != '\0');

    ok = is_directory_allowed(dir, NULL, NULL);

    return ok;
}


// check whether directory is allowed
// and determine screen title for directory
// [in]  dir - directory path
// [out] real_dir - directory path with all symlinks resolved
// [out] path_elements - a new GArray holding path_element_t elements
static gboolean is_directory_allowed (const gchar *dir, GString **real_dir, GArray **path_elements)
{
    gboolean            ok = TRUE;  // return value
    int                 rc;
    int                 n;
    char                *cp;
    char                *cp2;
    gboolean            done;
    char                path[PATH_MAX] = "";
    struct stat         statbuf;
    const dir_bw_list_t *bw_entry = NULL;
    const dir_bw_list_t *bw;
    gboolean            item_is_white;
    gboolean            is_white = FALSE;
    gboolean            is_black = FALSE;
    GArray              *path_arr = NULL;
    path_element_t      path_el;

    LOGPRINTF("entry: dir [%s]", dir);
    g_assert(dir                   && *dir           != '\0');
    g_assert(real_dir      == NULL || *real_dir      == NULL);
    g_assert(path_elements == NULL || *path_elements == NULL);

    // get real path
    cp = realpath(dir, path);
    if (cp != path)
    {
        ERRNOPRINTF("cannot determine realpath of [%s]", dir);
        ok = FALSE;
    }

    // check if this is a directory
    if (ok)
    {
        rc = stat(path, &statbuf);
        if (rc != 0)
        {
            ERRNOPRINTF("cannot stat [%s]", path);
            ok = FALSE;
        }
        else if ( !S_ISDIR(statbuf.st_mode) )
        {
            ERRNOPRINTF("not a directory [%s]", path);
            ok = FALSE;
        }
    }

    // check directory is white-listed and not black-listed
    if (ok)
    {
        // search directory black-and-white list
        // note: assumes path does not end with '/' for directories
        for (bw = DIR_BW_LIST ; bw->dir ; bw++)
        {
            n  = strlen(bw->dir);
            cp = path + n;
            if (   strncmp(path, bw->dir, n) == 0
                && (   *cp == '/'
                    || *cp == '\0' )              )
            {
                // match: is it black or white?
                if ( *cp == '\0' )
                {
                    // match on full path: check directory
                    item_is_white = bw->dir_is_white;
                }
                else
                {
                    // match on first part of path: check child
                    item_is_white = bw->child_is_white;
                }

                // does entry list the directory as black or white?
                if ( item_is_white )
                {
                    is_white = TRUE;
                    bw_entry = bw;
                }
                else
                {
                    is_black = TRUE;
                }
            }
        }

        // and the verdict is ...
        if ( is_black  ||  !is_white )
        {
            WARNPRINTF("directory [%s] not allowed", path);
            ok = FALSE;
        }
    }

    // set output parameters
    if (ok)
    {
        // set real path
        if (real_dir)
        {
            *real_dir = g_string_new( path );
        }

        // set path elements for title
        if (path_elements)
        {
            path_arr = g_array_new(TRUE, TRUE, sizeof(path_element_t));

            // first element always desktop
            path_el.title = g_string_new( _("Home") );
            path_el.path  = g_string_new( DIR_DESKTOP_INTERNAL );
            g_array_append_val(path_arr, path_el);
			LOGPRINTF("Added: title: Home path: %s", DIR_DESKTOP_INTERNAL);

            // main path element from bw list
            // note: title may be empty
            if ( bw_entry->title  &&  *(bw_entry->title) )
            {
                path_el.title = g_string_new( _(bw_entry->title) );
                path_el.path  = g_string_new(   bw_entry->dir    );
                g_array_append_val(path_arr, path_el);
				LOGPRINTF("Added: title: %s path: %s", bw_entry->title, bw_entry->dir);
            }

            // other path elements
            //   skip main path element
            done = FALSE;
            cp = path + strlen( bw_entry->dir );
            if (*cp)
            {
                cp++;
            }
            else
            {
                done = TRUE;
            }
            while ( !done )
            {
                // restrict path to this element
                cp2 = strchr(cp, '/');
                if (cp2)
                {
                    *cp2 = '\0';
                }

                // add element to array
                path_el.title = g_string_new( cp   );
                path_el.path  = g_string_new( path );
                g_array_append_val(path_arr, path_el);
				LOGPRINTF("Added: title: %s path: %s", cp, path);

                // restore path, prepare next element
                if (cp2)
                {
                    *cp2 = '/';
                    cp = cp2 + 1;
                }
                else
                {
                    done = TRUE;
                }
            }
        }
        LOGPRINTF( "return: ok [%d] real_dir [%s]", ok, real_dir ? (*real_dir)->str : "");
    }
    else
    {
        LOGPRINTF( "return: ok [%d]", ok);
    }

    // report our findings
    if (path_elements)
    {
        *path_elements = path_arr;
    }
    return ok;
}


// free an array of path elements, as generated by is_directory_allowed()
static void free_path_elements (GArray *path_arr)
{
    int                 i;
    path_element_t      *path_el;

    LOGPRINTF("entry: path_arr [%p]", path_arr);

    if (path_arr)
    {
        // free the elements
        path_el = (path_element_t*) path_arr->data;
        for (i = 0 ; i < path_arr->len ; i++)
        {
            // free strings
            if (path_el->title)
            {
                g_string_free(path_el->title, TRUE);
            }
            if (path_el->path)
            {
                g_string_free(path_el->path, TRUE);
            }

            // next element
            path_el++;
        }

        // free the array
        g_array_free(path_arr, TRUE);
    }
}


// create icon view
static GtkWidget* create_iconview (void)
{
    GtkWidget   *widget = NULL;   // return value
    GtkIconView *view   = NULL;

    LOGPRINTF("entry");

    widget = ergtk_icon_view_new_with_model( get_filemodel() );
    g_signal_connect(G_OBJECT(widget), "size-allocate",   G_CALLBACK(on_iconview_size_allocate),   NULL);
    g_signal_connect(G_OBJECT(widget), "item-activated",  G_CALLBACK(on_iconview_item_activated),  NULL);
    g_signal_connect(G_OBJECT(widget), "navigate-cursor", G_CALLBACK(on_iconview_navigate_cursor), NULL);
    gtk_widget_set_name(widget, "iconview-irex-ctb");
    view = GTK_ICON_VIEW(widget);
    ergtk_icon_view_set_focus_mode(ERGTK_ICON_VIEW(widget), TRUE, FALSE);
    gtk_icon_view_set_spacing(view, 8);    // pixels between icon and text

    // add columns
    //   filename
    gtk_icon_view_set_text_column(view, MODCOL_FILENAME_DISPLAY);
    //   thumbnail
    gtk_icon_view_set_pixbuf_column(view, MODCOL_THUMBNAIL);

    return widget;
}


// on size-allocate for icon view
static void on_iconview_size_allocate ( GtkWidget     *widget,
                                        GtkAllocation *allocation,
                                        gpointer      user_data   )
{
    LOGPRINTF("entry");

    update_view_size();
}


// on row-activated in icon view
static void on_iconview_item_activated ( GtkIconView *iconview,
                                         GtkTreePath *path,
                                         gpointer    user_data )
{
    LOGPRINTF("entry");

    (void) do_action_on_selected_items( CTB_ACTION_ACTIVATE );
}


// on navigate-cursor outside icon view
static void on_iconview_navigate_cursor ( erGtkIconView         *er_iconview,
                                          erGtkIconViewKeyPress keycode,
                                          gpointer              user_data )
{
    // old iconview details
    gint        old_row;
    gint        old_col;
    gint        num_rows;
    gint        num_cols;
    gint        num_items;

    // new iconview details
    gboolean    prev_page = FALSE;      // goto previous page
    gboolean    next_page = FALSE;      // goto next page
    gint        new_row = 0;
    gint        new_col = 0;

    LOGPRINTF("entry: keycode [%d]", keycode);

    // get iconview details
    ergtk_icon_view_get_cursor(er_iconview, &old_row, &old_col);
    ergtk_icon_view_get_view_size(er_iconview, &num_rows, &num_cols, &num_items);

    // determine new cursor position
    switch (keycode)
    {
        case ERGTK_ICON_VIEW_PRESS_SHORT_UP:
            // previous page, last row, same column
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_DOWN:
            // next page, first row, same column
            next_page = TRUE;
            new_row   = 0;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_LEFT:
            // previous page, last item
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = num_cols - 1;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_RIGHT:
            // next page, first item
            next_page = TRUE;
            new_row   = 0;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_UP:
            // previous page, same position
            prev_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_DOWN:
            // next page, same position
            next_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_LEFT:
            // previous page, start of last row
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_RIGHT:
            // next page, end of first row
            next_page = TRUE;
            new_row   = 0;
            new_col   = num_cols - 1;
            break;

        default:
            ERRORPRINTF("illegal erGtkIconViewKeyPress [%d]", keycode);
            ;  // ignore
    }

    // move page as needed, set new cursor position
    if ( prev_page  &&  g_has_prev_page )
    {
        filemodel_page_previous();
        set_cursor(new_row, new_col);
    }
    else if ( next_page  &&  g_has_next_page )
    {
        filemodel_page_next();
        set_cursor(new_row, new_col);
   }
}


// create list view
static GtkWidget* create_listview (void)
{
    GtkWidget           *ret         = NULL;    // return value
    GtkWidget           *widget      = NULL;
    GtkTreeView         *treeview    = NULL;
    erGtkListView       *er_listview = NULL;
    GtkTreeViewColumn   *column      = NULL;
    GtkTreeSelection    *selection   = NULL;
    GtkCellRenderer     *renderer    = NULL;

    LOGPRINTF("entry");

    widget = ergtk_list_view_new_with_model( get_filemodel() );
    g_signal_connect(G_OBJECT(widget), "size-allocate",   G_CALLBACK(on_listview_size_allocate),   NULL);
    g_signal_connect(G_OBJECT(widget), "row-activated",   G_CALLBACK(on_listview_row_activated),   NULL);
    g_signal_connect(G_OBJECT(widget), "navigate-cursor", G_CALLBACK(on_listview_navigate_cursor), NULL);
    gtk_widget_set_name(widget, "listview-irex-ctb");
    treeview = GTK_TREE_VIEW(widget);
    er_listview = ERGTK_LIST_VIEW(widget);
    selection = gtk_tree_view_get_selection(treeview);
    ergtk_list_view_set_focus_mode(er_listview, TRUE, FALSE);
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
    ret = widget;

    // add columns
    //
    //   icon
    renderer = gtk_cell_renderer_pixbuf_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 0.5,
                  "yalign", 0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( NULL,
                                                       renderer,
                                                       "pixbuf", MODCOL_THUMBNAIL,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  "expand", FALSE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);
    //
    //   filename
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  "ellipsize",     PANGO_ELLIPSIZE_END,
                  "ellipsize-set", TRUE,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILENAME_DISPLAY,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  "expand", TRUE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);
    //
    //   filetype
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILETYPE_DISPLAY,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  "expand", FALSE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);
    //
    //   author
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_AUTHOR,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  "expand", FALSE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);
    //
    //   filesize
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 1.0,
                  "yalign", 0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILESIZE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing",    GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  "alignment", 0.0,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);
    //
    //   filedate
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 0.5,
                  "yalign", 0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILEDATE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_AUTOSIZE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);


    return ret;
}


// set listview texts
static void listview_set_text (void)
{
    int                 i;
    GtkTreeViewColumn   *column = NULL;

    static const struct
    {
        char    *header;
        guint   index;
    }           column_headers[] =
                {
                    { N_("Name"),           2 },
                    { N_("Type"),           4 },
                    { N_("Author"),         6 },
                    { N_("Size"),           8 },
                    { N_("Date Modified"), 10 },
                    { NULL }  // end of list
                };

    LOGPRINTF("entry");

    // set listview column headers
    if (g_listview)
    {
        for ( i = 0 ; column_headers[i].header ; i++ )
        {
            column = gtk_tree_view_get_column( GTK_TREE_VIEW(g_listview), column_headers[i].index );
            if (column)
            {
                gtk_tree_view_column_set_title(column, _(column_headers[i].header));
            }
        }
    }
}


// on size-allocate for list view
static void on_listview_size_allocate ( GtkWidget     *widget,
                                        GtkAllocation *allocation,
                                        gpointer      user_data   )
{
    LOGPRINTF("entry");

    update_view_size();
}


// on row-activated in list view
static void on_listview_row_activated ( GtkTreeView       *view,
                                        GtkTreePath       *path,
                                        GtkTreeViewColumn *column,
                                        gpointer          user_data )
{
    LOGPRINTF("entry");

    (void) do_action_on_selected_items( CTB_ACTION_ACTIVATE );
}


// on navigate-cursor outside list view
static void on_listview_navigate_cursor ( erGtkListView         *er_listview,
                                          erGtkListViewKeyPress keycode,
                                          gpointer              user_data )
{
    // old listview details
    gint        old_row;
    gint        num_rows;
    gint        num_items;

    // new listview details
    gboolean    prev_page = FALSE;      // goto previous page
    gboolean    next_page = FALSE;      // goto next page
    gint        new_row = 0;


    LOGPRINTF("entry: keycode [%d]", keycode);

    // get listview details
    ergtk_list_view_get_cursor(er_listview, &old_row);
    ergtk_list_view_get_view_size(er_listview, &num_rows, &num_items);

    // determine new cursor position
    switch (keycode)
    {
        case ERGTK_LIST_VIEW_PRESS_SHORT_UP:
            // previous page, last row
            prev_page = TRUE;
            new_row   = num_rows - 1;
            break;

        case ERGTK_LIST_VIEW_PRESS_SHORT_DOWN:
            // next page, first row
            next_page = TRUE;
            new_row   = 0;
            break;

        case ERGTK_LIST_VIEW_PRESS_LONG_UP:
            // previous page, same position
            prev_page = TRUE;
            new_row   = old_row;
            break;

        case ERGTK_LIST_VIEW_PRESS_LONG_DOWN:
            // next page, same position
            next_page = TRUE;
            new_row   = old_row;
            break;

        default:
            ERRORPRINTF("illegal erGtkListViewKeyPress [%d]", keycode);
            ;  // ignore
    }

    // move page as needed, set new cursor position
    if ( prev_page  &&  g_has_prev_page )
    {
        filemodel_page_previous();
        set_cursor(new_row, 0);
    }
    else if ( next_page  &&  g_has_next_page )
    {
        filemodel_page_next();
        set_cursor(new_row, 0);
    }
}


// set filemodel number of items from current view
static void update_view_size ( void )
{
    int                 num_rows    = 1;
    int                 num_columns = 1;
    int                 num_items   = 1;
    ctb_viewtypes_t     current_view;
    erGtkIconView       *iconview;
    erGtkListView       *listview;

    LOGPRINTF("entry");
    g_assert(g_notebook);

    current_view = gtk_notebook_get_current_page(g_notebook);
    switch (current_view)
    {
        case CTB_ICONVIEW:
            if ( g_iconview  &&  GTK_WIDGET_VISIBLE(g_iconview) )
            {
                iconview = ERGTK_ICON_VIEW(g_iconview);
                ergtk_icon_view_get_view_size(iconview, &num_columns, &num_rows, NULL);
                num_items = num_rows * num_columns;
                LOGPRINTF("iconview: rows [%d] cols [%d] items [%d]", num_rows, num_columns, num_items);
                if (num_items > 0)
                {
                    filemodel_set_viewsize(num_items);
                }
            }
            else
            {
                LOGPRINTF("ignore - g_iconview not visible");
            }
            break;

        case CTB_LISTVIEW:
            if ( g_listview  &&  GTK_WIDGET_VISIBLE(g_listview) )
            {
                listview = ERGTK_LIST_VIEW(g_listview);
                ergtk_list_view_get_view_size(listview, &num_rows, &num_items);
                LOGPRINTF("listview: rows [%d] items [%d]", num_rows, num_items);
                if (num_rows > 0)
                {
                    filemodel_set_viewsize(num_rows);
                }
            }
            else
            {
                LOGPRINTF("ignore - g_listview not visible");
            }
            break;

        default:
            g_assert_not_reached();
    }
}


// open selected item(s)
void fileview_open_items (void)
{
    LOGPRINTF("entry");

    // activate selected item(s)
    (void) do_action_on_selected_items( CTB_ACTION_ACTIVATE );
}


// refresh screen data
void fileview_refresh (void)
{
    gchar       *cursor_item = NULL;

    LOGPRINTF("entry");

    // get current cursor position
    cursor_item = get_filename_at_cursor();

    // reiscan directory and position cursor at the same item
    fileview_show_dir(g_current_dir->str, cursor_item);

    // clean up
    g_free(cursor_item);
}


// delete selected item(s)
void fileview_delete_items (void)
{
    GtkTreePath *tree_path = NULL;
    gint        offset = 0;
    gint        item   = 0;
    gint        *pi;
    gboolean    ok = TRUE;

    LOGPRINTF("entry");

    // get current cursor position
    tree_path = get_cursor_tree_path();
    if (tree_path)
    {
        // convert to database offset
        pi = gtk_tree_path_get_indices( tree_path );
        if (pi)
        {
            item = *pi;
        }
        offset = filemodel_get_offset( item );
        if (offset < 0)
        {
            // cursor item not in database, probably parent directory
            ok = FALSE;
        }

        if (ok)
        {
            // delete selected items
            (void) do_action_on_selected_items( CTB_ACTION_DELETE );

            // reposition cursor to the next item
            set_cursor_at_item( offset );
        }
    }

    // clean up
    if (tree_path) { gtk_tree_path_free(tree_path); }
}


// create shortcut for selected item(s)
void fileview_create_shortcut (void)
{
    LOGPRINTF("entry");

    // create shortcut(s)
    (void) do_action_on_selected_items( CTB_ACTION_SHORTCUT );
}


// execute the specified action on the selected item(s)
static int do_action_on_selected_items (const char *action_name)
{
    int                     ret = ER_OK;    // return code
    int                     rc;
    int                     len;
    gboolean                found;
    const GString           *dir;
    ctb_viewtypes_t         current_view;
    const ctb_action_t      *action    = NULL;
    GtkIconView             *iconview  = NULL;
    GtkTreeView             *treeview  = NULL;
    GtkTreeSelection        *tree_sel  = NULL;
    GtkTreePath             *tree_path = NULL;
    GList                   *selected_items = NULL;
    GSList                  *filelist       = NULL;
    const filelist_entry_t  *filelist_entry = NULL;  // entry in filelist
    const GSList            *filelist_iter  = NULL;  // iterator over filelist

    LOGPRINTF("entry: action [%s]", action_name);

    // find action in list
    found = FALSE;
    for (action = CTB_ACTIONS ; action->name ; action++)
    {
        if ( strcmp(action_name, action->name) == 0 )
        {
            found = TRUE;
            break;  // exit for
        }
    }
    if ( !found )
    {
        ERRORPRINTF("unknown action [%s]", action_name);
        ret = ER_FAIL;
    }

    // get selected items
    if (ret == ER_OK)
    {
        current_view = gtk_notebook_get_current_page(g_notebook);
        switch (current_view)
        {
            case CTB_ICONVIEW:
                g_assert(g_iconview);
                iconview = GTK_ICON_VIEW(g_iconview);
                selected_items = gtk_icon_view_get_selected_items(iconview);
                break;
            case CTB_LISTVIEW:
                g_assert(g_listview);
                treeview = GTK_TREE_VIEW(g_listview);
                tree_sel = gtk_tree_view_get_selection(treeview);
                selected_items = gtk_tree_selection_get_selected_rows(tree_sel, NULL);
                break;
            default:
                g_assert_not_reached();
        }
    }

    // reduce to single selection when needed
    if (   ret == ER_OK
        && action->is_multi_select == FALSE )
    {
        // cancel select-all, if set
        g_do_select_all = FALSE;

        // reduce selection to a single item, if needed
        len = g_list_length(selected_items);
        if (len == 0)
        {
            WARNPRINTF("no selected items, action [%s] is void", action_name);
        }
        else if (len > 1)
        {
            // forget currently selected items
            treepath_list_free(&selected_items);
            selected_items = NULL;

            // select only the item which has the cursor (focus)
            switch (current_view)
            {
                case CTB_ICONVIEW:
                    gtk_icon_view_get_cursor(iconview, &tree_path, NULL);
                    g_assert(tree_path);
                    gtk_icon_view_unselect_all(iconview);
                    gtk_icon_view_select_path(iconview, tree_path);
                    selected_items = gtk_icon_view_get_selected_items(iconview);
                    break;

                case CTB_LISTVIEW:
                    gtk_tree_view_get_cursor(treeview, &tree_path, NULL);
                    g_assert(tree_path);
                    gtk_tree_selection_unselect_all(tree_sel);
                    gtk_tree_selection_select_path(tree_sel, tree_path);
                    selected_items = gtk_tree_selection_get_selected_rows(tree_sel, NULL);
                    break;

                default:
                    g_assert_not_reached();
            }

            // clean up
            if (tree_path)
            {
                gtk_tree_path_free(tree_path);
                tree_path= NULL;
            }
        }
    }

    // get filenames to perform action on
    if (ret == ER_OK)
    {
        if (g_do_select_all)
        {
            filelist = filemodel_get_filenames_all();
        }
        else
        {
            filelist = filemodel_get_filenames_from_treepath_list(selected_items);
        }

        if (filelist == NULL)
        {
            ERRORPRINTF("cannot get filenames from model");
            ret = ER_FAIL;
        }
    }

    // now DO the action
    //   pre-callback
    if (   ret == ER_OK
        && action->pre_callback)
    {
        rc = action->pre_callback(g_current_dir, filelist);
        if (rc != ER_OK)
        {
            ERRORPRINTF("pre-callback [%s] [%s] failed with error [%d]", action_name, g_current_dir->str, rc);
            ret = rc;
        }
    }
    //   item_callback per item
    if (   ret == ER_OK
        && action->item_callback)
    {
        for (filelist_iter = filelist ; filelist_iter  &&  ret == ER_OK ; filelist_iter = filelist_iter->next)
        {
            filelist_entry = (const filelist_entry_t*) filelist_iter->data;

            // call the item callback
            dir = filelist_entry->directory_path;
            if ( strcmp(dir->str, ".") == 0 )
            {
                dir = g_current_dir;
            }
            rc = action->item_callback( dir, filelist_entry );
            if (rc != ER_OK)
            {
                ERRORPRINTF( "item-callback [%s] [%s] failed with error [%d]",
                             action_name, filelist_entry->filename->str, rc   );
                ret = rc;
            }
        }
    }
    //   post-callback
    if (   ret == ER_OK
        && action->post_callback)
    {
        rc = action->post_callback(g_current_dir, filelist);
        if (rc != ER_OK)
        {
            ERRORPRINTF("post-callback [%s] [%s] failed with error [%d]", action_name, g_current_dir->str, rc);
            ret = rc;
        }
    }

    // clean up
    treepath_list_free(&selected_items);
    filemodel_filelist_free(filelist);

    return ret;
}


// free list of GtkTreePath elements
static void treepath_list_free (GList **list)
{
    LOGPRINTF("entry");

    if (list  && *list)
    {
        g_list_foreach(*list, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(*list);
        *list = NULL;
    }
}


// previous-page bar clicked
static gboolean on_prev_page_button_press (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    gint        row    = 0;
    gint        column = 0;

    LOGPRINTF("entry");

    if (event->type == GDK_BUTTON_PRESS)
    {
        get_cursor(&row, &column);
        filemodel_page_previous();
        set_cursor(row, column);
    }

    return FALSE;
}


// next-page bar clicked
static gboolean on_next_page_button_press (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    gint        row    = 0;
    gint        column = 0;

    LOGPRINTF("entry");

    if (event->type == GDK_BUTTON_PRESS)
    {
        get_cursor(&row, &column);
        filemodel_page_next();
        set_cursor(row, column);
    }

    return FALSE;
}


// get current cursor position
static GtkTreePath* get_cursor_tree_path ( void )
{
    GtkTreePath             *tree_path = NULL;  // return value
    gboolean                ok;
    ctb_viewtypes_t         current_view;
    GtkIconView             *iconview  = NULL;
    GtkTreeView             *treeview  = NULL;

    LOGPRINTF("entry");
    g_assert(g_notebook);

    // get tree-path for cursor item, if any
    current_view = gtk_notebook_get_current_page(g_notebook);
    switch (current_view)
    {
        case -1:
            WARNPRINTF("g_notebook page not set");
            break;

        case CTB_ICONVIEW:
            if (g_iconview)
            {
                iconview = GTK_ICON_VIEW(g_iconview);
                ok = gtk_icon_view_get_cursor(iconview, &tree_path, NULL);
                if ( !ok  &&  tree_path )
                {
                    gtk_tree_path_free( tree_path );
                    tree_path = NULL;
                }
            }
            break;

        case CTB_LISTVIEW:
            if (g_listview)
            {
                treeview = GTK_TREE_VIEW(g_listview);
                gtk_tree_view_get_cursor(treeview, &tree_path, NULL);
            }
            break;

        default:
            g_assert_not_reached();
    }

    return tree_path;
}


// get filename for current cursor item
static gchar* get_filename_at_cursor ( void )
{
    gchar                   *filename = NULL;  // return value
    GtkTreePath             *tree_path = NULL;
    GList                   *item_list = NULL;
    GSList                  *filelist       = NULL;
    const filelist_entry_t  *filelist_entry = NULL;  // entry in filelist

    LOGPRINTF("entry");
    g_assert(g_notebook);

    // get details for cursor item, if any
    tree_path = get_cursor_tree_path();
    if (tree_path)
    {
        item_list = g_list_append(item_list, tree_path);

        filelist = filemodel_get_filenames_from_treepath_list( item_list );
        if (filelist == NULL)
        {
            ERRORPRINTF("cannot get filenames from model");
        }
    }

    // extract filename
    if (filelist)
    {
        filelist_entry = (const filelist_entry_t*) filelist->data;
        if (   filelist_entry
            && filelist_entry->filename
            && filelist_entry->filename->len > 0 )
        {
            filename = g_strdup( filelist_entry->filename->str );
            LOGPRINTF("filename [%s]", filename);
        }
    }

    // clean up
    filemodel_filelist_free(filelist);
    treepath_list_free(&item_list);

    LOGPRINTF("leave: filename [%s]", filename);
    return filename;
}


// set cursor at requested item by filename
void fileview_set_cursor_at_filename ( const gchar *filename )
{
    gchar       *cp = NULL;

    LOGPRINTF("entry: filename [%s]", filename);

    if (filename  &&  *filename)
    {
        cp = g_strdup(filename);
        g_idle_add(on_idle_set_cursor_at_filename, cp);
    }
}

static gboolean on_idle_set_cursor_at_filename ( gpointer data )
{
    gchar           *filename = (gchar*)data;
    gint            row   = 0;
    gint            col   = 0;
    gint            item  = 0;
    gint            n_cols;
    erGtkIconView   *iconview = NULL;
    ctb_viewtypes_t current_view;

    LOGPRINTF("entry: filename [%s]", filename);
    g_assert(g_notebook);

    // load requested page in model
    item = filemodel_scroll_to_filename( filename );

    // set cursor to requested item
    if (item > 0)
    {
        current_view = gtk_notebook_get_current_page(g_notebook);
        switch (current_view)
        {
            case -1:
                WARNPRINTF("g_notebook page not set");
                break;
            case CTB_ICONVIEW:
                g_assert(g_iconview);
                iconview = ERGTK_ICON_VIEW(g_iconview);
                ergtk_icon_view_get_view_size(iconview, NULL, &n_cols, NULL);
                row = item / n_cols;
                col = item % n_cols;
                break;
            case CTB_LISTVIEW:
                row = item;
                col = 0;
                break;
            default:
                g_assert_not_reached();
        }
    }
    set_cursor(row, col);

    // clean up
    g_free(filename);

    return FALSE;  // don't call me again
}


// set cursor at requested item by offset
static void set_cursor_at_item ( const gint offset )
{
    LOGPRINTF("entry: offset [%d]", offset);

    g_idle_add( on_idle_set_cursor_at_item, (gpointer)offset );
}

static gboolean on_idle_set_cursor_at_item ( gpointer data )
{
    gint            offset = (gint)data;
    gint            row   = 0;
    gint            col   = 0;
    gint            item  = 0;
    gint            n_cols;
    erGtkIconView   *iconview = NULL;
    ctb_viewtypes_t current_view;

    LOGPRINTF("entry: offset [%d]", offset);
    g_assert(g_notebook);

    // load requested page in model
    item = filemodel_scroll_to_offset( offset );

    // set cursor to requested item
    if (item > 0)
    {
        current_view = gtk_notebook_get_current_page(g_notebook);
        switch (current_view)
        {
            case -1:
                WARNPRINTF("g_notebook page not set");
                break;
            case CTB_ICONVIEW:
                g_assert(g_iconview);
                iconview = ERGTK_ICON_VIEW(g_iconview);
                ergtk_icon_view_get_view_size(iconview, NULL, &n_cols, NULL);
                row = item / n_cols;
                col = item % n_cols;
                break;
            case CTB_LISTVIEW:
                row = item;
                col = 0;
                break;
            default:
                g_assert_not_reached();
        }
    }
    set_cursor(row, col);

    return FALSE;  // don't call me again
}


// registry access: get viewtype
ctb_viewtypes_t get_viewtype_from_registry ( void )
{
    ctb_viewtypes_t     view      = CTB_AUTOSELECT;   // return value
    GConfClient         *client   = NULL;
    const device_caps_t *dev_caps = NULL;
    gchar               *val;

    LOGPRINTF("entry");

    // get registry value
    client = gconf_client_get_default();
    val = gconf_client_get_string(client, REGKEY_VIEWTYPE, NULL);
    g_object_unref(client);

    // convert string value to viewtype
    if (val)
    {
        if ( strcmp(val, REGVAL_VIEWTYPE_ICON) == 0 )
        {
            view = CTB_ICONVIEW;
        }
        else if ( strcmp(val, REGVAL_VIEWTYPE_LIST) == 0 )
        {
            view = CTB_LISTVIEW;
        }
        g_free(val);
    } else {
        WARNPRINTF("error fetching GConf key %s", REGKEY_VIEWTYPE);
    }

    // for auto-select choose view depending on stylus availability
    if (view == CTB_AUTOSELECT)
    {
        dev_caps = ipc_sys_get_device_capabilities();
        g_assert(dev_caps);
        if (dev_caps->has_stylus)
        {
            view = CTB_ICONVIEW;
        }
        else
        {
            view = CTB_LISTVIEW;
        }
    }

    return view;
}


// registry access: set viewtype
void save_viewtype_to_registry ( ctb_viewtypes_t view )
{
    gboolean     ok;
    GConfClient  *client  = NULL;
    gchar        *val_old = NULL;
    const gchar  *val_new = NULL;
    GError       *err     = NULL;

    LOGPRINTF("entry");

    // convert viewtype to string value
    switch (view)
    {
        case CTB_ICONVIEW:
            val_new = REGVAL_VIEWTYPE_ICON;
            break;
        case CTB_LISTVIEW:
            val_new = REGVAL_VIEWTYPE_LIST;
            break;
        default:
            val_new = REGVAL_VIEWTYPE_AUTO;
    }

    // save to registry
    client = gconf_client_get_default();
    val_old = gconf_client_get_string(client, REGKEY_VIEWTYPE, NULL);
    if (   val_old == NULL
        || strcmp(val_old, val_new) != 0 )
    {
        ok = gconf_client_set_string(client, REGKEY_VIEWTYPE, val_new, &err);
        if ( !ok )
        {
            ERRORPRINTF("cannot write registry key [%s] - error [%s]", REGKEY_VIEWTYPE, err->message);
        }
        g_clear_error(&err);
    }
    g_object_unref(client);

    // clean up
    g_free(val_old);
}


static gboolean update_display(gpointer data)
{
    display_update_return_control((gint) data);
    // don't call again, thanks
    return FALSE;
}


static void idle_update_display(gint type)
{    
    g_idle_add(update_display, (gpointer) type);
}
