/*
 * File Name: filetypes.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gconf/gconf-client.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <string.h>

// local include files, between " "
#include "ctb_log.h"
#include "filetypes.h"
#include "i18n.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// detailed info per filetype
typedef struct
        {
            gchar       *extension;             // file extension excluding the '.'
            gchar       *reference_ext;         // extension from which to use this details
            gboolean    is_directory;           // extension used for directory or file
            gchar       *icon_name;             // name of the icon, also used for bitmap filename
            gchar       *description;           // type description for this extension
            gchar       *descr_template;        // type description when used as a template
            gchar       *viewer_cmd;            // command to start the corresponding viewer application
            GdkPixbuf   *pixbuf_icon    [N_FILEMODEL_THUMB_SIZES];  // cache for bitmaps loaded from icon files
            GdkPixbuf   *pixbuf_template[N_FILEMODEL_THUMB_SIZES];  // cache for bitmaps loaded from icon files
        } filetype_info_t;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const gchar      *ICONS_DIRECTORY = DATADIR;
static const gchar      *ICONFILE_PREFIX = "icon-";
static const gchar      *ICONFILE_POSTFIX[ N_FILEMODEL_THUMB_SIZES ] =
                        {
                            "-mini.png",
                            "-small.png",
                            "-medium.png",
                            "-large.png"
                        };

static const gchar      *ICONNAME_PARENTDIR   = "parent-dir";
static const gchar      *ICONNAME_APPLICATION = "application";
static const gchar      *ICONNAME_LIBRARY     = "library";
static const gchar      *ICONNAME_TEMPLATE    = "overlay-template";
static const gchar      *ICONNAME_SHORTCUT    = "overlay-shortcut";

static const gchar      *REGKEY_FILETYPE_DIR = "/apps/er/sys/ctb/filetypes";

// known filetypes
static filetype_info_t  g_hardcoded_filetypes[] =
                        {
                            // ext          ref    is_dir   icon_name    description                  viewer_cmd
                            //                                           descr_template                      
                            { ""         ,  NULL , TRUE ,  "folder"    , N_("Folder")              ,
                                                                         N_("Template")            ,  NULL         },
                            { ""         ,  NULL , FALSE,  "unknown"   , N_("Unknown")             ,
                                                                         N_("Unknown")             ,  NULL         },
                            { FILE_EXT_SHORTCUT
                                         ,  NULL , FALSE,  "unknown"   , N_("Shortcut")            ,
                                                                         N_("Invalid")             ,  NULL         },
                            { FILE_EXT_SHORTCUT_TO_DIR
                                         ,  NULL , FALSE,  "folder"    , N_("Shortcut")            ,
                                                                         N_("Invalid")             ,  NULL         },
                            { NULL }  // end of list
                        };

// hidden files and folders, based on file-/foldername
static const char       *HIDDEN_FILE_PREFIXES[] =
                        {
                            ".",
                            "_",
                            NULL
                        };
static const char       *HIDDEN_FILENAMES[] =
                        {
                            ERMETADB_DATABASE_FILE,
                            ERMETADB_DATABASE_FILE "-journal",
                            ERMETADB_DATABASE_FILE ".old",
                            FILENAME_WIN_AUTORUN,
                            FILENAME_WIN_THUMBNAILS,
                            NULL
                        };
static const char       *HIDDEN_FILE_EXTENSIONS[] =
                        {
                            "mbp",      // mobipocket metadata
                            NULL
                        };

// Mobipocket magic values
#define MOBIPOCKET_CHECK1_OFFSET            0x3C
#define MOBIPOCKET_FIRST_RECORD             0x4E
#define MOBIPOCKET_CHECK2_OFFSET            0x10
#define MOBIPOCKET_ENCODING_OFFSET          0x1C
#define MOBIPOCKET_CHECK3_OFFSET            0x24
#define MOBIPOCKET_TITLE_RECORD_OFFSET      0x54
#define MOBIPOCKET_TITLE_LENGTH_OFFSET      0x58
#define MOBIPOCKET_PDB_TITLE_OFFSET         0x00
#define MOBIPOCKET_PDB_TITLE_LENGTH         0x20

#define MOBIPOCKET_ENCODING_UTF8            0xFDE9
#define MOBIPOCKET_ENCODING_WIN1252         0x04E4

#define GCONF_BUFFER_SIZE                   128

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GArray           *g_filetype_info = NULL;

static GdkPixbuf        *g_icon_parent_dir [N_FILEMODEL_THUMB_SIZES];
static GdkPixbuf        *g_icon_application[N_FILEMODEL_THUMB_SIZES];
static GdkPixbuf        *g_icon_library    [N_FILEMODEL_THUMB_SIZES];
static GdkPixbuf        *g_icon_template   [N_FILEMODEL_THUMB_SIZES];
static GdkPixbuf        *g_icon_shortcut   [N_FILEMODEL_THUMB_SIZES];


//============================================================================
// Local Function Definitions
//============================================================================

static void             apply_icon_overlay      ( const filemodel_thumbsize_t icon_size,
                                                        GdkPixbuf             *icon,
                                                  const gchar                 *overlay_name,
                                                        GdkPixbuf             **overlay_bitmaps );

static filetype_info_t* find_filetype_info      ( const gchar                 *file_ext,
                                                  const gboolean              is_directory,
                                                  const gboolean              with_default );

static GdkPixbuf*       get_icon_impl           ( const gchar                 *icon_name,
                                                  const filemodel_thumbsize_t icon_size,
                                                        GdkPixbuf             **pixbuf_cache );

static void             init_filetype_info      ( void );


//============================================================================
// Functions Implementation
//============================================================================

static gchar *read_gconf_value(const gchar *dir,
                               const gchar *key,
                               GConfClient *client)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    gchar* value;
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);
    value = gconf_client_get_string(client, buffer, NULL);
    if (value == NULL) {
        ERRORPRINTF("gconf: cannot read %s\n", buffer);
        return NULL;
    }
    return value;
}


static void add_filetype_from_gconf(const gchar *dir,
                                    GConfClient *client)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    gchar *descr = NULL;
    gchar *template_descr = NULL;
    gchar *viewer_cmd = NULL;
    gchar *icon_name = NULL;
    gchar *reference_ext = NULL;
    GSList *ext = NULL;

    descr = read_gconf_value(dir, "descr", client);
    if (!descr) goto out_error;

    template_descr = read_gconf_value(dir, "template_descr", client);
    if (!template_descr) goto out_error;

    viewer_cmd = read_gconf_value(dir, "viewer_cmd", client);
    if (!viewer_cmd) goto out_error;

    icon_name = read_gconf_value(dir, "icon_name", client);
    if (!icon_name) goto out_error;

    g_assert(strlen(dir) + strlen("/extensions") < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/extensions", dir);
    GSList *extensions = gconf_client_get_list(client, buffer, GCONF_VALUE_STRING, NULL);
    if (extensions == NULL) {
        ERRORPRINTF("gconf: cannot read %s\n", buffer);
        goto out_error;
    }

    for (ext = extensions; ext; ext = g_slist_next(ext)) {
        if (find_filetype_info(ext->data, FALSE, FALSE) != NULL) {
            ERRORPRINTF("gconf: ext %s is already registered, skipping\n", (gchar*)ext->data);
            continue;
        }

        filetype_info_t newtype;
        memset(&newtype, 0x00, sizeof(newtype));
        newtype.extension       = g_strdup(ext->data);
        newtype.reference_ext   = g_strdup(reference_ext);
        newtype.is_directory    = FALSE;
        newtype.icon_name       = g_strdup(icon_name);
        newtype.description     = g_strdup(descr);
        newtype.descr_template  = g_strdup(template_descr);
        newtype.viewer_cmd      = g_strdup(viewer_cmd);
        g_array_append_val(g_filetype_info, newtype);

        if (reference_ext == NULL) reference_ext = ext->data;
    }
    g_slist_foreach (extensions, (GFunc)g_free, NULL);
    g_slist_free(extensions);
out_error:
    g_free(icon_name);
    g_free(viewer_cmd);
    g_free(template_descr);
    g_free(descr);
}


static void add_filetypes_from_gconf(void)
{
    GSList *types = NULL;
    GConfClient *client = gconf_client_get_default();

    gboolean exists = gconf_client_dir_exists(client, REGKEY_FILETYPE_DIR, NULL);
    if (exists == FALSE) {
        ERRORPRINTF("gconf: %s does not exist\n", REGKEY_FILETYPE_DIR);
        goto unref;
    }

    types = gconf_client_all_dirs(client, REGKEY_FILETYPE_DIR, NULL);
    if (types == NULL) {
        ERRORPRINTF("gconf: cannot read dir entries of %s\n", REGKEY_FILETYPE_DIR);
        goto unref;
    }

    g_slist_foreach (types, (GFunc)add_filetype_from_gconf, client);
    g_slist_foreach (types, (GFunc)g_free, NULL);
    g_slist_free(types);
unref:
    g_object_unref(client);
}


static void add_hardcoded_filetypes(void)
{
    filetype_info_t   *known_type = NULL;
    filetype_info_t   details;

    // fill array with hardcoded filetypes
    for ( known_type = g_hardcoded_filetypes ; known_type->extension ; known_type++ )
    {
        // set details
        memset(&details, 0x00, sizeof(details));
        details.extension       = g_strdup(known_type->extension     );
        details.reference_ext   = g_strdup(known_type->reference_ext );
        details.is_directory    = known_type->is_directory;
        details.icon_name       = g_strdup(known_type->icon_name     );
        details.description     = g_strdup(known_type->description   );
        details.descr_template  = g_strdup(known_type->descr_template);
        details.viewer_cmd      = g_strdup(known_type->viewer_cmd    );

        // add to array
        g_array_append_val(g_filetype_info, details);
    }
}


// initialise our local filetype info
static void init_filetype_info ( void )
{
    if (g_filetype_info)
    {
        // already initialised: ignore
        return;
    }
    LOGPRINTF("entry");

    g_filetype_info = g_array_new( TRUE, TRUE, sizeof(filetype_info_t) );
    g_assert(g_filetype_info);

    add_hardcoded_filetypes();
    add_filetypes_from_gconf();
}


// find filetype details
static filetype_info_t* find_filetype_info ( const gchar     *file_ext,
                                             const gboolean  is_directory,
                                             const gboolean  with_default )
{
    filetype_info_t *ret = NULL;      // return value
    filetype_info_t *details;
    gchar           *ref_ext;
    gboolean        done;
    gboolean        found;

    LOGPRINTF("entry: file_ext [%s] is_dir [%d]", file_ext, is_directory);
    if (file_ext == NULL)
    {
        file_ext = "";
    }

    // find filetype info
    init_filetype_info();
    done  = FALSE;
    found = FALSE;
    while ( !done )
    {
        details = (filetype_info_t*) g_filetype_info->data;
        while ( details->extension  &&  !found )
        {
            if (   details->is_directory == is_directory 
                && g_ascii_strcasecmp( details->extension, file_ext ) == 0 )
            {
                ret = details;
                found = TRUE;
            }
            else
            {
                details++;
            }
        }

        if ( found )
        {
            ref_ext = details->reference_ext;
            LOGPRINTF( "found filetype [%s] ref [%s] icon_name [%s] descr [%s]",
                       file_ext,
                       ref_ext,
                       details->icon_name,
                       details->description );
            if (ref_ext)
            {
                ret = find_filetype_info( ref_ext, is_directory, with_default );
            }
            done = TRUE;
        }
        else if ( *file_ext )
        {
            LOGPRINTF("unkown filetype [%s] is_dir [%d]", file_ext, is_directory);
            if (with_default)
            {
                file_ext = "";
            }
            else
            {
                done = TRUE;
            }
        }
        else
        {
            ERRORPRINTF("no default fileinfo for is_dir [%d]", is_directory);
            done = TRUE;
        }
    }

    // return filetype details, if found
    return ret;
}


// get icon for specified filename
GdkPixbuf* get_icon_from_file_extension ( const gchar                 *file_ext,
                                          const gboolean              is_directory,
                                          const gboolean              is_template,
                                          const filemodel_thumbsize_t icon_size     )
{
    GdkPixbuf           *icon               = NULL;   // return value
    gchar               *icon_file          = NULL;
    GdkPixbuf           **p_pixbuf_icon     = NULL;
    GdkPixbuf           **p_pixbuf_template = NULL;
    filetype_info_t     *details            = NULL;
    GError              *err                = NULL;

    LOGPRINTF("entry: file_ext [%s] is_dir [%d]", file_ext, is_directory);
    g_return_val_if_fail( (icon_size >= 0  &&  icon_size < N_FILEMODEL_THUMB_SIZES), NULL );


    // find filetype info
    details = find_filetype_info(file_ext, is_directory, TRUE);

    // load icon when not yet done
    if (details)
    {
        // get normal icon
        p_pixbuf_icon = &(details->pixbuf_icon[icon_size]);
        icon          = *p_pixbuf_icon;
        if ( icon == NULL )
        {
            // load filetype icon from file
            icon_file = g_strdup_printf( "%s/%s%s%s",
                                         ICONS_DIRECTORY,
                                         ICONFILE_PREFIX,
                                         details->icon_name,
                                         ICONFILE_POSTFIX[icon_size] );
            icon = gdk_pixbuf_new_from_file( icon_file, &err );
            if (icon)
            {
                *p_pixbuf_icon = icon;
            }
            else
            {
                ERRORPRINTF("cannot load iconfile [%s] error [%s]", icon_file, err->message);
                g_clear_error(&err);
            }
        }

        // apply template overlay, if needed
        if (icon  &&  is_template)
        {
            p_pixbuf_template = &(details->pixbuf_template[icon_size]);
            if (*p_pixbuf_template)
            {
                icon = *p_pixbuf_template;
            }
            else
            {
                icon = gdk_pixbuf_copy(icon);
                apply_icon_overlay_template( icon_size, icon);
                *p_pixbuf_template = icon;
            }
        }
    }

    // clean up
    if (icon_file) { g_free(icon_file) ; icon_file = NULL; }

    // return the icon, if found
    return icon;
}


// get icon file for specified filename
gchar* get_icon_file_from_file_extension ( const gchar                 *file_ext,
                                           const gboolean              is_directory,
                                           const filemodel_thumbsize_t icon_size     )
{
    gchar               *icon_file = NULL;      // return value
    filetype_info_t     *details;

    LOGPRINTF("entry: file_ext [%s] is_dir [%d]", file_ext, is_directory);
    g_return_val_if_fail( (icon_size >= 0  &&  icon_size < N_FILEMODEL_THUMB_SIZES), NULL );


    // find filetype info
    details = find_filetype_info(file_ext, is_directory, TRUE);

    // determine icon filename
    if (details)
    {
        icon_file = g_strdup_printf( "%s/%s%s%s",
                                     ICONS_DIRECTORY,
                                     ICONFILE_PREFIX,
                                     details->icon_name,
                                     ICONFILE_POSTFIX[icon_size] );
    }

    // return the icon file, if found
    return icon_file;
}


// get icon for parent directory
GdkPixbuf* get_icon_parent_dir ( const filemodel_thumbsize_t icon_size )
{
    GdkPixbuf  *icon = NULL;   // return value

    LOGPRINTF("entry");

    icon = get_icon_impl( ICONNAME_PARENTDIR, icon_size, g_icon_parent_dir );

    return icon;
}


// get icon for application
GdkPixbuf* get_icon_application ( const filemodel_thumbsize_t icon_size )
{
    GdkPixbuf  *icon = NULL;   // return value

    LOGPRINTF("entry");

    icon = get_icon_impl( ICONNAME_APPLICATION, icon_size, g_icon_application );

    return icon;
}


// get icon file for application
gchar* get_icon_file_application ( const filemodel_thumbsize_t icon_size )
{
    gchar       *icon_file = NULL;      // return value

    LOGPRINTF("entry");

    icon_file = g_strdup_printf( "%s/%s%s%s",
                                 ICONS_DIRECTORY,
                                 ICONFILE_PREFIX,
                                 ICONNAME_APPLICATION,
                                 ICONFILE_POSTFIX[icon_size] );

    return icon_file;
}


// get icon for library (SD card)
GdkPixbuf* get_icon_library ( const filemodel_thumbsize_t icon_size )
{
    GdkPixbuf  *icon = NULL;   // return value

    LOGPRINTF("entry");

    icon = get_icon_impl( ICONNAME_LIBRARY, icon_size, g_icon_library );

    return icon;
}


// get icon
static GdkPixbuf* get_icon_impl ( const gchar                 *icon_name,
                                  const filemodel_thumbsize_t icon_size,
                                        GdkPixbuf             **pixbuf_cache )
{
    GdkPixbuf           *icon      = NULL;   // return value
    GdkPixbuf           **p_icon   = NULL;
    gchar               *icon_file = NULL;
    GError              *err       = NULL;

    LOGPRINTF("entry: icon_name [%s] icon_size [%d]", icon_name, icon_size);
    g_assert(icon_name && *icon_name);

    // load icon when not yet done
    p_icon = pixbuf_cache + icon_size;
    icon = *p_icon;
    if ( icon == NULL )
    {
        // load filetype icon from file
        icon_file = g_strdup_printf( "%s/%s%s%s",
                                     ICONS_DIRECTORY,
                                     ICONFILE_PREFIX,
                                     icon_name,
                                     ICONFILE_POSTFIX[icon_size] );
        icon = gdk_pixbuf_new_from_file( icon_file, &err );
        if (icon)
        {
            *p_icon = icon;
        }
        else
        {
            ERRORPRINTF("cannot load iconfile [%s] error [%s]", icon_file, err->message);
            g_clear_error(&err);
        }
    }

    // clean up
    if (icon_file) { g_free(icon_file) ; icon_file = NULL; }

    // return the icon, if found
    return icon;
}


// add 'template' overlay to icon bitmap
void apply_icon_overlay_template ( const filemodel_thumbsize_t icon_size,
                                         GdkPixbuf             *icon     )
{
    LOGPRINTF("entry");

    apply_icon_overlay( icon_size, icon, ICONNAME_TEMPLATE, g_icon_template );
}


// add 'shortcut' overlay to icon bitmap
void apply_icon_overlay_shortcut ( const filemodel_thumbsize_t icon_size,
                                         GdkPixbuf             *icon     )
{
    LOGPRINTF("entry");

    apply_icon_overlay( icon_size, icon, ICONNAME_SHORTCUT, g_icon_shortcut );
}


// add overlay to icon bitmap
static void apply_icon_overlay ( const filemodel_thumbsize_t icon_size,
                                       GdkPixbuf             *icon,
                                 const gchar                 *overlay_name,
                                       GdkPixbuf             **overlay_bitmaps )
{
    GdkPixbuf           *overlay      = NULL;
    GdkPixbuf           **p_overlay   = NULL;
    gchar               *overlay_file = NULL;
    GError              *err          = NULL;

    LOGPRINTF("entry: icon_size [%d]", icon_size);
    g_assert(icon_size >= 0  &&  icon_size < N_FILEMODEL_THUMB_SIZES);

    if (icon == NULL)
    {
        return;
    }

    // get bitmap for 'shortcut' overlay
    // load icon when not yet done
    p_overlay = overlay_bitmaps + icon_size;
    overlay = *p_overlay;
    if ( overlay == NULL )
    {
        // load filetype icon from file
        overlay_file = g_strdup_printf( "%s/%s%s%s",
                                        ICONS_DIRECTORY,
                                        ICONFILE_PREFIX,
                                        overlay_name,
                                        ICONFILE_POSTFIX[icon_size] );
        overlay = gdk_pixbuf_new_from_file( overlay_file, &err );
        if (overlay)
        {
            *p_overlay = overlay;
        }
        else
        {
            ERRORPRINTF("cannot load overlay file [%s] error [%s]", overlay_file, err->message);
            g_clear_error(&err);
        }
    }

    // apply overlay to icon bitmap
    if (overlay)
    {
        gdk_pixbuf_composite( overlay,                      // src
                              icon,                         // dest
                              0, 0,                         // dest x, y
                              gdk_pixbuf_get_width(icon),   // dest width
                              gdk_pixbuf_get_height(icon),  // dest height
                              0.0, 0.0,                     // offset x, y
                              1.0, 1.0,                     // scale  x, y
                              GDK_INTERP_NEAREST,           // interpolation type: not needed, take the fastest
                              0xFF );                       // overall_alpha
    }

    // clean up
    if (overlay_file) { g_free(overlay_file) ; overlay_file = NULL; }
}


// get filetype description for specified filename
const gchar* get_type_descr_from_file_extension ( const gchar    *file_ext,
                                                  const gboolean is_directory,
                                                  const gboolean is_template  )
{
    const gchar         *description = "";      // return value
    const gchar         *cp;
    filetype_info_t     *details;

    LOGPRINTF("entry: file_ext [%s] is_dir [%d]", file_ext, is_directory);

    // find filetype info
    details = find_filetype_info(file_ext, is_directory, TRUE);

    // load icon when not yet done
    if (details)
    {
        if (is_template)
        {
            cp = details->descr_template;
        }
        else
        {
            cp = details->description;
        }
        if (cp)
        {
            description = _( cp );
        }
        else
        {
            ERRORPRINTF( "filetype [%s] is_dir [%d] is_templ [%d] has no description",
                         cp, is_directory, is_template );
        }
    }

    return description;
}


// tell whether the specified filename is a mobipocket file or not
gboolean is_mobipocket_file_extension ( const gchar *file_ext )
{
    gboolean            is_mobipocket = FALSE;  // return value
    filetype_info_t     *details;

    LOGPRINTF("entry: file_ext [%s]", file_ext);


    // find filetype info
    details = find_filetype_info(file_ext, FALSE, FALSE);

    // check for mobipocket
    if ( details  &&  strcmp( details->icon_name, "mobipocket") == 0 )
    {
        is_mobipocket = TRUE;
    }

    LOGPRINTF("leave: file_ext [%s] is_mobipocket [%d]", file_ext, is_mobipocket);
    return is_mobipocket;
}


// get title from Mobipocket file
// Note: code copied from iLiad contentLister getMobipocketTitle();
GString* get_mobipocket_title (const char* filepath)
{
    GString *title = NULL;      //return value
    GIOChannel *io;
    guint32 firstRecordIndex = 0;
    guint32 titleEncoding = 0;
    guint32 titleOffset = 0;
    guint32 titleLength = 0;
    guint32 check3Value = 0;
    int i;
    gchar *encoding = NULL;
    gchar *p;
    gunichar uniChar = 0;
    GIOStatus st;

    gchar buf[9];
    memset(buf, '\0', sizeof(buf));
    
    LOGPRINTF("entry: [%s]", filepath);
    
    io = g_io_channel_new_file(filepath, "r", NULL);
    if (io != NULL)
    {
        g_io_channel_set_encoding(io, NULL, NULL); // Binary data

        // First check: make sure this is a Mobipocket document
        st = g_io_channel_seek_position(io, MOBIPOCKET_CHECK1_OFFSET, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", MOBIPOCKET_CHECK1_OFFSET);
            goto return_false;
        }
        g_io_channel_read_chars(io, buf, 8, NULL, NULL);
        buf[8] = '\0';
        if (strcmp(buf, "BOOKMOBI") != 0)
        {
            ERRORPRINTF("Identifier 'BOOKMOBI' not found");
            if (strcmp(buf, "TEXtREAd") == 0)
            {
                WARNPRINTF("Identifier 'TEXtREAd' found; Assuming PDB title is valid.");
                // Assume UTF-8 encoding and read title from location 0 (PDB mode)
                // This skips all other tests
                firstRecordIndex = 0;
                titleOffset = MOBIPOCKET_PDB_TITLE_OFFSET;
                titleLength = MOBIPOCKET_PDB_TITLE_LENGTH;
                encoding = g_strdup("UTF-8");
                goto read_title;        
            }
            else
            {
                goto return_false;
            }
        }
        memset(buf, '\0', sizeof(buf));

        // Get firstRecordIndex
        st = g_io_channel_seek_position(io, MOBIPOCKET_FIRST_RECORD, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex);
            goto return_false;
        }
        g_io_channel_read_chars(io, buf, 4, NULL, NULL);
        for (i = 0; i < 4; i++)
        {
            firstRecordIndex = firstRecordIndex << 8;
            firstRecordIndex = firstRecordIndex + buf[i];
        }
        memset(buf, '\0', sizeof(buf));
        
        LOGPRINTF("firstRecordIndex = 0x%x", firstRecordIndex);

        // Second check: make sure this is a Mobipocket document
        st = g_io_channel_seek_position(io, firstRecordIndex + MOBIPOCKET_CHECK2_OFFSET, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + MOBIPOCKET_CHECK2_OFFSET);
            goto return_false;
        }
        g_io_channel_read_chars(io, buf, 4, NULL, NULL);
        buf[4] = '\0';
        if (strcmp(buf, "MOBI") != 0)
        {
            ERRORPRINTF("Identifier 'MOBI' not found");
            goto return_false;
        }
        memset(buf, '\0', sizeof(buf));
        
        // Get encoding of the title
        st = g_io_channel_seek_position(io, firstRecordIndex + MOBIPOCKET_ENCODING_OFFSET, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + MOBIPOCKET_ENCODING_OFFSET);
            goto return_false;
        }
        g_io_channel_read_chars(io, buf, 4, NULL, NULL);
        for (i = 0; i < 4; i++)
        {
            titleEncoding = titleEncoding << 8;
            titleEncoding = titleEncoding + buf[i];
        }
        if (titleEncoding == MOBIPOCKET_ENCODING_UTF8)
        {
            encoding = g_strdup("UTF-8");
        }
        else if (titleEncoding == MOBIPOCKET_ENCODING_WIN1252)
        {
            encoding = g_strdup("WINDOWS-1252");
        }
        else
        {
            ERRORPRINTF("Unknown encoding 0x%x", titleEncoding);
            goto return_false;
        }
        memset(buf, '\0', sizeof(buf));

        // Check 3: Make sure this is a Mobipocket document (value must be >= 2)
        st = g_io_channel_seek_position(io, firstRecordIndex + MOBIPOCKET_CHECK3_OFFSET, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + MOBIPOCKET_CHECK3_OFFSET);
            goto return_false;
        }
        g_io_channel_read_chars(io, buf, 4, NULL, NULL);
        for (i = 0; i < 4; i++)
        {
            check3Value = check3Value << 8;
            check3Value = check3Value + buf[i];
        }
        memset(buf, '\0', sizeof(buf));

        if (check3Value < 2)
        {
            // PDB title mode: title is in first 32 bytes of the file
            WARNPRINTF("Check 3: value < 2 ; fallback to PDB mode");
            firstRecordIndex = 0;
            titleOffset = MOBIPOCKET_PDB_TITLE_OFFSET;
            titleLength = MOBIPOCKET_PDB_TITLE_LENGTH;
        }
        else
        {
            // Normal Mobipocket document: get the title offset
        
            // Get title offset (big endian int)
            st = g_io_channel_seek_position(io, firstRecordIndex + MOBIPOCKET_TITLE_RECORD_OFFSET, G_SEEK_SET, NULL);
            if (st != G_IO_STATUS_NORMAL)
            {
                ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + MOBIPOCKET_TITLE_RECORD_OFFSET);
                goto return_false;
            }
            g_io_channel_read_chars(io, buf, 4, NULL, NULL);
            for (i = 0; i < 4; i++)
            {
                titleOffset = titleOffset << 8;
                titleOffset = titleOffset + buf[i];
            }
            memset(buf, '\0', sizeof(buf));
            
            // Get title length (big endian int)
            st = g_io_channel_seek_position(io, firstRecordIndex + MOBIPOCKET_TITLE_LENGTH_OFFSET, G_SEEK_SET, NULL);
            if (st != G_IO_STATUS_NORMAL)
            {
                ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + MOBIPOCKET_TITLE_LENGTH_OFFSET);
                goto return_false;
            }
            g_io_channel_read_chars(io, buf, 4, NULL, NULL);
            for (i = 0; i < 4; i++)
            {
                titleLength = titleLength << 8;
                titleLength = titleLength + buf[i];
            }
            memset(buf, '\0', sizeof(buf));
            
            titleLength = MIN(titleLength, 500);
            if (titleLength == 0)
            {
                // When the titleLength == 0, we fallback to PDB title mode 
                // and get the title from the first 32 bytes of the file
                WARNPRINTF("titleLength == 0");
                firstRecordIndex = 0;
                titleOffset = MOBIPOCKET_PDB_TITLE_OFFSET;
                titleLength = MOBIPOCKET_PDB_TITLE_LENGTH;
            }

        } // if check3Value < 2
        
read_title:

        LOGPRINTF("titleLength = %d, titleOffset = 0x%x", titleLength, titleOffset);

        // Get the title at titleOffset, read titleLength bytes
        st = g_io_channel_seek_position(io, firstRecordIndex + titleOffset, G_SEEK_SET, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to seek to 0x%x", firstRecordIndex + titleOffset);
            goto return_false;
        }
        // Set the source encoding for the string to be read
        // Strings are automatically converted to unicode when read
        st = g_io_channel_set_encoding(io, encoding, NULL);
        if (st != G_IO_STATUS_NORMAL)
        {
            ERRORPRINTF("Unable to change channel encoding to %s", encoding);
            goto return_false;
        }
        
        LOGPRINTF("Channel encoding changed to %s", encoding);

        // Read titleLength chars, not bytes (because some chars might be multibyte)
        title = g_string_sized_new(titleLength);
        for (i = 0; i < titleLength; i++)
        {
            st = g_io_channel_read_unichar(io, &uniChar, NULL);
            if (st != G_IO_STATUS_NORMAL)
            {
                ERRORPRINTF("Unable to read char from pos %d", i);
                goto return_false;
            }
            g_string_append_unichar(title, uniChar);
        }

        // Replace all '&amp;' references in title with only '&'
        while ((p = strstr(title->str, "&amp;")) != NULL)
        {
            strcpy(p + 1, p + 5);
            title->len -= 4;
        }

        LOGPRINTF("Found title [%s]", title->str);
    }
    else
    {
        ERRORPRINTF("Unable to open file [%s]", filepath);
        goto return_false;
    }
    
    // All OK, free stuff and close filepointer
    
return_false:
    if (encoding) g_free(encoding);
    if (io)
    {
        g_io_channel_shutdown(io, FALSE, NULL);
        g_io_channel_unref(io);
    }

    if (title == NULL)
    {
        WARNPRINTF("Not a valid Mobipocket file [%s]", filepath);
    }
    LOGPRINTF("leave");
    return title;
}



// tell whether the specified file extension is a shortcut file or not
gboolean is_shortcut_file_extension ( const gchar *file_ext )
{
    gboolean            is_shortcut = FALSE;  // return value

    LOGPRINTF("entry: file_ext [%s]", file_ext);


    // check for mobipocket
    if (   strcmp( file_ext, FILE_EXT_SHORTCUT       ) == 0
        || strcmp( file_ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
    {
        is_shortcut = TRUE;
    }

    return is_shortcut;
}


// check whether file is hidden or can be displayed
gboolean is_hidden_filename ( const gchar *filename )
{
    gboolean    is_hidden = FALSE;      // return value
    int         n;
    const char  *extension = NULL;
    const char  **cpp = NULL;

    // check on hidden prefix
    for (cpp = HIDDEN_FILE_PREFIXES ; *cpp && !is_hidden ; cpp++)
    {
        n = strlen(*cpp);
        if ( strncmp(filename, *cpp, n) == 0 )
        {
            is_hidden = TRUE;
        }
    }

    // check on hidden filename
    for (cpp = HIDDEN_FILENAMES ; *cpp && !is_hidden ; cpp++)
    {
        if ( strcmp(filename, *cpp) == 0 )
        {
            is_hidden = TRUE;
        }
    }

    // check on hidden extension
    if ( !is_hidden )
    {
        extension = g_extension_pointer(filename);
        g_assert(extension);
        for (cpp = HIDDEN_FILE_EXTENSIONS ; *cpp && !is_hidden ; cpp++)
        {
            if ( strcmp(extension, *cpp) == 0 )
            {
                is_hidden = TRUE;
            }
        }
    }

    LOGPRINTF("return: filename [%s] is_hidden [%d]", filename, is_hidden);
    return is_hidden;
}


// get viewer command for specified file extension
const gchar* get_viewer_from_file_extension ( const gchar *file_ext )
{
    const gchar         *viewer_cmd = NULL;   // return value
    filetype_info_t     *details;

    LOGPRINTF("entry: file_ext [%s]", file_ext);


    // find filetype info
    details = find_filetype_info(file_ext, FALSE, FALSE);

    // get viewer command
    if (details)
    {
        viewer_cmd = details->viewer_cmd;
    }

    if (viewer_cmd)
    {
        LOGPRINTF("found viewer_cmd [%s] for file_ext [%s]", viewer_cmd, file_ext);
    }
    else
    {
        ERRORPRINTF("no viewer registered for file_ext [%s]", file_ext);
    }

    // return the viewer command, if found
    return viewer_cmd;
}


// copied from libgnome-2.6.1.1 gnome-util.c:
/**
 * g_extension_pointer:
 * @path: A filename or file path.
 *
 * Extracts the extension from the end of a filename (the part after the final
 * '.' in the filename).
 *
 * Returns: A pointer to the extension part of the filename, or a
 * pointer to the end of the string if the filename does not
 * have an extension.
 */
const char *
g_extension_pointer (const char * path)
{
	char * s, * t;

	g_return_val_if_fail(path != NULL, NULL);

	/* get the dot in the last element of the path */
	t = strrchr(path, G_DIR_SEPARATOR);
	if (t != NULL)
		s = strrchr(t, '.');
	else
		s = strrchr(path, '.');

	if (s == NULL)
		return path + strlen(path); /* There is no extension. */
	else {
		++s;      /* pass the . */
		return s;
	}
}
