#ifndef __CTB_ACTIONS_H__
#define __CTB_ACTIONS_H__

/**
 * File Name  : ctb_actions.h
 *
 * Description: Content browser actions on selected file(s) or directory(ies)
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <glib.h>

#include "filemodel.h"

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

// actions on selected item(s)
#define CTB_ACTION_ACTIVATE     "activate"
#define CTB_ACTION_DELETE       "delete"
#define CTB_ACTION_SHORTCUT     "shortcut"


//============================================================================
// Public Functions
//============================================================================

/**---------------------------------------------------------------------------
 *
 * Name :  activate_item
 *
 * @brief  activate: launch viewer, goto directory, ...
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] fileinfo - details of the item
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int activate_item ( const GString           *directory,
                    const filelist_entry_t  *fileinfo  );


/**---------------------------------------------------------------------------
 *
 * Name :  activate_shortcut_to_application
 *
 * @brief  activate: launch command line as specified in shortcut to application
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] fileinfo - details of the item
 * @param  [in] command_line - command line to be executed
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int activate_shortcut_to_application ( const GString           *directory,
                                       const filelist_entry_t  *fileinfo,
                                       const GString           *command_line );


/**---------------------------------------------------------------------------
 *
 * Name :  pre_delete_item
 *
 * @brief  delete: delete file or directory
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] filelist - details of the item(s) to be deleted
 *
 * @return ER_OK = continue with delete
 *         other = cancel delete
 *
 *--------------------------------------------------------------------------*/
int pre_delete_item ( const GString  *directory,
                      const GSList   *filelist  );


/**---------------------------------------------------------------------------
 *
 * Name :  delete_item
 *
 * @brief  delete: delete file or directory
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] fileinfo - details of the item
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int delete_item ( const GString           *directory,
                  const filelist_entry_t  *fileinfo  );


/**---------------------------------------------------------------------------
 *
 * Name :  post_delete_item
 *
 * @brief  delete: delete file or directory
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] filelist - details of the item(s) to be deleted
 *
 * @return ER_OK = continue with delete
 *         other = cancel delete
 *
 *--------------------------------------------------------------------------*/
int post_delete_item ( const GString  *directory,
                       const GSList   *filelist  );


/**---------------------------------------------------------------------------
 *
 * Name :  shortcut_item
 *
 * @brief  shortcut: create shortcut on desktop for file or directory
 *
 * @param  [in] directory - directory in which the item is located
 * @param  [in] fileinfo - details of the item
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int shortcut_item ( const GString           *directory,
                    const filelist_entry_t  *fileinfo  );


/**---------------------------------------------------------------------------
 *
 * Name :  copy_file_or_folder_list
 *
 * @brief  copy list of files/folders and their metadata
 *
 * @param  [in] filenames_src  - list of file/folder names to be copied
 * @param  [in] dir_src        - directory in which the item files/folders located
 * @param  [in] filenames_dest - list of file/folder names of the copies to be created
 *                               or NULL when names are identical to source file/folder names
 * @param  [in] dir_dest       - directory in which the copies must be placed
 * @param  [in] db_dest        - metadata database in target directory
 *
 * @return ER_OK or error code
 *
 *--------------------------------------------------------------------------*/
int copy_file_or_folder_list( const GPtrArray  *filenames_src,
                              const gchar      *dir_src,
                              const GPtrArray  *filenames_dest,
                              const gchar      *dir_dest,
                                    erMetadb   *db_dest        );


/**---------------------------------------------------------------------------
 *
 * Name :  db_open
 *
 * @brief  open a metadata database and optionally create a new one
 *
 * @param  [in] directory          - directory in which to open or create a database
 * @param  [in] do_create_database - whether or not to create a database when none found
 *
 * @return a newly create database object
 *         or NULL on error
 *         Caller must release this database object using ermetadb_free().
 *
 *--------------------------------------------------------------------------*/
erMetadb* db_open ( const GString   *directory,
                    const gboolean  do_create_database );



G_END_DECLS

#endif // __CTB_ACTIONS_H__
