/*
 * File Name: recent.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gconf/gconf-client.h>
#include <stdlib.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "ctb_log.h"
#include "recent.h"
#include "filetypes.h"
#include "i18n.h"
//#include "ipc.h"
#include "main.h"
#include "shortcut.h"
#include "storage.h"
#include "ctb_actions.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

#define ER_ALREADY_IN_RECENT 0x00001001

static const gchar  *FILE_SHORTCUT   = "[Desktop Entry]\n"
                                        "Version=1.0\n"
                                        "Type=Link\n"
                                        "URL=file:%s\n"
                                        "Name=%s\n";

static const gchar  *FOLDER_SHORTCUT = "[Desktop Entry]\n"
                                       "Version=1.0\n"
                                       "Type=Directory\n"
                                       "Path=%s\n"
                                       "Name=%s\n";

#define METADATA_KEY_RECENT_COUNT  "ctb.recent-count"
#define METADATA_KEY_SORTORDER  "ctb.sort-order"

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// CAREFULL: These values need to match with the values in settings!!
static const gchar      *REGKEY_RECENTFILES       = "/apps/er/sys/ctb/recentfiles";
static const gchar      *REGVAL_RECENTFILES_OFF   = "off";
static const gchar      *REGVAL_RECENTFILES_STORE = "store";
static const gchar      *REGVAL_RECENTFILES_START = "start";
static const gchar      *REGVAL_RECENTFILES_AUTO  = "auto";

// Do not store recent files
#define RECENTFILES_OFF   0
// Only store
#define RECENTFILES_STORE 1
// After power on, start in recent files folder
#define RECENTFILES_START 2
// After power on, start the last read document
#define RECENTFILES_AUTO  3

static int g_recent_mode = 0; // OFF

//============================================================================
// Local Function Definitions
//============================================================================
static int create_recent_shortcut (  const gchar   *target_dir,
                                     const gchar   *target_file,
                                     const gchar   *target_display,
                                     const gchar   *shortcut_dir,
                                     GString *shortcut_file  );
static void write_recentfile_info_to_database (erMetadb *db, const gchar *filename);
static void write_sort_order_to_database_to_date (erMetadb *db);
int get_recentfilesmode_from_registry ( void );
void save_recentfilesmode_to_registry ( int mode );


//============================================================================
// Functions Implementation
//============================================================================

int recent_item_log ( const GString           *directory,
                  const filelist_entry_t  *fileinfo  )
{
	LOGPRINTF("directory: %s", directory->str);
	LOGPRINTF("fileinfo->filename:%s", fileinfo->filename->str);
	LOGPRINTF("fileinfo->filename_display: %s", fileinfo->filename_display->str);
	LOGPRINTF("fileinfo->filetype: %s", fileinfo->filetype->str);
	LOGPRINTF("fileinfo->directory_path: %s", fileinfo->directory_path->str);
	
	return ER_OK;
}

// init
void recent_init(void)
{
	// TEST CODE
	//save_recentfilesmode_to_registry(RECENTFILES_START);
	// read mode from registry
	g_recent_mode = get_recentfilesmode_from_registry();
}

// goto Recent Files folder
void recent_start (void)
{
    LOGPRINTF("entry");

	if ( (g_recent_mode >= RECENTFILES_START) &&
		 (g_file_test(DIR_LIBRARY_RECENTFILES, G_FILE_TEST_IS_DIR)) )
	{
    	fileview_show_dir(DIR_LIBRARY_RECENTFILES, NULL);
	}
	else
	{
		// else show desktop
		fileview_show_desktop();
	}
}

// add recent to recent-shortcuts
int recent_item ( const GString           *directory,
                  const filelist_entry_t  *fileinfo  )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename          && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display  && fileinfo->filename_display->str);
    g_assert(fileinfo->filetype          && fileinfo->filetype->str        );
    g_assert(fileinfo->directory_path    && fileinfo->directory_path->str  );

    int             ret = ER_OK;    // return code
    int             rc;
    const gchar     *target_dir;
    gchar           *error_msg       = NULL;
    GString         *shortcut_dir    = NULL;
    GString         *shortcut_file   = g_string_new("");
    GString         *shortcut_path   = g_string_new("");
    GString         *shortcut_target = g_string_new("");
    erMetadb        *db              = NULL;
    metadata_table  *values          = NULL;

    const gchar     *filename         = fileinfo->filename->str;
    const gchar     *filename_display = fileinfo->filename_display->str;
    const gchar     *directory_path   = fileinfo->directory_path->str;

    LOGPRINTF( "entry: dir [%s] filename [%s] dirpath [%s]", directory->str, filename, directory_path );

    // no shortcut to hidden file or folder
    if ( is_hidden_filename(filename) )
    {
        ret = ER_FORBIDDEN;
    }

	// WORKAROUND: I would like to subscribe for changes, but I am not sure if that
	//             works when another app is changing the gconf value, so reload
	//             every time, the function is called.
	// get the current value
	g_recent_mode = get_recentfilesmode_from_registry();

	if (g_recent_mode < RECENTFILES_STORE)
    {
        ret = ER_FORBIDDEN;
    }

    // get directory to create shortcut in,
    // create this directory if not yet present
    if (ret == ER_OK)
    {
        if ( strcmp(directory_path, ".") == 0 )
        {
            target_dir = directory->str;
        }
        else
        {
            target_dir = directory_path;
        }
        shortcut_dir = storage_get_recentfiles_dir(target_dir);
        if (shortcut_dir == NULL)
        {
            ret = ER_INVALID_DATA;
        }
        else
        {
            // create shortcut directory
            if ( !g_file_test(shortcut_dir->str, G_FILE_TEST_IS_DIR) )
            {
                rc = g_mkdir_with_parents( shortcut_dir->str, 0755 );
                if (rc != 0)
                {
                    ERRNOPRINTF("cannot create directory [%s]", shortcut_dir->str);
                    ret = ER_FAIL;
                }
            }
        }
    }

    // create shortcut file
	// It is not clear how the time of a shortcut works, should we touch the 
	// shortcut file to get it up to the front? or delete and add the shortcut?
    if (ret == ER_OK)
    {
        ret = create_recent_shortcut( target_dir,
                                      filename,
                                      filename_display,
                                      shortcut_dir->str,
                                      shortcut_file );
    }

	if (ret == ER_ALREADY_IN_RECENT)
	{
		// the file was already present in the recent files, need to adjust
		// parameters for file
	}
	
    // copy metadata to desktop database
    if (ret == ER_OK)
    {
		// get db of filemodel (i.e. current folder)
		db = db_open( fileinfo->directory_path, FALSE );
        if (db == NULL)
		{
				WARNPRINTF("Could not open datbase for target: %s", fileinfo->directory_path->str);
		}
		else
        {
            // get metadata from original database
            rc = ermetadb_get_file_metadata( db, fileinfo->filename, NULL, &values );
			
			// Close target db
			ermetadb_free(db);
			db = NULL;
			
            if (rc == ER_OK  &&  values)
            {

				// store metadata in recent files database on this storage device
                db = db_open( shortcut_dir, TRUE );
				
                if (db == NULL)
                {
                    WARNPRINTF("cannot open database [%s]", shortcut_dir->str);
					// TODO:
                    // ret = ER_FAIL;
                }
                else
                {
					// check sort order
					write_sort_order_to_database_to_date(db);
					
                    // insert folder or document
                    rc = ermetadb_add_document(db, shortcut_file, 0, 0);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot add folder/document, error [%d]", rc);
                        ret = rc;
                    }

                    // set file metadata
                    if (ret == ER_OK)
                    {
                        rc = ermetadb_set_file_metadata( db, shortcut_file, values );
                        if (rc != ER_OK)
                        {
                            ERRORPRINTF("cannot set file_metadata, error [%d]", rc);
                            ret = rc;
                        }
                    }
					
					// Update recent specific values in application data
					write_recentfile_info_to_database(db, shortcut_file->str);

                    // close desktop database
                    rc = ermetadb_close_database(db);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot close database [%s], error [%d]", shortcut_dir->str, rc);
                        if (ret == ER_OK)
                        {
                            ret = rc;
                        }
                    }

                    // clean up
                    ermetadb_free(db);
                    db = NULL;
                }
            }
			else
			{
				WARNPRINTF("No data found from original file: %s", filename);
			}
            db = NULL;
        }
    }
	else if (ret == ER_ALREADY_IN_RECENT)
	{
		// store metadata in desktop database on this storage device
		db = db_open( shortcut_dir, TRUE );
		if (db == NULL)
		{
			WARNPRINTF("cannot open database [%s]", shortcut_dir->str);
			// TODO: create database
			// ret = ER_FAIL;
		}
		else
		{
			// Update recent specific values in application data
			write_recentfile_info_to_database(db, shortcut_file->str);

			// close desktop database
			rc = ermetadb_close_database(db);
			if (rc != ER_OK)
			{
				ERRORPRINTF("cannot close database [%s], error [%d]", shortcut_dir->str, rc);
				if (ret == ER_OK)
				{
					ret = rc;
				}
			}

			// clean up
			ermetadb_free(db);
			db = NULL;
		}
	}

	
	// TODO: prune short-cuts which criteria to use
	//       - used longest ago
	//       - used least frequently
	//       Combination used time/frequency seems best
	
	// If we are looking into the recent folder, we need to refresh
    if ((ret == ER_OK) || (ret == ER_ALREADY_IN_RECENT))
	{
		GString *current_dir = fileview_get_current_dir();
		
		LOGPRINTF("Currentdir: %s", current_dir->str);
		if (strcmp(current_dir->str, shortcut_dir->str) == 0)
		{
			LOGPRINTF("Refresh recent Files directories");
			// TODO: This seems to lead to two extra refreshes, which may be considered
			//       very anoying ... (mainly seen on qemu, not that clear on DR)
			fileview_refresh();
		}
	}

    // clean up
    if (values) { metadata_table_free(values); }
    if (shortcut_target) { g_string_free(shortcut_target, TRUE); }
    if (shortcut_path)   { g_string_free(shortcut_path,   TRUE); }
    if (shortcut_file)   { g_string_free(shortcut_file,   TRUE); }
    if (shortcut_dir)    { g_string_free(shortcut_dir,    TRUE); }
    if (error_msg) { g_free(error_msg); }

    return ret;
}

// create a shortcut file
static int create_recent_shortcut (  const gchar   *target_dir,
                                     const gchar   *target_file,
                                     const gchar   *target_display,
                                     const gchar   *shortcut_dir,
                                     GString *shortcut_file  )
{
    int             ret = ER_OK;     // return code
    int             sequence = 1;    // sequence number for shortcut file
    int             rc;
    gboolean        done;
    shortcut_t      *shortcut        = NULL;
    shortcut_type_t shortcut_type    = SHORTCUT_EMPTY;
    gchar           *cp1;
    gchar           *cp2;
    gchar           *target_path     = NULL;
    gchar           *temp_uri        = NULL;
    const gchar     *shortcut_ext    = "";
    GString         *file            = g_string_new("");;
    GString         *shortcut_path   = g_string_new("");;
    GString         *shortcut_target = g_string_new("");
    FILE            *fp              = NULL;

    LOGPRINTF("entry: target [%s] [%s] shortcut_dir [%s]", target_dir, target_file, shortcut_dir);
    g_assert( target_dir            && *target_dir   == '/'    );
    g_assert( target_file           && *target_file            );
    g_assert( target_display        && *target_display         );
    g_assert( shortcut_dir          && *shortcut_dir == '/'    );
    g_assert( shortcut_file == NULL || shortcut_file->len == 0 );

    // check target/shortcut directories allowed
    if (   fileview_is_directory_allowed(target_dir  ) == FALSE
        || fileview_is_directory_allowed(shortcut_dir) == FALSE )
    {
        ret = ER_FORBIDDEN;
    }

    // determine shortcut type
    if (ret == ER_OK)
    {
        target_path = g_strdup_printf("%s/%s", target_dir, target_file);
        if ( g_file_test(target_path, G_FILE_TEST_IS_REGULAR) )
        {
            shortcut_type = SHORTCUT_TO_FILE;
            shortcut_ext  = FILE_EXT_SHORTCUT;
        }
        else if ( g_file_test(target_path, G_FILE_TEST_IS_DIR) )
        {
            shortcut_type = SHORTCUT_TO_FOLDER;
            shortcut_ext  = FILE_EXT_SHORTCUT_TO_DIR;
        }
        else
        {
            ret = ER_NOT_FOUND;
        }
    }

    // determine shortcut filename
    if (ret == ER_OK)
    {
        done = FALSE;
        while ( !done )
        {
            g_string_printf( file,
                             "%s_%03d.%s",
                              target_file,
                                 sequence,
                                      shortcut_ext );
            g_string_printf( shortcut_path,
                             "%s/%s",
                              shortcut_dir,
                                 file->str );

            if ( g_file_test(shortcut_path->str, G_FILE_TEST_EXISTS) )
            {
				// Determine if the available short-cut points to the same file
				// if so, do not add an extra short cut
    			ret = parse_shortcut_file(shortcut_dir, file->str, &shortcut);
				if (shortcut->type == SHORTCUT_TO_FILE)
				{
					// only handle files, check if this shortcut points to same file
					if ( (strcmp(shortcut->details.file.filename, target_file) == 0) &&
						 (strcmp(shortcut->details.file.directory, target_dir) == 0) )
					{
						int         argc = 0;
						const char  *argv[10];
					    gint        stat;
						
                		WARNPRINTF("filename [%s][%s] already linked", target_file, target_dir);
						// same file, delete it so it can be re-created
						done = TRUE;
						ret = ER_ALREADY_IN_RECENT;
						
						// remove old file so it will be created again below
						// NOTE: this will lead to extra flash activity for every read file
						argc = 0;
						argv[argc++] = "rm";
						argv[argc++] = "-rf";
						argv[argc++] = shortcut_path->str;
						argv[argc++] = NULL;
						g_assert(argc < sizeof(argv)/sizeof(argv[0]));
						(void) g_spawn_sync( NULL,                   // working directory: inherit
											 (char**)argv,           // child's argument vector
											 NULL,                   // environment: inherit
											 G_SPAWN_SEARCH_PATH,    // flags
											 NULL,                   // child_setup: none
											 NULL,                   // child setup data: none
											 NULL,                   // stdout: not interested
											 NULL,                   // stderr: not interested
											 &stat,                  // exit status
											 NULL                );  // error return
					}
				}
				
                WARNPRINTF("filename [%s] already present", shortcut_path->str);
                sequence++;
            }
            else
            {
                LOGPRINTF("shortcut_path [%s]", shortcut_path->str);
                done = TRUE;
            }
        }
    }

    // determine relative path shortcut -> target
    if ((ret == ER_OK) || (ret == ER_ALREADY_IN_RECENT))
    {
        // skip identical pre-fix
        cp1 = target_path;
        cp2 = shortcut_path->str;
        while ( *cp1 != '\0'  &&  *cp1 == *cp2 )
        {
            cp1++;
            cp2++;
        }
        while ( *(cp1 - 1) != '/' )
        {
            cp1--;
            cp2--;
        }
        // count parent directories to traverse
        done = FALSE;
        while ( !done )
        {
            while ( *cp2 == '/' )
            {
                cp2++;
            }
            cp2 = strchr(cp2, '/');
            if (cp2)
            {
                g_string_append(shortcut_target, "../");
            }
            else
            {
                done = TRUE;
            }
        }
        // append remaining part of shortcut target
        g_string_append(shortcut_target, cp1);
    }

    // create shortcut file
    if ((ret == ER_OK) || (ret == ER_ALREADY_IN_RECENT))
    {
        fp = fopen(shortcut_path->str, "w");
        if (fp == NULL)
        {
            ERRNOPRINTF("cannot open file [%s]", shortcut_path->str);
            ret = ER_OPEN_ERROR;
        }
        else
        {
            // write shortcut details
            switch (shortcut_type)
            {
                case SHORTCUT_TO_FILE:
                    temp_uri = g_uri_escape_string(shortcut_target->str, G_URI_RESERVED_CHARS_ALLOWED_IN_PATH, TRUE);
                    rc = fprintf( fp,
                                  FILE_SHORTCUT,
                                  temp_uri,                 // URL=
                                  target_display        );  // Name=
                    break;

                case SHORTCUT_TO_FOLDER:
                    rc = fprintf( fp,
                                  FOLDER_SHORTCUT,
                                  shortcut_target->str,     // Path=
                                  target_display        );  // Name=
                    break;

                default:
                    g_assert_not_reached();
            }
            if (rc <= 0)
            {
                ERRNOPRINTF("cannot write file [%s]", shortcut_path->str);
                ret = ER_WRITE_ERROR;
            }

            // close file
            rc = fclose(fp);
            if (rc != 0)
            {
                ERRNOPRINTF("cannot close file [%s]", shortcut_path->str);
                ret = ER_WRITE_ERROR;
            }
        }
    }

    // set output parameters
    if ((ret == ER_OK) || (ret == ER_ALREADY_IN_RECENT))
    {
        if (shortcut_file)
        {
            g_string_assign(shortcut_file, file->str);
        }
    }

    // clean up
    if (shortcut_target) { g_string_free(shortcut_target, TRUE); }
    if (shortcut_path  ) { g_string_free(shortcut_path,   TRUE); }
    if (file           ) { g_string_free(file,            TRUE); }
    g_free(target_path);
    g_free(temp_uri   );

	LOGPRINTF("leave: [%d]", ret);
	
    return ret;
}

static void write_recentfile_info_to_database (erMetadb *db, const gchar *filename)
{
    gboolean            ok = TRUE;
    int                 rc;
    const char          *cp;
    GString             *this_file = g_string_new(filename);
    metadata_table      *values = NULL;
    metadata_table      *names  = NULL;
    const metadata_cell *cell   = NULL;
	int                 count = 0; // number of times item is opened while in recent-files

	LOGPRINTF("entry:%s", filename);
    g_return_if_fail(db);

    // set metadata key we are interested in
    names = metadata_table_new();
    if (names == NULL)
    {
        ok = FALSE;
    }
    if (ok)
    {
        rc = metadata_table_add_column(names, METADATA_KEY_RECENT_COUNT);
        if (rc != ER_OK)
        {
            ok = FALSE;
			LOGPRINTF("FAIL:could not add to names:%d", rc);
        }
    }

    // retrieve recent count from database
    if (ok)
    {
        rc = ermetadb_get_application_data( db,
                                            this_file,
                                            names,
                                            &values   );
        if (rc != ER_OK)
        {
			WARNPRINTF("cannot retrieve recent count for file %s :%d", this_file->str, rc);
            ok = FALSE;
        }
    }

    // process the retrieved data, if any
    if (ok  &&  values)
    {
        cell = (metadata_cell*) (values->cell_data->data);
        if (cell->type != METADATA_TEXT)
        {
            ERRORPRINTF("illegal cell type [%d] for metadata key [%s]", cell->type, METADATA_KEY_RECENT_COUNT);
        }
        else
        {
            LOGPRINTF("recent count from database [%s]", cell->value.v_text->str);
			count = atoi(cell->value.v_text->str);
		}
		
	}
	
    // clean up
    metadata_table_free(values);
    metadata_table_free(names);
	ok = TRUE;

	// increase the (retrieved or new) count
	count++;
	
    // set new metadata for item
    //   create values table
    values = metadata_table_new();
    if (values == NULL)
    {
        ok = FALSE;
    }
    //   set recent used count
    if (ok)
    {
        rc = metadata_table_add_column(values, METADATA_KEY_RECENT_COUNT);
        if (rc != ER_OK)
        {
            ok = FALSE;
			LOGPRINTF("FAIL:could not add to names:%d", rc);
        }
        else
        {
            cp = g_strdup_printf( "%d", count );
            metadata_table_set_text( values, 0, cp );
        }
    }
    //   write to database
    if (ok)
    {
        rc = ermetadb_set_application_data(db, this_file, values);
        if (rc != ER_OK)
        {
			ERRORPRINTF("cannot write recent count to database for file %s:%d", this_file->str, rc);
            ok = FALSE;
        }
    }
	// TODO: add date/time of using this file

    if (this_file) { g_string_free(this_file, TRUE); }
    metadata_table_free(values);
}

static void write_sort_order_to_database_to_date (erMetadb *db)
{
    gboolean            ok = TRUE;
    int                 rc;
    const char          *cp;
    GString             *this_file = g_string_new(".");
    metadata_table      *names  = NULL;
    metadata_table      *values = NULL;
    const metadata_cell *cell   = NULL;

	LOGPRINTF("entry");
    g_return_if_fail(db);
	
	// TODO: make sure that the "." folder is in the metadata.db file, otherwise
	//       sort order will not be stored.
	
    // set metadata key we are interested in
    names = metadata_table_new();
    if (names == NULL)
    {
        ok = FALSE;
    }
    if (ok)
    {
        rc = metadata_table_add_column(names, METADATA_KEY_SORTORDER);
        if (rc != ER_OK)
        {
            ok = FALSE;
			LOGPRINTF("FAIL:could not add sort order:%d", rc);
        }
    }

    // retrieve sort order from database
    if (ok)
    {
        rc = ermetadb_get_application_data( db,
                                            this_file,
                                            names,
                                            &values   );
        if (rc != ER_OK)
        {
			WARNPRINTF("cannot retrieve sort order for file %s :%d", this_file->str, rc);
            ok = FALSE;
        }
    }

    // process the retrieved data, if any
    if (ok  &&  values)
    {
        cell = (metadata_cell*) (values->cell_data->data);
        if (cell->type != METADATA_TEXT)
        {
            ERRORPRINTF("illegal cell type [%d] for metadata key [%s]", cell->type, METADATA_KEY_SORTORDER);
        }
        else
        {
            LOGPRINTF("recent count from database [%s]", cell->value.v_text->str);
			if ( (strstr(cell->value.v_text->str, "file_last_modified") != NULL) &&
				 (strstr(cell->value.v_text->str, "DESC") != NULL) )
			{
				// sort order is already OK, make sure we will not write
				LOGPRINTF("sort order OK");
				ok = FALSE;
			}
		}
		
	}
	else
	{
		// reading failed, but prepare for write
		ok = TRUE;
	}
	
    // clean up
    metadata_table_free(values);
    metadata_table_free(names);

    //   create values table
    values = metadata_table_new();
    if (values == NULL)
    {
        ok = FALSE;
    }
    //   set recent used count
    if (ok)
    {
        rc = metadata_table_add_column(values, METADATA_KEY_SORTORDER);
        if (rc != ER_OK)
        {
            ok = FALSE;
			LOGPRINTF("FAIL:could not add to values:%d", rc);
        }
        else
        {
			cp = g_strdup_printf( "%s %s", "file_last_modified", "DESC" );
            metadata_table_set_text( values, 0, cp );
        }
    }
    //   write to database
    if (ok)
    {
        rc = ermetadb_set_application_data(db, this_file, values);
        if (rc != ER_OK)
        {
			ERRORPRINTF("cannot write sort order to database for file %s:%d", this_file->str, rc);
            ok = FALSE;
        }
    }

	LOGPRINTF("leave: %d", ok);
	
    if (this_file) { g_string_free(this_file, TRUE); }
    metadata_table_free(values);
}

#if 0
void
key_changed_callback(GConfClient* client,
                     guint cnxn_id,
                     GConfEntry *entry,
                     gpointer user_data)
	

{
	(void) client;
	(void) cnxn_id;
	(void) entry;
	(void) user_data;
	
	// get the new value
	g_recent_mode = get_recentfilesmode_from_registry();
	LOGPRINTF("RecentMode change to : %d", g_recent_mode);
}

#endif

// registry access: get recent files mode
int get_recentfilesmode_from_registry ( void )
{
    int                 mode      = 0;   // return value
    GConfClient         *client   = NULL;
    gchar               *val;
	//static int          subscribed_on_change = 0;
	
    LOGPRINTF("entry");

    // get registry value
    client = gconf_client_get_default();
    val = gconf_client_get_string(client, REGKEY_RECENTFILES, NULL);
	
#if 0
// This does not seem to work
	// subscribe to changes
	if (subscribed_on_change == 0)
	{
		subscribed_on_change = 1;
		gconf_client_notify_add(client, REGKEY_RECENTFILES,
							  key_changed_callback,
							  NULL,
							  NULL, NULL);
	}
#endif
    g_object_unref(client);

    // convert string value to viewtype
    if (val)
    {
		LOGPRINTF("Registry:%s", val);
		
        if ( strcmp(val, REGVAL_RECENTFILES_OFF) == 0 )
        {
			// Do not use the recent file option
            mode = RECENTFILES_OFF;
        }
        else if ( strcmp(val, REGVAL_RECENTFILES_STORE) == 0 )
        {
			// Store the used files in the recent folder
            mode = RECENTFILES_STORE;
        }
        else if ( strcmp(val, REGVAL_RECENTFILES_START) == 0 )
        {
			// Store in folder and start-up with that folder
            mode = RECENTFILES_START;
        }
        else if ( strcmp(val, REGVAL_RECENTFILES_AUTO) == 0 )
        {
			// Store in folder and start-up with that folder
			// and start with the latest document
            mode = RECENTFILES_AUTO;
        }
		else 
		{
			WARNPRINTF("Unknown mode:%s", val);
		}
        g_free(val);
    } 
	else 
	{
        WARNPRINTF("error fetching GConf key %s", REGKEY_RECENTFILES);
    }

    return mode;
}


// registry access: set viewtype
void save_recentfilesmode_to_registry ( int mode )
{
    gboolean     ok;
    GConfClient  *client  = NULL;
    gchar        *val_old = NULL;
    const gchar  *val_new = NULL;
    GError       *err     = NULL;

    LOGPRINTF("entry");

    // convert viewtype to string value
    switch (mode)
    {
        case RECENTFILES_OFF:
            val_new = REGVAL_RECENTFILES_OFF;
            break;
        case RECENTFILES_STORE:
            val_new = REGVAL_RECENTFILES_STORE;
            break;
        case RECENTFILES_START:
            val_new = REGVAL_RECENTFILES_START;
            break;
        case RECENTFILES_AUTO:
            val_new = REGVAL_RECENTFILES_AUTO;
            break;
        default:
            val_new = REGVAL_RECENTFILES_OFF;
    }

    // save to registry
    client = gconf_client_get_default();
    val_old = gconf_client_get_string(client, REGKEY_RECENTFILES, NULL);
    if (   val_old == NULL
        || strcmp(val_old, val_new) != 0 )
    {
        ok = gconf_client_set_string(client, REGKEY_RECENTFILES, val_new, &err);
        if ( !ok )
        {
            ERRORPRINTF("cannot write registry key [%s] - error [%s]", REGKEY_RECENTFILES, err->message);
        }
        g_clear_error(&err);
    }
    g_object_unref(client);

    // clean up
    g_free(val_old);
}

