/*
 * File Name: ipc.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

// ereader include files, between < >
#include <liberipc/eripc.h>
#include <liberutils/er_error.h>

// local include files, between " "
#include "ctb_log.h"
#include "fileview.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "storage.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct 
{
    eripc_handler_t *handler;
    const gchar     *message_name;
    const gchar     *interface;
    gint            handler_id;
} functionEntry;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// IPC content browser
#define DBUS_APPL_NAME                  PACKAGE_NAME
#define DBUS_SERVICE                     "com.irexnet." DBUS_APPL_NAME
#define DBUS_PATH                       "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE                   "com.irexnet." DBUS_APPL_NAME

// IPC system control
#define DBUS_SERVICE_SYSTEM_CONTROL     "com.irexnet.sysd"

// IPC popup menu
#define DBUS_SERVICE_POPUP_MENU         "com.irexnet.popupmenu"

// IPC settings application
#define DBUS_SERVICE_SETTINGS           "com.irexnet.settings"


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_context_t  *eripcContext = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void on_goto_location     ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_window_activated  ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_menu_item         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_mounted           ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_prepare_storage   ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_prepare_unmount   ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_unmounted         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_prepare_hibernate ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_locale    ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );


// Exported DBUS API list
static functionEntry    service_functions[] =
                        {
                            { on_goto_location,     "gotoLocation",        NULL                        },
                            { on_menu_item,         "menuItemActivated",   NULL                        },
                            { on_mounted,           "sysVolumeMounted",    NULL                        },
                            { on_window_activated,  "activatedWindow",     NULL                        },
                            { on_mounted,           "sysVolumeMounted",    DBUS_SERVICE_SYSTEM_CONTROL },
                            { on_prepare_storage,   "ctbPrepareStorage",   NULL                        },
                            { on_prepare_unmount,   "sysPrepareUnmount",   DBUS_SERVICE_SYSTEM_CONTROL },
                            { on_unmounted,         "sysVolumeUnmounted",  DBUS_SERVICE_SYSTEM_CONTROL },
                            { on_prepare_hibernate, "sysPrepareHibernate", DBUS_SERVICE_SYSTEM_CONTROL },
                            { on_changed_locale,    "sysChangedLocale",    DBUS_SERVICE_SYSTEM_CONTROL },
                            { NULL }  // end of list
                        };


//============================================================================
// Functions Implementation
//============================================================================


//----------------------------------------------------------------------------
// Generic
//----------------------------------------------------------------------------

    
/*
// function to test mounting and unmounting feature in content browser
static gboolean mount_unmount_test (gpointer data) 
{
    static int count = 0;
    const char *mountpoint = "/media/mmcblk0p1";
 
    count++;
    WARNPRINTF("entry: count [%d]", count);
 
    switch (count % 2)
    {
        case 0:
            WARNPRINTF("simulate SD card mounted");
            storage_media_mounted(mountpoint);
            break;
        case 1:
            WARNPRINTF("simulate SD card unmount");
            storage_media_prepare_unmount(mountpoint);
            break;
        default:
            g_assert_not_reached();
   }
 
   return TRUE;
}
*/

         
// initialise
void ipc_set_services (void)
{
    gint          i;
    eripc_error_t retval;

    LOGPRINTF("entry");
    g_assert(eripcContext == NULL);

    // initialise IPC context
    eripcContext = eripc_init(DBUS_APPL_NAME, "1.0", NULL);
    if (eripcContext == NULL) 
    {
        ERRORPRINTF("Failed to initialize eripc context\n");
        exit(1);
    }

    // register message/signal handlers
    for (i = 0 ; service_functions[i].handler != NULL ; i++)
    {
        if (service_functions[i].interface)
        {
            // install signal handler
            retval = eripc_set_signal_handler( eripcContext,
                                               service_functions[i].handler,
                                               NULL,
                                               ERIPC_BUS_SESSION,
                                               service_functions[i].interface,
                                               service_functions[i].message_name,
                                               &(service_functions[i].handler_id) );
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_signal_handler [%s] returns [%d]",
                             service_functions[i].message_name,
                             retval );
            }
        }
        else
        {
            // install message handler
            retval = eripc_set_message_handler( eripcContext,
                                                service_functions[i].handler,
                                                NULL,
                                                ERIPC_BUS_SESSION,
                                                DBUS_INTERFACE,
                                                service_functions[i].message_name,
                                                &(service_functions[i].handler_id) );
              
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_message_handler [%s] returns [%d]",
                             service_functions[i].message_name,
                             retval );
            }
        }
    }

    // remove "Please wait" dialog, might be left over after crash
    ipc_menu_busy_show(FALSE);

    // test mounting and unmounting feature in content browser
    //g_timeout_add(60*1000, mount_unmount_test, NULL);
}


// un-initialise
void ipc_unset_services (void)
{
    gint i;

    for (i = 0 ; service_functions[i].handler_id != 0 ; i++)
    {
        eripc_unset_handler(eripcContext, service_functions[i].handler_id);
    }    
}


//----------------------------------------------------------------------------
// System control
//----------------------------------------------------------------------------

// report "content browser started"
void ipc_sys_startup_complete ( void )
{
    eripc_error_t   result;

    // parameters for ipc message
    const int       my_pid = getpid();
    const gboolean  is_multidoc = FALSE;

    LOGPRINTF("entry");
    g_assert( eripcContext );

    // broadcast signal over session bus
    result = eripc_send_signal_varargs( eripcContext,
                                        ERIPC_BUS_SESSION,
                                        DBUS_PATH,
                                        DBUS_SERVICE_SYSTEM_CONTROL,
                                        "startupComplete",
                                        ERIPC_TYPE_STRING, DBUS_APPL_NAME,
                                        ERIPC_TYPE_INT,    my_pid,
                                        ERIPC_TYPE_BOOL,   is_multidoc,
                                        ERIPC_TYPE_STRING, DBUS_SERVICE,
                                        ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
    } 
}


// start task (application)
gint ipc_sys_start_task ( const gchar  *cmd_line,
                          const gchar  *work_dir,
                          const gchar  *label,
                          const gchar  *thumbnail_path,
                          gchar        **err_message )
{
    gint                retval = 0;    // success
    eripc_error_t       result;
    eripc_event_info_t  *reply = NULL;
    const eripc_arg_t   *arg_array = NULL;
    int                 old_timeout = 0;
    gboolean            timeout_set = FALSE;

    WARNPRINTF("entry: cmd_line [%s] work_dir [%s] label [%s] thumbnail_path [%s]",
                       cmd_line,     work_dir,     label,     thumbnail_path       );
    g_assert( eripcContext );
    g_assert( cmd_line && *cmd_line );

    // set busy dialog
    ipc_menu_busy_show( TRUE );

    // set ipc timeout to "long"
    eripc_error_t eripc_get_timeout (eripc_context_t *context, int *timeout);
    result = eripc_get_timeout(eripcContext, &old_timeout);
    if (result == ERIPC_ERROR_SUCCESS)
    {
        result = eripc_set_timeout(eripcContext, 60*1000);
        if (result == ERIPC_ERROR_SUCCESS)
        {
            timeout_set = TRUE;
        }
    }

    // broadcast signal over session bus
    result = eripc_send_varargs_and_wait( eripcContext,
                                          &reply,                // reply structure
                                          ERIPC_BUS_SESSION,
                                          DBUS_SERVICE_SYSTEM_CONTROL,
                                          "startTask",
                                          ERIPC_TYPE_STRING, cmd_line,
                                          ERIPC_TYPE_STRING, work_dir,
                                          ERIPC_TYPE_STRING, label,
                                          ERIPC_TYPE_STRING, thumbnail_path,
                                          ERIPC_TYPE_INVALID );
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs_and_wait returns [%d]", result);
    }
    else if (reply == NULL || reply->args == NULL)
    {
        ERRORPRINTF("sysd returns OK but no reply structure");
    }
    else
    {
        // parse the reply strcuture
        arg_array = reply->args;
        if (arg_array[0].type == ERIPC_TYPE_INT)
        {
            retval = arg_array[0].value.i;
            
            if (arg_array[1].type == ERIPC_TYPE_STRING)            
            {
                gchar *message = arg_array[1].value.s;
                if ( (message != NULL) && (err_message != NULL))
                {
                    *err_message = g_strdup(message);
                }
            }
        }
        else
        {
            ERRORPRINTF("unexpected argument: type [%d]", arg_array[0].type);
        }
    }

    // restore ipc timeout
    if (timeout_set)
    {
        eripc_set_timeout(eripcContext, old_timeout);
    }

    // remove busy dialog
    ipc_menu_busy_show( FALSE );

    LOGPRINTF("leave: retval [%d]", retval);
    return retval;
}


// stop task (application)
gboolean ipc_sys_stop_task ( const gchar  *cmd_line )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    WARNPRINTF("entry: cmd_line [%s]", cmd_line );
    g_assert( eripcContext );
    g_assert( cmd_line && *cmd_line );

    // broadcast signal over session bus
    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_SYSTEM_CONTROL,
                                 "stopTask",
                                 ERIPC_TYPE_STRING, cmd_line,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}


// query device capabilities
const device_caps_t* ipc_sys_get_device_capabilities ( void )
{
    int                 i;
    gboolean            ok;
    eripc_error_t       result;
    eripc_event_info_t  *reply = NULL;
    const eripc_arg_t   *arg = NULL;
    int                 old_timeout = 0;
    gboolean            timeout_set = FALSE;

    static device_caps_t dev_caps;

    LOGPRINTF("entry");
    g_assert( eripcContext );

    // set default reply
    dev_caps.has_stylus    = FALSE;
    dev_caps.has_wifi      = FALSE;
    dev_caps.has_bluetooth = FALSE;

    // set ipc timeout to "long"
    result = eripc_get_timeout(eripcContext, &old_timeout);
    if (result == ERIPC_ERROR_SUCCESS)
    {
        result = eripc_set_timeout(eripcContext, 10*1000);
        if (result == ERIPC_ERROR_SUCCESS)
        {
            timeout_set = TRUE;
        }
    }

    // query sysd for device capabilities
    result = eripc_send_varargs_and_wait( eripcContext,
                                          &reply,                // reply structure
                                          ERIPC_BUS_SESSION,
                                          DBUS_SERVICE_SYSTEM_CONTROL,
                                          "sysGetDeviceCapabilities",
                                          ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs_and_wait returns [%d]", result);
    }
    else if (reply == NULL || reply->args == NULL)
    {
        ERRORPRINTF("sysd returns OK but no reply structure");
    }
    else
    {
        // parse the reply strcuture
        for ( i = 0, arg = reply->args ;
                     arg->type != ERIPC_TYPE_INVALID ;
              i++,   arg++ )
        {
            ok = FALSE;
            switch (i)
            {
                case 0:
                    if (arg->type == ERIPC_TYPE_BOOL)
                    {
                        dev_caps.has_stylus = arg->value.b ? TRUE : FALSE;
                        ok = TRUE;
                    }
                    WARNPRINTF("has_stylus [%d]", dev_caps.has_stylus);
                    break;
                case 1:
                    if (arg->type == ERIPC_TYPE_BOOL)
                    {
                        dev_caps.has_wifi = arg->value.b ? TRUE : FALSE;
                        ok = TRUE;
                    }
                    WARNPRINTF("has_wifi [%d]", dev_caps.has_wifi);
                    break;
                case 2:
                    if (arg->type == ERIPC_TYPE_BOOL)
                    {
                        dev_caps.has_bluetooth = arg->value.b ? TRUE : FALSE;
                        ok = TRUE;
                    }
                    WARNPRINTF("has_bluetooth [%d]", dev_caps.has_bluetooth);
                    break;
                default:
                    ; // ignore
            }
            if (!ok)
            {
                ERRORPRINTF("unexpected argument: i [%d] type [%d]", i, arg->type);
            }
        }
    }

    // restore ipc timeout
    if (timeout_set)
    {
        eripc_set_timeout(eripcContext, old_timeout);
    }
    
    eripc_event_info_free(eripcContext, reply);

    return &dev_caps;
}


//----------------------------------------------------------------------------
// Popup menu
//----------------------------------------------------------------------------


// add a menu set
gboolean ipc_menu_add_menu( const char *name,
                            const char *group1,
                            const char *group2,
                            const char *group3 )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] groups [%s] [%s] [%s]",
                       name,     group1, group2, group3 );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, DBUS_SERVICE,
                                 ERIPC_TYPE_STRING, group1,
                                 ERIPC_TYPE_STRING, group2,
                                 ERIPC_TYPE_STRING, group3,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// add a menu group
gboolean ipc_menu_add_group( const char *name,
                             const char *parent, 
                             const char *image )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] image [%s]",
                       name,     parent,     image       );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addGroup",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, image,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// add a menu item
gboolean ipc_menu_add_item( const char *name,
                            const char *parent, 
                            const char *image  )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] image [%s]",
                       name,     parent,     image       );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addItem",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, image,
                                 ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for a menu set
gboolean ipc_menu_set_menu_label ( const char *name,
                                   const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] label [%s]", name, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setMenuLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for a menu group
gboolean ipc_menu_set_group_label ( const char *name,
                                    const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] label [%s]", name, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setGroupLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for an menu item
gboolean ipc_menu_set_item_label ( const char *name,
                                   const char *parent, 
                                   const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] label [%s]", name, parent, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setItemLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// show the given menu set
gboolean ipc_menu_show_menu( const char *name )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s]", name );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "showMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set the state of a menu group
gboolean ipc_menu_set_group_state( const char *name,
                                   const char *state )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] state [%s]", name, state );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setGroupState",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set the state of a menu item
gboolean ipc_menu_set_item_state( const char *name,
                                  const char *parent,
                                  const char *state  )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] state [%s]", name, parent, state );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setItemState",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// show/hide the busy dialog
gboolean ipc_menu_busy_show( gboolean look_busy )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    g_assert( eripcContext );

    // broadcast signal over session bus
    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setBusyShow",
                                 ERIPC_TYPE_STRING, look_busy ? "show" : "hide",
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}

// Set the statusbar stylus state
gboolean ipc_status_set_stylus( const char *state )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;
    
    g_assert( eripcContext );
    g_return_val_if_fail( (state != NULL), FALSE);
    
    result = eripc_send_varargs(eripcContext,
                                NULL,                  // reply handler
                                NULL,                  // user data
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_POPUP_MENU,
                                "setStatusItemState",
                                ERIPC_TYPE_STRING, "statusbar_stylus",
                                ERIPC_TYPE_STRING, state,
                                ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}

//============================================================================
// Local Function Implementation
//============================================================================

//----------------------------------------------------------------------------
// Signal/message handlers
//----------------------------------------------------------------------------

// show specified location
static void on_goto_location ( eripc_context_t          *context,
                               const eripc_event_info_t *info,
                               void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *location  = arg_array[0].value.s;

    int                   rc;
    gint64                diff_ms = 0;
    gint64                now_ms  = 0;
    struct timeval        now_time;

    static char           *previous_location = NULL;
    static gint64         previous_ms = 0;

    if ( location  &&  location[0] != '\0' )
    {
        // get current time in milliseconds
        // get time since previous signal handling, in milliseconds
        rc = gettimeofday( &now_time, NULL );
        if ( rc == 0 )
        {
            now_ms = (gint64)now_time.tv_sec  * 1000 +
                     (gint64)now_time.tv_usec / 1000;
            diff_ms = now_ms - previous_ms;
        }

        // now handle the signal, depending on timing
        if ( previous_location  &&  diff_ms < 1000 )
        {
            // called too short after the previous: ignore this call
            WARNPRINTF("ignore location [%s] because called [%lld] ms after previous", location, diff_ms);
        }
        else
        {
            // handle the signal
            menu_on_goto_location( location );

            // remember we have handled this signal now
            g_free( previous_location );
            rc = gettimeofday( &now_time, NULL );
            if ( rc == 0 )
            {
                previous_location = g_strdup( location );

                now_ms = (gint64)now_time.tv_sec  * 1000 +
                         (gint64)now_time.tv_usec / 1000;
                previous_ms = now_ms;
            }
        }
    }

    return;
}

// activate context
static void on_window_activated( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    LOGPRINTF("entry");
    
    menu_show();
    fileview_grab_focus();

    return;
}


// menu item activated
static void on_menu_item ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[1].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[2].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[3].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *item      = arg_array[0].value.s;
    const char        *group     = arg_array[1].value.s;
    const char        *menu      = arg_array[2].value.s;
    const char        *state     = arg_array[3].value.s;

    if (item && group && menu && state)
    {
        menu_on_item_activated( item, group, menu, state );
    }

    return;
}


// storage device has been mounted
static void on_mounted ( eripc_context_t          *context,
                         const eripc_event_info_t *info,
                         void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    if (mountpoint && *mountpoint)
    {
        storage_media_mounted(mountpoint);
    }
}


// prepare storage device for iOn use
static void on_prepare_storage ( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    int             rc;
    eripc_error_t   result;
    gboolean        reply_ok = FALSE;
    const gchar     *mountpoint = DIR_LIBRARY;

    LOGPRINTF("entry");

    // prepare storage device
    rc = storage_media_prepare(mountpoint);
    if (rc == ER_OK)
    {
        reply_ok = TRUE;
    }
    else
    {
        ERRORPRINTF("failed to prepare storage, mountpoint [%s] error [%d]", mountpoint, rc);
    }

    // reply ok/not-ok
    WARNPRINTF("reply_ok [%d]", reply_ok);
    result = eripc_reply_varargs( eripcContext,
                                  info->message_id,
                                  ERIPC_TYPE_BOOL, reply_ok,
                                  ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_reply_varargs returns [%d]", result);
    } 
}


// storage device will be unmounted
static void on_prepare_unmount ( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    if (mountpoint && *mountpoint)
    {
        storage_media_prepare_unmount(mountpoint);
    }
}


// storage device has been unmounted
static void on_unmounted ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    if (mountpoint && *mountpoint)
    {
        storage_media_unmounted(mountpoint);
    }
}


// system will be shutdown or will go into hibernate
static void on_prepare_hibernate ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data )
{
    LOGPRINTF("entry");

    main_quit();
}


// LOCALE setting (language) has changed
static void on_changed_locale ( eripc_context_t          *context,
                                const eripc_event_info_t *info,
                                void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *locale    = arg_array[0].value.s;

    if (locale)
    {
        main_set_locale(locale);
    }
}
