
#include <stdlib.h>
#include <stdio.h>
#include "endian.h"

#include <zlib.h>

#include <sys/stat.h>
#include <fcntl.h>

#ifndef O_BINARY
#define O_BINARY 0
#endif
#define CREAT_PERMS S_IREAD|S_IWRITE

/* Magic.. Ask Sony why all LRF files start here */
WORD global_subfile = 0x32;

typedef unsigned char * SPTR;
#define SPTRDATA(s)  (s + sizeof(unsigned int) )
#define SPTRLEN(s)  (*((unsigned int *)s))
#define SPTRSETLEN(s,len)  {*((unsigned int *)s) = len;}

BYTE * checked_malloc(unsigned int len)
{
    unsigned char * p;

    p = malloc(len); 
    if (!p)
    {
        fprintf(stderr,"malloc(%d) failed. Bad. Won't continue.\n");
        exit(-1);
    }    
    return p;
}

SPTR salloc(unsigned int len)
{
    unsigned char * p;

    p = checked_malloc(len + sizeof(unsigned int));
    *((unsigned int *)p) = len;
    return p;
}

SPTR read_whole_file(char * fname)
{
        off_t   filesize;
        unsigned char * fileptr;
        int     fd;
        int     nbytes;

        fd = open(fname, O_RDONLY|O_BINARY);
        if (fd < 0) {
                fprintf(stderr,"Unable to open \"%s\" for reading.\n",
                        fname);
                perror("Open");
                return NULL;
        }
        filesize = lseek(fd, 0, SEEK_END);
        (void)lseek(fd, 0, SEEK_SET);

        fileptr = salloc(filesize);

        nbytes = read(fd, SPTRDATA(fileptr), filesize);
        if (nbytes != filesize) {
                fprintf(stderr,"Failed reading \"%s\" - Only got %d bytes "
                        "out of %d.\n", fname, nbytes, filesize);
                perror("Reading");
                free(fileptr);
                close(fd);
                return NULL;
        }
        close(fd);
        return fileptr;
}

typedef struct tag
{
    struct tag * next;
    BYTE   tagid;
    SPTR   data;
    int    len;
    BYTE   shortdata[16];
} tag;

typedef struct subfile
{
    struct  subfile * next;
    int     size;   /* size of data associated with this */
    DWORD   location; /* Where was I put in the file */
    DWORD   id;       /* what # am I */
    WORD    type;     
    WORD    dataflags;
    SPTR    data;
    tag     * taglist;
} subfile;

tag * new_tag(BYTE tagid, int len, const BYTE * data, DWORD val)
{
    tag * q;

    q = (tag *)checked_malloc(sizeof(tag));
    memset(q, 0, sizeof(tag));
    q->tagid = tagid;
    if (len <= sizeof(q->shortdata)) 
    {
        q->len = len;
        if (data)
            memcpy(&q->shortdata[0], data, len);
        else 
            memcpy(&q->shortdata[0], &val, len);
    } else {
        q->len = 0;
        q->data = salloc(len);
        memcpy(SPTRDATA(q->data),data, len);
    }
    return q;
}

void free_tag(tag * q)
{
    if (q->data) free(q->data);
    free(q);
}

void add_tag_to_subfile(subfile * q, tag * r)
{
    tag * p;
    /* inefficient. fix. */
    if (q->taglist) {
        p = q->taglist;
        while (p->next) {
            p = p->next;
        }        
        p->next = r;
        return;
    }
    else 
        q->taglist = r;
}

subfile * new_subfile(WORD type)
{
    subfile * q;

    q = (subfile *)checked_malloc(sizeof(subfile));
    memset(q, 0, sizeof(subfile));
    q->id = global_subfile++;
    q->type = type;
    return q;
}

void free_subfile(subfile * q)
{
    if (q->data) free(q->data);
    if (q->taglist) 
    {
        tag * newp, *p = q->taglist;
        while (p)
        {
            newp = p->next;
            free_tag(p);
            p = newp;
        } 
    }
    free(q);
}

#define LRF_VERSION     0x08
#define LRF_PSUEDOKEY   0x0A
#define LRF_HEADSUBFILE 0x0C
#define LRF_SUBFILECOUNT   0x10
#define LRF_SUBFILEOFFSET  0x18
#define LRF_DIRECTION   0x24  
#define LRF_DIRECTION_FORWARDS  0x01
#define LRF_DIRECTION_BACKWARDS  0x10
#define LRF_UNK1     0x26
#define LRF_UNK2     0x2A
#define LRF_UNK3     0x2C
#define LRF_UNK4     0x2E
#define LRF_ALTSUBFILE  0x44        /* doesn't seem to be used */
#define LRF_UNK5     0x48
#define LRF_INFOLEN  0x4C
#define LRF_IMGLEN   0x50
#define LRF_UNK6     0x4E

int write_lrf_file( char * fname, SPTR infop, SPTR imagep, 
    subfile * subfilelist)
{
    BYTE  lrfheader[0x54];       
    DWORD running;
    int     nsubfiles;
    int     newlen;
    int     needed;
    SPTR    dataptr;
    int     fd = -1;
    int     error = -1, zstatus;
    unsigned char align[16];
    subfile * subp;

    memset(align, 0, sizeof(align));

    memset(lrfheader, 0, sizeof(lrfheader));
    running = sizeof(lrfheader);
    lrfheader[0] = 'L';
    lrfheader[2] = 'R';
    lrfheader[4] = 'F';

    /* Why? I don't know why. */
    WRITE_LE_WORD(&lrfheader[LRF_VERSION], 0x3E7);

    /* Completely arbitrary - I expect. I don't use this encryption anyway */
    WRITE_LE_WORD(&lrfheader[LRF_PSUEDOKEY], 0x30);
   
    lrfheader[LRF_DIRECTION] = LRF_DIRECTION_FORWARDS; 

    /* Dimensions?  */
    WRITE_LE_WORD(&lrfheader[LRF_UNK1],800*2);
    WRITE_LE_WORD(&lrfheader[LRF_UNK2],600);
    WRITE_LE_WORD(&lrfheader[LRF_UNK3],800);
    /* No clue at all*/
    lrfheader[LRF_UNK4] = 0x18;
    WRITE_LE_WORD(&lrfheader[LRF_UNK5], 0x1536);
    lrfheader[LRF_UNK6] = 0x14;

    dataptr = NULL;
    do {
        fd = open(fname, O_WRONLY|O_BINARY|O_EXCL|O_CREAT, CREAT_PERMS);
        if (fd < 0) {
                fprintf(stderr,"Unable to open \"%s\" for writing -- "
                        "Maybe it already exists?\n", fname);
                perror("Open (writing)");
                return -1;
        }
        running = sizeof(lrfheader);
        lseek(fd, running, SEEK_SET);

        /* XML file containing metadata */
        newlen = SPTRLEN(infop) + ((SPTRLEN(infop) / 1000) + 1) + 12 + 4;
        dataptr = checked_malloc(newlen + sizeof(DWORD));
        if (!dataptr) break;
        WRITE_LE_DWORD(dataptr, SPTRLEN(infop));
        zstatus = compress(dataptr + sizeof(DWORD),&newlen,
            SPTRDATA(infop), SPTRLEN(infop));
        if (zstatus != Z_OK) {
            fprintf(stderr,"Compress failed = %d\n", zstatus);
            break;
        }    
        newlen += sizeof(DWORD);
        if (newlen != write(fd, dataptr, newlen)) break;

        running += newlen;
        WRITE_LE_WORD(&lrfheader[LRF_INFOLEN], newlen);
        free(dataptr); dataptr = NULL; 

        /* Image file, 60x80 */
        WRITE_LE_DWORD(&lrfheader[LRF_IMGLEN], SPTRLEN(imagep));
        if (SPTRLEN(imagep) != write(fd, SPTRDATA(imagep), SPTRLEN(imagep)))
            break;

        running += SPTRLEN(imagep);;
    
        /* 16-byte align the first subfile. Unnecessary but done in 
         * every sample LRF file 
         */
        needed = (16 - (running & 0xF)) & 0x0f;
        if (needed != write(fd, align, needed)) break;
        running += needed;

        subp = subfilelist;
        /* First, write out the files */
        /* Lots of silly redundant writes here */
        nsubfiles = 0;
        while (subp) 
        {
            int     size;
            BYTE    tagbuffer[18];
            tag     * tagp;
            
            nsubfiles++;
            if ( subp->type == 0x1C) 
                WRITE_LE_WORD(&lrfheader[LRF_HEADSUBFILE], subp->id);
            if ( subp->type == 0x1e) 
                WRITE_LE_WORD(&lrfheader[LRF_ALTSUBFILE], subp->id);
            /* Lots of silly writes here */
            tagbuffer[0] = 0x0;
            tagbuffer[1] = 0xF5;
            WRITE_LE_DWORD(&tagbuffer[2], subp->id);
            WRITE_LE_WORD(&tagbuffer[6], subp->type);
            if (8 != write(fd, tagbuffer, 8)) break;
            size = 8;

            tagp = subp->taglist;
            while (tagp)
            {
                int len = 2;

                tagbuffer[0] = tagp->tagid;
                tagbuffer[1] = 0xF5;
                if (tagp->len < 16) {
                    memcpy(&tagbuffer[2], tagp->shortdata, tagp->len);
                    len += tagp->len;
                } 
                if (len != write(fd,tagbuffer, len)) break;
                size += len;
                if (tagp->data)
                {
                    if (SPTRLEN(tagp->data) != write(fd, SPTRDATA(tagp->data), 
                        SPTRLEN(tagp->data))) break;
                    size += SPTRLEN(tagp->data); 
                }
                tagp = tagp->next;
            }    
            if (tagp) break;

            if (subp->data)
            {
                tagbuffer[0] = 0x54; /* dataflags */
                tagbuffer[1] = 0xF5;
                WRITE_LE_WORD(&tagbuffer[2], subp->dataflags);
                tagbuffer[4] = 0x04; /* length */
                tagbuffer[5] = 0xF5; 
                WRITE_LE_DWORD(&tagbuffer[6], SPTRLEN(subp->data));
                tagbuffer[10] = 0x05; /* start the data already */
                tagbuffer[11] = 0xF5; 
                if (12 != write(fd, tagbuffer, 12)) break;
                size += 12;
                
                if (SPTRLEN(subp->data) != write(fd, SPTRDATA(subp->data), 
                        SPTRLEN(subp->data))) break;
                size += SPTRLEN(subp->data); 

                tagbuffer[0] = 0x06; /* end of data */
                tagbuffer[1] = 0xF5;
                if (2 != write(fd, tagbuffer, 2)) break;
                size += 2;
            }
            tagbuffer[0] = 0x01; /* end of subfile */
            tagbuffer[1] = 0xF5;
            if (2 != write(fd, tagbuffer, 2)) break;
            size += 2;

            subp->location = running;
            running += size;
            subp->size = size;
            subp = subp->next;
        }
        if (subp) break;

        /* 16-byte align the subfile list. Unnecessary but done in 
         * every sample LRF file 
         */
        needed = (16 - (running & 0xF)) & 0x0f;
        if (needed != write(fd, align, needed)) break;
        running += needed;

        WRITE_LE_DWORD(&lrfheader[LRF_SUBFILECOUNT], nsubfiles);
        WRITE_LE_DWORD(&lrfheader[LRF_SUBFILEOFFSET], running);

        subp = subfilelist;
        while (subp)
        {
            BYTE   subrec[16];
           
            WRITE_LE_DWORD(&subrec[12],0);
            WRITE_LE_DWORD(&subrec[0], subp->id);
            WRITE_LE_DWORD(&subrec[4], subp->location);
            WRITE_LE_DWORD(&subrec[8], subp->size);
            if (sizeof(subrec) != write(fd, subrec, sizeof(subrec))) break;
            running += sizeof(subrec);
            subp = subp->next;
        }
        if (subp) break;
        lseek(fd, 0, SEEK_SET);
        write(fd, lrfheader, sizeof(lrfheader));
        error = 0;
    } while (0);
    if (dataptr) free(dataptr);
    if (fd < 0) close(fd);
    if (error) {
        perror("lRF writing process failed");
    }
    return error;
}

subfile * make_one_long_para(BYTE * page_ptr, DWORD len)
{
    subfile * psub;
    SPTR    ppara;
    BYTE    *p, *q;
    char    *s;
    WORD    c;
    DWORD   j;

    psub = new_subfile(0x0A);
    ppara = salloc( (2*len) + 6 + 4 );
    p = SPTRDATA(ppara);
    q = page_ptr;

    *(p++) = 0xA1;
    *(p++) = 0xF5;
    WRITE_LE_DWORD(p, 0); p+=sizeof(DWORD);
    for (j = 0; j < len; j++) { 
        c = *(q++);
        if (c == '\r') continue;
        if (c == '\n') c = 0xF5D2;
        WRITE_LE_WORD(p,c); p+=2;
    }
    *(p++) = 0xA2;
    *(p++) = 0xF5;
    psub->data = ppara;
    return psub;
}

/* simple way of splitting into pages */
void get_next_page(SPTR file, BYTE ** page_ptr, DWORD * page_len)
{
    int i, lines, nls;
    DWORD len, maxlen;
    BYTE * q;

    if (!*page_len) *page_ptr = SPTRDATA(file);
    if ( (*page_ptr - (SPTRDATA(file)) + *page_len) == SPTRLEN(file)) {
        *page_len = 0;
        return;
    }
    *page_ptr = (*page_ptr) + *page_len;
    q = *page_ptr;
    len = lines = nls = 0;
    maxlen = SPTRLEN(file) - (*page_ptr - SPTRDATA(file));
    while (len < maxlen) 
    {
        if ( (*(q + len) == '\n') || (*(q+len) == '\r') ) {
            nls++; lines++;
        }
        else {
            nls = 0;
        }
        len++;
        if (lines > 199) break;
        if ((nls == 6) && (lines > 30)) break;
    };
    *page_len = len; 
}

const BYTE fontname78[] = {22, 0,'I',0,'W',0,'A',0,0x0E,0x66,'-',0,
        0x2D,0x4E,0x30,0x7D,'N',0,'-',0,'e',0,'b',0, /* HACK for 78 */
        0xFB,0x30};
const BYTE fontname[] = {22, 0,'I',0,'W',0,'A',0,0x0E,0x66,'-',0,
        0x2D,0x4E,0x30,0x7D,'N',0,'-',0,'e',0,'b',0};

#define MAX_ENTS 1000

subfile * make_lrf_book(SPTR textfile)
{
    subfile * head, *tail;
    subfile * fontrec, * para, * box, * something, *page, *book;
    subfile * first_box;
    subfile * layout, * page_layout, * fifth, * seven, *eleven;
    static BYTE pagefiles[MAX_ENTS*4 + 2];
    int  nfile = 0;
    int  saved_nfile;
    static BYTE boxes[MAX_ENTS*6];
    int nbox = 0;
    static BYTE pages[MAX_ENTS*6+2];
    static BYTE pages2[MAX_ENTS*6+4];
    int npage = 0;
    int i;
    BYTE * page_ptr;
    DWORD page_len;

    head = new_subfile(0x1C); /* Head node */
    add_tag_to_subfile(head, new_tag(0x75,sizeof(WORD),0,2));
    add_tag_to_subfile(head, new_tag(0x76,sizeof(WORD),0,0));
    add_tag_to_subfile(head, new_tag(0x77,sizeof(WORD),0,1));

    /* Eek, Handling 78 is aweful. This needs fixed */
    add_tag_to_subfile(head, new_tag(0x78,sizeof(DWORD),0,0));
    add_tag_to_subfile(head, new_tag(0x16,sizeof(fontname78),fontname78,0));
    /* End of 78/16 pair */

    add_tag_to_subfile(head, new_tag(0x79,sizeof(WORD),0,2));
    add_tag_to_subfile(head, new_tag(0x7A,sizeof(WORD),0,0x10));
    add_tag_to_subfile(head, new_tag(0xDA,sizeof(WORD),0,2));

    tail = head;
    {
        subfile * temp; 
        SPTR unkdat;

        temp = new_subfile(0x1E); /* special ? */
        unkdat = salloc(4);
        memset(SPTRDATA(unkdat),0,SPTRLEN(unkdat));
        temp->dataflags = 0x51; 
        temp->data = unkdat;
        tail->next = temp;
    }
    tail = tail->next;

    book = new_subfile(0x01);
    /* will be filled in later */
    tail->next = book; tail = tail->next;

    seven = new_subfile(0x07);
    add_tag_to_subfile(seven,new_tag(0x31,sizeof(WORD),0,0x204));
    add_tag_to_subfile(seven,new_tag(0x32,sizeof(WORD),0,0x64));
    add_tag_to_subfile(seven,new_tag(0x33,sizeof(WORD),0,0x12));
    add_tag_to_subfile(seven,new_tag(0x34,sizeof(DWORD),0,0xFF));
    add_tag_to_subfile(seven,new_tag(0x35,sizeof(WORD),0,0x34));
    // To display nothing...
    // add_tag_to_subfile(seven,new_tag(0x35,sizeof(WORD),0,34));
    add_tag_to_subfile(seven,new_tag(0x36,sizeof(WORD),0,0));
    add_tag_to_subfile(seven,new_tag(0x37,sizeof(DWORD),0,0));
    add_tag_to_subfile(seven,new_tag(0x2e,sizeof(WORD),0,1));
    add_tag_to_subfile(seven,new_tag(0x38,sizeof(WORD),0,0));
    add_tag_to_subfile(seven,new_tag(0x39,sizeof(WORD),0,0));
    {
        BYTE sixbytes[6];
        memset(sixbytes, 0, sizeof(sixbytes));
        sixbytes[0] = 1;
        add_tag_to_subfile(seven,new_tag(0x29,6,sixbytes,0));
    }
    tail->next = seven;  tail = tail->next;
    WRITE_LE_DWORD(&pagefiles[nfile*4+2],seven->id); nfile++;

    eleven = new_subfile(0x0b);
    add_tag_to_subfile(eleven,new_tag(0x76,sizeof(WORD),0,0x0));
    add_tag_to_subfile(eleven,new_tag(0x77,sizeof(WORD),0,0x1));
    add_tag_to_subfile(eleven,new_tag(0x79,sizeof(WORD),0,0x1));
    add_tag_to_subfile(eleven,new_tag(0x7A,sizeof(WORD),0,0x0));
    add_tag_to_subfile(eleven,new_tag(0x11,sizeof(WORD),0,0x64));
    add_tag_to_subfile(eleven,new_tag(0x12,sizeof(WORD),0,0xFFF6));
    add_tag_to_subfile(eleven,new_tag(0x13,sizeof(WORD),0,0x0));
    add_tag_to_subfile(eleven,new_tag(0x14,sizeof(WORD),0,0x0));
    add_tag_to_subfile(eleven,new_tag(0x15,sizeof(WORD),0,0x190));
    add_tag_to_subfile(eleven,new_tag(0x16,sizeof(fontname),fontname,0));
    add_tag_to_subfile(eleven,new_tag(0x17,sizeof(DWORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0x18,sizeof(DWORD),0,0xff));
    add_tag_to_subfile(eleven,new_tag(0x19,sizeof(WORD),0,0x19));
    add_tag_to_subfile(eleven,new_tag(0x1a,sizeof(WORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0x1b,sizeof(WORD),0,0x8C));
    add_tag_to_subfile(eleven,new_tag(0x1c,sizeof(WORD),0,0x0a));
    add_tag_to_subfile(eleven,new_tag(0x1d,sizeof(WORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0x1e,sizeof(WORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0xf1,sizeof(WORD),0,2));
    add_tag_to_subfile(eleven,new_tag(0xf2,sizeof(DWORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0x3c,sizeof(WORD),0,1));
    add_tag_to_subfile(eleven,new_tag(0x3d,sizeof(WORD),0,1));
    add_tag_to_subfile(eleven,new_tag(0x3e,sizeof(WORD),0,0));
    add_tag_to_subfile(eleven,new_tag(0x75,sizeof(WORD),0,1));
    /* 78 here, but skipping it */
    tail->next = eleven;  tail = tail->next;
    WRITE_LE_DWORD(&pagefiles[nfile*4+2],eleven->id); nfile++;
    
    fifth = new_subfile(0x05);
    add_tag_to_subfile(fifth,new_tag(0x21,sizeof(WORD),0,5));
    add_tag_to_subfile(fifth,new_tag(0x22,sizeof(WORD),0,0x35));
    add_tag_to_subfile(fifth,new_tag(0x23,sizeof(WORD),0,5));
    add_tag_to_subfile(fifth,new_tag(0x24,sizeof(WORD),0,0x2a));
    add_tag_to_subfile(fifth,new_tag(0x2C,sizeof(WORD),0,0x2a));
    add_tag_to_subfile(fifth,new_tag(0x25,sizeof(WORD),0,0x2a2));
    add_tag_to_subfile(fifth,new_tag(0x26,sizeof(WORD),0,0x204));
    add_tag_to_subfile(fifth,new_tag(0x27,sizeof(WORD),0,0x3a));
    add_tag_to_subfile(fifth,new_tag(0x28,sizeof(WORD),0,0x35));
    add_tag_to_subfile(fifth,new_tag(0x35,sizeof(WORD),0,0x34));
    add_tag_to_subfile(fifth,new_tag(0x2b,sizeof(WORD),0,0));
    add_tag_to_subfile(fifth,new_tag(0x2a,sizeof(WORD),0,1));
    add_tag_to_subfile(fifth,new_tag(0xda,sizeof(WORD),0,2));
    {
        BYTE sixbytes[6];
        memset(sixbytes, 0, sizeof(sixbytes));
        add_tag_to_subfile(fifth,new_tag(0x29,6,sixbytes,0));
    }
    tail->next = fifth;  tail = tail->next;

    saved_nfile = nfile;
    page_len = 0;
    do {
        get_next_page(textfile,&page_ptr, &page_len);
        if (!page_len) break;
        if (npage >= MAX_ENTS) {
            fprintf(stderr,"TOO MANY PAGES! Limit = %d\n", MAX_ENTS);
            break;
        }

        nfile = saved_nfile;
        nbox = 0;

        first_box = NULL;
        for (i = 0; i < 1; i++) {
            para = make_one_long_para(page_ptr, page_len);

            add_tag_to_subfile(para,new_tag(0x03,sizeof(DWORD),0,eleven->id));
            tail->next = para;  tail = tail->next;
            WRITE_LE_DWORD(&pagefiles[nfile*4+2],para->id); nfile++;

            /* the para goes into a bounding box */
            box = new_subfile(0x06);
            if (!first_box) first_box = box;
            add_tag_to_subfile(box, new_tag(0x03,sizeof(DWORD),0, seven->id));
            add_tag_to_subfile(box, new_tag(0x32,sizeof(WORD),0, 800)); 
            add_tag_to_subfile(box, new_tag(0x31,sizeof(WORD),0, 600)); 
            /* shouldn't be needed? */
            add_tag_to_subfile(box, new_tag(0x33,sizeof(WORD),0,0x14));
            /* The box's data is the paragraph I just made */
            {
                SPTR boxdat;
                BYTE * p;
                boxdat = salloc(6);
                p = SPTRDATA(boxdat);
                *(p++) = 0x03; *(p++) = 0xF5;
                WRITE_LE_DWORD(p, para->id);
                box->data = boxdat;
            }
            tail->next = box;  tail = tail->next;
            WRITE_LE_DWORD(&pagefiles[nfile*4+2],box->id); nfile++;
            boxes[(nbox*6)] = 0x03;
            boxes[(nbox*6)+1] = 0xF5;
            WRITE_LE_DWORD(&boxes[nbox*6 + 2], box->id); nbox++;
        }

        page = new_subfile(0x02);
        WRITE_LE_WORD(&pagefiles[0], nfile);
        /* page points back to book */ 
        add_tag_to_subfile(page, new_tag(0x7C,sizeof(DWORD),0,book->id));
        add_tag_to_subfile(page,new_tag(0x0b,nfile*4+2,pagefiles,0));
        add_tag_to_subfile(page,new_tag(0x03,sizeof(DWORD),0,fifth->id));
        /* The pages's data is the box I just made */
        {
            SPTR pagedat;
            BYTE * p;
    
            pagedat = salloc(6 * nbox);
            memcpy(SPTRDATA(pagedat),boxes, 6*nbox);    
            page->data = pagedat;
        }
        WRITE_LE_DWORD(&pages[ (npage * 4) + 2 ],page->id);
        /* For the layout?! */
        WRITE_LE_DWORD(&pages2[ (npage * 6) + 4 ],page->id);
        WRITE_LE_WORD(&pages2[ (npage * 6) + 8 ],1);
        npage++;
        tail->next = page;  tail = tail->next;

 
        page_layout = new_subfile(0x1a);
        page_layout->dataflags = 0x82;
        {
            SPTR laydat;
            BYTE * p;
            laydat = salloc(24); /* even for 1 */
            p = SPTRDATA(laydat);
#if 1
        p[18] = 0x19; /* MAGIC */
#endif
            WRITE_LE_DWORD(p, 1); p +=  sizeof(DWORD);
            WRITE_LE_DWORD(p, first_box->id); p +=  sizeof(DWORD);
            page_layout->data = laydat;
        }
        tail->next = page_layout;  tail = tail->next;
        add_tag_to_subfile(page,new_tag(0x02,sizeof(DWORD),0,page_layout->id));

    } while (1);
    WRITE_LE_WORD(&pages[0], npage);
    WRITE_LE_DWORD(&pages2[0], npage);

    layout = new_subfile(0x1A);
    layout->dataflags = 0x81;
    {
        SPTR laydat;
        BYTE * p;
#if 0
        laydat = salloc(4 + 4 + 2);
        p = SPTRDATA(laydat);
        WRITE_LE_DWORD(p, 1); p +=  sizeof(DWORD);
        WRITE_LE_DWORD(p, page->id); p +=  sizeof(DWORD);
        WRITE_LE_WORD(p, 1); p +=  sizeof(WORD);
#endif
        laydat = salloc(npage * 6 + 4);
        memcpy(SPTRDATA(laydat),pages2, npage*6+4);
        layout->data = laydat;
    }
    tail->next = layout;  tail = tail->next;

    add_tag_to_subfile(book, new_tag(0x02,sizeof(DWORD),0,layout->id)); 
    /* the book has a tag which lists the pages... you thought it might
     * use the data again?! */
    {
        SPTR bookdat;
        BYTE * p;
        tag * t;
        bookdat = salloc(4 + 2);
        p = SPTRDATA(bookdat);
        /*
        WRITE_LE_WORD(p, 1); p+=2;
        WRITE_LE_DWORD(p, page->id); p+=4;
        t = new_tag(0x5C, 4+2, SPTRDATA(bookdat), 0); 
        */
        t = new_tag(0x5C, npage*4 + 2, pages, 0); 
        add_tag_to_subfile(book, t);
        free(bookdat);
    }
    add_tag_to_subfile(head, new_tag(0x7B,sizeof(DWORD),0,book->id));
    return head;
}

int main(int argc, char ** argv)
{
    SPTR imagep, infop, textp;
    int  ret;
    subfile * sublist;

    if (argc != 3) { 
        fprintf(stderr," destlrffile sourcetxtfile\n");
        exit(-1);
    } 

    textp = read_whole_file(argv[2]);
    if (!textp) {
        fprintf(stderr,"Unable to read textfile.\n");
        exit(-1);
    }

    infop = read_whole_file("info.xml");
    if (!infop) {
        fprintf(stderr,"no info.xml -create one.\n");
        exit(-1);
    }
    imagep = read_whole_file("image.gif");
    if (!imagep) {
        fprintf(stderr,"no image.gif - make one.\n");
        exit(-1);
    }
    sublist = make_lrf_book(textp);
    ret = write_lrf_file(argv[1],infop,imagep,sublist);
    if (ret) { 
        fprintf(stderr,"Failure Code - %d\n", ret);
    }
    free(infop);
    free(imagep);
    free_subfile(sublist);
    exit(ret);
}
