/*
 * File Name: time_selection_dialog.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >

// ereader include files, between < >

// local include files, between " "
#include "i18n.h"
#include "log.h"
#include "ipc.h"
#include "settings.h"
#include "settings_utils.h"
#include "time_selection_dialog.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

#define NUMERIC_BUTTONS_PER_LINE      3
#define TIME_ENTRY_WIDTH             40
#define FIELD_BUTTON_SPACING          2


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget* g_hh_entry = NULL;
static GtkWidget* g_mm_entry = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void     on_backspace_clicked    (GtkButton* button, gpointer data);
static void     on_number_clicked       (GtkButton* button, gpointer data);
static void     on_ok_clicked           (GtkButton* button, gpointer data);
static gboolean on_idle_set_focus       (gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--table
//       |--label  "hh", "mm"
//       |--entry  "hh", "mm", "<-"
//       |--button "1", "2", "3"
//       |--button "4", "5", "6"
//       |--button "7", "8", "9"
//       |--button "0", "OK"
GtkWidget* create_time_selection_dialog(GtkWindow* parent)
{
    // Need to popup a dialog with numeric buttons.
    GtkWidget* dialog   = NULL;
    GtkWidget* table    = NULL;
    GtkWidget* label    = NULL;
    GtkWidget* button   = NULL;
    GtkWidget* image    = NULL;
    int        i        = 0;
    char       tmp[4];

    dialog = gtk_dialog_new_with_buttons("",
        parent,
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        NULL);
    set_popup_window_style(GTK_WINDOW(dialog));

    table = gtk_table_new(6, 3, TRUE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

    // Label "hh"
    label = gtk_label_new("hh");
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);

    // Label "mm"
    label = gtk_label_new("mm");
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);

    // Entry "hh"
    g_hh_entry = gtk_entry_new();
    gtk_widget_set_size_request(g_hh_entry, TIME_ENTRY_WIDTH, -1);
    gtk_entry_set_max_length(GTK_ENTRY(g_hh_entry), 2);
    gtk_editable_set_editable(GTK_EDITABLE(g_hh_entry), FALSE);
    gtk_widget_set(g_hh_entry, "can-focus", FALSE, NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), g_hh_entry, 0, 1, 1, 2);

    // Entry "mm"
    g_mm_entry = gtk_entry_new();
    gtk_widget_set_size_request(g_mm_entry, TIME_ENTRY_WIDTH, -1);
    gtk_entry_set_max_length(GTK_ENTRY(g_mm_entry), 2);
    gtk_editable_set_editable(GTK_EDITABLE(g_mm_entry), FALSE);
    gtk_widget_set(g_mm_entry, "can-focus", FALSE, NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), g_mm_entry, 1, 2, 1, 2);

    // The "<-" button.
    button = gtk_button_new();
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(on_backspace_clicked), button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 1, 2);

    // Add bitmap for "<-" button.
    image = gtk_image_new_from_file(DATADIR"/time_entry_backspace.png");
    gtk_container_add(GTK_CONTAINER(button), image);

    for (i=0; i<10; i++)
    {
        sprintf(tmp, "%d", (i+1) % 10);
        button = gtk_button_new_with_label(tmp);
        if (i == 0)
        {
            // Set focus on the button "1".
            g_idle_add(on_idle_set_focus, button);
        }
        g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(on_number_clicked), (gpointer)((i+1) % 10));
        gtk_table_attach_defaults(GTK_TABLE(table),
            button,
            i % NUMERIC_BUTTONS_PER_LINE,
            i % NUMERIC_BUTTONS_PER_LINE+1,
            i / NUMERIC_BUTTONS_PER_LINE+2,
            i / NUMERIC_BUTTONS_PER_LINE+3);
    }

    // The "OK" button.
    button = gtk_button_new_with_label(_("OK"));
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(on_ok_clicked), dialog);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 3, 5, 6);

    gtk_table_set_row_spacings(GTK_TABLE(table), FIELD_BUTTON_SPACING);
    gtk_table_set_col_spacings(GTK_TABLE(table), FIELD_BUTTON_SPACING);
    
    gtk_widget_show_all(dialog);
    return dialog;
}


int get_hh()
{
    const char* hh = gtk_entry_get_text(GTK_ENTRY(g_hh_entry));
    return (hh[0] - '0') * 10 + (hh[1] - '0');
}


int get_mm()
{
    const char* mm = gtk_entry_get_text(GTK_ENTRY(g_mm_entry));
    return (mm[0] - '0') * 10 + (mm[1] - '0');
}


//============================================================================
// Local Functions Implementation
//============================================================================

static void on_backspace_clicked(GtkButton* button, gpointer data)
{
    char mm[3] = {0};
    char hh[3] = {0};

    strcpy(mm, gtk_entry_get_text(GTK_ENTRY(g_mm_entry)));
    strcpy(hh, gtk_entry_get_text(GTK_ENTRY(g_hh_entry)));

    if (mm[0] != '\0')
    {
        mm[strlen(mm) - 1] = '\0';
        gtk_entry_set_text(GTK_ENTRY(g_mm_entry), mm);
        return;
    }
    if (hh[0] != '\0')
    {
        hh[strlen(hh) - 1] = '\0';
        gtk_entry_set_text(GTK_ENTRY(g_hh_entry), hh);
    }
}


static void on_number_clicked(GtkButton* button, gpointer data)
{
    char mm[3] = {0};
    char hh[3] = {0};
    int  digit = 0;
    int  num_entered = (int)data;

    strcpy(mm, gtk_entry_get_text(GTK_ENTRY(g_mm_entry)));
    strcpy(hh, gtk_entry_get_text(GTK_ENTRY(g_hh_entry)));

    if (hh[0] == '\0')
    {
        // We are entering the first digit of hh.
        // Only 0,1,2 is valid.
        if (num_entered > 2)
        {
            return;
        }
        hh[0] = '0' + num_entered;
    }
    else if (hh[1] == '\0')
    {
        // We are entering the second digit of hh.
        // Only 00 ~ 23 is valid.
        digit = hh[0] - '0';
        if (digit * 10 + num_entered > 23)
        {
            return;
        }
        hh[1] = '0' + num_entered;
    }
    else if (mm[0] == '\0')
    {
        // We are entering the first digit of mm.
        // Only 0 ~ 5 is valid.
        if (num_entered > 5)
        {
            return;
        }
        mm[0] = '0' + num_entered;
    }
    else if (mm[1] == '\0')
    {
        // We are entering the second digit of mm.
        // Only 00 ~ 59 is valid.
        digit = mm[0] - '0';
        if (digit * 10 + num_entered > 59)
        {
            return;
        }
        mm[1] = '0' + num_entered;
    }
    else
    {
        // Nothing to do.
        return;
    }

    gtk_entry_set_text(GTK_ENTRY(g_hh_entry), hh);
    gtk_entry_set_text(GTK_ENTRY(g_mm_entry), mm);
}


static void on_ok_clicked(GtkButton* button, gpointer data)
{
    const char* hh = gtk_entry_get_text(GTK_ENTRY(g_hh_entry));
    const char* mm = gtk_entry_get_text(GTK_ENTRY(g_mm_entry));
    GtkDialog* dialog = GTK_DIALOG(data);

    if (strlen(hh) == 2 && strlen(mm) == 2)
    {
        gtk_dialog_response(dialog, GTK_RESPONSE_ACCEPT);
    }
    else
    {
        gtk_dialog_response(dialog, GTK_RESPONSE_CANCEL);
    }
}


static gboolean on_idle_set_focus(gpointer data)
{
    gtk_widget_grab_focus(data);

    // Don't call me again.
    return FALSE;
}
