/*
 * File Name: settings_utils.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#ifndef WIN32
#include <sys/time.h>
#endif
#include <time.h>
#include <stdlib.h>
#include <gconf/gconf-client.h>

// ereader include files, between < >

// local include files, between " "
#include "ipc.h"
#include "settings.h"
#include "settings_utils.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GConfClient* client = NULL;


//============================================================================
// Local Function Definitions
//============================================================================


//============================================================================
// Functions Implementation
//============================================================================

GtkWidget* create_separator_widgets()
{
    GtkWidget* separator = gtk_event_box_new();
    gtk_widget_set_name(separator, "irex-settings-separator");
    gtk_widget_set_size_request(separator, -1, SETTINGS_SEPARATOR_HEIGHT);
    return separator;
}


void set_popup_window_style(GtkWindow* window)
{
    gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_POPUP_MENU);
    gtk_window_set_decorated(window, FALSE);
    gtk_widget_realize(GTK_WIDGET(window));
    gdk_window_set_decorations(GTK_WIDGET(window)->window, GDK_DECOR_BORDER);

    // Put the window in the center of its parent.
    gtk_window_set_position(window, GTK_WIN_POS_CENTER_ON_PARENT);
}


DeviceType get_device_type()
{
    gboolean has_stylus    = FALSE;
    gboolean has_wifi      = FALSE;
    gboolean has_bluetooth = FALSE;

    // Read device capabilities from sysd.
    get_device_capabilities(&has_stylus, &has_wifi, &has_bluetooth);

    if (has_stylus == TRUE)
    {
        if (has_wifi == TRUE && has_bluetooth == TRUE)
        {
            return IREX_DR1000SW;
        }
        else
        {
            return IREX_DR1000S;
        }
    }
    else
    {
        return IREX_DR1000;
    }
}


void set_date_time(time_t t)
{
#ifndef WIN32
    struct timeval tv;
    gettimeofday(&tv, NULL);

    tv.tv_sec = t;
    settimeofday(&tv, NULL);

    // Sync RTC clock with system clock.
    system("/sbin/hwclock -w -u");
#endif
}


void gconf_initialize()
{
    if (client == NULL)
    {
        client = gconf_client_get_default();
    }
}

void gconf_finalize()
{
    if (client)
    {
        g_object_unref(client);
        client = NULL;
    }
}


int get_value_int(const char* key)
{
    return gconf_client_get_int(client, key, NULL);
}


gboolean set_value_int(const char* key, int new_value)
{
    return gconf_client_set_int(client, key, new_value, NULL);
}


gboolean get_value_bool(const char* key)
{
    return gconf_client_get_bool(client, key, NULL);
}


gboolean set_value_bool(const char* key, gboolean new_value)
{
    return gconf_client_set_bool(client, key, new_value, NULL);
}


const char* get_value_string(const char* key)
{
    return gconf_client_get_string(client, key, NULL);
}


gboolean set_value_string(const char* key, const char* new_value)
{
    return gconf_client_set_string(client, key, new_value, NULL);
}


GSList* get_value_string_list(const char* key)
{
    return gconf_client_get_list(client, key, GCONF_VALUE_STRING, NULL);
}
