/*
 * File Name: sd_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#ifndef WIN32
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/types.h>
#include <sys/wait.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <gtk/gtk.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "i18n.h"
#include "log.h"
#include "ipc.h"
#include "settings.h"
#include "settings_utils.h"
#include "sd_settings.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char  *SD_FORMAT_COMMAND = "mkfs.vfat -F 32 " SD_CARD_DEVICE;


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget* g_sd_settings_dialog    = NULL;
static GtkWidget* g_used_space_label      = NULL;
static GtkWidget* g_free_space_label      = NULL;
static GtkWidget* g_capacity_bar          = NULL;
static GtkWidget* g_format_sd_button      = NULL;
static GtkWidget* g_setup_sd_button       = NULL;
static GtkWidget* g_reset_settings_button = NULL;
static GtkWidget* g_busy_dialog           = NULL;

static gboolean g_sd_card_mounted = FALSE;  // SD card currently mounted or not
static gboolean g_sd_format_ok    = FALSE;  // SD format finished successfully


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_sd_capacity_widgets    (GtkBox* parent);
static GtkWidget* create_format_sd_widgets      (GtkBox* parent);
static GtkWidget* create_setup_sd_widgets       (GtkBox* parent);
static GtkWidget* create_reset_settings_widgets (GtkBox* parent);
static void       update_sd_widgets             (void);
static void       on_mount_done                 (void);
static void       on_umount_done                (void);

static void       on_format_sd_clicked          (GtkWidget *widget, gpointer data);
static void       on_setup_sd_clicked           (GtkWidget *widget, gpointer data);
static void       on_reset_settings_clicked     (GtkWidget *widget, gpointer data);

static gboolean   on_format_sd                  (gpointer data);
static gboolean   on_setup_sd                   (gpointer data);

static void       (*g_cb_post_mount)            (void);  // callback function when mounting is done
static void       (*g_cb_post_umount)           (void);  // callback function when umounting is done


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--Card Storage Capacity section vbox
//          |--separator
//          |--Format the SD Card section vbox
//          |--separator
//          |--Setting Up a SD Card for the Device section vbox
//          |--separator
//          |--Reset the Settings section vbox
//          |--separator
GtkWidget* create_sd_settings_window(GtkWidget* parent)
{
    GtkWidget* top_window = NULL;
    GtkWidget* alignment  = NULL;
    GtkWidget* vbox       = NULL;
    GtkWidget* label      = NULL;
    GtkWidget* filler     = NULL;
    GtkWidget* separator  = NULL;

    LOGPRINTF("entry");

    top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        _("Close"),
        GTK_RESPONSE_ACCEPT,
        NULL);
    set_popup_window_style(GTK_WINDOW(top_window));
    g_sd_settings_dialog = top_window;

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("SD Card Properties").
    label = gtk_label_new(_("SD Card Properties"));
    gtk_widget_set_name(label, "irex-settings-pathbar");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    // The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The Card Storage Capacity section.
    create_sd_capacity_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The Format the SD Card section.
    create_format_sd_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The Setting Up a SD Card section.
    create_setup_sd_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The Reset the Settings section.
    create_reset_settings_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // Make sure widget pointer is cleared when widget is destroyed
    g_signal_connect(g_sd_settings_dialog, "destroy",
                                  G_CALLBACK (gtk_widget_destroyed),
                                  &g_sd_settings_dialog);
    
    update_sd_widgets();

    gtk_widget_show_all(top_window);
    return top_window;
}


void sd_card_mounted(gboolean is_mounted)
{
    LOGPRINTF("entry: is_mounted [%d]", is_mounted);

    // remember card state
    g_sd_card_mounted = is_mounted;

    if (g_sd_settings_dialog != NULL)
    {
        // show new card state
        update_sd_widgets();
    }

    // Do post-mount stuff.
    if (is_mounted)
    {
        if (g_cb_post_mount)
        {
            (*g_cb_post_mount)();
            g_cb_post_mount = NULL;
        }
    }
    else
    {
        if (g_cb_post_umount)
        {
            (*g_cb_post_umount)();
            g_cb_post_umount = NULL;
        }
    }
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Card Storage Capacity")
//    |--GtkProgressBar
//    |--label
//    |--label
static GtkWidget* create_sd_capacity_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;

    LOGPRINTF("entry");

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Card Storage Capacity".
    label = gtk_label_new(_("Card Storage Capacity"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The GtkProgressBar.
    g_capacity_bar = gtk_progress_bar_new();
    gtk_widget_set_name(g_capacity_bar, "capacity");
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_capacity_bar, FALSE, FALSE, 0);

    // The label "Used Space:"
    g_used_space_label = gtk_label_new(NULL);
    gtk_widget_set_name(g_used_space_label, "irex-normal-text");
    gtk_widget_set_size_request(g_used_space_label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_used_space_label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_used_space_label, FALSE, FALSE, 0);

    // The label "Free Space:"
    g_free_space_label = gtk_label_new(NULL);
    gtk_widget_set_name(g_free_space_label, "irex-normal-text");
    gtk_widget_set_size_request(g_free_space_label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(g_free_space_label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), g_free_space_label, FALSE, FALSE, 0);

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Format the SD Card")
//    |--label ("When you format...")
//    |--alignment
//       |--button
static GtkWidget* create_format_sd_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* alignment      = NULL;
    GtkWidget* button         = NULL;

    LOGPRINTF("entry");

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Format the SD Card".
    label = gtk_label_new(_("Format the SD Card"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "When you format..."
    label = gtk_label_new(_("When you format the SD card, ALL the information will be completely deleted from the card."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The alignment.
    alignment = gtk_alignment_new(1.0, 0.5, 0, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), alignment, FALSE, FALSE, 0);

    // The button "Format SD Card".
    button = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(alignment), button);
    g_signal_connect(G_OBJECT(button),
        "clicked",
        G_CALLBACK(on_format_sd_clicked),
        NULL);
    g_format_sd_button = button;

    alignment = gtk_alignment_new(0.5, 0.5, 0, 0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              0,
                              0,
                              LABEL_PADDING_WITHIN_BUTTON,
                              LABEL_PADDING_WITHIN_BUTTON);
    gtk_container_add(GTK_CONTAINER(button), alignment);

    label = gtk_label_new(_("Format SD Card"));
    gtk_container_add(GTK_CONTAINER(alignment), label);

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Setting Up a SD Card for the Device")
//    |--label ("This function will...")
//    |--alignment
//       |--button
static GtkWidget* create_setup_sd_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* alignment      = NULL;
    GtkWidget* button         = NULL;

    LOGPRINTF("entry");

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Setting Up a SD Card for the Device".
    label = gtk_label_new(_("Setting Up a SD Card for the Device"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "This function will..."
    label = gtk_label_new(_("This function will install a folder called \"System\", the settings, and the Companion Software to your card; "
                            "it won't touch existing information on the card."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The alignment.
    alignment = gtk_alignment_new(1.0, 0.5, 0, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), alignment, FALSE, FALSE, 0);

    // The button "Set Up Card for this Device".
    button = gtk_button_new();
    g_signal_connect(G_OBJECT(button),
        "clicked",
        G_CALLBACK(on_setup_sd_clicked),
        NULL);
    gtk_container_add(GTK_CONTAINER(alignment), button);
    g_setup_sd_button = button;

    alignment = gtk_alignment_new(0.5, 0.5, 0, 0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              0,
                              0,
                              LABEL_PADDING_WITHIN_BUTTON,
                              LABEL_PADDING_WITHIN_BUTTON);
    gtk_container_add(GTK_CONTAINER(button), alignment);

    label = gtk_label_new(_("Set Up Card for this Device"));
    gtk_container_add(GTK_CONTAINER(alignment), label);
    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Reset the Settings")
//    |--label ("With one click...")
//    |--alignment
//       |--button
static GtkWidget* create_reset_settings_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* alignment      = NULL;
    GtkWidget* button         = NULL;

    LOGPRINTF("entry");

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Reset the Settings".
    label = gtk_label_new(_("Reset the Settings"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "With one click..."
    label = gtk_label_new(_("With one click of the button, all the settings will be returned to their original, factory settings."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The alignment.
    alignment = gtk_alignment_new(1.0, 0.5, 0, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), alignment, FALSE, FALSE, 0);

    // The button "Reset the Settings".
    button = gtk_button_new();
    g_signal_connect(G_OBJECT(button),
        "clicked",
        G_CALLBACK(on_reset_settings_clicked),
        NULL);
    gtk_container_add(GTK_CONTAINER(alignment), button);
    g_reset_settings_button = button;

    alignment = gtk_alignment_new(0.5, 0.5, 0, 0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              0,
                              0,
                              LABEL_PADDING_WITHIN_BUTTON,
                              LABEL_PADDING_WITHIN_BUTTON);
    gtk_container_add(GTK_CONTAINER(button), alignment);

    label = gtk_label_new(_("Reset the Settings"));
    gtk_container_add(GTK_CONTAINER(alignment), label);

    return top_level_vbox;
}


static gboolean on_format_sd(gpointer data)
{
    LOGPRINTF("entry");

    // format not succeeded (yet)
    g_sd_format_ok = FALSE;

    // Before formatting sd card, unmount it first.
    if ( g_sd_card_mounted )
    {
        g_cb_post_umount = on_umount_done;
        unmount_sd_card();
    }
    else
    {
        on_umount_done();
    }

    return FALSE;  // Don't call me again.
}


static void on_umount_done()
{
    int         rc;
    int         exitcode;

    LOGPRINTF("entry");

    // Function void if no SD screen
    if (g_sd_settings_dialog == NULL)
    {
        return;
    }

    // Update storage capacity indicator
    update_sd_widgets();

    // SD card unmounted, start formatting.
    rc = system(SD_FORMAT_COMMAND);
    exitcode = WEXITSTATUS(rc);
    if (rc == -1)
    {
        ERRORPRINTF("format failed, cmd [%s]", SD_FORMAT_COMMAND);
    }
    else if (exitcode != 0)
    {
        ERRORPRINTF("format failed with exitcode [%d], cmd [%s]", exitcode, SD_FORMAT_COMMAND);
    }
    else
    {
        LOGPRINTF("format succeeded, cmd [%s]", SD_FORMAT_COMMAND);
        g_sd_format_ok = TRUE;
    }

    if (g_sd_format_ok)
    {
        // re-mount SD card after succesful formatting
        g_cb_post_mount = on_mount_done;
        mount_sd_card();
    }
    else
    {
        // format failed, no use to remount SD card
        on_mount_done();
    }
}


static void on_mount_done()
{
    GtkWidget  *dlg = NULL;

    LOGPRINTF("entry: g_sd_format_ok [%d]", g_sd_format_ok);

    // clear busy dialog.
    if (g_busy_dialog)
    {
        gtk_widget_destroy(g_busy_dialog);
        g_busy_dialog = NULL;
        ipc_sys_set_busy(FALSE);
    }

    // Update storage capacity indicator
    update_sd_widgets();

    // Done. Popup a dialog indicating the result
    if (g_sd_format_ok)
    {
        dlg = gtk_message_dialog_new( GTK_WINDOW(g_sd_settings_dialog),
                                      GTK_DIALOG_MODAL,
                                      GTK_MESSAGE_INFO,
                                      GTK_BUTTONS_OK,
                                      _("The SD card has been formatted.") );
    }
    else
    {
        dlg = gtk_message_dialog_new( GTK_WINDOW(g_sd_settings_dialog),
                                      GTK_DIALOG_MODAL,
                                      GTK_MESSAGE_ERROR,
                                      GTK_BUTTONS_OK,
                                      _("A problem occurred while formatting the SD card.\n"
                                        "\n"
                                        "Please insert another SD card, then try again") );
    }
    gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);
}


static void on_format_sd_clicked(GtkWidget *widget, gpointer data)
{
    GtkWidget* dlg = NULL;
    int result;

    LOGPRINTF("entry");

    dlg = gtk_message_dialog_new(GTK_WINDOW(g_sd_settings_dialog),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_YES_NO,
        _("Formatting will erase all the information from the card.\n"
          "\n"
          "Do you wish to continue formatting?"));

    result = gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);

    if (result != GTK_RESPONSE_YES)
    {
        return;
    }

    // Popup a erGtkBusyDialog.
    g_busy_dialog = ergtk_busy_dialog_new(_("Please wait..."));
    gtk_widget_show(g_busy_dialog);
    ipc_sys_set_busy(TRUE);

    g_idle_add(on_format_sd, NULL);
}


static gboolean on_setup_sd(gpointer data)
{
    gboolean   ok;
    GtkWidget* dlg = NULL;

    LOGPRINTF("entry");

    // Have content browser prepare the SD card
    ok = prepare_sd_card();

    // Make erGtkBusyDialog disappear.
    if (g_busy_dialog)
    {
        gtk_widget_destroy(g_busy_dialog);
        g_busy_dialog = NULL;
        ipc_sys_set_busy(FALSE);
    }

    // Done. Popup a dialog indicating the actions are successfully completed.
    if (ok)
    {
        dlg = gtk_message_dialog_new( GTK_WINDOW(g_sd_settings_dialog),
                                      GTK_DIALOG_MODAL,
                                      GTK_MESSAGE_INFO,
                                      GTK_BUTTONS_OK,
                                      _("The SD card has been set up for this device.") );
    }
    else
    {
        dlg = gtk_message_dialog_new( GTK_WINDOW(g_sd_settings_dialog),
                                       GTK_DIALOG_MODAL,
                                       GTK_MESSAGE_ERROR,
                                       GTK_BUTTONS_OK,
                                      _("A problem occurred while setting up the SD card.") );
    }

    gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);

    return FALSE;  // Don't call me again.
}


static void on_setup_sd_clicked(GtkWidget *widget, gpointer data)
{
    GtkWidget* dlg = NULL;
    int result;

    LOGPRINTF("entry");

    dlg = gtk_message_dialog_new(GTK_WINDOW(g_sd_settings_dialog),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_YES_NO,
        _("Set up SD card for this device will place system folders, settings, and the PC Application on the card.\n"
          "\n"
          "Do you wish to set up the card?"));

    result = gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);

    if (result != GTK_RESPONSE_YES)
    {
        return;
    }

    // Popup a erGtkBusyDialog.
    g_busy_dialog = ergtk_busy_dialog_new(_("Please wait..."));
    gtk_widget_show(g_busy_dialog);
    ipc_sys_set_busy(TRUE);

    g_idle_add(on_setup_sd, NULL);
}


static void on_reset_settings_clicked(GtkWidget *widget, gpointer data)
{
    GtkWidget* dlg = NULL;
    int result;

    LOGPRINTF("entry");

    dlg = gtk_message_dialog_new(GTK_WINDOW(g_sd_settings_dialog),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_YES_NO,
        _("Resetting will change all the settings on the device to the factory default.\n"
          "\n"
          "Do you wish to reset the settings?"));

    result = gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);

    if (result != GTK_RESPONSE_YES)
    {
        return;
    }

    // Delete tree in GConf registry
    system("/usr/bin/erconftool -d /apps/er");
    
    // Force export to INI file
    system("/usr/bin/erconftool -f -e /apps/er");
    
    // Make sure everything is written
    sync();

    // Done. Popup a dialog indicating the actions are successfully completed.
    dlg = gtk_message_dialog_new(GTK_WINDOW(g_sd_settings_dialog),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_INFO,
        GTK_BUTTONS_OK,
        _("The settings have been returned to their original, factory defaults."));

    gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);
}


static void update_sd_widgets()
{
#ifndef WIN32
    int           rc;
    struct statfs stat;
    long          used_blocks;
    gdouble       usage;
    char          buf[32];

    LOGPRINTF("entry");

    rc = statfs(SD_CARD_MOUNTPOINT, &stat);
    if ( rc == 0  &&  g_sd_card_mounted )
    {
        // update filesystem usage
        used_blocks = stat.f_blocks - stat.f_bfree;
        usage       = (gdouble)used_blocks / (gdouble)stat.f_blocks;

        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(g_capacity_bar), usage);

        sprintf(buf, _("Used Space: %ld MB"), used_blocks / (1048576 / stat.f_bsize));
        gtk_label_set_label(GTK_LABEL(g_used_space_label), buf);

        sprintf(buf, _("Free Space: %ld MB"), stat.f_bfree / (1048576 / stat.f_bsize));
        gtk_label_set_label(GTK_LABEL(g_free_space_label), buf);

        // enable buttons, except format.
        gtk_widget_set_sensitive(g_setup_sd_button, TRUE);
        gtk_widget_set_sensitive(g_reset_settings_button, TRUE);
    }
    else
    {
        gtk_label_set_label(GTK_LABEL(g_used_space_label), _("Used Space: N/A"));
        gtk_label_set_label(GTK_LABEL(g_free_space_label), _("Free Space: N/A"));

        // disable buttons, except format.
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(g_capacity_bar), 0.0);
        gtk_widget_set_sensitive(g_setup_sd_button, FALSE);
        gtk_widget_set_sensitive(g_reset_settings_button, FALSE);
    }
#endif
}
