/*
 * File Name: power_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <string.h>
#include <stdlib.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "power_settings.h"
#include "settings.h"
#include "settings_utils.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct
{
    gboolean hibernate_if_plugged;
    int seconds_suspend;
    int minutes_hibernate;
} PowerSettings;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static PowerSettings g_orig_power_settings;
static PowerSettings g_cur_power_settings;

static GtkWidget* g_plugged_radios[2]           = {NULL, NULL};
static GtkWidget* g_hibernate_radios[5]         = {NULL, NULL, NULL, NULL, NULL};
static GtkWidget* g_use_powersaving_check       = NULL;

static int hibernate_times[5] = { -1,  15,  30,  45,  60}; // minutes


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_auto_shutdown_widgets  (GtkBox* parent);
static GtkWidget* create_suspend_widgets        (GtkBox* parent);

static void on_plugged_changed                  (GtkWidget *widget, gpointer data);
static void on_hibernate_changed                (GtkWidget *widget, gpointer data);
static void on_suspend_changed                  (GtkWidget *widget, gpointer data);
static void init_widgets_with_settings          (void);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--Auto shutdown section vbox
//          |--separator
//          |--Suspend section vbox
//          |--separator
GtkWidget* create_power_settings_window(GtkWidget* parent)
{
    GtkWidget* top_window = NULL;
    GtkWidget* alignment  = NULL;
    GtkWidget* vbox       = NULL;
    GtkWidget* label      = NULL;
    GtkWidget* filler     = NULL;
    GtkWidget* separator  = NULL;

    bzero(g_plugged_radios, 2);
    bzero(g_hibernate_radios, 5);
    //bzero(g_suspend_radios, 5);
    
    top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        _("Cancel"),
        GTK_RESPONSE_REJECT,
        _("Save & Close"),
        GTK_RESPONSE_ACCEPT,
        NULL);
    set_popup_window_style(GTK_WINDOW(top_window));

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("Power Management").
    label = gtk_label_new(_("Power Management"));
    gtk_widget_set_name(label, "irex-settings-pathbar");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    // The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The suspend settings section.
    create_suspend_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The auto shutdown settings section.
    create_auto_shutdown_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // Update widget with current settings.
    init_widgets_with_settings();
    gtk_widget_show_all(top_window);
    return top_window;
}


void load_power_settings()
{
    g_orig_power_settings.hibernate_if_plugged = get_value_bool(GCONF_HIBERNATE_IF_PLUGGED);
    g_orig_power_settings.seconds_suspend      = get_value_int (GCONF_SEC_SUSPEND);
    g_orig_power_settings.minutes_hibernate    = get_value_int (GCONF_MIN_HIBERNATE);

    g_cur_power_settings = g_orig_power_settings;
}


void save_power_settings()
{
    if (g_cur_power_settings.hibernate_if_plugged != g_orig_power_settings.hibernate_if_plugged)
    {
        set_value_bool(GCONF_HIBERNATE_IF_PLUGGED, g_cur_power_settings.hibernate_if_plugged);
    }

    if (g_cur_power_settings.seconds_suspend != g_orig_power_settings.seconds_suspend)
    {
        set_value_int(GCONF_SEC_SUSPEND, g_cur_power_settings.seconds_suspend);
    }

    if (g_cur_power_settings.minutes_hibernate != g_orig_power_settings.minutes_hibernate)
    {
        set_value_int(GCONF_MIN_HIBERNATE, g_cur_power_settings.minutes_hibernate);
    }

    LOGPRINTF("Saving power settings, done.");
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Automatic shutdown")
//    |--vbox
//       |--radio
//       |--radio
//    |--label ("Choose the automatic shutdown time")
//    |--vbox
//       |--radio
//       |--radio
static GtkWidget* create_auto_shutdown_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;
    char       name[16];

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Automatic Shutdown".
    label = gtk_label_new(_("Automatic Shutdown"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);
    
    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i=0; i<2; i++)
    {
        if (i == 0)
        {
            // The radio button "Always".
            g_plugged_radios[i] = gtk_radio_button_new_with_label(NULL, _("Always use automatic shutdown"));
        }
        else
        {
            // The radio button "Only when running on battery".
            g_plugged_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_plugged_radios[0]), _("Only use automatic shutdown when running on battery"));
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_plugged_radios[i]),
            "toggled",
            G_CALLBACK(on_plugged_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_plugged_radios[i], FALSE, FALSE, 0);
    }

    // The label "Choose the automatic shutdown time"
    label = gtk_label_new(_("Choose the automatic shutdown time:"));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i = 0; i < 5; i++)
    {
        if (i == 0)
        {
            // The radio button "Never".
            g_hibernate_radios[i] = gtk_radio_button_new_with_label(NULL, _("Never"));
        }
        else
        {
            // The radio button "after ...".
            // TRANSLATORS: this is an ngettext(3) call which needs a singular and plural translation
            sprintf(name, ngettext("after %d minute", "after %d minutes", hibernate_times[i]), hibernate_times[i]);
            g_hibernate_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_hibernate_radios[0]), name);
        }

        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_hibernate_radios[i]),
            "toggled",
            G_CALLBACK(on_hibernate_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_hibernate_radios[i], FALSE, FALSE, 0);
    }

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Power Saving")
//    |--label ("Turn on sleep mode when the device is not in use:")
//    |--vbox
//       |--check
static GtkWidget* create_suspend_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);
    
    // The label "Power Saving".
    label = gtk_label_new(_("Power Saving"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "Suspend the device ..."
    label = gtk_label_new(_("Power saving is managed automatically and is optimised for battery performance. Disabling this setting will seriously impact the battery life of your device."));
    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The vbox containing the checkbox.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);
    
    g_use_powersaving_check = gtk_check_button_new_with_label(_("Use power saving for better battery performance"));
    g_signal_connect(G_OBJECT(g_use_powersaving_check),
        "toggled",
        G_CALLBACK(on_suspend_changed),
        NULL);
    gtk_box_pack_start(GTK_BOX(vbox), g_use_powersaving_check, FALSE, FALSE, 0);

    return top_level_vbox;
}

static void on_plugged_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_power_settings.hibernate_if_plugged = (index == 0) ? TRUE : FALSE;
    }
}


static void on_hibernate_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    g_cur_power_settings.minutes_hibernate = hibernate_times[index];
}

static void on_suspend_changed(GtkWidget *widget, gpointer data)
{
    g_cur_power_settings.seconds_suspend = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 10 : -1;
}

static void init_widgets_with_settings()
{
    int i;
    int to_be_activated = 0;

    to_be_activated = g_cur_power_settings.hibernate_if_plugged == TRUE ? 0 : 1;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_plugged_radios[to_be_activated]),
                                 TRUE);

    for (i = 0; i < 5; i++)
    {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_hibernate_radios[i]), 
                (hibernate_times[i]==g_cur_power_settings.minutes_hibernate));
    }
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_use_powersaving_check),
            (g_cur_power_settings.seconds_suspend!=-1));
}

