/*
 * File Name: ipc.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>

// ereader include files, between < >
#include <liberipc/eripc.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "settings.h"
#include "sd_settings.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct 
{
    eripc_handler_t *handler;
    const gchar     *message_name;
    const gchar     *interface;
    gint            handler_id;
} FuncEntry;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

// IPC settings
#define DBUS_APPL_NAME           PACKAGE_NAME
#define DBUS_SERVICE             "com.irexnet."DBUS_APPL_NAME
#define DBUS_PATH                "/com/irexnet/"DBUS_APPL_NAME
#define DBUS_INTERFACE           "com.irexnet."DBUS_APPL_NAME

// IPC popup menu
#define DBUS_SERVICE_POPUP_MENU  "com.irexnet.popupmenu"
#define DBUS_SERVICE_SYSD        "com.irexnet.sysd"
#define DBUS_SERVICE_CTB         "com.irexnet.ctb"

// Popup menu strings
static const char* action_group  = "settings_action";
static const char* action_close  = "close";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_context_t *g_eripc_context = NULL;

                 
//============================================================================
// Local Function Definitions
//============================================================================

static void on_menu_item         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_mounted           ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_unmounted         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_locale    ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_window_activated  ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static gboolean ipc_menu_add_menu(const char *name,
                                  const char *label,
                                  const char *group1,
                                  const char *group2,
                                  const char *group3,
                                  const char *group4);

static gboolean ipc_menu_add_group(const char *name,
                                   const char *parent,
                                   const char *label,
                                   const char *icon_name);
                                   
static gboolean ipc_menu_add_item(const char* name,
                                  const char* parent,
                                  const char* label);

gboolean ipc_menu_set_item_state (const char *name,
                                  const char *parent,
                                  const char *state);

// Exported DBUS API list
static FuncEntry g_service_functions[] =
                 {
                     { on_menu_item,         "menuItemActivated",   NULL,              0 },
                     { on_window_activated,  "activatedWindow",     NULL,              0 },
                     { on_mounted,           "sysVolumeMounted",    NULL,              0 },
                     { on_mounted,           "sysVolumeMounted",    DBUS_SERVICE_SYSD, 0 },
                     { on_unmounted,         "sysVolumeUnmounted",  DBUS_SERVICE_SYSD, 0 },
                     { on_changed_locale,    "sysChangedLocale",    DBUS_SERVICE_SYSD, 0 },
                     { NULL }  // end of list
                 };


//============================================================================
// Functions Implementation
//============================================================================

gboolean ipc_set_services()
{
    eripc_error_t retval;
    int i = 0;

    LOGPRINTF("entry");
    g_assert(g_eripc_context == NULL);

    // Initialize IPC context
    g_eripc_context = eripc_init(DBUS_APPL_NAME, "1.0", NULL);
    if (g_eripc_context == NULL)
    {
        ERRORPRINTF("Failed to initialize eripc context\n");
        return FALSE;
    }

    for (i = 0; g_service_functions[i].handler != NULL; i++)
    {
        if (g_service_functions[i].interface)
        {
            // install signal handler
            retval = eripc_set_signal_handler( g_eripc_context,
                                               g_service_functions[i].handler,
                                               NULL,
                                               ERIPC_BUS_SESSION,
                                               g_service_functions[i].interface,
                                               g_service_functions[i].message_name,
                                               &(g_service_functions[i].handler_id) );
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_signal_handler [%s] returns [%d]",
                             g_service_functions[i].message_name,
                             retval );
                return FALSE;
            }
        }
        else
        {
            // install message handler
            retval = eripc_set_message_handler( g_eripc_context,
                                                g_service_functions[i].handler,
                                                NULL,
                                                ERIPC_BUS_SESSION,
                                                DBUS_INTERFACE,
                                                g_service_functions[i].message_name,
                                                &(g_service_functions[i].handler_id) );
              
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_message_handler [%s] returns [%d]",
                             g_service_functions[i].message_name,
                             retval );
                return FALSE;
            }
        }
    }

    return TRUE;
}


void ipc_unset_services()
{
    int i;

    LOGPRINTF("entry");

    // Uninitialize.
    for (i = 0; g_service_functions[i].handler != NULL; i++)
    {
        eripc_unset_handler(g_eripc_context, g_service_functions[i].handler_id);
    }   
}


void ipc_sys_startup_complete ( void )
{
    eripc_error_t   result;

    // parameters for ipc message
    const int       my_pid = getpid();
    const gboolean  is_multidoc = FALSE;

    LOGPRINTF("entry");
    g_assert(g_eripc_context);

    // broadcast signal over session bus
    result = eripc_send_signal_varargs( g_eripc_context,
                                        ERIPC_BUS_SESSION,
                                        DBUS_PATH,
                                        DBUS_SERVICE_SYSD,
                                        "startupComplete",
                                        ERIPC_TYPE_STRING, DBUS_APPL_NAME,
                                        ERIPC_TYPE_INT,    my_pid,
                                        ERIPC_TYPE_BOOL,   is_multidoc,
                                        ERIPC_TYPE_STRING, DBUS_SERVICE,
                                        ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
    } 
}


gboolean ipc_sys_set_busy( gboolean on )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;
     
    LOGPRINTF( "entry: on [%d]", on );
    g_assert( g_eripc_context );
     
    result = eripc_send_string( g_eripc_context,
                                NULL,                       // reply handler
                                NULL,                       // user data
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_SYSD,
                                "sysSetBusyLed",
                                (on ? "on" : "off"));
     
    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }
     
    return ok;
}


gboolean ipc_menu_show_menu( const char *name )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s]", name );
    g_assert( g_eripc_context );

    result = eripc_send_varargs( g_eripc_context,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "showMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}


gboolean ipc_remove_menu( const char *name )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s]", name );
    g_assert( g_eripc_context );

    result = eripc_send_varargs( g_eripc_context,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "removeMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}


gboolean ipc_menu_popup(const char* state)
{
    eripc_error_t result;

    LOGPRINTF("entry");

    result = eripc_send_varargs(g_eripc_context,
                                 NULL,
                                 NULL,
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_SYSD,
                                 "menuRequestPopup",
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        return FALSE;
    }
    return TRUE;
}


void ipc_menu_enable()
{
    ipc_menu_set_item_state(action_close, action_group, "NORMAL");
}


void ipc_menu_disable()
{
    ipc_menu_set_item_state(action_close, action_group, "DISABLED");
}

gboolean ipc_set_stylus(const char* state)
{
    eripc_error_t result;

    LOGPRINTF("entry");

    result = eripc_send_varargs(g_eripc_context,
                                 NULL,
                                 NULL,
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_SYSD,
                                 "sysSetStylus",
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        return FALSE;
    }
    return TRUE;
}


void menu_init()
{
    LOGPRINTF("entry");
    
    ipc_menu_add_group(action_group,  NULL,  _("Actions"), NULL);
    ipc_menu_add_item(action_close, action_group, _("Close"));

    ipc_menu_add_menu(SETTINGS_MENU, _("Menu"), action_group, NULL, NULL, NULL);

    ipc_menu_show_menu(SETTINGS_MENU);
}


gboolean get_device_capabilities(gboolean* has_stylus,
                                 gboolean* has_wifi,
                                 gboolean* has_bluetooth)
{
    gboolean result = FALSE;
    eripc_error_t retval;
    eripc_event_info_t *info = NULL;

    LOGPRINTF("entry");

    retval = eripc_send_varargs_and_wait(g_eripc_context,
                                         &info,
                                         ERIPC_BUS_SESSION,
                                         DBUS_SERVICE_SYSD,
                                         "sysGetDeviceCapabilities",
                                         ERIPC_TYPE_INVALID);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", retval);
    }
    else if (info == NULL || info->args == NULL)
    {
        ERRORPRINTF("sysd returns OK but no reply structure");
    }
    else
    {
        const eripc_arg_t *arg_array = info->args;

        if ((has_stylus != NULL) && (arg_array[0].type == ERIPC_TYPE_BOOL))
        {
            *has_stylus = (gboolean) arg_array[0].value.b;
        }

        if ((has_wifi!= NULL) && (arg_array[1].type == ERIPC_TYPE_BOOL))
        {
            *has_wifi = (gboolean) arg_array[1].value.b;
        }

        if ((has_bluetooth != NULL) && (arg_array[2].type == ERIPC_TYPE_BOOL))
        {
            *has_bluetooth = (gboolean) arg_array[2].value.b;
        }

        LOGPRINTF("Reply received: capabilities stylus %d, wifi %d, bluetooth %d", 
                *has_stylus, *has_wifi, *has_bluetooth);
        
        result = TRUE;
    }
    
    eripc_event_info_free(g_eripc_context, info);
    
    return result;
}


void mount_sd_card()
{
    eripc_error_t retval;

    retval = eripc_send_varargs(g_eripc_context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_SYSD,
                                "sysCardMount",
                                ERIPC_TYPE_INVALID);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", retval);
    }
}


void unmount_sd_card()
{
    eripc_error_t retval;

    retval = eripc_send_varargs(g_eripc_context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_SYSD,
                                "sysCardUnmount",
                                ERIPC_TYPE_INVALID);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", retval);
    }
}


gboolean prepare_sd_card( void )
{
    gboolean            result = FALSE;    // return value: TRUE = ok, FALSE = error
    eripc_error_t       retval;
    eripc_event_info_t  *reply = NULL;
    int                 old_timeout = 0;
    gboolean            timeout_set = FALSE;

    LOGPRINTF("entry");

    // set ipc timeout to "long"
    result = eripc_get_timeout(g_eripc_context, &old_timeout);
    if (result == ERIPC_ERROR_SUCCESS)
    {
        // set timeout in milliseconds
        result = eripc_set_timeout(g_eripc_context, 60*1000);
        if (result == ERIPC_ERROR_SUCCESS)
        {
            timeout_set = TRUE;
        }
    }

    // send ipc message and wait for a reply
    retval = eripc_send_varargs_and_wait( g_eripc_context,
                                          &reply,                // reply structure
                                          ERIPC_BUS_SESSION,
                                          DBUS_SERVICE_CTB,
                                          "ctbPrepareStorage",
                                          ERIPC_TYPE_INVALID );

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", retval);
    }
    else if ( reply == NULL || reply->args == NULL)
    {
        ERRORPRINTF("ctb returns OK but no reply structure");
    }
    else
    {
        const eripc_arg_t *arg_array = reply->args;

        if (arg_array[0].type == ERIPC_TYPE_BOOL)
        {
            result = (gboolean) arg_array[0].value.b;
        }

        LOGPRINTF("Reply received: result %d", result);
    }

    // restore ipc timeout
    if (timeout_set)
    {
        eripc_set_timeout(g_eripc_context, old_timeout);
    }

    eripc_event_info_free(g_eripc_context, reply);

    return result;
}


//============================================================================
// Local Functions Implementation
//============================================================================

static gboolean ipc_menu_add_menu(const char *name,
                                  const char *label,
                                  const char *group1,
                                  const char *group2,
                                  const char *group3,
                                  const char *group4)
{
    eripc_error_t result;

    LOGPRINTF("entry");

    result = eripc_send_varargs(g_eripc_context,
                                0,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_POPUP_MENU,
                                "addMenu",
                                ERIPC_TYPE_STRING, name,
                                ERIPC_TYPE_STRING, label,
                                ERIPC_TYPE_STRING, DBUS_SERVICE,
                                ERIPC_TYPE_STRING, group1,
                                ERIPC_TYPE_STRING, group2,
                                ERIPC_TYPE_STRING, group3,
                                ERIPC_TYPE_STRING, group4,
                                ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        return FALSE;
    }
    return TRUE;
}


static gboolean ipc_menu_add_group(const char *name,
                                   const char *parent,
                                   const char *label,
                                   const char *icon_name)
{
    eripc_error_t result;

    LOGPRINTF("entry");

    result = eripc_send_varargs(g_eripc_context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_POPUP_MENU,
                                "addGroup",
                                ERIPC_TYPE_STRING, name,
                                ERIPC_TYPE_STRING, parent,
                                ERIPC_TYPE_STRING, label,
                                ERIPC_TYPE_STRING, icon_name,
                                ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        return FALSE;
    }
    else
    {
        LOGPRINTF("Add group: %s %s!", name, parent);
    }
    return TRUE;
}


static gboolean ipc_menu_add_item(const char* name,
                                  const char* parent,
                                  const char* label)
{
    eripc_error_t result;

    LOGPRINTF("entry");

    result = eripc_send_varargs(g_eripc_context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_POPUP_MENU,
                                "addItem",
                                ERIPC_TYPE_STRING, name,
                                ERIPC_TYPE_STRING, parent,
                                ERIPC_TYPE_STRING, label,
                                ERIPC_TYPE_STRING, name,
                                ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        return FALSE;
    }
    else
    {
        LOGPRINTF("Add item %s %s by ipc done!", name, parent);
    }
    return TRUE;
}


// set the state of a menu item
gboolean ipc_menu_set_item_state( const char *name,
                                  const char *parent,
                                  const char *state  )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] state [%s]", name, parent, state );

    result = eripc_send_varargs( g_eripc_context,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setItemState",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// user has pressed a menu button
static void menu_on_item_activated(const gchar *item,
                                   const gchar *group,
                                   const gchar *menu,
                                   const gchar *state)
{
    if (strcmp(group, action_group) == 0)
    {
        if (strcmp(item, action_close) == 0)
        {
            // Close settings application.
            main_quit();
        }
    }
}


// menu item activated
static void on_menu_item ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[1].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[2].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[3].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *item      = arg_array[0].value.s;
    const char        *group     = arg_array[1].value.s;
    const char        *menu      = arg_array[2].value.s;
    const char        *state     = arg_array[3].value.s;

    if (item && group && menu && state)
    {
        menu_on_item_activated( item, group, menu, state );
    }

    return;
}


static void on_mounted(eripc_context_t          *context,
                       const eripc_event_info_t *info,
                       void                     *user_data)
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    LOGPRINTF("mountpoint=%s", mountpoint);
    if (mountpoint && strcmp(mountpoint, SD_CARD_MOUNTPOINT) == 0)
    {
        sd_card_mounted(TRUE);
    }
}


static void on_unmounted(eripc_context_t          *context,
                         const eripc_event_info_t *info,
                         void                     *user_data)
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    LOGPRINTF("mountpoint=%s", mountpoint);
    if (mountpoint && strcmp(mountpoint, SD_CARD_MOUNTPOINT) == 0)
    {
        sd_card_mounted(FALSE);
    }
}


static void on_window_activated(eripc_context_t          *context,
                                const eripc_event_info_t *info,
                                void                     *user_data)
{
    LOGPRINTF("entry");
    gboolean          result      = FALSE; 
    const eripc_arg_t *arg_array  = info->args;

    if (arg_array[0].type == ERIPC_TYPE_INT)
    {
        ipc_menu_show_menu(SETTINGS_MENU);
        result = TRUE;
    }

    // return result to caller
    eripc_reply_bool(context, info->message_id, result);
}


static void on_changed_locale (eripc_context_t          *context,
                               const eripc_event_info_t *info,
                               void                     *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (arg_array[0].type == ERIPC_TYPE_STRING)
    {
        const char *locale = arg_array[0].value.s;
        if (locale)
        {
            const char *old_locale = g_getenv("LANG");
            
            if (!old_locale || (strcmp(old_locale, locale) != 0))
            {
                LOGPRINTF("Locale has changed to %s", locale);

                // set locale in environment 
                g_setenv("LANG", locale, TRUE);
                setlocale(LC_ALL, "");
                
                // Update text in main window.
                update_main_window_text();

                // Re-initialize popup menu.
                ipc_remove_menu(SETTINGS_MENU);
                menu_init();
            }
        }
    }
}
