/*
 * File Name: firstboot.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

#define KEYS_SYSTEM                 "/apps/er/sys"
#define KEY_GLOBAL_LOCALE           KEYS_SYSTEM "/global/locale"
#define KEY_GLOBAL_FIRST_BOOT       KEYS_SYSTEM "/global/is_first_boot"
#define KEY_GLOBAL_LIST_LOCALES     KEYS_SYSTEM "/global/list_locales"
#define KEY_GLOBAL_LIST_LANGUAGES   KEYS_SYSTEM "/global/list_languages"

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// screen layout
static const int        TITLE_TOP_PADDING       =  40;
static const int        TITLE_BOTTOM_PADDING    =  40;
static const int        TITLE_LEFT_PADDING      =   0;
static const int        TITLE_RIGHT_PADDING     =   0;

static const int        BUTTONS_TOP_PADDING     =   0;
static const int        BUTTONS_BOTTOM_PADDING  =  24;
static const int        BUTTONS_LEFT_PADDING    =   0;
static const int        BUTTONS_RIGHT_PADDING   =   0;

static const int        BUTTONS_SPACING         =   2;

// screen elements
static GtkLabel         *g_settings_title       = NULL;
static GtkButton        *g_button_active        = NULL;

// locale elements
static GSList           *locale_code_list       = NULL;
static GSList           *locale_lang_list       = NULL;
static const char       *locale_set             = NULL;

static GConfClient      *g_gconfclient          = NULL;

FILE *fplog = NULL;

//============================================================================
// Local Function Definitions
//============================================================================

static void       on_sigterm(int signo);
static void       on_destroy(GtkWidget * widget, gpointer data);

static GtkWidget* create_language_screen_screen_layout(void);
static GtkWidget* create_set_language(void);
static void       language_screen_screen_set_text();

static gboolean   event_button_focussed (GtkButton *button, GdkEventFocus *event, gchar *locale);
static void       event_button_clicked  (GtkButton *button, gchar *locale);


//============================================================================
// Functions Implementation
//============================================================================

int main(int argc, char *argv[])
{
    GtkWindow        *window;
    GtkWidget        *widget;
    struct sigaction on_term;

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);

    // init domain for translations
    textdomain(GETTEXT_PACKAGE);
    
    // init gtk
    gtk_init(&argc, &argv);

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // create the main, top level, window
    widget = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    window = GTK_WINDOW(widget);
    gtk_window_set_title(window, PACKAGE " " VERSION);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_window_fullscreen(GTK_WINDOW(window));
    gtk_widget_show(widget);

    g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(on_destroy), NULL);

    // add screen details
    widget = create_set_language();
    gtk_container_add(GTK_CONTAINER(window), widget);

    // set focus to active language button
    if (g_button_active)
    {
        gtk_widget_grab_focus(GTK_WIDGET(g_button_active));
    }
    
    // run the main loop
    WARNPRINTF("-- before gtk_main");
    gtk_main();

    return 0;
}


//============================================================================
// Local Functions Implementation
//============================================================================

static GtkWidget* create_set_language (void)
{
    GtkWidget   *background = NULL;     // return value
    GtkWidget   *widget     = NULL;
    char        *old_locale = NULL;
    
    LOGPRINTF("entry");

    g_gconfclient = gconf_client_get_default();
    if (g_gconfclient==NULL)
    {
        ERRORPRINTF("Failed to connect to GConf");
        exit(1);
    }

    
    locale_set = g_getenv("LANG");
    if (locale_set == NULL)
    {
        WARNPRINTF("env LANG not set");
    }
    
    old_locale  = gconf_client_get_string(g_gconfclient, KEY_GLOBAL_LOCALE, NULL);
    if (old_locale == NULL)
    {
        WARNPRINTF("GConf locale not set");
    }

    if (locale_set == NULL)
    {
        if (old_locale != NULL)
        {
            // use GConf setting
            locale_set = g_strdup(old_locale);
            LOGPRINTF("using locale from GConf: %s", locale_set);
        }
        else
        {
            // fall back to default
            locale_set = g_strdup_printf("en_GB");
            LOGPRINTF("using default locale: %s", locale_set);
        }
    }
    else
    {
        LOGPRINTF("using locale from env LANG: %s", locale_set);
    }
       
    // create screen layout
    widget = create_language_screen_screen_layout();
    background = widget;
    
    return background;
}


static GtkWidget* create_language_screen_screen_layout()
{
    GtkWidget   *background = NULL;  // return value
    GtkWidget   *widget;
    GtkWidget   *alignment;
    GtkBox      *vbox;
    GtkBox      *buttonbox;
    
    LOGPRINTF("entry");

    locale_code_list = gconf_client_get_list(g_gconfclient, KEY_GLOBAL_LIST_LOCALES, GCONF_VALUE_STRING, NULL);
    locale_lang_list = gconf_client_get_list(g_gconfclient, KEY_GLOBAL_LIST_LANGUAGES, GCONF_VALUE_STRING, NULL);
    
    if (g_slist_length(locale_code_list) != g_slist_length(locale_lang_list))
    {
        ERRORPRINTF("Locale code and language lists are not of equal length");
        return NULL;
    }

    // object hierarchy:
    //     background (eventbox)
    //       |
    widget = gtk_event_box_new();
    gtk_widget_set_name(widget, "irex-first-boot-background");
    gtk_widget_show(widget);
    background = widget;

    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
    
    //             |
    //             |-- alignment
    //             |     |
    widget = gtk_alignment_new(0.5, 0.0, 0.0, 0.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               TITLE_TOP_PADDING,
                               TITLE_BOTTOM_PADDING,
                               TITLE_LEFT_PADDING,
                               TITLE_RIGHT_PADDING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    alignment = widget;
    
    //             |     |
    //             |     |-- label
    //             |
    widget = gtk_label_new(NULL);
    g_settings_title = GTK_LABEL(widget);
    gtk_widget_set_name(widget, "irex-first-boot-title");
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);
    
    //             |
    //             |-- alignment (expanding spacer)
    //             |
    widget = gtk_alignment_new(0.0, 1.0, 0.0, 1.0);
    gtk_box_pack_start(vbox, widget, TRUE, FALSE, 0);
    gtk_widget_show(widget);
    alignment = widget;
    
    //             |
    //             |-- alignment
    //                   |
    widget = gtk_alignment_new(0.5, 0.0, 0.0, 0.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               BUTTONS_TOP_PADDING,
                               BUTTONS_BOTTOM_PADDING,
                               BUTTONS_LEFT_PADDING,
                               BUTTONS_RIGHT_PADDING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    alignment = widget;
    
    //                   |
    //                   |-- vbox
    //                         |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);
    buttonbox = GTK_BOX(widget);

    //                         |
    //                         |-- buttons (*)
    //
    GSList *code_ptr = locale_code_list;
    GSList *lang_ptr = locale_lang_list;
    
    while ((code_ptr != NULL) && (lang_ptr != NULL))
    {
        widget = gtk_button_new_with_label(lang_ptr->data);
        gtk_widget_set_name(widget, "irex-first-boot-button");
        gtk_signal_connect(GTK_OBJECT(widget), "focus-in-event", GTK_SIGNAL_FUNC(event_button_focussed), (gpointer) code_ptr->data);
        gtk_signal_connect(GTK_OBJECT(widget), "clicked", GTK_SIGNAL_FUNC(event_button_clicked), (gpointer) code_ptr->data);
        gtk_box_pack_start(buttonbox, widget, FALSE, FALSE, BUTTONS_SPACING);
        gtk_widget_show (widget);
        
        // save button for active languange to focus later
        if (strcmp(locale_set, code_ptr->data) == 0)
        {
            g_button_active = GTK_BUTTON(widget);
        }
 
        code_ptr = code_ptr->next;
        lang_ptr = lang_ptr->next;
    }
       
    // add screen texts
    language_screen_screen_set_text();

    return background;
}


static void language_screen_screen_set_text()
{
    if (g_settings_title)
    {
        gtk_label_set_text(g_settings_title, _("Please select your language"));
    }
}


static gboolean event_button_focussed(GtkButton *button, GdkEventFocus *event, gchar *locale)
{
    // temp save setting
    LOGPRINTF("set language %s", (char*) locale);

    // change locale immediately
    g_setenv("LANG", locale, TRUE);
    setlocale(LC_ALL, "");
    locale_set = locale;

    // update screen texts
    language_screen_screen_set_text();
    
    // propagate the event further
    return FALSE;
}


static void event_button_clicked(GtkButton *button, gchar *locale)
{
    char *old_locale;
    
    // store new locale in gconf
    //
    old_locale  = gconf_client_get_string(g_gconfclient, KEY_GLOBAL_LOCALE, NULL);
    
    if ((old_locale == NULL) || (strcmp(old_locale, locale_set) != 0) )
    {
        LOGPRINTF("store new locale in gconf: %s (was: %s)", (char*) locale_set, old_locale);
        gconf_client_set_string(g_gconfclient, KEY_GLOBAL_LOCALE, (char*) locale_set, NULL);
    }
    else
    {
        LOGPRINTF("locale not changed: %s", (char*) locale_set);
    }

    // reset First Boot flag 
    gconf_client_set_bool(g_gconfclient, KEY_GLOBAL_FIRST_BOOT, FALSE, NULL);
    
    // sleep a second to allow system to switch language
    usleep(1000*1000);
    
    // release resources and quit
    //
    g_object_unref(g_gconfclient);
    g_slist_free(locale_code_list);
    g_slist_free(locale_lang_list);

    gtk_main_quit();
}


// signal handlers
static void on_sigterm(int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());
    gtk_main_quit();
    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


// terminate the main loop
static void on_destroy(GtkWidget * widget, gpointer data)
{
    WARNPRINTF("entry");
    gtk_main_quit();
}
