/*
 * File Name: about_info.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#ifndef WIN32
#include <sys/utsname.h>
#include <dirent.h>
#endif

// ereader include files, between < >

// local include files, between " "
#include "i18n.h"
#include "ipc.h"
#include "log.h"
#include "settings.h"
#include "settings_utils.h"
#include "about_info.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct
{
    char* title;
    char* value;
} InfoItem;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

#define INFORMATION_ITEMS_INDENT     40
#define ITEM_TITLE_VALUE_SPACING     30

enum InfoItemType
{
    SERIAL_NUMBER,
    PRODUCT_TYPE,
    PCBA_REVISION,
    EPD_ID,
    SOFTWARE_VERSION,
    SOFTWARE_BUILDNO,
    KERNEL_VERSION,
    MICRO_APP_VER,
    MOBIPOCKET_PID,
    ITEM_TYPE_COUNT
};


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static InfoItem g_info_items[ITEM_TYPE_COUNT] =
{
    {NULL, NULL},
    {NULL, NULL},
    {NULL, NULL},
    {NULL, NULL},
    {NULL, NULL},
    {NULL, NULL},
    {NULL, NULL}
};


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_device_info_widgets(GtkBox* parent);
static GtkWidget* create_acknowledgements_widgets(GtkBox* parent);
static char* get_kernel_version();
static char* get_sub_dir(const char* parent, const char* sub_dir);
static char* get_micro_app_version();
static gchar* get_file_content(const gchar* file_path);


//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--device info section vbox
//          |--separator
//          |--acknowledgements section vbox
//          |--separator
GtkWidget* create_about_info_window(GtkWidget* parent)
{
    GtkWidget* top_window = NULL;
    GtkWidget* alignment  = NULL;
    GtkWidget* vbox       = NULL;
    GtkWidget* label      = NULL;
    GtkWidget* filler     = NULL;
    GtkWidget* separator  = NULL;

    top_window = gtk_dialog_new_with_buttons("",
        GTK_WINDOW(parent),
        GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
        _("Close"),
        GTK_RESPONSE_ACCEPT,
        NULL);
    set_popup_window_style(GTK_WINDOW(top_window));

    // Alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                              WINDOW_TOP_PADDING,
                              WINDOW_BOTTOM_PADDING,
                              WINDOW_H_PADDING,
                              WINDOW_H_PADDING);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(top_window)->vbox), alignment);

    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SECTION_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), vbox);

    // The label ("About").
    label = gtk_label_new(_("About"));
    gtk_widget_set_name(label, "irex-settings-pathbar");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

    // The filler.
    filler = gtk_event_box_new();
    gtk_widget_set_name(filler, "irex-settings-prev-page-filler");
    gtk_widget_set_size_request(filler, -1, SETTINGS_FILLER_HEIGHT);
    gtk_box_pack_start(GTK_BOX(vbox), filler, FALSE, FALSE, 0);

    // The device info section.
    create_device_info_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The acknowledgements section.
    create_acknowledgements_widgets(GTK_BOX(vbox));

    // Add separator here.
    separator = create_separator_widgets();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);
    
    gtk_widget_show_all(top_window);
    return top_window;
}


void load_about_info()
{
    int i = 0;
    for (i=0; i<ITEM_TYPE_COUNT; i++)
    {
        // Free existing values
        // TODO: fix memory leak in this function
        if (g_info_items[i].value)
        {
            g_free(g_info_items[i].value);
        }

        switch (i)
        {
        case SERIAL_NUMBER:
            g_info_items[i].title = _("Serial Number:");
            g_info_items[i].value = get_file_content("/sys/devices/system/sysset/sysset0/fasm/serial");
            break;

        case PRODUCT_TYPE:
            g_info_items[i].title = _("Product Type:");
            g_info_items[i].value = g_strconcat(
                            get_file_content("/sys/devices/system/sysset/sysset0/fasm/category"), 
                            get_file_content("/sys/devices/system/sysset/sysset0/fasm/model"), 
                            "-",
                            get_file_content("/sys/devices/system/sysset/sysset0/fasm/region"), 
                            NULL);
            break;

        case PCBA_REVISION:
            g_info_items[i].title = _("PCBA Revision:");
            g_info_items[i].value = get_file_content("/sys/devices/system/sysset/sysset0/mpcb/pcb_rev");
            break;

        case EPD_ID:
            g_info_items[i].title = _("EPD ID:");
            g_info_items[i].value = g_strconcat(
                            get_file_content("/sys/devices/system/sysset/sysset0/display/fpl"),
                            get_file_content("/sys/devices/system/sysset/sysset0/display/waveform"),
                            NULL);
            break;

        case SOFTWARE_VERSION:
            g_info_items[i].title = _("Software Version:");
            g_info_items[i].value = get_file_content("/etc/er_version_commercial");
            break;

        case SOFTWARE_BUILDNO:
            g_info_items[i].title = _("Software Build Number:");
            g_info_items[i].value = g_strconcat(
                            get_file_content("/etc/er_version"),
                            " (",
                            get_file_content("/etc/version"),
                            ")",
                            NULL);
            break;

        case KERNEL_VERSION:
            g_info_items[i].title = _("Kernel Version:");
            g_info_items[i].value = get_kernel_version();
            break;

        case MICRO_APP_VER:
            g_info_items[i].title = _("Micro Application Version:");
            g_info_items[i].value = get_micro_app_version();
            break;

        case MOBIPOCKET_PID:
            g_info_items[i].title = _("Mobipocket PID:");
            g_info_items[i].value = get_file_content("/sys/devices/system/sysset/sysset0/software/mobi_pid");
            break;
        
        default:
            break;
        }
    }
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("iRex Digital Reader 1000 S")
//    |--alignment
//       |--hbox
//          |--vbox
//             |--item title 1
//             |--item title 2
//             |--item title 3
//          |--vbox
//             |--item value 1
//             |--item value 2
//             |--item value 3
static GtkWidget* create_device_info_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;
    GtkWidget* alignment      = NULL;
    GtkWidget* hbox           = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "iRex Digital Reader 1000*".
    switch (get_device_type())
    {
    case IREX_DR1000:
        label = gtk_label_new(_("iRex Digital Reader 1000"));
        break;

    case IREX_DR1000S:
        label = gtk_label_new(_("iRex Digital Reader 1000 S"));
        break;

    case IREX_DR1000SW:
        label = gtk_label_new(_("iRex Digital Reader 1000 SW"));
        break;
    }

    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The alignment.
    alignment = gtk_alignment_new(0, 0, 1.0, 1.0);
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment),
                          0,
                          0,
                          INFORMATION_ITEMS_INDENT,
                          0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), alignment, FALSE, FALSE, 0);

    hbox = gtk_hbox_new(FALSE, ITEM_TITLE_VALUE_SPACING);
    gtk_container_add(GTK_CONTAINER(alignment), hbox);

    // The vbox containing the item titles.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    // The item titles goes here.
    for (i=0; i<ITEM_TYPE_COUNT; i++)
    {
        label = gtk_label_new(g_info_items[i].title);
        gtk_widget_set_name(label, "irex-info-item-title");
        gtk_widget_set_size_request(label, -1, -1);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
        gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    }

    // The vbox containing the item value.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    // The item titles goes here.
    for (i=0; i<ITEM_TYPE_COUNT; i++)
    {
        label = gtk_label_new(g_info_items[i].value);
        gtk_widget_set_name(label, "irex-normal-text");
        gtk_widget_set_size_request(label, -1, -1);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
        gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    }

    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Acknowledgements")
//    |--label ("iRex would like...")
static GtkWidget* create_acknowledgements_widgets(GtkBox* parent)
{
    GtkWidget* top_level_vbox = NULL;
    GtkWidget* label          = NULL;

    // Top level vbox.
    top_level_vbox = gtk_vbox_new(FALSE, ITEM_SPACING);
    gtk_box_pack_start(parent, top_level_vbox, FALSE, FALSE, 0);

    // The label "Lights".
    label = gtk_label_new(_("Acknowledgements"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    // The label "iRex would like..."
    label = gtk_label_new(_("The realization of this Digital Reader product is the result of a collective effort; "
                            "our collaboration with many, many people over the years. iRex is grateful to all of "
                            "them because we learned everything we know about how to create the best possible "
                            "e-paper reading solution from them. Thanks are also due to the iRex team members who "
                            "cheerfully accepted the challenge to articulate our collective vision and ideas, "
                            "and who relentlessly worked to turn these into reality. We owe special thanks to "
                            "the spouses and partners of our iRex team members, who suffered by having been alone "
                            "for many, many long days, nights and weekends."));

    gtk_widget_set_name(label, "irex-normal-text");
    gtk_widget_set_size_request(label, SETTINGS_LABEL_MAX_WIDTH, -1);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);

    return top_level_vbox;
}


static gchar* get_file_content(const gchar* file_path)
{
    gchar* contents = NULL;
    gsize  len      = 0;

    if (g_file_get_contents(file_path, &contents, &len, NULL) == FALSE)
    {
        return NULL;
    }

    // Remove trailing '\n' characters
    // End of string may have more than one \0 char
    while (contents[len - 1] == '\n' || contents[len - 1] == '\0')
    {
        contents[len - 1] = '\0';
        len--;
    }
    return contents;
}


static char* get_kernel_version()
{
#ifndef WIN32
    struct utsname utsbuf;
    unsigned int   build_no;

    if (-1 == uname(&utsbuf))
    {
        // An error occurred.
        return NULL;
    }

    sscanf(utsbuf.version, "#%d", &build_no);
    return g_strdup_printf("%s (%d)", utsbuf.release, build_no);
#else
    return NULL;
#endif
}


static char* get_sub_dir(const char* parent, const char* sub_dir)
{
    GDir* dir  = g_dir_open(parent, 0, NULL);
    const char* file = NULL;
    char  full_path[256];

    if (dir == NULL)
    {
        return NULL;
    }

    while ((file = g_dir_read_name(dir)) != NULL)
    {
        // Skip regular file.
        sprintf(full_path, "%s/%s", parent, file);

        if (g_file_test(full_path, G_FILE_TEST_IS_DIR) == TRUE &&
            g_str_has_prefix(file, sub_dir) == TRUE)
        {
            // Sub dir found.
            return g_strdup_printf("%s/%s", parent, file);
        }
    }

    g_dir_close(dir);
    return NULL;
}


static char* get_micro_app_version()
{
    char* serio = get_sub_dir("/sys/devices", "serio");
    char* app_ver_path = NULL;
    char* content      = NULL;

    if (serio == NULL)
    {
        ERRORPRINTF("Can't find the micro device name!");
        return NULL;
    }

    app_ver_path = g_strdup_printf("%s/micro_ver", serio);
    content = get_file_content(app_ver_path);

    g_free(app_ver_path);
    g_free(serio);
    return content;
}
