#ifndef __FILETYPES_H__
#define __FILETYPES_H__

/**
 * File Name  : filetypes.h
 *
 * Description: Content browser filetype specific utilities
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <gdk/gdk.h>
#include <glib.h>

#include "filemodel.h"

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 

#define FILE_EXT_SHORTCUT               "desktop"
#define FILE_EXT_SHORTCUT_TO_DIR        "directory"

#define FILENAME_WIN_AUTORUN            "autorun.inf"       // MS-Windows autorun file
#define FILENAME_WIN_THUMBNAILS         "Thumbs.db"         // MS-Windows thumbnails file


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================

/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_from_file_extension
 *
 * @brief  Get icon for specified file extension
 *
 * @param  [in]  file_ext - file extension for which icon is requested
 *                          or empty string
 * @param  [in]  is_directory - whether file_ext is a directory
 * @param  [in]  is_template - whether file_ext is a template
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pointer to pixbuf holding the requested icon
 *         or NULL on error.
 *         Lifetime of the pixbuf is guaranteed until the next call to this function,
 *         caller must g_object_ref() / _unref() the pixbuf when it needs the pixbuf longer than that.
 *
 *--------------------------------------------------------------------------*/
GdkPixbuf* get_icon_from_file_extension ( const gchar                 *file_ext,
                                          const gboolean              is_directory,
                                          const gboolean              is_template,
                                          const filemodel_thumbsize_t icon_size     );


/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_file_from_file_extension
 *
 * @brief  Get pathname of the icon file for the specified file extension
 *
 * @param  [in]  file_ext - file extension for which icon is requested
 *                          or empty string
 * @param  [in]  is_directory - whether file_ext is a directory
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pathname of icon file for filetype of the specified filename
 *         or NULL on error.
 *         Caller must free this with g_free()
 *
 *--------------------------------------------------------------------------*/
gchar* get_icon_file_from_file_extension ( const gchar                 *file_ext,
                                           const gboolean              is_directory,
                                           const filemodel_thumbsize_t icon_size     );


/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_parent_dir
 *
 * @brief  Get icon for parent directory
 *
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pointer to pixbuf holding the requested icon
 *         or NULL on error.
 *         Lifetime of the pixbuf is guaranteed until the next call to this function,
 *         caller must g_object_ref() / _unref() the pixbuf when it needs the pixbuf longer than that.
 *
 *--------------------------------------------------------------------------*/
GdkPixbuf* get_icon_parent_dir ( const filemodel_thumbsize_t icon_size );


/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_application
 *
 * @brief  Get icon for application (executable file)
 *
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pointer to pixbuf holding the requested icon
 *         or NULL on error.
 *         Lifetime of the pixbuf is guaranteed until the next call to this function,
 *         caller must g_object_ref() / _unref() the pixbuf when it needs the pixbuf longer than that.
 *
 *--------------------------------------------------------------------------*/
GdkPixbuf* get_icon_application ( const filemodel_thumbsize_t icon_size );


/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_file_application
 *
 * @brief  Get pathname of the icon file for application (executable file)
 *
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pathname of icon file for application
 *         or NULL on error.
 *         Caller must free this with g_free()
 *
 *--------------------------------------------------------------------------*/
gchar* get_icon_file_application ( const filemodel_thumbsize_t icon_size );


/**---------------------------------------------------------------------------
 *
 * Name :  get_icon_library
 *
 * @brief  Get icon for library (SD card)
 *
 * @param  [in]  icon_size - specifies which icon is requested
 *
 * @return Pointer to pixbuf holding the requested icon
 *         or NULL on error.
 *         Lifetime of the pixbuf is guaranteed until the next call to this function,
 *         caller must g_object_ref() / _unref() the pixbuf when it needs the pixbuf longer than that.
 *
 *--------------------------------------------------------------------------*/
GdkPixbuf* get_icon_library ( const filemodel_thumbsize_t icon_size );


/**---------------------------------------------------------------------------
 *
 * Name :  apply_icon_overlay_template
 *
 * @brief  Apply overlay for templates to the given icon
 *
 * @param  [in]     icon_size - specifies which icon overlay is requested
 * @param  [in/out] icon - a pixbuf holding a bitmap of the icon to be modified
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void apply_icon_overlay_template ( const filemodel_thumbsize_t icon_size,
                                         GdkPixbuf             *icon     );


/**---------------------------------------------------------------------------
 *
 * Name :  apply_icon_overlay_shortcut
 *
 * @brief  Apply overlay for shortcuts to the given icon
 *
 * @param  [in]     icon_size - specifies which icon overlay is requested
 * @param  [in/out] icon - a pixbuf holding a bitmap of the icon to be modified
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void apply_icon_overlay_shortcut ( const filemodel_thumbsize_t icon_size,
                                         GdkPixbuf             *icon     );


/**---------------------------------------------------------------------------
 *
 * Name :  get_type_descr_from_file_extension
 *
 * @brief  Get filetype description for specified file extension
 *
 * @param  [in]  file_ext - file extension for which a description is requested
 *                          or empty string
 * @param  [in]  is_directory - whether filename is a directory
 * @param  [in]  is_template - whether file_ext is a template
 *
 * @return filetype description
 *         or NULL when no description for the specified file extension
 *
 *--------------------------------------------------------------------------*/
const gchar* get_type_descr_from_file_extension ( const gchar    *file_ext,
                                                  const gboolean is_directory,
                                                  const gboolean is_template  );


/**---------------------------------------------------------------------------
 *
 * Name :  get_viewer_from_file_extension
 *
 * @brief  Get executable for viewer application to be used for specified file extension
 *
 * @param  [in]  file_ext - file extension for which viewer is requested
 *                          or empty string
 *
 * @return Command-line command for the requested viewer application
 *         or NULL when no viewer known for the specified filename
 *
 *--------------------------------------------------------------------------*/
const gchar* get_viewer_from_file_extension ( const gchar *file_ext );


/**---------------------------------------------------------------------------
 *
 * Name :  is_mobipocket_file_extension
 *
 * @brief  Tell whether the specified file extension is a mobipocket file or not
 *
 * @param  [in]  file_ext - file extension to be checked
 *                          or empty string
 *
 * @return TRUE  when file extension is a mobipocket file or
 *         FALSE when file extension not a mobipocket file, or error
 *
 *--------------------------------------------------------------------------*/
gboolean is_mobipocket_file_extension ( const gchar *file_ext );


/**---------------------------------------------------------------------------
 *
 * Name :  get_mobipocket_title
 *
 * @brief  Find document title in a mobipocket file
 *
 * @param  [in]  filepath - file in which to find a title
 *
 * @return title copied from the specified file
 *         or NULL on error
 *         Caller must free this string using g_string_free().
 *
 *--------------------------------------------------------------------------*/
GString* get_mobipocket_title ( const char* filepath );


/**---------------------------------------------------------------------------
 *
 * Name :  is_shortcut_file_extension
 *
 * @brief  Tell whether the specified file extension is a shortcut file or not
 *
 * @param  [in]  file_ext - file extension to be checked
 *                          or empty string
 *
 * @return TRUE  when file extension is a shortcut file or
 *         FALSE when file extension not a shortcut file, or error
 *
 *--------------------------------------------------------------------------*/
gboolean is_shortcut_file_extension ( const gchar *file_ext );


/**---------------------------------------------------------------------------
 *
 * Name :  is_hidden_filename
 *
 * @brief  Tell whether the specified filename is hidden or can be displayed
 *
 * @param  [in]  filename - filename to be checked
 *
 * @return TRUE  when filename is a hidden file or
 *         FALSE when filename can be displayed, or error
 *
 *--------------------------------------------------------------------------*/
gboolean is_hidden_filename ( const gchar *filename );


// copied from libgnome-2.6.1.1 gnome-util.c:
/**---------------------------------------------------------------------------
 *
 * g_extension_pointer:
 * @path: A filename or file path.
 *
 * Extracts the extension from the end of a filename (the part after the final
 * '.' in the filename).
 *
 * Returns: A pointer to the extension part of the filename, or a
 * pointer to the end of the string if the filename does not
 * have an extension.
 *
 *--------------------------------------------------------------------------*/
const char* g_extension_pointer ( const char *path );


G_END_DECLS

#endif // __FILETYPES_H__
