/*
 * File Name: storage.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <dirent.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

// local include files, between " "
#include "ctb_log.h"
#include "ctb_actions.h"
#include "filetypes.h"
#include "i18n.h"
#include "shortcut.h"
#include "storage.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const gchar      *STORAGE_SHORTCUT = "[Desktop Entry]\n"
                                            "Version=1.0\n"
                                            "Type=Directory\n"
                                            "Path=%s\n"
                                            "Name=%s\n";

static const gchar      *ERCONFTOOL_CMD_INIT_SD = "erconftool -c /apps/er";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// available storage devices
static GPtrArray        *g_storage_media = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static int       create_device_shortcut  ( const gchar      *filepath,
                                           const gchar      *mountpoint,
                                           const gchar      *title      );
                 
static gint      find_device             ( const gchar      *path );

static erMetadb* open_or_create_database ( const gchar      *directory );

static int       create_symlinks_per_device (
                                           const gchar      *dir_symlink,
                                           const gchar      *dir_desktop );

static int       prepare_desktop         ( const gchar *mountpoint );

static int       prepare_pc_application  ( const gchar *mountpoint );

static int       update_database         ( const gchar      *filename,
                                           const gboolean   is_internal,
                                           const gboolean   is_mounted  );
                 
static void      update_desktop_item     ( const gchar      *mountpoint,
                                           const gboolean   is_mounted );


//============================================================================
// Functions Implementation
//============================================================================

// initialise local data
void storage_init ( void )
{
    LOGPRINTF("entry");

    // enforce function is called only once
    static gboolean firsttime = TRUE;
    g_assert(firsttime);
    firsttime = FALSE;

    // set current storage devices to none
    g_storage_media = g_ptr_array_new();

    // remove desktop shortcuts to storage devices
    (void) system( "rm -f " DIR_DESKTOP_INTERNAL "/storage_*.directory" );
}


// update screen texts
void storage_set_text ( void )
{
    gint        idx;

    LOGPRINTF("entry");

    // re-create shortcut to internal SD card,
    // because the "Name=" key is language dependent
    idx = find_device(DIR_LIBRARY);
    if (idx >= 0)
    {
        update_desktop_item(DIR_LIBRARY, FALSE);
        update_desktop_item(DIR_LIBRARY, TRUE );
    }
}


// prepare for power down
void storage_quit ( void )
{
    gint        idx;
    gchar       **media = NULL;

    LOGPRINTF("entry");

    // remove all mountpoints from desktop
    for ( media = (gchar**)g_storage_media->pdata,  idx = 0 ;
          idx < g_storage_media->len ;
          media++,  idx++                                    )
    {
        update_desktop_item(*media, FALSE);
    }
}


// get currently mounted storage media
const GPtrArray* storage_get_media ( void )
{
    LOGPRINTF("entry");

    return g_storage_media;
}


// storage media mounted/unmounted
// return: -1 = not found, or index in g_storage_media (0 ..)
static gint find_device ( const gchar *path )
{
    gint        media_index = -1;       // return value
    gint        idx;
    gint        n;
    gboolean    found = FALSE;
    gchar       **media = NULL;

    LOGPRINTF("entry: mountpoint [%s]", path);

    // search media in available storage devices
    if (g_storage_media)
    {
        for ( media = (gchar**)g_storage_media->pdata,  idx = 0 ;
              !found  &&  idx < g_storage_media->len ;
              media++,  idx++                                    )
        {
            n = strlen( *media );
            if (   strncmp(path, *media, n) == 0
                && (   path[n] == '/'
                    || path[n] == '\0'            ) )
            {
                media_index = idx;
                found = TRUE;
            }
        }
    }

    LOGPRINTF("leave: media_index [%d]", media_index);
    return media_index;
}


// get iRex desktop directory for path
GString* storage_get_desktop_dir ( const gchar *location )
{
    GString     *irex_dir = NULL;     // return value
    const gchar *mountpoint = NULL;
    int         idx;
    gboolean    ok = TRUE;
    char        *cp;
    char        path[PATH_MAX];

    LOGPRINTF("entry: location [%s]", location);

    // get real path
    cp = realpath( location, path );
    if (cp != path)
    {
        ERRORPRINTF("location not present [%s]", location);
        ok = FALSE;
    }

    // get storage media
    if (ok)
    {
        idx = find_device( path );
        if (idx < 0)
        {
            ERRORPRINTF("no storage media for location [%s]", location);
            ok = FALSE;
        }
        else
        {
            mountpoint = g_ptr_array_index(g_storage_media, idx);
        }
    }

    // determine iRex private directory
    if (ok)
    {
        irex_dir = g_string_new( mountpoint );
        g_string_append(irex_dir, "/" DIR_IREX_PRIVATE "/" DIR_DESKTOP);
    }


    LOGPRINTF("leave [%s]", irex_dir ? irex_dir->str : NULL);
    return irex_dir;
}


// get recent files directory for path
GString* storage_get_recentfiles_dir ( const gchar *location )
{
    GString     *irex_dir = NULL;     // return value
    const gchar *mountpoint = NULL;
    int         idx;
    gboolean    ok = TRUE;
    char        *cp;
    char        path[PATH_MAX];

    LOGPRINTF("entry: location [%s]", location);

    // get real path
    cp = realpath( location, path );
    if (cp != path)
    {
        ERRORPRINTF("location not present [%s]", location);
        ok = FALSE;
    }

    // get storage media
    if (ok)
    {
        idx = find_device( path );
        if (idx < 0)
        {
            ERRORPRINTF("no storage media for location [%s]", location);
            ok = FALSE;
        }
        else
        {
            mountpoint = g_ptr_array_index(g_storage_media, idx);
        }
    }

    // determine iRex private directory
    if (ok)
    {
        irex_dir = g_string_new( mountpoint );
        //g_string_append(irex_dir, "/" DIR_IREX_PRIVATE "/" DIR_RECENTFILES);
        g_string_append(irex_dir, "/" DIR_RECENTFILES);
    }


    LOGPRINTF("leave [%s]", irex_dir ? irex_dir->str : NULL);
    return irex_dir;
}


// get filename used for desktop shortcut
GString *storage_get_shortcut_filename( const gchar *mountpoint )
{
    GString *filename = NULL;       // return value
    gchar   *cp       = NULL;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    // determine filename
    cp = g_path_get_basename(mountpoint);
    filename = g_string_new("");
    g_string_printf(filename, "storage_%s.directory", cp);

    // clean up
    g_free(cp);

    LOGPRINTF("leave: filename [%s]", filename->str);
    return filename;
}

// storage device mounted
void storage_media_mounted ( const gchar *mountpoint )
{
    gint        idx;
    GString     *dir = NULL;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    idx = find_device(mountpoint);
    if (idx >= 0)
    {
        ERRORPRINTF("storage already present [%s]", mountpoint);
    }
    else
    {
        // check media allowed
        if ( !fileview_is_directory_allowed(mountpoint) )
        {
            ERRORPRINTF("storage media not allowed [%s]", mountpoint);
        }
        else
        {
            // add mountpoint to list of available media
            g_ptr_array_add(g_storage_media, g_strdup(mountpoint) );

            // add mountpoint to desktop
            update_desktop_item(mountpoint, TRUE);

            // report device mounted
            fileview_media_mounted(mountpoint);
        }
    }

    // clean up
    if (dir) { g_string_free(dir, TRUE); }
}


// storage device will be unmounted
void storage_media_prepare_unmount ( const gchar *mountpoint )
{
    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    storage_media_unmounted(mountpoint);
}


// storage device has been unmounted
void storage_media_unmounted ( const gchar *mountpoint )
{
    gint        idx;
    gchar       *cp;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    idx = find_device(mountpoint);
    if (idx < 0)
    {
        WARNPRINTF("storage not present [%s]", mountpoint);
    }
    else
    {
        // remove mountpoint from list of available media
        cp = g_ptr_array_remove_index( g_storage_media, idx );
        g_free(cp);

        // remove mountpoint from desktop
        update_desktop_item(mountpoint, FALSE);

        // report device unmounted
        fileview_media_unmounted(mountpoint);
    }
}


// prepare storage media for iOn use
int storage_media_prepare ( const gchar *mountpoint )
{
    int             ret = ER_OK;    // return value

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    // initialise desktop directory
    ret = prepare_desktop(mountpoint);
    if (ret != ER_OK)
    {
        ERRORPRINTF("cannot install desktop for [%s]", mountpoint);
    }

    // install pc application
    if (ret == ER_OK)
    {
        ret = prepare_pc_application(mountpoint);
        if (ret != ER_OK)
        {
            ERRORPRINTF("cannot install pc_application for [%s]", mountpoint);
        }
    }

    return ret;
}


// prepare storage media for iOn use: desktop folder
static int prepare_desktop ( const gchar *mountpoint )
{
    int             ret = ER_OK;    // return value
    int             rc;
    gboolean        done;
    gboolean        is_database_transaction_started = FALSE;
    DIR             *dir       = NULL;
    struct dirent   *dirent    = NULL;;
    GPtrArray       *filenames = g_ptr_array_new();
    const gchar     *ext;
    gchar           *file;
    gchar           **p_file;
    shortcut_t      *shortcut     = NULL;
    GString         *desktop_dir  = NULL;
    const char      *template_dir = DIR_DESKTOP_TEMPLATE_CARD;
    erMetadb        *db           = NULL;
    erMetadb        *db_new       = NULL;

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    // create desktop directory
    desktop_dir = storage_get_desktop_dir(mountpoint);
    if (desktop_dir == NULL)
    {
        ret = ER_NOT_FOUND;
    }
    else
    {
        // create directory, if not present
        if ( !g_file_test(desktop_dir->str, G_FILE_TEST_IS_DIR) )
        {
            rc = g_mkdir_with_parents( desktop_dir->str, 0755 );
            if (rc != 0)
            {
                ERRNOPRINTF("cannot create directory [%s]", desktop_dir->str);
                ret = ER_FAIL;
            }
        }
    }

    // create a fresh config file
    if (ret == ER_OK)
    {
        rc = system(ERCONFTOOL_CMD_INIT_SD);
        sync();
        if (rc != 0)
        {
            ERRORPRINTF("erconftool error [%d] cmd [%s]", WEXITSTATUS(rc), ERCONFTOOL_CMD_INIT_SD);
            ret = ER_FAIL;
        }
    }

    // get list of desktop files for copying to storage device
    if (ret == ER_OK)
    {
        // open directory
        dir = opendir(template_dir);
        if (dir == NULL)
        {
            ERRNOPRINTF("cannot open directory [%s]", template_dir );
            // not considered fatal, so we can still return "ok"
        }
        else
        {
            // read filenames
            done = FALSE;
            while ( !done )
            {
                dirent = readdir(dir);
                if (   dirent
                    && dirent->d_name )
                {
                    LOGPRINTF("file [%s]", dirent->d_name);
                    if (   strcmp(dirent->d_name, ".") != 0
                        && strcmp(dirent->d_name, "..") != 0
                        && strcmp(dirent->d_name, ERMETADB_DATABASE_FILE) != 0 )
                    {
                        // directory entry: store filename
                        file = g_strdup(dirent->d_name);
                        g_ptr_array_add( filenames, file );
                    }
                }
                else
                {
                    // error or end-of-file, assume end-of-file
                    g_ptr_array_add( filenames, NULL );
                    done = TRUE;
                }
            }

            // close directory
            rc = closedir(dir);
            if (rc != 0)
            {
                ERRNOPRINTF("cannot close directory");
            }
            dir = NULL;
        }
    }

    // open or create database in desktop directory
    if (ret == ER_OK)
    {
        db = get_database();
        if (   db
            && db->directory
            && db->directory->str
            && strcmp( db->directory->str, desktop_dir->str) == 0 )
        {
            // database already open: use it
        }
        else
        {
            // database not open: create and open a new database object
            db_new = db_open( desktop_dir, TRUE );
            db     = db_new;
        }
        if (db == NULL)
        {
            ERRORPRINTF("cannot use/open/create target database [%s]", desktop_dir->str);
            ret = ER_FAIL;
        }
    }

    // start database transaction
    if (ret == ER_OK)
    {
        ret = ermetadb_begin_transaction(db);
        if (ret == ER_OK)
        {
            is_database_transaction_started = TRUE;
        }
    }

    // copy files/folders
    if (ret == ER_OK)
    {
        ret = copy_file_or_folder_list( filenames, template_dir,            // source
                                        NULL,      desktop_dir->str, db );  // target
        sync();
    }

    // write database changes now
    if (is_database_transaction_started)
    {
        rc = ermetadb_end_transaction( db );
        if (ret == ER_OK)
        {
            ret = rc;
        }
    }

    // create targets for shortcut to directory
    if (ret == ER_OK)
    {
        for ( p_file = (gchar**)(filenames->pdata);
              *p_file ;
              p_file++ )
        {
            file = *p_file;
            ext = g_extension_pointer(file);
            if ( strcmp(ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
            {
                LOGPRINTF("shortcut to folder [%s]", file);
                rc = parse_shortcut_file(desktop_dir->str, file, &shortcut);
                if (   rc != ER_OK
                    || shortcut->type != SHORTCUT_TO_FOLDER )
                {
                    ERRORPRINTF("invalid shortcut file [%s] [%s] error [%d]", desktop_dir->str, file, rc);
                }
                else
                {
                    // rename or create target folder, if needed
                    (void) shortcut_propagate_name_to_folder(shortcut);
                    (void) shortcut_create_target_folder(shortcut);
                }
                shortcut_free(shortcut);
            }
        }
    }
    sync();

    // update symlinks to target directory of desktop shortcuts
    storage_refresh_symlinks();

    // clean up
    if (db_new)      { ermetadb_free(db_new); }
    if (desktop_dir) { g_string_free(desktop_dir, TRUE); }
    if (filenames)
    {
        for ( p_file = (gchar**)(filenames->pdata);
              *p_file ;
              p_file++ )
        {
            g_free(*p_file);
        }
        g_ptr_array_free(filenames, TRUE);
    }

    return ret;
}


// prepare storage media for iOn use: pc application
static int prepare_pc_application ( const gchar *mountpoint )
{
    int             ret = ER_OK;    // return value
    int             rc;
    gchar           *target_dir = g_strdup_printf("%s/%s", mountpoint, DIR_PC_APP);
    GString         *cmd        = g_string_new("");

    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);
    g_assert(target_dir);

    // remove old pc application and autorun.inf, if any
    g_string_printf( cmd, "rm -rf %s %s/%s",
                                  target_dir,
                                     mountpoint,
                                        FILENAME_WIN_AUTORUN );
    (void)system(cmd->str);

    // create target directory
    rc = g_mkdir_with_parents(target_dir, 0755);
    if (rc != 0)
    {
        ERRNOPRINTF("cannot mkdir [%s]", target_dir);
        ret = ER_FAIL;
    }

    // copy pc application
    if (ret == ER_OK)
    {
        g_string_printf(cmd, "cp -r %s/* %s", DIR_PC_APP_TEMPLATE, target_dir);
        rc = system(cmd->str);
        sync();
        if (rc != 0)
        {
            ERRORPRINTF("copy error [%d] cmd [%s]", WEXITSTATUS(rc), cmd->str);
            ret = ER_FAIL;
        }
        else
        {
            // move autorun.inf file to storage device root
            g_string_printf(cmd, "mv %s/%s %s", target_dir, FILENAME_WIN_AUTORUN, mountpoint);
            rc = system(cmd->str);
            sync();
            if (rc != 0)
            {
                ERRORPRINTF("move error [%d] cmd [%s]", WEXITSTATUS(rc), cmd->str);
                ret = ER_FAIL;
            }
        }
    }

    // clean up
    g_free(target_dir);
    g_string_free(cmd, TRUE);

    return ret;
}


// refresh symlinks for desktop shortcuts
void storage_refresh_symlinks ( void )
{
    int         rc;
    int         i;
    gchar       *cp;
    const gchar *mountpoint;
    const gchar *dir_new     = DIR_DESKTOP_SYMLINK ".new";
    GString     *dir_symlink = g_string_new("");
    GString     *dir_desktop;
    erMetadb    *db;

    LOGPRINTF("entry");

    // create new shortcuts directory
    //   delete old one, if any
    cp = g_strdup_printf( "rm -rf %s", dir_new );
    (void) system( cp );
    g_free(cp);
    //   create new one
    rc = mkdir( dir_new, 0755 );
    if (rc != 0)
    {
        ERRNOPRINTF("cannot create directory [%s]", dir_new);
    }

    // refresh symlinks for each device mounted
    if (g_storage_media)
    {
        for (i = 0 ; i < g_storage_media->len ; i++)
        {
            // get desktop directory
            mountpoint = g_ptr_array_index(g_storage_media, i);
            dir_desktop = storage_get_desktop_dir(mountpoint);
            if (dir_desktop)
            {
                if ( g_file_test(dir_desktop->str, G_FILE_TEST_IS_DIR) )
                {
                    // get symlinks directory
                    cp = g_path_get_basename(mountpoint);
                    g_string_printf(dir_symlink, "%s/%s", dir_new, cp);
                    g_free(cp);

                    // create symlinks
                    db = db_open(dir_desktop, FALSE);
                    rc = create_symlinks_per_device(dir_symlink->str, dir_desktop->str);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot create symlinks for device [%s]", mountpoint);
                    }
                    ermetadb_free(db);
                }
                g_string_free(dir_desktop, TRUE);
            }
        }
    }

    // replace shortcuts directory with new one
    //   delete old one, if any
    (void) system( "rm -rf " DIR_DESKTOP_SYMLINK );
    //   rename new one
    rc = rename( dir_new, DIR_DESKTOP_SYMLINK );
    if (rc != 0)
    {
        ERRNOPRINTF("cannot rename [%s] to [%s]", dir_new, DIR_DESKTOP_SYMLINK);
    }

    // clean up
    if (dir_symlink) { g_string_free(dir_symlink, TRUE); }
}


// create symlinks for desktop shortcuts on specified device
static int create_symlinks_per_device ( const gchar    *dir_symlink,
                                        const gchar    *dir_desktop )
{
    int             ret = ER_OK;    // return value
    int             rc;
    int             i;
    gboolean        done = FALSE;
    const gchar     *ext;
    const gchar     *target;
    DIR             *dir      = NULL;
    struct dirent   *dirent   = NULL;;
    shortcut_t      *shortcut = NULL;
    GString         *linkname = g_string_new("");

    LOGPRINTF("entry");

    // create shortcut directory
    rc = mkdir(dir_symlink, 0755);
    if (rc != 0)
    {
        ERRNOPRINTF("cannot create directory [%s]", dir_symlink);
        ret = ER_FAIL;
    }

    // open desktop directory
    dir = opendir(dir_desktop);
    if (dir == NULL)
    {
        ERRNOPRINTF("cannot open directory [%s]", dir_desktop );
        ret = ER_NOT_FOUND;
    }

    // read file entries
    while ( ret == ER_OK  &&  !done )
    {
        dirent = readdir(dir);
        if (dirent == NULL)
        {
            // error or end-of-file, assume end-of-file
            done = TRUE;
        }
        else
        {
            // directory entry: check if shortcut to directory
            ext = g_extension_pointer(dirent->d_name);
            if ( strcmp(ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
            {
                LOGPRINTF("shortcut file [%s]", dirent->d_name);
                rc = parse_shortcut_file(dir_desktop, dirent->d_name, &shortcut);
                if (   rc != ER_OK
                    || shortcut->type != SHORTCUT_TO_FOLDER )
                {
                    ERRORPRINTF("invalid shortcut file [%s] [%s] error [%d]", dir_desktop, dirent->d_name, rc);
                }
                else
                {
                    // rename target folder, if needed
                    (void) shortcut_propagate_name_to_folder(shortcut);

                    // create symlink to target folder
                    i = strlen(dirent->d_name) - strlen(ext) - 1;
                    g_string_printf(linkname, "%s/%.*s", dir_symlink, i, dirent->d_name);

                    target = shortcut->details.folder.directory;
                    if (target)
                    {
                        rc = symlink( target, linkname->str );
                        if (rc != 0)
                        {
                            ERRNOPRINTF("cannot create symlink [%s]->[%s]", linkname->str, target);
                        }
                    }
                    else
                    {
                        ERRNOPRINTF("target not present, shortcut file [%s] [%s]", dir_desktop, dirent->d_name);
                    }
                }
                shortcut_free(shortcut);
            }
        }
    }

    // close directory
    if (dir)
    {
        rc = closedir(dir);
        if (rc != 0)
        {
            ERRNOPRINTF("cannot close directory");
        }
        dir = NULL;
    }

    // clean up
    if (linkname) { g_string_free(linkname, TRUE); }

    return ret;
}


// add or remove desktop item
static void update_desktop_item ( const gchar *mountpoint, const gboolean is_mounted )
{
    int         i;
    int         rc;
    gboolean    is_internal = FALSE;
    GString     *filename   = NULL;
    gchar       *filepath   = NULL;

    LOGPRINTF("entry: mountpoint [%s] is_mounted [%d]", mountpoint, is_mounted);

    if ( strcmp(mountpoint, DIR_LIBRARY) == 0 )
    {
        is_internal = TRUE;
    }

    // determine filename
    filename = storage_get_shortcut_filename(mountpoint);
    filepath = g_strdup_printf("%s/%s", DIR_DESKTOP_INTERNAL, filename->str);

    // update shortcut file
    if (is_mounted)
    {
        // mounted: add shortcut file
        rc = create_device_shortcut( filepath,
                                     mountpoint,
                                     is_internal ? _("Documents") : mountpoint );
        if (rc == ER_OK)
        {
            update_database(filename->str, is_internal, is_mounted);
        }
    }
    else
    {
        // unmounted: remove shortcut file
        i = unlink(filepath);
        if (i != 0)
        {
            ERRNOPRINTF("cannot remove file [%s]", filepath);
        }
        update_database(filename->str, is_internal, is_mounted);
    }

    // update symlinks to target directory of desktop shortcuts
    storage_refresh_symlinks();

    // clean up
    g_free(filepath);
    if (filename) { g_string_free(filename, TRUE); }
}


// create shortcut file for a storage device
static int create_device_shortcut ( const gchar *filepath, const gchar *mountpoint, const gchar *title )
{
    int         ret = ER_OK;    // return code
    int         i;
    FILE        *fp = NULL;

    LOGPRINTF("entry: mountpoint [%s] title [%s]", mountpoint, title);

    // mounted: create shortcut file
    fp = fopen(filepath, "w");
    if (fp == NULL)
    {
        ERRNOPRINTF("cannot open file [%s]", filepath);
        ret = ER_OPEN_ERROR;
    }
    else
    {
        // write shortcut details
        i = fprintf( fp,
                     STORAGE_SHORTCUT,
                     mountpoint,          // Path=
                     title            );  // Name=
        if (i <= 0)
        {
            ERRNOPRINTF("cannot write file [%s]", filepath);
            ret = ER_WRITE_ERROR;
        }

        // close file
        i = fclose(fp);
        if (i != 0)
        {
            ERRNOPRINTF("cannot close file [%s]", filepath);
            ret = ER_WRITE_ERROR;
        }
    }

    return ret;
}


// open database, create a new one if needed
static erMetadb* open_or_create_database ( const gchar *directory )
{
    erMetadb        *db = NULL;     // return value
    int             rc;

    LOGPRINTF("entry: directory [%s]", directory);
    g_assert(directory && *directory);

    GString     *dir = g_string_new( directory );

    // open database, create new one if needed
    db = ermetadb_new();
    if (db)
    {
        rc = ermetadb_open_database( db, dir );
        if (rc != ER_OK)
        {
            rc = ermetadb_create_database( dir );
            if (rc == ER_OK)
            {
                rc = ermetadb_open_database( db, dir );
            }
        }
        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot open or create database in [%s]", directory);
            ermetadb_free(db);
            db = NULL;
        }
    }

    // clean up
    if (dir) { g_string_free(dir, TRUE); }

    LOGPRINTF("leave: db [%p]", db);
    return db;
}


// update database for mounted/unmounted storage device
static int update_database ( const gchar *filename, const gboolean is_internal, const gboolean is_mounted)
{
    int             ret = ER_OK;    // return code
    int             rc;
    erMetadb        *db     = NULL;
    metadata_table  *values = NULL;

    LOGPRINTF("entry: filename [%s] is_mounted [%d]", filename, is_mounted);

    GString     *db_dir   = g_string_new( DIR_DESKTOP_INTERNAL );
    GString     *file_str = g_string_new( filename );

    // open database, create new one if needed
    db = open_or_create_database( DIR_DESKTOP_INTERNAL );
    if (db == NULL)
    {
        ret = ER_FAIL;
    }

    // update database
    if (ret == ER_OK )
    {
        // remove filename
        ret = ermetadb_delete_document( db, file_str );

        // add filename when device is mounted
        if ( is_mounted  &&  ret == ER_OK )
        {
            ret = ermetadb_add_document( db, file_str, 0, 0 );

            // set priority
            if (ret == ER_OK)
            {
                values = metadata_table_new();
                g_assert(values);
                metadata_table_add_column( values, "sort_priority" );
                metadata_table_set_int64( values,
                                          0,
                                          is_internal ? MDB_SORT_PRIORITY_LIB_INTERNAL : MDB_SORT_PRIORITY_LIB_EXTERNAL );
                ret = ermetadb_set_file_metadata( db, file_str, values );
            }
        }
    }

    // close database
    if ( db )
    {
        rc = ermetadb_close_database( db );
        ermetadb_free( db );
    }

    // clean up
    metadata_table_free( values );
    if ( file_str ) { g_string_free( file_str, TRUE ); }
    if ( db_dir   ) { g_string_free( db_dir,   TRUE ); }

    LOGPRINTF("return: ret [%d]", ret);
    return ret;
}
