/*
 * File Name: menu.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >

// ereader include files, between < >

// local include files, between " "
#include "ctb_log.h"
#include "fileview.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// menus for content browser
static const char *MENU_CONTENT             = "ctb_menu_content";
//static const char *MENU_TRASHCAN            = "ctb_menu_trashcan";
                                            
// menu groups                              
static const char *GROUP_VIEWS              = "ctb_view";
static const char *GROUP_SORT               = "ctb_sort";
static const char *GROUP_ACTIONS            = "ctb_actions";
                                            
// menu items                               
//   group "Views"                          
static const char *ITEM_VIEW_ICON           = "view_small";
static const char *ITEM_VIEW_LIST           = "view_detail";
//static const char *ITEM_VIEW_COVER          = "view_large";
//   group "Sort"                           
static const char *ITEM_SORT_NAME           = "sort_name";
static const char *ITEM_SORT_TYPE           = "sort_type";
static const char *ITEM_SORT_AUTHOR         = "sort_author";
static const char *ITEM_SORT_SIZE           = "sort_size";
static const char *ITEM_SORT_DATE           = "sort_date";
//   group "Actions"                        
static const char *ITEM_ACTION_OPEN         = "open";
static const char *ITEM_ACTION_SEARCH       = "search_library";
static const char *ITEM_ACTION_SHORTCUT     = "create_shortcut";
static const char *ITEM_ACTION_DELETE       = "delete";
static const char *ITEM_ACTION_SELECT_ALL   = "select_all";
static const char *ITEM_ACTION_COPY         = "copy";
static const char *ITEM_ACTION_CUT          = "cut";
static const char *ITEM_ACTION_PASTE        = "paste";
                                            
// item states                              
static const char *STATE_NORMAL             = "normal";
static const char *STATE_SELECTED           = "selected";
static const char *STATE_DISABLED           = "disabled";

// stylus states
static const char *STYLUS_POINTER           = "pointer";

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GString  *g_sort_item            = NULL;
static gboolean g_is_sort_item_fixed    = FALSE;
static gboolean g_current_is_desktop    = FALSE;


//============================================================================
// Local Function Definitions
//============================================================================


//============================================================================
// Functions Implementation
//============================================================================

// initialise popup menu
void menu_init ( void ) 
{
    static gboolean firsttime = TRUE;
    const char      *group;

    // execute only once
    if ( !firsttime )
    {
        WARNPRINTF("function called twice");
        return;
    }
    firsttime = FALSE;

    // allocate global variables
    g_sort_item = g_string_new("");

    // TODO: implement all and uncomment from this list

    // add groups
    group = GROUP_VIEWS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_VIEW_ICON,         group, "view_small"      );
    ipc_menu_add_item ( ITEM_VIEW_LIST,         group, "view_detail"     );
//    ipc_menu_add_item ( ITEM_VIEW_COVER,        group, "view_large"      );
    //                                                                                                                
    group = GROUP_SORT;                         
    ipc_menu_add_group( group,                  "",    "folder"          );                                                  
    ipc_menu_add_item ( ITEM_SORT_NAME,         group, "sort_name"       );
    ipc_menu_add_item ( ITEM_SORT_TYPE,         group, "sort_type"       );
    ipc_menu_add_item ( ITEM_SORT_AUTHOR,       group, "sort_author"     );
    ipc_menu_add_item ( ITEM_SORT_SIZE,         group, "sort_size"       );
    ipc_menu_add_item ( ITEM_SORT_DATE,         group, "sort_date"       );
    //                                                 
    group = GROUP_ACTIONS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_ACTION_OPEN,       group, "open"            );
//    ipc_menu_add_item ( ITEM_ACTION_SEARCH,     group, "search_library"  );
    ipc_menu_add_item ( ITEM_ACTION_SHORTCUT,   group, "create_shortcut" );
    ipc_menu_add_item ( ITEM_ACTION_DELETE,     group, "delete"          );
//    ipc_menu_add_item ( ITEM_ACTION_SELECT_ALL, group, "select_all"      );
//    ipc_menu_add_item ( ITEM_ACTION_COPY,       group, "copy"            );
//    ipc_menu_add_item ( ITEM_ACTION_CUT,        group, "cut"             );
//    ipc_menu_add_item ( ITEM_ACTION_PASTE,      group, "paste"           );
    //
//    group = "ctb_trashcan_tools";          
//    ipc_menu_add_group( group,                  "",    "folder"          );
//    ipc_menu_add_item ( "empty_trash",          group, "empty_trash"     );
//    ipc_menu_add_item ( "delete",               group, "delete"          );
//    ipc_menu_add_item ( "copy",                 group, "copy"            );
//    ipc_menu_add_item ( "cut",                  group, "cut"             );
//    ipc_menu_add_item ( "paste",                group, "paste"           );

    // add menus
    ipc_menu_add_menu( MENU_CONTENT,  GROUP_VIEWS, GROUP_SORT,           GROUP_ACTIONS        );
//    ipc_menu_add_menu( MENU_TRASHCAN, GROUP_VIEWS, GROUP_SORT,           "ctb_trashcan_tools" );

    // set menu texts
    menu_set_text();

    // show the main menu
    ipc_menu_show_menu( MENU_CONTENT );
}


// initialise popup menu
void menu_set_text ( void ) 
{
    const char      *group;

    // TODO: implement all and uncomment from this list

    // set groups + items
    group = GROUP_VIEWS;
    ipc_menu_set_group_label( group,                         _("View")              );
    ipc_menu_set_item_label ( ITEM_VIEW_ICON,         group, _("Thumbnails")        );
    ipc_menu_set_item_label ( ITEM_VIEW_LIST,         group, _("Details")           );
//    ipc_menu_set_item_label ( ITEM_VIEW_COVER,        group, _("Covers")            );
    //                                                                                                                      
    group = GROUP_SORT;                         
    ipc_menu_set_group_label( group,                         _("Sort by")           );                                                  
    ipc_menu_set_item_label ( ITEM_SORT_NAME,         group, _("Name")              );
    ipc_menu_set_item_label ( ITEM_SORT_TYPE,         group, _("Type")              );
    ipc_menu_set_item_label ( ITEM_SORT_AUTHOR,       group, _("Author")            );
    ipc_menu_set_item_label ( ITEM_SORT_SIZE,         group, _("Size")              );
    ipc_menu_set_item_label ( ITEM_SORT_DATE,         group, _("Date")              );
    //                                                           
    group = GROUP_ACTIONS;
    ipc_menu_set_group_label( group,                         _("Actions")           );
    ipc_menu_set_item_label ( ITEM_ACTION_OPEN,       group, _("Open")              );
//    ipc_menu_set_item_label ( ITEM_ACTION_SEARCH,     group, _("Search Documents")  );
    ipc_menu_set_item_label ( ITEM_ACTION_SHORTCUT,   group, _("Add Shortcut")      );
    ipc_menu_set_item_label ( ITEM_ACTION_DELETE,     group, _("Delete")            );
//    ipc_menu_set_item_label ( ITEM_ACTION_SELECT_ALL, group, _("Select All")        );
//    ipc_menu_set_item_label ( ITEM_ACTION_COPY,       group, _("Copy")              );
//    ipc_menu_set_item_label ( ITEM_ACTION_CUT,        group, _("Cut")               );
//    ipc_menu_set_item_label ( ITEM_ACTION_PASTE,      group, _("Paste")             );
    //
//    group = "ctb_trashcan_tools";          
//    ipc_menu_set_group_label( group,                         _("Actions")           );
//    ipc_menu_set_item_label ( "empty_trash",          group, _("Empty Recycle Bin") );
//    ipc_menu_set_item_label ( "delete",               group, _("Delete")            );
//    ipc_menu_set_item_label ( "copy",                 group, _("Copy")              );
//    ipc_menu_set_item_label ( "cut",                  group, _("Cut")               );
//    ipc_menu_set_item_label ( "paste",                group, _("Paste")             );

    // set menus
    ipc_menu_set_menu_label( MENU_CONTENT,  _("Menu") );
//    ipc_menu_set_menu_label( MENU_TRASHCAN, _("Menu") );
}


void menu_show ( void ) 
{
    // TODO: need to pass current context when more than one menu used
    ipc_menu_show_menu( MENU_CONTENT );
    ipc_status_set_stylus( STYLUS_POINTER );
}

// select one of the view types
void menu_select_view_type ( const ctb_viewtypes_t view_type )
{
    static const char   *item_old = "";
           const char   *item_new = "";

    LOGPRINTF("entry: view_type [%d]", view_type);
    g_assert( view_type >= 0  &&  view_type < N_CTB_VIEWS );

    switch (view_type)
    {
        case CTB_ICONVIEW:
            item_new = ITEM_VIEW_ICON;
            break;
        case CTB_LISTVIEW:
            item_new = ITEM_VIEW_LIST;
            break;
        default:
            ;  // ignore: handled by assert at function entry
    }

    if ( strcmp(item_old, item_new) != 0 )
    {
        if ( item_old[0] != '\0' )
        {
            ipc_menu_set_item_state ( item_old, GROUP_VIEWS, STATE_NORMAL   );
        }
        ipc_menu_set_item_state ( item_new, GROUP_VIEWS, STATE_SELECTED );

        item_old = item_new;
    }
}


// select one of the sorting orders
void menu_select_sort_order ( const ctb_sort_order_t sort_order )
{
    const char   *item_new = "";

    LOGPRINTF("entry: sort_order [%d]", sort_order);
    g_assert( sort_order >= 0  &&  sort_order < N_CTB_SORT_ORDER );
    g_assert(g_sort_item && g_sort_item->str);

    g_return_if_fail(g_is_sort_item_fixed == FALSE);

    switch (sort_order)
    {
        case CTB_SORT_BY_NAME:
            item_new = ITEM_SORT_NAME;
            break;
        case CTB_SORT_BY_TYPE:
            item_new = ITEM_SORT_TYPE;
            break;
        case CTB_SORT_BY_SIZE:
            item_new = ITEM_SORT_SIZE;
            break;
        case CTB_SORT_BY_DATE:
            item_new = ITEM_SORT_DATE;
            break;
        case CTB_SORT_BY_AUTHOR:
            item_new = ITEM_SORT_AUTHOR;
            break;
        default:
            ;  // ignore: handled by assert at function entry
    }

    if ( strcmp(g_sort_item->str, item_new) != 0 )
    {
        if ( g_sort_item->str[0] )
        {
            ipc_menu_set_item_state ( g_sort_item->str, GROUP_SORT, STATE_NORMAL   );
        }
        ipc_menu_set_item_state ( item_new, GROUP_SORT, STATE_SELECTED );

        g_string_assign( g_sort_item, item_new );
    }
}


// force one of the sorting orders, disable all other sort orders
void menu_set_fixed_sort_order ( const gboolean is_fixed )
{
    const gchar **cpp;
    const gchar *sort_item_names[] =
                {
                    ITEM_SORT_NAME,
                    ITEM_SORT_TYPE,
                    ITEM_SORT_SIZE,
                    ITEM_SORT_DATE,
                    ITEM_SORT_AUTHOR,
                    NULL
                };

    LOGPRINTF("entry: is_fixed [%d]", is_fixed);
    g_return_if_fail( g_sort_item );

    for ( cpp = sort_item_names ; *cpp ; cpp++ )
    {
        if ( strcmp(*cpp, g_sort_item->str) != 0 )
        {
            ipc_menu_set_item_state ( *cpp,
                                      GROUP_SORT,
                                      is_fixed ? STATE_DISABLED : STATE_NORMAL );
        }
    }

    g_is_sort_item_fixed = is_fixed;
}


// show menu for desktop or normal directories
void menu_set_current_is_desktop ( const gboolean is_desktop )
{
    LOGPRINTF("entry: is_desktop [%d]", is_desktop);

    if (g_current_is_desktop != is_desktop)
    {
        g_current_is_desktop = is_desktop;

        ipc_menu_set_item_state( ITEM_ACTION_SHORTCUT,
                                 GROUP_ACTIONS,
                                 is_desktop ? STATE_DISABLED : STATE_NORMAL );
    }
}


//----------------------------------------------------------------------------
// Callbacks from popupmenu
//----------------------------------------------------------------------------

// go to specified location
void menu_on_goto_location ( const gchar *location )
{
    g_assert( location  &&  location[0] != '\0' );

    LOGPRINTF("entry: location [%s]", location );

    if ( strcmp(location, "library") == 0 )
    {
        // content browser back on top:
        //   rescan current directory
        fileview_refresh();
        //   set content browser main menu
        menu_show();
    }
    else if ( strcmp(location, "desktop") == 0 )
    {
        // goto desktop :
        //   show desktop directory
        fileview_show_desktop();
        //   set content browser main menu
        menu_show();
    }
    else
    {
        ERRORPRINTF("unknown location [%s]", location);
    }
}


// user has pressed a menu button
void menu_on_item_activated ( const gchar *item,
                              const gchar *group,
                              const gchar *menu,
                              const gchar *state )
{
    gboolean            ok = TRUE;
    ctb_viewtypes_t     view_type  = 0;
    ctb_sort_order_t    sort_order = 0;

    LOGPRINTF("entry: item [%s] group [%s]", item, group);

    if ( strcmp(group, GROUP_VIEWS) == 0 )
    {
        if ( strcmp(state, "normal") == 0 )
        {
            if (      strcmp(item, ITEM_VIEW_ICON) == 0 )
            {
                view_type = CTB_ICONVIEW;
            }
            else if ( strcmp(item, ITEM_VIEW_LIST) == 0 )
            {
                view_type = CTB_LISTVIEW;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok)
            {
                fileview_set_view_type( view_type );
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_SORT) == 0 )
    {
        if ( strcmp(state, "normal") == 0 )
        {
            if (      strcmp(item, ITEM_SORT_NAME  ) == 0 )
            {
                sort_order = CTB_SORT_BY_NAME;
            }
            else if ( strcmp(item, ITEM_SORT_TYPE  ) == 0 )
            {
                sort_order = CTB_SORT_BY_TYPE;
            }
            else if ( strcmp(item, ITEM_SORT_SIZE  ) == 0 )
            {
                sort_order = CTB_SORT_BY_SIZE;
            }
            else if ( strcmp(item, ITEM_SORT_DATE  ) == 0 )
            {
                sort_order = CTB_SORT_BY_DATE;
            }
            else if ( strcmp(item, ITEM_SORT_AUTHOR) == 0 )
            {
                sort_order = CTB_SORT_BY_AUTHOR;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok)
            {
                fileview_set_sort_order( sort_order );
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_ACTIONS) == 0 )
    {
        if ( strcmp(state, "normal") == 0 )
        {
            if (      strcmp(item, ITEM_ACTION_OPEN      ) == 0 )
            {
                fileview_open_items();
            }
            else if ( strcmp(item, ITEM_ACTION_SEARCH    ) == 0 )
            {
                // TODO: handle ITEM_ACTION_SEARCH
            }
            else if ( strcmp(item, ITEM_ACTION_SHORTCUT  ) == 0 )
            {
                fileview_create_shortcut();
            }
            else if ( strcmp(item, ITEM_ACTION_DELETE    ) == 0 )
            {
                fileview_delete_items();
            }
            else if ( strcmp(item, ITEM_ACTION_SELECT_ALL) == 0 )
            {
                // TODO: handle ITEM_ACTION_SELECT_ALL
            }
            else if ( strcmp(item, ITEM_ACTION_COPY      ) == 0 )
            {
                // TODO: handle ITEM_ACTION_COPY
            }
            else if ( strcmp(item, ITEM_ACTION_CUT       ) == 0 )
            {
                // TODO: handle ITEM_ACTION_CUT
            }
            else if ( strcmp(item, ITEM_ACTION_PASTE     ) == 0 )
            {
                // TODO: handle ITEM_ACTION_PASTE
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else
    {
        WARNPRINTF("unexpected menu group [%s]", group);
    }
}
