/*
 * File Name: filemodel.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gtk/gtk.h>
#include <dirent.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

// local include files, between " "
#include "ctb_log.h"
#include "fileinfo.h"
#include "filemodel.h"
#include "fileview.h"
#include "filetypes.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"
#include "shortcut.h"
#include "storage.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// metadata key for sorting order
#define METADATA_KEY_SORTORDER  "ctb.sort-order"
#define SORT_ORDER_DEFAULT       CTB_SORT_BY_NAME
#define SORT_IS_ASC_DEFAULT      TRUE

// maximum time to wait for a database transaction (seconds)
static const int        MAX_WAIT_DATABASE = 30;

// data layout for thumbnails retrieved from metadata database
static const char       *THUMBNAIL_IMAGE_TYPE = "png";

// columns in database table file_metadata
//   index in MDB_COLUMN_NAMES
typedef enum
        {
            COL_FILENAME,
            COL_DIRECTORY_PATH,
            COL_SORT_PRIORITY,
            COL_IS_DIRECTORY,
            COL_IS_HIDDEN,
            COL_IS_TEMPLATE,
            COL_FILETYPE,
            COL_FILESIZE,
            COL_FILEDATE,
            COL_TITLE,
            COL_AUTHOR,
            COL_THUMB_MINI,
            COL_THUMB_SMALL,
            COL_THUMB_MEDIUM,
            COL_THUMB_LARGE,
            N_METADATA_COLUMNS
        } mdb_column_idx_t;
//
// column names
//   index in this table is a mdb_column_idx_t
//   note: keep synchronised with mdb_column_idx_t above
static  const struct
        {
            const char  *name;
            const int   filemodel_column;
        }               MDB_COLUMN_NAMES[] =
                        {
                            { "filename"          , MODCOL_FILENAME       },
                            { "directory_path"    , MODCOL_DIRECTORY_PATH },
                            { "sort_priority"     , -1                    },
                            { "is_directory"      , MODCOL_IS_DIRECTORY   },
                            { "is_hidden"         , -1                    },
                            { "is_template"       , MODCOL_IS_TEMPLATE    },
                            { "file_type"         , MODCOL_FILETYPE       },
                            { "file_size"         , MODCOL_FILESIZE       },
                            { "file_last_modified", MODCOL_FILEDATE       },
                            { "title"             , -1                    },
                            { "author"            , MODCOL_AUTHOR         },
                            { "thumb_data_mini"   , MODCOL_THUMBNAIL      },
                            { "thumb_data_small"  , MODCOL_THUMBNAIL      },
                            { "thumb_data_medium" , MODCOL_THUMBNAIL      },
                            { "thumb_data_large"  , MODCOL_THUMBNAIL      },
                            { NULL }  /* end of list */
                        };

// database columns allowed as sorting order
//   index in this table is a ctb_sort_order_t
//   data  in this table is a mdb_column_idx_t, an index in MDB_COLUMN_NAMES[]
static const int    SORT_COLUMN_NAME_IDX[ N_CTB_SORT_ORDER ] =
                    {
                        COL_FILENAME,
                        COL_FILETYPE,
                        COL_FILESIZE,
                        COL_FILEDATE,
                        COL_AUTHOR
                    };


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkListStore          *g_filestore    = NULL;             // details of files
static gint                  g_num_items     = 1;                // max. number of items to be displayed
static gint                  g_item_offset   = 0;                // offset of first item to be displayed
static gint                  g_total_items   = 0;                // total number of items
static gboolean              g_has_next_page = FALSE;            // do we have a next page to be displayed?
static const gchar           *g_sort_column  = "filename";       // item sorting order
static gboolean              g_is_sort_asc   = TRUE;             // item sorting order ascending or descending
static filemodel_thumbsize_t g_thumbnail_size= MODTHUMB_MEDIUM;  // thumbail size to retrieve from database

static erMetadb              *g_metadb       = NULL;             // database with file metadata
static gchar                 *g_parent_dir   = NULL;             // directory for "Up" entry


//============================================================================
// Local Function Definitions
//============================================================================

static void           close_database                    ( void );
static int            compare_fs_fileinfo               ( gconstpointer a, gconstpointer b );
static int            create_database_item              ( const fs_fileinfo_t *inf );
static void           create_filemodel                  ( void );
static int            delete_database_item              ( const fs_fileinfo_t *inf );
static int            end_database_transaction          ( void );
static void           filestore_insert_previous_dir     ( void );
static gchar*         format_size                       ( const gint64 bytes );
static void           free_fs_fileinfo_list             ( GPtrArray *fileinfo );
static void           free_fs_fileinfo_item             ( gpointer data, gpointer user_data );
static gboolean       db_is_desktop_directory           ( erMetadb *db );
static int            load_items_in_model               ( void );
static int            open_database                     ( const gchar *current_dir );
static fs_fileinfo_t* parse_dirent                      ( const struct dirent *dirent );
static GPtrArray*     read_fs_fileinfo                  ( const gchar *current_dir );
static void           read_sort_order_from_database     ( void );
static int            start_database_transaction        ( void );
static int            sync_database                     ( const gchar *current_dir );
static int            update_database_item              ( const fs_fileinfo_t *inf );
static int            update_database_item_sort_order   ( const GString *filename, int sort_order);
static void           write_sort_order_to_database      ( void );


//============================================================================
// Functions Implementation
//============================================================================

// prepare for power down
void filemodel_quit (void)
{
    LOGPRINTF("entry");

    close_database();
}


// get store for file details
GtkTreeModel* get_filemodel (void)
{
    static GtkTreeModel *filemodel = NULL;

    LOGPRINTF("entry");

    // create file store as needed
    if (g_filestore == NULL)
    {
        create_filemodel();
        filemodel = GTK_TREE_MODEL(g_filestore);
    }

    return filemodel;
}


// create store for file details
static void create_filemodel (void)
{
    GtkListStore *store;

    LOGPRINTF("entry");
    g_assert(g_filestore == NULL);

    // create list store
    store = gtk_list_store_new(N_FILEMODEL_COLUMNS, FILEMODEL_COLUMN_TYPES);
    g_assert(store);

    g_filestore = store;
}


// get database
erMetadb* get_database (void)
{
    LOGPRINTF("entry: g_metadb [%p]", g_metadb);

    return g_metadb;
}


// set sorting order
gboolean filemodel_set_sortorder (const ctb_sort_order_t sort_order, const gboolean is_ascending)
{
    gboolean    order_has_changed = FALSE;  // return value

    // sorting order
    const gchar     *sort_column;

    LOGPRINTF("entry: sort_order [%d] is_asc [%d]", sort_order, is_ascending);
    g_assert(sort_order >= 0  &&  sort_order < N_CTB_SORT_ORDER);


    sort_column = MDB_COLUMN_NAMES[ SORT_COLUMN_NAME_IDX[ sort_order ] ].name;

    if (   strcmp(sort_column, g_sort_column) != 0
        || is_ascending   !=   g_is_sort_asc       )
    {
        // set new sort order
        g_sort_column = sort_column;
        g_is_sort_asc = is_ascending;

        // update popup menu
        menu_select_sort_order( sort_order );

        // save in database
        write_sort_order_to_database();

        order_has_changed = TRUE;
    }

    return order_has_changed;
}


// set number of items
void filemodel_set_viewsize (const int n_items)
{
    LOGPRINTF("entry: n_items [%d]", n_items);

    g_return_if_fail(n_items > 0  &&  n_items < 100);

    if (n_items != g_num_items)
    {
        g_num_items = n_items;
        load_items_in_model();
    }
}


// set thumbnail size to retrieve from database
void filemodel_set_thumbsize (const filemodel_thumbsize_t thumb_size)
{
    LOGPRINTF("entry: thumb_size [%d]", thumb_size);

    g_return_if_fail(thumb_size >= 0  &&  thumb_size < N_FILEMODEL_THUMB_SIZES);

    if (thumb_size != g_thumbnail_size)
    {
        g_thumbnail_size = thumb_size;
        g_item_offset    = 0;
        load_items_in_model();
    }
}


// goto previous page
void filemodel_page_previous (void)
{
    LOGPRINTF("entry: old offset [%d]", g_item_offset);

    g_item_offset -= g_num_items;
    if (g_item_offset < 0)
    {
        g_item_offset = 0;
    }
    load_items_in_model();
}


// goto next page
void filemodel_page_next (void)
{
    LOGPRINTF("entry");

    if (g_has_next_page)
    {
        LOGPRINTF("entry: old offset [%d] page size [%d]", g_item_offset, g_num_items);
        if (g_item_offset == 0  &&  g_parent_dir)
        {
            g_item_offset += g_num_items - 1;
        }
        else
        {
            g_item_offset += g_num_items;
        }
        load_items_in_model();
    }
}


// report page offset
gint filemodel_get_offset (const guint item)
{
    gint        offset = 0;

    LOGPRINTF("entry");

    // note: g_item_offset excludes the "parent-dir" entry
    offset = g_item_offset + item;
    if (g_parent_dir  &&  g_item_offset == 0)
    {
        offset--;
    }

    if (offset >= 0)
    {
        return offset;
    }
    else
    {
        ERRORPRINTF("return -1");
        return -1;
    }
}


// got specified item by offset
gint filemodel_scroll_to_offset (const guint offset)
{
    gint        item_idx  = 0;  // return value

    LOGPRINTF("entry: offset [%d]", offset);

    item_idx = MIN(offset, g_total_items - 1);
    if (g_parent_dir)
    {
        item_idx++;
    }
    g_item_offset = ((item_idx / g_num_items) * g_num_items);
    item_idx      =   item_idx % g_num_items;
    if (g_parent_dir  &&  g_item_offset > 0)
    {
        g_item_offset--;
    }
    LOGPRINTF("offset [%d] g_item_offset [%d] g_total_items, [%d] index [%d]", offset, g_item_offset, g_total_items, item_idx);

    // load selected page from database
    load_items_in_model();

    return item_idx;
}


// goto specified item by filename
gint filemodel_scroll_to_filename (const gchar *filename)
{
    gint                item_idx  = 0;  // return value
    gint                rc;
    gint                i;
    gint                n_rows = 0;
    gboolean            found = FALSE;
    GString             *sort_key = NULL;
    metadata_table      *names    = NULL;
    metadata_table      *values   = NULL;
    const metadata_cell *cell     = NULL;

    LOGPRINTF("entry: filename [%s] g_num_items [%d]", filename, g_num_items);

    // find index of specified item
    if (filename  &&  *filename)
    {
        // select all from database
        //   set sort key
        sort_key = g_string_new(g_sort_column);
        //   set column names we are interested in
        names = metadata_table_new();
        if (names)
        {
            metadata_table_add_column(names, MDB_COLUMN_NAMES[ COL_FILENAME ].name);
        }
        //   retrieve item details from database
        rc = ermetadb_select_file_metadata( g_metadb,
                                            sort_key,
                                            g_is_sort_asc,
                                            0,
                                            -1,
                                            names,
                                            &values       );
        if (rc == ER_OK  &&  values)
        {
            // note: values is NULL when no files/folders selected
            n_rows = metadata_table_n_rows(values);
            cell   = (const metadata_cell*) (values->cell_data->data);
        }

        // find the requested item
        found = FALSE;
        for (i = 0 ; i < n_rows  &&  !found ; i++, cell++)
        {
            if (   cell->type == METADATA_TEXT
                && strcmp( filename, cell->value.v_text->str) == 0 )
            {
                found = TRUE;
                item_idx = i;
            }
        }

        // calculate page offset + item-index within page
        if (found)
        {
            // note: g_item_offset excludes the "parent-dir" entry
            if (g_parent_dir)
            {
                item_idx++;
            }
            g_item_offset = ((item_idx / g_num_items) * g_num_items);
            item_idx      =   item_idx % g_num_items;
            if (g_parent_dir  &&  g_item_offset > 0)
            {
                g_item_offset--;
            }
        }
        else
        {
            g_item_offset = 0;
            item_idx      = 0;
        }
        LOGPRINTF("found [%d] offset [%d] index [%d]", found, g_item_offset, item_idx);
    }

    // load selected page from database
    load_items_in_model();

    // clean up
    metadata_table_free(values);
    metadata_table_free(names );
    if (sort_key) { g_string_free(sort_key, TRUE); }

    return item_idx;
}


// get details for current directory
int filemodel_chdir (const gchar *dir, const gchar *parent_dir)
{
    int         ret = ER_OK;    // return value

    LOGPRINTF("entry: dir [%s] parent [%s]", dir, parent_dir);
    g_assert(g_filestore);
    g_assert(dir && *dir);

    // synchronise database
    ret = sync_database(dir);
    if (ret != ER_OK)
    {
        ERRORPRINTF("cannot synchronise database, keeping model unchanged");
    }
    else
    {
        read_sort_order_from_database();

        // update the filemodel, which triggers update of current view
        g_free(g_parent_dir);
        g_parent_dir  = g_strdup(parent_dir);
        g_item_offset = 0;
        load_items_in_model();
    }

    return ret;
}


// get details for desktop directory
void filemodel_chdir_desktop (void)
{
    int             rc;
    int             i;
    gboolean        ok = TRUE;    // no problems so far
    gchar           *cp;
    GString         *dir;
    GString         *dir_desktop     = g_string_new(DIR_DESKTOP_INTERNAL);
    GString         *dir_desktop_tmp = g_string_new(DIR_DESKTOP_TMP     );
    const GPtrArray *storage_media   = NULL;
    const gchar     *mountpoint      = NULL;
    metadata_table  *names           = NULL;
    metadata_table  *values          = NULL;

    LOGPRINTF( "entry" );
    g_assert(g_filestore);

    // Note: Don't synchronise database on internal desktop,
    //       as this will find conflicts in filesize/filedate and clear the thumbnails.
    //       Internal desktop items are static, so no need to synchronise the database.

    // synchronise database(s) on removable media
    storage_media = storage_get_media();
    if (storage_media)
    {
        for (i = 0 ; i < storage_media->len ; i++)
        {
            mountpoint = g_ptr_array_index(storage_media, i);
            dir = storage_get_desktop_dir(mountpoint);
            if (dir)
            {
                if ( g_file_test(dir->str, G_FILE_TEST_IS_DIR) )
                {
                    // synchronise removable database
                    rc = sync_database(dir->str);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot synchronise database [%s]", mountpoint);
                    }
                }
                g_string_free(dir, TRUE);
            }
        }

        storage_media = NULL;
    }

    // create and open temporary database
    if (ok)
    {
        // close and delete current database
        close_database();
        cp = g_strdup_printf("%s/%s", dir_desktop_tmp->str, ERMETADB_DATABASE_FILE);
        (void) unlink(cp);
        g_free(cp);

        // create and open new database
        rc = open_database(dir_desktop_tmp->str);
        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot open temporary database [%s]", dir_desktop_tmp->str);
            ok = FALSE;
        }
    }

    // copy details from internal desktop to temporary database
    if (ok)
    {
        (void) ermetadb_copy_all_documents( g_metadb,
                                            dir_desktop,    // directory_src
                                            TRUE,           // skip_hidden
                                            TRUE );         // skip_template
    }

    // add desktop data from removable media to temporary database
    if (ok)
    {
        storage_media = storage_get_media();
        if (storage_media)
        {
            for (i = 0 ; i < storage_media->len ; i++)
            {
                mountpoint = g_ptr_array_index(storage_media, i);
                dir = storage_get_desktop_dir(mountpoint);
                if (dir)
                {
                    if ( g_file_test(dir->str, G_FILE_TEST_IS_DIR) )
                    {
                        // copy details from removable database
                        (void) ermetadb_copy_all_documents( g_metadb,
                                                            dir,        // directory_src
                                                            TRUE,       // skip_hidden
                                                            TRUE );     // skip_template
                    }
                    g_string_free(dir, TRUE);
                }
            }

            storage_media = NULL;
        }
    }

    // get total number of items
    if (ok)
    {
        names = metadata_table_new();
        metadata_table_add_column(names, MDB_COLUMN_NAMES[ COL_IS_HIDDEN ].name);
        rc = ermetadb_select_file_metadata( g_metadb,
                                            NULL,           // sort_key
                                            FALSE,          // sort_is_ascending
                                            0,              // start_item
                                            -1,             // num_items
                                            names,
                                            &values );
        if (rc == ER_OK  && values)
        {
            g_total_items = metadata_table_n_rows(values);
        }
    }

    // load desktop items in model, so the view will display them
    if (ok)
    {
        read_sort_order_from_database();

        // update the filemodel, which triggers update of current view
        g_free(g_parent_dir);
        g_parent_dir  = NULL;
        g_item_offset = 0;
        load_items_in_model();
    }

    // clean up
    metadata_table_free(values);
    metadata_table_free(names);
    g_string_free(dir_desktop_tmp, TRUE);
    g_string_free(dir_desktop    , TRUE);
}


// item has been deleted
void filemodel_item_deleted()
{
    LOGPRINTF("entry");

    if (g_total_items > 0)
    {
        g_total_items--;
    }
    if (g_item_offset >= g_total_items)
    {
        g_item_offset = g_total_items - 1;
    }
}


// synchronise database with filesystem
static int sync_database (const gchar *current_dir)
{
    int                 ret = ER_OK;    // return code
    int                 rc;
    int                 col;
    int                 i;
    int                 file_cmp;
    int                 n;
    int                 idx;
    gchar               *cp;
    const char          *name;
    const char          *ext;
    gpointer            *gpp;
    const metadata_cell *cell;
    gboolean            ok;
    gboolean            done;
    gboolean            exists;
    gboolean            do_delete_item;
    gboolean            do_create_item;
    gboolean            do_update_item;
    gboolean            is_database_transaction_started = FALSE;
    fs_fileinfo_t       fs_tmp;
    const fs_fileinfo_t *p_fs_tmp = &fs_tmp;

    // data from filesystem
    GPtrArray           *fileinfo = NULL;
    const fs_fileinfo_t **p_fs    = NULL;
    const fs_fileinfo_t *fs       = NULL;
    int                 fs_n_rows = 0;
    int                 fs_row    = 0;

    // data from metadata database
    metadata_table      *values = NULL;
    const metadata_cell *db     = NULL;
    int                 db_n_rows = 0;
    int                 db_n_cols = 0;
    int                 db_row    = 0;
    const GString       *db_filename = NULL;
    const GString       *db_dirpath  = NULL;
    gint64              db_isdir;
    gint64              db_ishidden;
    gint64              db_filesize;
    gint64              db_filedate;

    // columns to fetch from database
    int                 column_idx[ N_METADATA_COLUMNS ];
    static const int    COLUMN_NAME_IDX[] =
                        {
                            COL_FILENAME,
                            COL_DIRECTORY_PATH,
                            COL_IS_DIRECTORY,
                            COL_IS_HIDDEN,
                            COL_FILESIZE,
                            COL_FILEDATE
                        };


    LOGPRINTF("entry: dir [%s]", current_dir);
    g_assert(current_dir && *current_dir);

    // open metadata database
    if (ret == ER_OK)
    {
        ret = open_database(current_dir);
    }

    // read filenames from database
    if (ret == ER_OK)
    {
        ret = ermetadb_get_filenames(g_metadb, &values);

        if (ret == ER_OK  &&  values == NULL)
        {
            // no filenames in database: make a dummy result table
            values = metadata_table_new();
        }
    }
    if (    ret == ER_OK
        &&  metadata_table_n_rows(values) > 0 )
    {
        // find column indices in values table
        n = sizeof(COLUMN_NAME_IDX) / sizeof(COLUMN_NAME_IDX[0]);
        for (i = 0 ; i < n ; i++)
        {
            idx  = COLUMN_NAME_IDX[i];
            name = MDB_COLUMN_NAMES[idx].name;
            col = metadata_table_find_column(values, name);
            if (col < 0)
            {
                ERRORPRINTF("no column [%s] from ermetadb_get_filenames()", name);
                ret = ER_FAIL;
            }
            else
            {
                column_idx[idx] = col;
            }
        }
    }

    // read directory entries
    fileinfo = read_fs_fileinfo(current_dir);
    if (fileinfo == NULL)
    {
        ERRORPRINTF("cannot read filesystem info for dir [%s]", current_dir);
        ret = ER_FAIL;
    }

    // synchronise database
    if (ret == ER_OK)
    {
        // pointers in fileinfo
        fs_n_rows = fileinfo->len;
        gpp  = fileinfo->pdata;
        p_fs = (const fs_fileinfo_t**)gpp;

        // pointers in database values
        db_n_rows = metadata_table_n_rows(values);
        db_n_cols = metadata_table_n_columns(values);
        db = (const metadata_cell*) (values->cell_data->data);

        // current index of tables
        fs_row = 0;
        db_row = 0;

        // no displayable items yet
        g_total_items = 0;

        // compare filesystem info with database info
        while (fs_row < fs_n_rows  ||  db_row < db_n_rows)
        {
            ok = TRUE;

            fs = *p_fs;

            db_filename = NULL;
            db_dirpath  = NULL;
            db_isdir    = 0;
            db_ishidden = 0;
            db_filesize = 0;
            db_filedate = 0;

            do_delete_item = FALSE;
            do_create_item = FALSE;
            do_update_item = FALSE;

            // skip database entries from other directories,
            // i.e. entries with directory_path not "."
            done = FALSE;
            while ( !done  &&  db_row < db_n_rows )
            {
                // get basic fields from database entry
                //   get filename
                cell = db + column_idx[COL_FILENAME];
                if (cell->type == METADATA_TEXT)
                {
                    db_filename = cell->value.v_text;
                }
                else
                {
                    ERRORPRINTF("invalid filename type [%d] db_row [%d]", cell->type, db_row);
                    ok = FALSE;
                }
                //   get directory_path
                cell = db + column_idx[COL_DIRECTORY_PATH];
                if (cell->type == METADATA_TEXT)
                {
                    db_dirpath = cell->value.v_text;
                }
                else
                {
                    ERRORPRINTF("invalid directory_path type [%d] db_row [%d]", cell->type, db_row);
                }
                //   get is_directory
                cell = db + column_idx[COL_IS_DIRECTORY];
                if (cell->type == METADATA_INT64)
                {
                    db_isdir = cell->value.v_int64;
                }
                else
                {
                    ERRORPRINTF("invalid is_directory type [%d] filename [%s]", cell->type, fs->filename->str);
                    ok = FALSE;
                }
                //   get is_hidden
                cell = db + column_idx[COL_IS_HIDDEN];
                if (cell->type == METADATA_INT64)
                {
                    db_ishidden = cell->value.v_int64;
                }
                else
                {
                    ERRORPRINTF("invalid is_hidden type [%d] filename [%s]", cell->type, fs->filename->str);
                    ok = FALSE;
                }

                // check whether file is local or in another directory
                if (   db_filename
                    && db_dirpath
                    && strcmp(db_dirpath->str, ".") != 0 )
                {
                    // file in another directory
                    //   check target exists
                    cp = g_strdup_printf("%s/%s", db_dirpath->str, db_filename->str);
                    exists = g_file_test(cp, G_FILE_TEST_EXISTS);
                    if ( !exists )
                    {
                        // target not found: remove from database
                        //   lock database
                        if ( !is_database_transaction_started )
                        {
                            rc = start_database_transaction();
                            if (rc == ER_OK)
                            {
                                is_database_transaction_started = TRUE;
                            }
                        }
                        //   remove item from database
                        if (db_isdir)
                        {
                            (void) ermetadb_delete_folder_with_dirpath(g_metadb, db_filename, db_dirpath);
                        }
                        else
                        {
                            (void) ermetadb_delete_document_with_dirpath(g_metadb, db_filename, db_dirpath);
                        }
                    }
                    g_free(cp);

                    //   use next database entry
                    db_row++;
                    db += db_n_cols;
                }
                else
                {
                    // local file: use this database entry
                    done = TRUE;
                }
            }

            // check table pointers synchronised
            if (db_row >= db_n_rows)
            {
                // out of database entries:
                //   filesystem entry not in database
                file_cmp = -1;
            }
            else if (fs_row >= fs_n_rows)
            {
                // out of filesystem entries:
                //   database entry not on filesystem
                fs_tmp.filename     = (GString*) db_filename;  // const_cast
                fs_tmp.is_directory = db_isdir;
                file_cmp = 1;
            }
            else
            {
                // compare filenames and is_directory
                //   compare
                fs_tmp.filename     = (GString*) db_filename;  // const_cast
                fs_tmp.is_directory = db_isdir;
                file_cmp = compare_fs_fileinfo(&fs, &p_fs_tmp);
            }

            // decide what database action is needed
            if (file_cmp > 0)
            {
                // database entry not on filesystem
                do_delete_item = TRUE;

                // next database entry
                db_row++;
                db += db_n_cols;
            }
            else if (file_cmp < 0)
            {
                // filesystem entry not in database
                do_create_item = TRUE;

                if ( !(fs->is_hidden) )
                {
                    g_total_items += 1;
                }

                // next filesystem entry
                fs_row++;
                p_fs++;
            }
            else
            {
                // filesystem and database filename match, check additional fields
                if ( !(fs->is_hidden) )
                {
                    g_total_items += 1;
                }

                // get is_directory
                cell = db + column_idx[COL_IS_DIRECTORY];
                if (cell->type == METADATA_INT64)
                {
                    db_isdir = cell->value.v_int64;
                }
                else
                {
                    ERRORPRINTF("invalid is_directory type [%d] filename [%s]", cell->type, fs->filename->str);
                    ok = FALSE;
                }

                // get filesize
                cell = db + column_idx[COL_FILESIZE];
                if (cell->type == METADATA_INT64)
                {
                    db_filesize = cell->value.v_int64;
                }
                else
                {
                    ERRORPRINTF("invalid file_size type [%d] filename [%s]", cell->type, fs->filename->str);
                    ok = FALSE;
                }

                // get filedate
                cell = db + column_idx[COL_FILEDATE];
                if (cell->type == METADATA_INT64)
                {
                    db_filedate = cell->value.v_int64;
                }
                else
                {
                    ERRORPRINTF("invalid file_last_modified type [%d] filename [%s]", cell->type, fs->filename->str);
                    ok = FALSE;
                }

                // compare fields and decide if/how to update the database
                if (   !ok
                    || (db_isdir == 0  &&  fs->is_directory != FALSE)
                    || (db_isdir == 1  &&  fs->is_directory != TRUE ) )
                {
                    // wrong data from database or is_directory values differ: re-create database item
                    do_delete_item = TRUE;
                    do_create_item = TRUE;
                    WARNPRINTF( "is_directory differs: db [%lld] fs [%d] filename [%s]",
                                                       db_isdir,
                                                                 fs->is_directory,
                                                                                   fs->filename->str);
                }
                else
                {
                    // is_directory values match: check other fields

                    //   check filesize
                    if (db_filesize != fs->filesize)
                    {
                        // filesize differs
                        ext = g_extension_pointer(fs->filename->str);
                        if (   db_ishidden == 1
                            || strcmp(ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
                        {
                            // accept filesize changes for these files:
                            //   hidden files are not shown, so database update is useless
                            //     without this check the 'metadata.db', '.', and possibly '..' keep
                            //     changing when navigating the filesystem, which results in many
                            //     database updates.
                            //   shortcuts to directory may have the propagate-name-to-folder option enabled,
                            //     in which case the filesize may change but thumbnail data must stay in database
                        }
                        else
                        {
                            // filesize values differ: re-create database item
                            do_delete_item = TRUE;
                            do_create_item = TRUE;
                            WARNPRINTF( "file_size differs [%lld] [%lld] filename [%s]",
                                                            db_filesize,
                                                                   fs->filesize,   fs->filename->str);
                        }
                    }

                    //   check filedate (timestamp)
                    if (   !do_delete_item
                        && db_ishidden == 0
                        && db_filedate != fs->filedate )
                    {
                        do_update_item = TRUE;
                    }
                }

                // next filesystem entry
                fs_row++;
                p_fs++;
                // next database entry
                db_row++;
                db += db_n_cols;
            }

            LOGPRINTF( "fs.file [%s] db.file [%s] do_del/_ins/_upd [%d %d %d]", 
                       (fs_row < fs_n_rows) ? fs->filename->str : NULL,
                       (db_row < db_n_rows) ? db_filename->str  : NULL,
                       do_delete_item,
                       do_create_item,
                       do_update_item );

            // lock database, if needed
            if (   ( do_delete_item  ||  do_create_item  ||  do_update_item )
                && !is_database_transaction_started )
            {
                rc = start_database_transaction();
                if (rc == ER_OK)
                {
                    is_database_transaction_started = TRUE;
                }
            }

            // update database as needed
            if (do_delete_item)
            {
                // remove database entry
                (void) delete_database_item(&fs_tmp);
            }
            if (do_create_item)
            {
                // insert database entry
                (void) create_database_item(fs);

                // make shortcut to directory sort with directories
                ext = g_extension_pointer(fs->filename->str);
                if ( strcmp(ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
                {
                    update_database_item_sort_order(fs->filename, MDB_SORT_PRIORITY_FOLDER);
                }
            }
            if (do_update_item)
            {
                // update database entry
                (void) update_database_item(fs);
            }
        } // end while
    }

    // write database changes now
    if (is_database_transaction_started)
    {
        rc = end_database_transaction();
        if (ret == ER_OK)
        {
            ret = rc;
        }
    }

    // clean up
    metadata_table_free(values);
    free_fs_fileinfo_list(fileinfo);
    fileinfo = NULL;

    LOGPRINTF("leave");
    return ret;
}


// retrieve item info from database and load these in file model
static int load_items_in_model (void)
{
    int                 ret = ER_OK;    // return code
    int                 rc;
    int                 i;
    int                 n;
    int                 n_row = 0;
    int                 num_requ;
    time_t              t;
    gint64              i64;
    gboolean            ok;
    gboolean            is_shortcut      = FALSE;
    gboolean            prepend_prev_dir = FALSE;
    char                *cp;
    const gchar         *dir;
    const gchar         *ext;
    GString             *mobi_title = NULL;
    shortcut_t          *shortcut   = NULL;
    GdkPixbufLoader     *thumb_ldr  = NULL;
    GError              *err        = NULL;
    const metadata_cell *cell       = NULL;

    // data to be released at end of function
    GString             *sort_key = NULL;
    metadata_table      *names    = NULL;
    metadata_table      *values   = NULL;

    // values to be stored in model:
    const char          *filename         = NULL;
    const char          *filename_display = NULL;
    const char          *filetype         = NULL;
    const char          *filetype_display = NULL;
    const char          *dirpath          = NULL;
    gboolean            is_directory;
    gboolean            is_template;
    gchar               *filesize  = NULL;
    char                filedate[100];
    const char          *title     = NULL;
    const char          *author    = NULL;
    GdkPixbuf           *thumbnail = NULL;

    // columns to fetch from database
    static const int    COLUMN_NAME_IDX[] =
                        {
                            COL_FILENAME,
                            COL_DIRECTORY_PATH,
                            COL_IS_DIRECTORY,
                            COL_IS_TEMPLATE,
                            COL_FILETYPE,
                            COL_FILESIZE,
                            COL_FILEDATE,
                            COL_TITLE,
                            COL_AUTHOR
                        };

    LOGPRINTF( "entry: g_item_offset [%d] g_num_items [%d] g_parent_dir [%s]",
               g_item_offset, g_num_items, g_parent_dir );
    if (g_metadb == NULL)
    {
        return ER_OK;
    }

    // prepare database retrieval
    //   set sort key
    sort_key = g_string_new(g_sort_column);
    //   set column names we are interested in
    names = metadata_table_new();
    if (names == NULL)
    {
        ret = ER_FAIL;
    }
    else
    {
        // set default columns
        n = sizeof(COLUMN_NAME_IDX) / sizeof(COLUMN_NAME_IDX[0]);
        for (i = 0 ; i < n ; i++)
        {
            rc = metadata_table_add_column(names, MDB_COLUMN_NAMES[ COLUMN_NAME_IDX[i] ].name);
            if (rc != ER_OK)
            {
                ret = ER_FAIL;
            }
        }

        // set thumbnail column
        switch (g_thumbnail_size)
        {
            case MODTHUMB_MINI:
                i = COL_THUMB_MINI;
                break;
            case MODTHUMB_SMALL:
                i = COL_THUMB_SMALL;
                break;
            case MODTHUMB_MEDIUM:
                i = COL_THUMB_MEDIUM;
                break;
            case MODTHUMB_LARGE:
                i = COL_THUMB_LARGE;
                break;
            default:
                ERRORPRINTF("unknown g_thumbnail_size [%d]", g_thumbnail_size);
                i = -1;
        }
        if (i >= 0)
        {
            rc = metadata_table_add_column(names, MDB_COLUMN_NAMES[i].name);
            if (rc != ER_OK)
            {
                ret = ER_FAIL;
            }
        }
    }

    // retrieve item details from database
    if (ret == ER_OK)
    {
        // need to show previous directory ?
        // how many items to request
        //   always request one more than we need, to know whether there is a next page
        num_requ = g_num_items + 1;
        if (g_item_offset == 0  &&  g_parent_dir)
        {
            // first page: prepend prev_dir
            prepend_prev_dir = TRUE;
            num_requ = g_num_items;
        }

        // get items
        rc = ermetadb_select_file_metadata( g_metadb,
                                            sort_key,
                                            g_is_sort_asc,
                                            g_item_offset,
                                            num_requ,
                                            names,
                                            &values       );
        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot select metadata");
            ret = rc;
        }
        else if (values)
        {
            // note: values is NULL when no files/folders selected
            n_row = metadata_table_n_rows(values);
        }
    }

    // report number of items to be displayed
    if (ret == ER_OK)
    {
        fileview_set_num_items( MIN(g_item_offset + 1, g_total_items),          // first item
                                g_item_offset + MIN(n_row, num_requ - 1),       // last item
                                g_total_items );                                // total items
    }

    // process the retrieved data
    if (ret == ER_OK)
    {
        // determine previous/next page available
        g_has_next_page = (n_row == num_requ) ? TRUE : FALSE;
        fileview_show_next_button( g_has_next_page );
        fileview_show_prev_button( (g_item_offset > 0) ? TRUE : FALSE );

        // remove old items from filestore
        gtk_list_store_clear(g_filestore);

        // insert previous directory at start
        if (prepend_prev_dir)
        {
            filestore_insert_previous_dir();
        }

        // add new items to filestore
        if (values)
        {
            cell = (const metadata_cell*) (values->cell_data->data);
        }
        for (i = 0 ; i < n_row  &&  i < num_requ - 1 ; i++, cell++)
        {
            // get filename
            filename = "";
            if (cell->type == METADATA_TEXT)
            {
                filename = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for filename", cell->type);
            }
            filename_display = filename;

            // get directory_path
            cell++;
            dirpath = "";
            if (cell->type == METADATA_TEXT)
            {
                dirpath = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for directory_path", cell->type);
            }

            // get is_directory
            cell++;
            is_directory = FALSE;
            if (cell->type == METADATA_INT64)
            {
                i64 = cell->value.v_int64;
                if (i64 == 1)
                {
                    is_directory = TRUE;
                }
                else if (i64 != 0)
                {
                    ERRORPRINTF("illegal value [%lld] for is_directory", i64);
                }
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for is_directory", cell->type);
            }

            // get is_template
            cell++;
            is_template = FALSE;
            if (cell->type == METADATA_INT64)
            {
                i64 = cell->value.v_int64;
                if (i64 == 1)
                {
                    is_template = TRUE;
                }
                else if (i64 != 0)
                {
                    ERRORPRINTF("illegal value [%lld] for is_template", i64);
                }
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for is_template", cell->type);
            }

            // get filetype
            cell++;
            filetype = "";
            if (cell->type == METADATA_TEXT)
            {
                filetype = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for file_type", cell->type);
            }

            // get filetype description
            filetype_display = "";
            if (filetype)
            {
                filetype_display = get_type_descr_from_file_extension( filetype, is_directory, is_template );
            }

            // get filesize
            cell++;
            if (cell->type == METADATA_INT64)
            {
                i64 = cell->value.v_int64;
                if (i64 >= 0)
                {
                    if ( !is_directory )
                    {
                        filesize = format_size(i64);
                    }
                }
                else
                {
                    ERRORPRINTF("illegal value [%lld] for filesize", i64);
                }
            }
            else if (cell->type != METADATA_NULL)
            {
                ERRORPRINTF("illegal cell type [%d] for filesize", cell->type);
            }

            // get filedate
            cell++;
            filedate[0] = '\0';
            if (cell->type == METADATA_INT64)
            {
                t = cell->value.v_int64;
                if (t >= 0)
                {
                    /* TRANSLATORS: This is a date and time format specifier, according to the strftime(3) man page.
                    *            The corresponding string should be something like 13:23 Monday, July 30 2007.
                    *                       Keep %R unchanged. */
                    // xgettext:no-c-format
                    strftime(filedate, sizeof(filedate), _("%x %X"), localtime(&t));  // xgettext:no-c-format
                }
                else
                {
                    ERRORPRINTF("illegal value [%ld] for filedate", t);
                }
            }
            else if (cell->type != METADATA_NULL)
            {
                ERRORPRINTF("illegal cell type [%d] for filedate", cell->type);
            }

            // get title
            cell++;
            title = "";
            if (cell->type == METADATA_TEXT)
            {
                title = cell->value.v_text->str;
            }
            else if (cell->type != METADATA_NULL)
            {
                ERRORPRINTF("illegal cell type [%d] for title", cell->type);
            }
            // for Mobipocket documents use title as filename
            if (   is_directory == FALSE
                && is_mobipocket_file_extension(filetype) )
            {
                if ( title && title[0] )
                {
                    // title specified in database: use this one
                    filename_display = title;
                }
                else
                {
                    // fallback: extract title from document
                    cp = g_strdup_printf("%s/%s", dirpath, filename);
                    mobi_title = get_mobipocket_title(cp);
                    if (mobi_title)
                    {
                        filename_display = mobi_title->str;
                    }
                    g_free(cp);
                }
            }

            // get author
            cell++;
            author = "";
            if (cell->type == METADATA_TEXT)
            {
                author = cell->value.v_text->str;
            }
            else if (cell->type != METADATA_NULL)
            {
                ERRORPRINTF("illegal cell type [%d] for author", cell->type);
            }

            // get thumbnail
            thumbnail = NULL;
            cell++;
            if (cell->type == METADATA_BLOB)
            {
                ok = TRUE;

                // load pixbuf with data from blob
                thumb_ldr = gdk_pixbuf_loader_new_with_type(THUMBNAIL_IMAGE_TYPE, NULL);
                if (thumb_ldr == NULL)
                {
                    ERRORPRINTF("cannot create GdkPixbufLoader for thumbnail type [%s]", THUMBNAIL_IMAGE_TYPE);
                    ok = FALSE;
                }

                if (ok)
                {
                    ok = gdk_pixbuf_loader_write(thumb_ldr, (guchar*)(cell->value.v_blob.data), cell->value.v_blob.len, &err);
                    if ( !ok )
                    {
                        ERRORPRINTF("cannot load thumbnail for file [%s] - [%s]", filename, err->message);
                        g_clear_error(&err);
                    }
                }

                if (ok)
                {
                    ok = gdk_pixbuf_loader_close(thumb_ldr, &err);
                    if ( !ok )
                    {
                        ERRORPRINTF("cannot close thumbnail for file [%s] - [%s]", filename, err->message);
                        g_clear_error(&err);
                    }
                }

                // get a pointer to the resulting pixbuf, if any
                if (ok)
                {
                    thumbnail = gdk_pixbuf_loader_get_pixbuf(thumb_ldr);
                    if (thumbnail)
                    {
                        // take a reference on the object
                        g_object_ref(thumbnail);

                        // add icon overlay as needed
                        if (is_template)
                        {
                            apply_icon_overlay_template(g_thumbnail_size, thumbnail);
                        }
                    }
                    else
                    {
                        ERRORPRINTF("cannot get thumbnail pixbuf for file [%s]", filename);
                    }
                }
            }
            else if (cell->type != METADATA_NULL)
            {
                ERRORPRINTF("illegal cell type [%d] for thumbnail", cell->type);
            }

            // shortcut files need special care
            is_shortcut = is_shortcut_file_extension(filetype);
            if (is_shortcut)
            {
                // get details from shortcut file
                if ( strcmp(dirpath, ".") == 0 )
                {
                    dir = g_metadb->directory->str;
                }
                else
                {
                    dir = dirpath;
                }
                rc = parse_shortcut_file(dir, filename, &shortcut);
                if (shortcut)
                {
                    // take display name from shortcut
                    if (shortcut->name)
                    {
                        filename_display = shortcut->name;
                    }

                    if (thumbnail == NULL)
                    {
                        // get default icon
                        switch (shortcut->type)
                        {
                            case SHORTCUT_TO_FILE:
                                // get icon for extension of target file
                                ext = g_extension_pointer(shortcut->details.file.filename);
                                thumbnail = get_icon_from_file_extension( ext,
                                                                          is_directory,
                                                                          is_template,
                                                                          g_thumbnail_size );
                                break;

                            case SHORTCUT_TO_FOLDER:
                                // get icon for library
                                cp = shortcut->details.folder.directory;
                                if (   cp
                                    && shortcut->details.folder.filename == NULL )
                                {
                                    if ( strcmp(cp, DIR_LIBRARY) == 0 )
                                    {
                                        // internal library root
                                        thumbnail = get_icon_library( g_thumbnail_size );
                                    }
                                    else
                                    {
                                        n = strlen(DIR_STORAGE_MNT);
                                        if (   strncmp(cp, DIR_STORAGE_MNT, n) == 0
                                            && cp[n] == '/' )
                                        {
                                            // path: /media/
                                            cp = strchr( cp + n + 1, '/' );
                                            if ( cp == NULL )
                                            {
                                                // external library root
                                                thumbnail = get_icon_library( g_thumbnail_size );
                                            }
                                        }
                                    }
                                }

                                // get icon for folder
                                if (thumbnail == NULL)
                                {
                                    thumbnail = get_icon_from_file_extension( "",
                                                                              TRUE,        // is_directory
                                                                              is_template,
                                                                              g_thumbnail_size );
                                }
                                break;

                            case SHORTCUT_TO_APPLICATION:
                                // get icon for application
                                thumbnail = get_icon_application( g_thumbnail_size );
                                break;

                            default:
                                WARNPRINTF("unknown shortcut type [%d] file [%s]", shortcut->type, filename);

                                // get icon from file extension
                                thumbnail = get_icon_from_file_extension( filetype,
                                                                          is_directory,
                                                                          is_template,
                                                                          g_thumbnail_size );
                        }

                        // add shortcut overlay
                        if (thumbnail)
                        {
                            if ( strcmp(dir, DIR_DESKTOP_INTERNAL) != 0 )
                            {
                                thumbnail = gdk_pixbuf_copy(thumbnail);
                                apply_icon_overlay_shortcut(g_thumbnail_size, thumbnail);
                            }
                            else
                            {
                                g_object_ref(thumbnail);
                            }
                        }
                    }
                }
            }

            // get default icon when no thumbnail available
            if (thumbnail == NULL)
            {
                thumbnail = get_icon_from_file_extension( filetype, is_directory, is_template, g_thumbnail_size );
                if (thumbnail)
                {
                    g_object_ref(thumbnail);
                }
            }

            // add to filestore
            gtk_list_store_insert_with_values( g_filestore,
                                               NULL,
                                               g_num_items,
                                               MODCOL_FILENAME,         filename,
                                               MODCOL_FILENAME_DISPLAY, filename_display,
                                               MODCOL_FILETYPE,         filetype,
                                               MODCOL_FILETYPE_DISPLAY, filetype_display,
                                               MODCOL_DIRECTORY_PATH,   dirpath,
                                               MODCOL_IS_DIRECTORY,     is_directory,
                                               MODCOL_IS_TEMPLATE,      is_template,
                                               MODCOL_FILESIZE,         filesize,
                                               MODCOL_FILEDATE,         filedate,
                                               MODCOL_AUTHOR,           author,
                                               MODCOL_THUMBNAIL,        thumbnail,
                                               -1 );

            // clean up
            if (mobi_title)
            {
                g_string_free(mobi_title, TRUE);
                mobi_title = NULL;
            }
            if (shortcut)
            {
                shortcut_free(shortcut);
                shortcut = NULL;
            }
            if (thumb_ldr)
            {
                g_object_unref(thumb_ldr);
                thumb_ldr = NULL;
            }
            if (thumbnail)
            {
                g_object_unref(thumbnail);
            }
            g_free(filesize);
            filesize = NULL;
        } // end for
    } // end if (ret == ER_OK)

    // clean up
    metadata_table_free(values);
    metadata_table_free(names);
    if (sort_key) { g_string_free(sort_key, TRUE); }

    LOGPRINTF("leave");
    return ret;
}

// add previous directory to filestore
static void filestore_insert_previous_dir (void)
{
    LOGPRINTF("entry");

    gtk_list_store_insert_with_values( g_filestore,
                                       NULL,
                                       g_num_items,
                                       MODCOL_FILENAME,         g_parent_dir,
                                       /* TRANSLATORS: This text indicates the parent directory in the file browser. */
                                       MODCOL_FILENAME_DISPLAY, _("Up a Level"),
                                       MODCOL_FILETYPE,         "",
                                       MODCOL_FILETYPE_DISPLAY, "",
                                       MODCOL_DIRECTORY_PATH,   ".",
                                       MODCOL_IS_DIRECTORY,     TRUE,
                                       MODCOL_IS_TEMPLATE,      FALSE,
                                       MODCOL_FILESIZE,         "",
                                       MODCOL_FILEDATE,         "",
                                       MODCOL_AUTHOR,           "",
                                       MODCOL_THUMBNAIL,        get_icon_parent_dir(g_thumbnail_size),
                                       -1 );
}


// format filesize into a string
static gchar* format_size(const gint64 bytes)
{
    gchar       *size = NULL;   // return value
    long        n_int;
    long        n_frac;

    const int   MB = 1024 * 1024;
    const int   KB = 1024;

    if (bytes >= MB)
    {
        if (bytes >= 10 * MB)
        {
            n_int = (bytes + (MB / 2)) / MB;                    // MB round upward
            size = g_strdup_printf( "%ld %s",
                                     n_int,
                                          _("MB") );
        }
        else
        {
            n_int  = bytes / MB;                                //     MB round downward
            n_frac = ((bytes % MB) + (MB / 20)) / (MB / 10);    // 0.1 MB round upward
            n_frac = MIN(n_frac, 9);
            size = g_strdup_printf( "%ld.%ld %s",
                                     n_int,
                                         n_frac,
                                             _("MB") );
        }
    }
    else if (bytes >= KB)
    {
        if (bytes >= 10 * KB)
        {
            n_int = (bytes + (KB / 2)) / KB;                    // KB round upward
            size = g_strdup_printf( "%ld %s",
                                     n_int,
                                          _("KB") );
        }
        else
        {
            n_int  = bytes / KB;                                //     KB round downward
            n_frac = ((bytes % KB) + (KB / 20)) / (KB / 10);    // 0.1 KB round upward
            n_frac = MIN(n_frac, 9);
            size = g_strdup_printf( "%ld.%ld %s",
                                     n_int,
                                         n_frac,
                                             _("KB") );
        }
    }
    else
    {
        size = g_strdup_printf( "%lld %s",
                                 bytes,
                                    _("B") );
    }

    return size;
}


//------------------------------------------------------------------------------
// Functions to access filesystem info
//------------------------------------------------------------------------------

// read file details from filesystem
static GPtrArray* read_fs_fileinfo (const gchar *current_dir)
{
    GPtrArray       *fileinfo = NULL;       // return value
    int             rc;
    gboolean        done;
    DIR             *dir     = NULL;
    struct dirent   *dirent  = NULL;;
    fs_fileinfo_t   *inf     = NULL;
    gchar           *old_dir = NULL;

    LOGPRINTF("entry: dir [%s]", current_dir);

    // open directory
    dir = opendir(current_dir);
    if (dir == NULL)
    {
        ERRNOPRINTF("cannot open directory [%s]", current_dir );
    }
    else
    {
        fileinfo = g_ptr_array_new();
        if (fileinfo == NULL)
        {
            ERRORPRINTF("cannot create fileinfo array");
        }
    }

    // read file entries
    if (dir && fileinfo)
    {
        old_dir = g_get_current_dir();
        chdir(current_dir);

        done = FALSE;
        while ( !done )
        {
            dirent = readdir(dir);
            if (dirent == NULL)
            {
                // error or end-of-file, assume end-of-file
                done = TRUE;
            }
            else
            {
                // get details and add to fileinfo array
                inf = parse_dirent(dirent);
                if (inf)
                {
                    inf->is_hidden = is_hidden_filename(dirent->d_name);
                    g_ptr_array_add(fileinfo, inf);
                }
            }
        }

        chdir(old_dir);
    }

    // close directory
    if (dir)
    {
        rc = closedir(dir);
        if (rc != 0)
        {
            ERRNOPRINTF("cannot close directory");
        }
        dir = NULL;
    }

    // sort file entries by name
    if (fileinfo)
    {
        g_ptr_array_sort(fileinfo, compare_fs_fileinfo);
    }

    // clean up
    g_free(old_dir);

    return fileinfo;
}


// compare two fs_fileinfo_t objects
// assuming sort order filename (ascending)
// note that a and b point to the GPtrArray elements
static int compare_fs_fileinfo (gconstpointer a, gconstpointer b)
{
    int ret = 0;  // return code

    const fs_fileinfo_t  *left  = *( (const fs_fileinfo_t**) a );
    const fs_fileinfo_t  *right = *( (const fs_fileinfo_t**) b );

    // LOGPRINTF("entry: left [%s] [%d] right [%s] [%d]",
    //                         left->filename->str,
    //                              left->is_directory,
    //                                         right->filename->str,
    //                                              right->is_directory );

    ret = strcmp(left->filename->str, right->filename->str);

    // LOGPRINTF("leave: return [%d]", ret);
    return ret;
}


// parse directory entry and read additional details
static fs_fileinfo_t* parse_dirent (const struct dirent *dirent)
{
    fs_fileinfo_t   *inf = NULL;        // return value
    int             rc;
    gboolean        ok = TRUE;
    struct stat     statbuf;

    LOGPRINTF("entry: filename [%s]", dirent->d_name);

    // get details from filesystem
    rc = stat(dirent->d_name, &statbuf);
    if (rc != 0)
    {
        ERRNOPRINTF("cannot stat [%s]", dirent->d_name);
        ok = FALSE;
    }
    else
    {
        inf = fs_fileinfo_new();
        if (inf == NULL)
        {
            ERRORPRINTF("cannot allocate a new fs_fileinfo_t object");
            ok = FALSE;
        }
    }

    // parse filesystem info
    if (ok)
    {
        g_string_assign(inf->filename, dirent->d_name);
        if ( S_ISREG(statbuf.st_mode) )
        {
            inf->is_directory = FALSE;
            inf->filesize     = statbuf.st_size;
            inf->filedate     = statbuf.st_mtime;
        }
        else if ( S_ISDIR(statbuf.st_mode) )
        {
            inf->is_directory = TRUE;
            inf->filesize     = 0;
            inf->filedate     = statbuf.st_mtime;
        }
        else
        {
            WARNPRINTF("ignore unknown file type [%s]", dirent->d_name);
            ok = FALSE;
        }
    }

    // discard results when problems
    if ( !ok )
    {
        fs_fileinfo_free(inf);
        inf = NULL;
    }

    return inf;
}


// free array holding fs_fileinfo_t entries
static void free_fs_fileinfo_list (GPtrArray *fileinfo)
{
    LOGPRINTF("entry");

    if (fileinfo)
    {
        // free array elements
        g_ptr_array_foreach(fileinfo, free_fs_fileinfo_item, NULL);

        // free the array
        g_ptr_array_free(fileinfo, TRUE);
    }
}

static void free_fs_fileinfo_item (gpointer data, gpointer user_data)
{
    fs_fileinfo_t *inf = (fs_fileinfo_t*)data;

    fs_fileinfo_free(inf);
}


//------------------------------------------------------------------------------
// Functions to access database holding metadata for files
//------------------------------------------------------------------------------

// chek if database is in desktop directory
static gboolean db_is_desktop_directory ( erMetadb *db )
{
    gboolean    is_desktop = FALSE;     // return value

    LOGPRINTF("entry");
    g_assert(db);

    if (   db
        && db->directory
        && db->directory->str
        && strcmp( db->directory->str, DIR_DESKTOP_INTERNAL ) == 0 )
    {
        is_desktop = TRUE;
    }

    return is_desktop;
}

 
// open or create metadata database
static int open_database (const gchar *directory)
{
    int           ret = ER_OK;    // return code
    int           rc;
    const time_t  start = time(NULL);
    GString       *dir = g_string_new(directory);

    LOGPRINTF("entry: dir [%s]", directory);

    // check whether requested database already open
    if (   g_metadb
        && g_metadb->directory
        && strcmp(g_metadb->directory->str, directory) == 0 )
    {
        // requested database already open: nothing to do
        LOGPRINTF("database already open: quit, dir [%s]", directory);
        return ER_OK;
    }

    // create database object when not yet done
    if (g_metadb == NULL)
    {
        g_metadb = ermetadb_new();
        if (g_metadb == NULL)
        {
            ERRORPRINTF("cannot create database object");
            ret = ER_FAIL;
        }
    }
    else
    {
        // database object exists: close it
        close_database();
    }

    // open database in specified directory
    if (ret == ER_OK)
    {
        rc = ermetadb_open_database(g_metadb, dir);
        if (rc == ERMETADB_DATABASE_BUSY)
        {
            // database locked, wait till database becomes available
            ipc_menu_busy_show(TRUE);
            while (   rc == ERMETADB_DATABASE_BUSY
                   && (time(NULL) - start) <= MAX_WAIT_DATABASE )
            {
                WARNPRINTF("database locked, dir [%s]", dir->str);
                rc = ermetadb_open_database(g_metadb, dir);
            }
            ipc_menu_busy_show(FALSE);
        }

        if (rc == ER_NOT_FOUND)
        {
            // database not present: create a new database and open it
            WARNPRINTF("cannot open database [%s], error [%d]", dir->str, rc);
            rc = ermetadb_create_database(dir);
            if (rc == ER_OK)
            {
                rc = ermetadb_open_database(g_metadb, dir);
            }
        }

        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot open or create database in dir [%s] rc [%d]", directory, rc);
            ret = rc;
        }
    }

    // clean up
    g_string_free(dir, TRUE);

    return ret;
}


// close metadata database
static void close_database (void)
{
    LOGPRINTF("entry");

    if (g_metadb)
    {
        // database object exists: close it
        (void) ermetadb_close_database(g_metadb);
        sync();
    }
}


// start a database transaction
static int start_database_transaction (void)
{
    int           ret = ER_OK;    // return code
    const time_t  start = time(NULL);

    LOGPRINTF("entry");

    if (g_metadb)
    {
        ret = ermetadb_begin_transaction(g_metadb);
        if (ret == ERMETADB_DATABASE_BUSY)
        {
            // database locked, wait till database becomes available
            ipc_menu_busy_show(TRUE);
            while (   ret == ERMETADB_DATABASE_BUSY
                   && (time(NULL) - start) <= MAX_WAIT_DATABASE )
            {
                WARNPRINTF("database locked");
                ret = ermetadb_begin_transaction(g_metadb);
            }
            ipc_menu_busy_show(FALSE);
        }
    }

    return ret;
}


// end a database transaction
static int end_database_transaction (void)
{
    int  ret = ER_OK;    // return code

    LOGPRINTF("entry");

    if (g_metadb)
    {
        ret = ermetadb_end_transaction(g_metadb);
    }

    return ret;
}


// insert a new database item
static int create_database_item (const fs_fileinfo_t *inf)
{
    int  ret = ER_OK;    // return code

    LOGPRINTF("entry: file [%s] is_dir [%d]", inf->filename->str, inf->is_directory);


    // create item
    if (inf->is_directory)
    {
        ret = ermetadb_add_folder(g_metadb, inf->filename, inf->filedate);
    }
    else
    {
        ret = ermetadb_add_document(g_metadb, inf->filename, inf->filesize, inf->filedate);
    }

    // set details, if needed
    if (ret == ER_OK)
    {
        if (inf->is_hidden)
        {
            ret = update_database_item(inf);
        }
    }

    return ret;
}


// delete an existing database item
static int delete_database_item (const fs_fileinfo_t *inf)
{
    int         ret = ER_OK;    // return code

    LOGPRINTF("entry: file [%s] is_dir [%d]", inf->filename->str, inf->is_directory);

    // delete item
    if (inf->is_directory)
    {
        ret = ermetadb_delete_folder(g_metadb, inf->filename);
    }
    else
    {
        ret = ermetadb_delete_document(g_metadb, inf->filename);
    }

    return ret;
}


// update an existing database item
// note: leave is_template unchanged, as content browser uses it read-only for now.
//       add is_template as a separate parameter,
//       do not include it in fs_fileinfo_t because it is not a details read from filesystem
static int update_database_item (const fs_fileinfo_t *inf)
{
    int             ret = ER_OK;    // return code
    int             col = 0;
    metadata_table  *values   = NULL;

    LOGPRINTF( "entry: file [%s] is_dir [%d]",
               inf->filename->str,
               inf->is_directory  );

    // set new metadata for item
    //   create values table
    values = metadata_table_new();
    if (values == NULL)
    {
        ret = ER_FAIL;
    }
    //   set is_directory
    col = 0;
    if (ret == ER_OK)
    {
        ret = metadata_table_add_column(values, MDB_COLUMN_NAMES[ COL_IS_DIRECTORY ].name);
    }
    if (ret == ER_OK)
    {
        metadata_table_set_int64( values, col, (inf->is_directory ? 1 : 0) );
    }
    //   set is_hidden
    if (ret == ER_OK)
    {
        ret = metadata_table_add_column(values, MDB_COLUMN_NAMES[ COL_IS_HIDDEN ].name);
    }
    if (ret == ER_OK)
    {
        col++;
        metadata_table_set_int64( values, col, (inf->is_hidden ? 1 : 0) );
    }
    //   set file size
    if (ret == ER_OK)
    {
        ret = metadata_table_add_column(values, MDB_COLUMN_NAMES[ COL_FILESIZE ].name);
    }
    if (ret == ER_OK)
    {
        col++;
        metadata_table_set_int64(values, col, inf->filesize);
    }
    //   set file date+time
    if (ret == ER_OK)
    {
        ret = metadata_table_add_column(values, MDB_COLUMN_NAMES[ COL_FILEDATE ].name);
    }
    if (ret == ER_OK)
    {
        col++;
        metadata_table_set_int64(values, col, inf->filedate);
    }
    //   write to database
    if (ret == ER_OK)
    {
        ret = ermetadb_set_file_metadata(g_metadb, inf->filename, values);
    }

    // clean up
    if (values)
    {
        metadata_table_free(values);
    }

    if (ret != ER_OK)
    {
        ERRORPRINTF("error [%d]", ret);
    }
    return ret;
}


// set sort_order for an existing database item
static int update_database_item_sort_order (const GString *filename, int sort_order)
{
    int             ret = ER_OK;    // return code
    int             col;
    metadata_table  *values   = NULL;

    LOGPRINTF( "entry: file [%s] sort_order [%d]", filename->str, sort_order );

    // set new metadata for item
    //   create values table
    values = metadata_table_new();
    if (values == NULL)
    {
        ret = ER_FAIL;
    }
    //   set sort_order
    col = 0;
    if (ret == ER_OK)
    {
        ret = metadata_table_add_column(values, MDB_COLUMN_NAMES[COL_SORT_PRIORITY].name);
    }
    if (ret == ER_OK)
    {
        metadata_table_set_int64( values, col, sort_order );
    }
    //   write to database
    if (ret == ER_OK)
    {
        ret = ermetadb_set_file_metadata(g_metadb, filename, values);
    }

    // clean up
    if (values)
    {
        metadata_table_free(values);
    }

    if (ret != ER_OK)
    {
        ERRORPRINTF("error [%d]", ret);
    }
    return ret;
}


// get sort order from database
// [out] g_sort_column
// [out] g_is_sort_ascending
static void read_sort_order_from_database (void)
{
    gboolean            ok = TRUE;
    int                 rc;
    int                 i;
    const char          *cp;
    gchar               *str_col = NULL;
    gchar               *str_asc;
    GString             *this_dir = g_string_new(".");
    metadata_table      *names  = NULL;
    metadata_table      *values = NULL;
    const metadata_cell *cell   = NULL;

    // sorting order, fill in default
    gboolean            is_sort_valid     = FALSE;
    ctb_sort_order_t    sort_order        = SORT_ORDER_DEFAULT;
    gboolean            is_sort_ascending = SORT_IS_ASC_DEFAULT;

    LOGPRINTF("entry");
    g_return_if_fail(g_metadb);

    // set metadata key we are interested in
    names = metadata_table_new();
    if (names == NULL)
    {
        ok = FALSE;
    }
    if (ok)
    {
        rc = metadata_table_add_column(names, METADATA_KEY_SORTORDER);
        if (rc != ER_OK)
        {
            ok = FALSE;
        }
    }

    // retrieve sort order from database
    if (ok)
    {
        rc = ermetadb_get_application_data( g_metadb,
                                            this_dir,
                                            names,
                                            &values   );
        if (rc != ER_OK)
        {
            WARNPRINTF("cannot retrieve sort order");
            ok = FALSE;
        }
    }

    // process the retrieved data, if any
    if (ok  &&  values)
    {
        cell = (metadata_cell*) (values->cell_data->data);
        if (cell->type != METADATA_TEXT)
        {
            ERRORPRINTF("illegal cell type [%d] for metadata key [%s]", cell->type, METADATA_KEY_SORTORDER);
        }
        else
        {
            LOGPRINTF("sort_order from database [%s]", cell->value.v_text->str);

            // split sort value into column + asc/desc
            str_col = g_strdup(cell->value.v_text->str);
            str_asc = strchr(str_col, ' ');
            while (str_asc  &&  *str_asc == ' ')
            {
                *str_asc = '\0';
                str_asc++;
            }

            // check valid sort column
            for (i = 0 ; !is_sort_valid  &&  i < N_CTB_SORT_ORDER ; i++)
            {
                cp = MDB_COLUMN_NAMES[ SORT_COLUMN_NAME_IDX[i] ].name;
                if ( strcmp(str_col, cp) == 0 )
                {
                    is_sort_valid  = TRUE;
                    sort_order     = i;
                }
            }

            // check valid ascending/descending
            if ( is_sort_valid )
            {
                if ( g_ascii_strcasecmp(str_asc, "ASC") == 0 )
                {
                    is_sort_ascending = TRUE;
                }
                else if ( g_ascii_strcasecmp(str_asc, "DESC") == 0 )
                {
                    is_sort_ascending = FALSE;
                }
            }
        }
    }

    // save results
    cp = MDB_COLUMN_NAMES[ SORT_COLUMN_NAME_IDX[ sort_order ] ].name;
    g_sort_column = cp;
    g_is_sort_asc = is_sort_ascending;
    LOGPRINTF("sort_column [%s] is_ascending [%d]", cp, is_sort_ascending);

    // update popup menu
    menu_select_sort_order( sort_order );

    // clean up
    metadata_table_free(values);
    metadata_table_free(names);
    if (this_dir) { g_string_free(this_dir, TRUE); }
    g_free(str_col);  // note: str_asc points into str_col and must not be free'ed
}


// store sort order in database
// [in] g_sort_column
// [in] g_is_sort_ascending
static void write_sort_order_to_database (void)
{
    gboolean            ok = TRUE;
    int                 rc;
    const char          *cp;
    GString             *this_dir = g_string_new(".");
    metadata_table      *values = NULL;

    LOGPRINTF("entry");
    g_return_if_fail(g_metadb);

    // function void for desktop directory
    if ( db_is_desktop_directory( g_metadb ) )
    {
        return;
    }

    // set new metadata for item
    //   create values table
    values = metadata_table_new();
    if (values == NULL)
    {
        ok = FALSE;
    }
    //   set sort order
    if (ok)
    {
        rc = metadata_table_add_column(values, METADATA_KEY_SORTORDER);
        if (rc != ER_OK)
        {
            ok = FALSE;
        }
        else
        {
            cp = g_strdup_printf( "%s %s",
                                  g_sort_column,
                                  g_is_sort_asc ? "ASC" : "DESC" );
            metadata_table_set_text( values, 0, cp );
        }
    }
    //   write to database
    if (ok)
    {
        rc = ermetadb_set_application_data(g_metadb, this_dir, values);
        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot write sorting order to database");
            ok = FALSE;
        }
    }

    // clean up
    metadata_table_free(values);
    if (this_dir) { g_string_free(this_dir, TRUE); }
}


//------------------------------------------------------------------------------
// Functions to manipulate data in filestore
//------------------------------------------------------------------------------

// get list of filenames from treepath
GSList* filemodel_get_filenames_from_treepath_list (const GList *selected_items)
{
    GSList              *filelist       = NULL;  // return value
    filelist_entry_t    *filelist_entry = NULL;  // entry in filelist
    gboolean            ok;
    gboolean            is_directory;
    gboolean            is_template;
    GtkTreeIter         iter;
    GtkTreePath         *treepath         = NULL;
    gchar               *treepath_string  = NULL;
    gchar               *filename         = NULL;
    gchar               *filename_display = NULL;
    gchar               *filetype         = NULL;
    gchar               *directory_path   = NULL;
    const GList         *item             = NULL;

    GtkTreeModel        *filemodel = get_filemodel();

    LOGPRINTF("entry");
    g_assert(g_filestore);

    // add filenames from selected items
    for (item = selected_items ; item ; item = item->next )
    {
        treepath        = (GtkTreePath*) item->data;
        treepath_string = gtk_tree_path_to_string( treepath );
     
        // get filename from model
        //   get iterator on item
        ok = gtk_tree_model_get_iter(filemodel, &iter, treepath);
        if ( !ok )
        {
            ERRORPRINTF("cannot get iter for treepath [%s]", treepath_string);
        }
        else
        {
            //   get details for item
            is_directory = FALSE;
            is_template  = FALSE;
            gtk_tree_model_get( filemodel, &iter, MODCOL_FILENAME,         &filename,
                                                  MODCOL_FILENAME_DISPLAY, &filename_display,
                                                  MODCOL_FILETYPE,         &filetype,
                                                  MODCOL_DIRECTORY_PATH,   &directory_path,
                                                  MODCOL_IS_DIRECTORY,     &is_directory,
                                                  MODCOL_IS_TEMPLATE,      &is_template,
                                                  -1 );

            LOGPRINTF("treepath [%s]:",          treepath_string );
            LOGPRINTF("  filename         [%s]", filename        );
            LOGPRINTF("  filename_display [%s]", filename_display);
            LOGPRINTF("  filetype         [%s]", filetype        );
            LOGPRINTF("  directory_path   [%s]", directory_path  );
            LOGPRINTF("  is_directory     [%d]", is_directory    );
            LOGPRINTF("  is_template      [%d]", is_template     );
            if (filename == NULL)
            {
                ERRORPRINTF("cannot get filename for treepath [%s]", treepath_string);
            }
            if (filename_display == NULL)
            {
                ERRORPRINTF("cannot get filename_display for treepath [%s]", treepath_string);
            }
            if (filetype == NULL)
            {
                ERRORPRINTF("cannot get filetype for treepath [%s]", treepath_string);
            }
            if (directory_path == NULL)
            {
                ERRORPRINTF("cannot get directory_path for treepath [%s]", treepath_string);
            }
        }

        // append filename to list
        if (filename)
        {
            filelist_entry                   = filelist_entry_new();
            filelist_entry->filename         = g_string_new(filename);
            filelist_entry->filename_display = g_string_new(filename_display);
            filelist_entry->filetype         = g_string_new(filetype);
            filelist_entry->directory_path   = g_string_new(directory_path);
            filelist_entry->is_directory     = is_directory;
            filelist_entry->is_template      = is_template;

            filelist = g_slist_prepend(filelist, filelist_entry);
            filelist_entry = NULL;
        }

        // clean up
        g_free(directory_path);    directory_path   = NULL;
        g_free(filetype);          filetype         = NULL;
        g_free(filename_display);  filename_display = NULL;
        g_free(filename);          filename         = NULL;
        g_free(treepath_string);   treepath_string  = NULL;
    } // end for
    filelist = g_slist_reverse(filelist);

    // report our findings
    return filelist;
}


// get all filenames
GSList* filemodel_get_filenames_all (void)
{
    GSList              *filelist       = NULL;  // return value
    filelist_entry_t    *filelist_entry = NULL;  // entry in filelist
    gboolean            ok = TRUE;
    gboolean            is_directory;
    gboolean            is_template;
    int                 rc;
    int                 i;
    int                 n;
    const metadata_cell *cell           = NULL;
    const char          *filename       = NULL;
    const char          *filetype       = NULL;
    gchar               *directory_path = NULL;
    metadata_table      *names          = NULL;
    metadata_table      *values         = NULL;

    LOGPRINTF("entry");
    g_assert(g_metadb);

    // get filenames from database
    names = metadata_table_new();
    metadata_table_add_column( names, MDB_COLUMN_NAMES[COL_FILENAME      ].name );
    metadata_table_add_column( names, MDB_COLUMN_NAMES[COL_DIRECTORY_PATH].name );
    metadata_table_add_column( names, MDB_COLUMN_NAMES[COL_FILETYPE      ].name );
    metadata_table_add_column( names, MDB_COLUMN_NAMES[COL_IS_DIRECTORY  ].name );
    metadata_table_add_column( names, MDB_COLUMN_NAMES[COL_IS_TEMPLATE   ].name );
    rc = ermetadb_select_file_metadata( g_metadb,
                                        NULL,           // sort_key
                                        FALSE,          // sort_is_ascending
                                        0,              // start_item
                                        -1,             // num_items
                                        names,
                                        &values );
    if (rc != ER_OK)
    {
        ERRORPRINTF("cannot select metadata");
        ok = FALSE;
    }

    // append filenames to list
    if (values)
    {
        n = metadata_table_n_rows(values);
        cell = (const metadata_cell*) (values->cell_data->data);
        for (i = 0 ; i < n ; i++, cell++)
        {
            // get filename
            filename = NULL;
            if (cell->type == METADATA_TEXT)
            {
                filename = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for filename", cell->type);
            }

            // get directory_path
            directory_path = NULL;
            if (cell->type == METADATA_TEXT)
            {
                directory_path = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for directory_path", cell->type);
            }

            // get filetype
            cell++;
            filetype = NULL;
            if (cell->type == METADATA_TEXT)
            {
                filetype = cell->value.v_text->str;
            }
            else
            {
                ERRORPRINTF("illegal cell type [%d] for filetype", cell->type);
            }

            // get is_directory
            cell++;
            is_directory = FALSE;
            if (cell->type == METADATA_INT64)
            {
                if (cell->value.v_int64 == 1)
                {
                    is_directory = TRUE;
                }
            }
            else
            {
                ERRORPRINTF("invalid is_directory type [%d] filename [%s]", cell->type, filename);
            }

            // get is_template
            cell++;
            is_template = FALSE;
            if (cell->type == METADATA_INT64)
            {
                if (cell->value.v_int64 == 1)
                {
                    is_template = TRUE;
                }
            }
            else
            {
                ERRORPRINTF("invalid is_template type [%d] filename [%s]", cell->type, filename);
            }

            // append filename to list
            if (filename)
            {
                filelist_entry                   = filelist_entry_new();
                filelist_entry->filename         = g_string_new(filename);
                filelist_entry->filename_display = g_string_new(filename);
                filelist_entry->filetype         = g_string_new(filetype);
                filelist_entry->directory_path   = g_string_new(directory_path);
                filelist_entry->is_directory     = is_directory;
                filelist_entry->is_template      = is_template;

                filelist = g_slist_prepend(filelist, filelist_entry);
                filelist_entry = NULL;
            }
        }
        filelist = g_slist_reverse(filelist);
    }


    // clean up
    metadata_table_free(values);
    metadata_table_free(names);

    // report our findings
    return filelist;
}


// free list of filenames
void filemodel_filelist_free ( GSList *filelist )
{
    GSList  *filelist_iter  = NULL;  // iterator over filelist

    LOGPRINTF("entry: filelist [%p]", filelist);
    
    if (filelist)
    {
        for (filelist_iter = filelist ; filelist_iter ; filelist_iter = filelist_iter->next)
        {
            filelist_entry_free( (filelist_entry_t*) filelist_iter->data );
        }
        g_slist_free(filelist);
    }
}


// allocate a single filelist_entry
filelist_entry_t* filelist_entry_new ( void )
{
    filelist_entry_t    *thiz = NULL;   // return value

    LOGPRINTF("entry");
    
    thiz = g_new0(filelist_entry_t, 1);
    g_assert(thiz);
    thiz->is_directory = FALSE;
    thiz->is_template  = FALSE;

    return thiz;
}


// free a single filelist_entry
void filelist_entry_free ( filelist_entry_t *thiz )
{
    LOGPRINTF("entry: thiz [%p]", thiz);
    
    if (thiz)
    {
        if (thiz->filename)
        {
            g_string_free(thiz->filename, TRUE);
        }
        if (thiz->filename_display)
        {
            g_string_free(thiz->filename_display, TRUE);
        }
        if (thiz->filetype)
        {
            g_string_free(thiz->filetype, TRUE);
        }
        if (thiz->directory_path)
        {
            g_string_free(thiz->directory_path, TRUE);
        }
        g_free(thiz);
    }
}


// copy a filelist_entry
filelist_entry_t* filelist_entry_copy ( const filelist_entry_t *src )
{
    filelist_entry_t    *thiz = NULL;    // return value

    LOGPRINTF("entry");
    g_assert(src);
    g_assert(src->filename         && src->filename->str);
    g_assert(src->filename_display && src->filename_display->str);
    g_assert(src->filetype         && src->filetype->str);

    thiz = filelist_entry_new();
    if (src->filename)
    {
        thiz->filename = g_string_new(src->filename->str);
    }
    if (src->filename_display)
    {
        thiz->filename_display = g_string_new(src->filename_display->str);
    }
    if (src->filetype)
    {
        thiz->filetype = g_string_new(src->filetype->str);
    }
    thiz->is_directory = src->is_directory;
    thiz->is_template  = src->is_template;

    return thiz;
}


// get thumbnail
GByteArray* filemodel_get_thumbnail( const filelist_entry_t      *file_info,
                                     const filemodel_thumbsize_t thumbnail_size )
{
    GByteArray          *thumbnail = NULL;      // return value
    gboolean            ok         = TRUE;
    int                 rc;
    int                 i;
    metadata_table      *names     = NULL;
    metadata_table      *values    = NULL;
    const metadata_cell *cell      = NULL;
    const guint8        *data;
    guint               len;

    g_assert(g_metadb);
    g_assert(file_info  &&  file_info->filename  &&  file_info->filename->str);
    g_assert(thumbnail_size >= 0  &&  thumbnail_size < N_FILEMODEL_THUMB_SIZES);
    LOGPRINTF( "entry: filename [%s] thumbnail_size [%d]", file_info->filename->str, thumbnail_size);

    // prepare database retrieval
    //   set column name we are interested in
    if (ok)
    {
        names = metadata_table_new();
        if (names == NULL)
        {
            ok = FALSE;
        }
    }
    //   set thumbnail column
    if (ok)
    {
        switch (thumbnail_size)
        {
            case MODTHUMB_MINI:
                i = COL_THUMB_MINI;
                break;
            case MODTHUMB_SMALL:
                i = COL_THUMB_SMALL;
                break;
            case MODTHUMB_MEDIUM:
                i = COL_THUMB_MEDIUM;
                break;
            case MODTHUMB_LARGE:
                i = COL_THUMB_LARGE;
                break;
            default:
                g_assert_not_reached();
        }
        rc = metadata_table_add_column(names, MDB_COLUMN_NAMES[i].name);
        if (rc != ER_OK)
        {
            ok = FALSE;
        }
    }
    //   retrieve thumbnail from database
    if (ok)
    {
        rc = ermetadb_get_file_metadata_with_dirpath( g_metadb,
                                                      file_info->filename,
                                                      file_info->directory_path,
                                                      names,
                                                      &values );
        if (rc != ER_OK)
        {
            WARNPRINTF("cannot select thumbnail");
            ok = FALSE;
        }
    }
    //   process the retrieved data
    if (ok  &&  values)
    {
        // get thumbnail data
        cell = metadata_table_get_cell(values, 0);
        if (cell  &&  cell->type == METADATA_BLOB)
        {
            len  = cell->value.v_blob.len;
            data = (guint8*)(cell->value.v_blob.data);
            thumbnail = g_byte_array_sized_new(len);
            if (thumbnail)
            {
                thumbnail = g_byte_array_append(thumbnail, data, len);
            }
        }
    }

    // clean up
    metadata_table_free(names);
    metadata_table_free(values);

    return thumbnail;
}
