/*
 * File Name: actions.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <stdlib.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "ctb_log.h"
#include "ctb_actions.h"
#include "filetypes.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "shortcut.h"
#include "storage.h"
#include "recent.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const gchar      *CTB_THUMBNAIL_PATH = "/tmp/ctb_thumbnail.png";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GString          *g_cmdline = NULL;      // command-line to be executed

static gboolean         g_is_folder_shortcut_deleted = FALSE;
                                                // one or more shortcuts to a folder have been deleted


//============================================================================
// Local Function Definitions
//============================================================================

static int       copy_from_template ( const GString           *directory,
                                      const filelist_entry_t  *fileinfo,
                                            filelist_entry_t  **p_fileinfo_copy );


//============================================================================
// Functions Implementation
//============================================================================

// activate: launch viewer, goto directory, ...
int activate_item ( const GString           *directory,
                    const filelist_entry_t  *fileinfo  )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename         && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display && fileinfo->filename_display->str);

    int              ret = ER_OK;    // return code
    gboolean         ok;
    const gchar      *viewer_cmd     = NULL;
    gchar            *filepath       = NULL;
    gchar            *cmd_line       = NULL;
    GByteArray       *thumbnail_blob = NULL;
    gchar            *thumbnail_path = NULL;
    gchar            *error_msg      = NULL;
    GError           *err            = NULL;
    filelist_entry_t *fileinfo_copy  = NULL;

    // When activating a (nested) shortcut, the 'Tasks' menu shows
    //   (1) the thumbnail read from database for the top-level shortcut, or
    //   (2) the default thumbnail from the target filename
    // In such situation, the current function will be called recursively so we avoid global data.
    //
    // Default situation when not handling a shortcut:
    //   is_toplevel_thumbnail = FALSE, is_thumbnail = FALSE, thumbnail = NULL
    // When handling top-level shortcut:
    //   is_toplevel_thumbnail = TRUE,  is_thumbnail = TRUE,  thumbnail = <ptr>
    // When handling the target file or folder:
    //   is_toplevel_thumbnail = FALSE, is_thumbnail = TRUE,  thumbnail = <ptr>
    //
    // Note that static variables have only a single instance on recursive calls,
    // where non-static variables have a separate instance for each recursive call level.
    //
           gboolean   is_toplevel_shortcut = FALSE;
    static gboolean   is_shortcut          = FALSE;

    // shorthand access to fileinfo details
    const gchar      *filename         = fileinfo->filename->str;
    const gchar      *filename_display = fileinfo->filename_display->str;
    const gchar      *filetype         = fileinfo->filetype->str;
    const gboolean   is_directory      = fileinfo->is_directory;
    const gboolean   is_template       = fileinfo->is_template;

    LOGPRINTF( "entry: dirpath [%s] filename [%s] is_dir [%d] is_templ [%d]",
               directory->str,
               filename,
               is_directory,
               is_template  );

    if (   is_shortcut_file_extension(filetype) 
        && !is_shortcut                         )
    {
        // top-level shortcut: activate shortcut
        is_shortcut          = TRUE;
        is_toplevel_shortcut = TRUE;

        ret = activate_shortcut(directory, fileinfo);
    }
    else if (is_template)
    {
        // template: make a fresh copy, then activate (open) it
        ret = copy_from_template(directory, fileinfo, &fileinfo_copy);
        if (ret == ER_OK)
        {
            ret = activate_item(directory, fileinfo_copy);
            if ( !is_directory )
            {
                // give viewer some time to start, then reposition cursor
                // note: wait implemented with sleep() not g_timeout_add()
                //       to deliberately block the content browser's GUI during the wait
                sleep(3);
                fileview_set_cursor_at_filename( fileinfo_copy->filename->str );
            }
        }
    }
    else if (is_directory)
    {
        // directory: change directory
        if ( strcmp(filename, "..") == 0 )
        {
            // get name of current directory, to position the cursor on it
            filepath = g_path_get_basename(directory->str);
        }
        fileview_show_dir(filename, filepath);
    }
    else
    {
        if ( is_shortcut && g_cmdline )
        {
            // shortcut to application
            viewer_cmd = g_cmdline->str;
        }
        else
        {
            // file: launch viewer for this file
            //   get viewer command
            viewer_cmd = get_viewer_from_file_extension(filetype);
            filepath   = g_strdup_printf("%s/%s", directory->str, filename);
            if (viewer_cmd == NULL)
            {
                error_msg = g_strdup_printf( _("'%s' cannot be opened; the format is unknown."),
                                             fileinfo->filename->str                            );
                run_error_dialog(error_msg);
                ret = ER_FAIL;
            }
        }
        //   get thumbnail from database
        if (ret == ER_OK)
        {
            thumbnail_blob = filemodel_get_thumbnail(fileinfo, MODTHUMB_SMALL);
            if (thumbnail_blob)
            {
                // thumbnail found: save to file
                ok = g_file_set_contents( CTB_THUMBNAIL_PATH,
                                          (gchar*)(thumbnail_blob->data),
                                          thumbnail_blob->len,
                                          &err );
                if (ok)
                {
                    thumbnail_path = g_strdup(CTB_THUMBNAIL_PATH);
                }
                else
                {
                    // cannot save thumbnail: continue without, no error reported
                    ERRORPRINTF("cannot save thumbnail file - [%s]", err->message);
                    g_clear_error(&err);
                }
            }
            else
            {
                WARNPRINTF("no thumbnail in database for file [%s] dir [%s]", filename, directory->str);
            }
        }
        //   get default thumbnail for filetype, if needed
        if ( ret == ER_OK  &&  thumbnail_path == NULL )
        {
            if ( is_shortcut && g_cmdline )
            {
                thumbnail_path = get_icon_file_application( MODTHUMB_SMALL );
            }
            else
            {
                thumbnail_path = get_icon_file_from_file_extension( filetype, is_directory, MODTHUMB_SMALL );
            }
            if (thumbnail_path == NULL)
            {
                // no default thumbnail: continue without, no error reported
                ERRORPRINTF("no icon file for filename [%s] is_dir [%d]", filename, is_directory);
                thumbnail_path = g_strdup("");
            }
        }
        //   launch viewer (or application)
        if (ret == ER_OK)
        {
            if (filepath == NULL)
            {
                cmd_line = g_strdup(viewer_cmd);
            }
            else
            {
               cmd_line = g_strdup_printf("%s \"%s\"", viewer_cmd, filepath);
            }

            gchar *ret_message = NULL;
            gint errcode = ipc_sys_start_task(cmd_line, directory->str, filename_display, thumbnail_path, &ret_message);
            
			LOGPRINTF("ipc_sys_start_task=[%d]", errcode);
            if ( errcode > 0 )
            {
                ERRORPRINTF("cannot launch viewer, cmd_line [%s], errcode %d [%s]", cmd_line, errcode, ret_message);
                if (ret_message && (ret_message[0] != '\0'))
                {
                    // show custom error message
                    run_error_dialog(ret_message);
                }
                else if ( is_shortcut && g_cmdline )
                {
                    switch (errcode)
                    {
                    case 1: // application cannot be started
                    default:
                        error_msg = g_strdup_printf(
                            _("The '%s' application cannot be opened.\n"
                              "Try closing some documents or reinstalling the software and try again."),
                            filename_display );
                        run_error_dialog(error_msg);
                        break;
                    case 4: // application failed to open file (may be handled above with custom message)
                        error_msg = g_strdup_printf(
                            _("'%s' cannot be opened."), 
                            filename_display );
                        run_error_dialog(error_msg);
                        break;
                    case 2: // timeout waiting for application window
                    case 3: // application has exited before creating a window
                        break;
                    }
                }
            }
			else
			{
				LOGPRINTF("filepath= %x", (int)filepath);
				// successfull add to recent opened file list
				if (filepath != NULL)
				{
					// only add if it is not an application
					LOGPRINTF("calling recent_item [%s] [%s]", directory->str, fileinfo->filename->str);
					recent_item (directory, fileinfo);
				}
			}
            g_free(ret_message);
        }
    }

    // clean up shortcut thumbnail variables
    if (is_toplevel_shortcut)
    {
        if (g_cmdline)
        {
            g_string_free(g_cmdline, TRUE);
            g_cmdline = NULL;
        }
        is_shortcut = FALSE;
    }

    // clean up
    if (fileinfo_copy ) { filelist_entry_free(fileinfo_copy);      }
    if (thumbnail_blob) { g_byte_array_free(thumbnail_blob, TRUE); }
    g_free(thumbnail_path);
    g_free(error_msg);
    g_free(cmd_line);
    g_free(filepath);

    LOGPRINTF("leave");
    return ret;
}


// activate: launch command line as specified in shortcut to application
int activate_shortcut_to_application ( const GString           *directory,
                                       const filelist_entry_t  *fileinfo,
                                       const GString           *command_line )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename         && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display && fileinfo->filename_display->str);
    g_assert(command_line               && command_line->str              );

    int              ret = ER_OK;    // return code

    LOGPRINTF( "entry: dirpath [%s] filename [%s] command_line [%s]",
               directory->str,
               fileinfo->filename->str,
               command_line->str        );

    g_assert(g_cmdline == NULL);
    g_cmdline = g_string_new(command_line->str);

    ret = activate_item(directory, fileinfo);

    return ret;
}


// copy a template file or directory to a new instance
static int copy_from_template ( const GString           *directory,
                                const filelist_entry_t  *fileinfo,
                                      filelist_entry_t  **p_fileinfo_copy )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename         && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display && fileinfo->filename_display->str);
    g_assert(fileinfo->is_template);
    g_assert(p_fileinfo_copy && *p_fileinfo_copy == NULL);

    int              ret = ER_OK;    // return code
    int              i;
    int              rc;
    int              n_rows;
    int              col;
    int              sequence = 1;   // sequence number for copied file-/foldername
    gboolean         done;
    gboolean         exists;
    gboolean         is_database_transaction_started = FALSE;
    const gchar      *viewer_cmd     = NULL;
    gchar            *file_prefix    = NULL;
    gchar            *file_suffix    = NULL;
    gchar            *error_msg      = NULL;
    gchar            *cp;
    GString          *file_glob      = g_string_new("");
    GString          *filename_copy  = g_string_new("");
    GString          *str            = NULL;
    filelist_entry_t *fileinfo_copy  = NULL;
    metadata_table   *values         = NULL;
    erMetadb         *db             = get_database();
    GPtrArray        *filenames_src  = NULL;
    GPtrArray        *filenames_dest = NULL;

    const gchar      *filename         = fileinfo->filename->str;
    const gchar      *filename_display = fileinfo->filename_display->str;
    const gchar      *filetype         = fileinfo->filetype->str;
    const gboolean   is_directory      = fileinfo->is_directory;

    LOGPRINTF( "entry: dirpath [%s] filename [%s] is_dir [%d]",
               directory->str,
               filename,
               is_directory );

    // abort if unknown filetype
    if ( !is_directory )
    {
        viewer_cmd = get_viewer_from_file_extension(filetype);
        if (viewer_cmd == NULL)
        {
            error_msg = g_strdup_printf( _("'%s' cannot be opened; the format is unknown."),
                                         fileinfo->filename->str                            );
            run_error_dialog(error_msg);
            ret = ER_FAIL;
        }
    }

    // determine name for copy
    if (ret == ER_OK)
    {
        // split prefix and suffix in separate strings
        file_prefix = g_strdup(filename);
        cp = (gchar*) g_extension_pointer(file_prefix);
        if (*cp)
        {
            // include the '.'
            cp--;
        }
        file_suffix = g_strdup(cp);
        *cp = '\0';

        //   get highest sequence number + 1
        g_string_printf( file_glob, "%s [0-9][0-9][0-9]%s", file_prefix, file_suffix );
        rc = ermetadb_get_filenames_with_glob( db, file_glob, &values );
        if (rc == ER_OK  &&  values)
        {
            n_rows = metadata_table_n_rows(values);
            col    = metadata_table_find_column(values, "filename");
            if (n_rows > 0  &&  col >= 0)
            {
                i = metadata_table_cell_index( values, n_rows - 1, col );
                str = metadata_table_get_string( values, i );
                if (str->len > 0)
                {
                    cp  = str->str + str->len - strlen(file_suffix);
                    *cp = '\0';
                    cp  = cp - 3;
                    sequence = atoi(cp) + 1;
                }
                g_string_free(str, TRUE);
                str = NULL;
            }
        }
        if (values)
        {
            metadata_table_free(values);
            values = NULL;
        }
        //   find first free sequence number
        done = FALSE;
        while ( !done )
        {
            g_string_printf( filename_copy, "%s %03d%s", file_prefix, sequence, file_suffix );
            exists = g_file_test(filename_copy->str, G_FILE_TEST_EXISTS);
            if (exists)
            {
                WARNPRINTF("filename [%s] already present", filename_copy->str);
                sequence++;
            }
            else
            {
                LOGPRINTF("filename_copy [%s]", filename_copy->str);
                done = TRUE;
            }
        }
    }

    // create the new file_info structure
    fileinfo_copy = filelist_entry_copy(fileinfo);
    fileinfo_copy->is_template = FALSE;
    g_string_assign(fileinfo_copy->filename,         filename_copy->str);
    g_string_assign(fileinfo_copy->filename_display, filename_copy->str);

    // start database transaction
    if (ret == ER_OK)
    {
        ret = ermetadb_begin_transaction( db );
        if (ret == ER_OK)
        {
            is_database_transaction_started = TRUE;
        }
    }

    // copy the file or folder
    if (ret == ER_OK)
    {
        filenames_src  = g_ptr_array_new();
        filenames_dest = g_ptr_array_new();
        g_assert(filenames_src);
        g_assert(filenames_dest);
        g_ptr_array_add( filenames_src,  (gchar*)(filename) );
        g_ptr_array_add( filenames_dest, (gchar*)(filename_copy->str) );

        ret = copy_file_or_folder_list( filenames_src,  directory->str,        // source
                                        filenames_dest, directory->str, db );  // target
        if (ret != ER_OK)
        {
            // show error message, wait for user input
            error_msg = g_strdup_printf( _("'%s' cannot be opened and saved;"
                                           " the memory card may be write-protected or damaged."
                                           " Try unlocking the card and try again."),
                                         filename_display );
            run_error_dialog(error_msg);
        }
    }

    // copy database info
    if (ret == ER_OK)
    {
        // adjust the is_template flag and sort_priority
        col = 0;
        values = metadata_table_new();
        if (values == NULL)
        {
            ret = ER_FAIL;
        }
        if (ret == ER_OK)
        {
            ret = metadata_table_add_column(values, "is_template");
            if (ret == ER_OK)
            {
                ret = metadata_table_set_int64( values, col++, 0 );
            }
        }
        if (ret == ER_OK)
        {
            ret = metadata_table_add_column(values, "sort_priority");
            if (ret == ER_OK)
            {
                ret = metadata_table_set_int64( values,
                                                col++,
                                                is_directory ? MDB_SORT_PRIORITY_FOLDER : MDB_SORT_PRIORITY_FILE );
            }
        }
        if (ret == ER_OK)
        {
            ret = ermetadb_set_file_metadata( db, filename_copy, values );
        }
        if (ret != ER_OK)
        {
            ERRORPRINTF("cannot adjust file_metadata for [%s]", filename_copy->str);
        }
        if (values)
        {
            metadata_table_free(values);
            values = NULL;
        }
    }

    // write database changes now
    if (is_database_transaction_started)
    {
        rc = ermetadb_end_transaction( db );
        if (ret == ER_OK)
        {
            ret = rc;
        }
    }

    // report our findings
    if (ret == ER_OK)
    {
        *p_fileinfo_copy = fileinfo_copy;
        fileinfo_copy = NULL;
    }

    // clean up
    if (filenames_src ) { g_ptr_array_free(filenames_src,  TRUE); }
    if (filenames_dest) { g_ptr_array_free(filenames_dest, TRUE); }
    if (fileinfo_copy ) { filelist_entry_free(fileinfo_copy); }
    if (filename_copy ) { g_string_free(filename_copy, TRUE); }
    if (file_glob     ) { g_string_free(file_glob,     TRUE); }
    g_free(error_msg    );
    g_free(file_suffix  );
    g_free(file_prefix  );

    return ret;
}


int copy_file_or_folder_list( const GPtrArray  *filenames_src,
                              const gchar      *dir_src,
                              const GPtrArray  *filenames_dest,
                              const gchar      *dir_dest,
                                    erMetadb   *db_dest        )
{
    int         ret = ER_OK;    // return value
    int         idx = 0;
    gboolean    ok;
    const gchar *file_src;
    const gchar *file_dest;
    GString     *s_file_src  = g_string_new("");
    GString     *s_file_dest = g_string_new("");
    GString     *s_dir_src   = g_string_new("");
    GString     *s_dir_dest  = g_string_new("");
    GString     *s_path_src  = g_string_new("");
    GString     *s_path_dest = g_string_new("");
    int         argc = 0;
    const char  *argv[10];
    gint        stat;
    GError      *err = NULL;

    LOGPRINTF("entry: dir_src [%s] dir_dest [%s]", dir_src, dir_dest);
    g_assert(filenames_src);
    g_assert(dir_src  && *dir_src );
    g_assert(dir_dest && *dir_dest);
    for ( idx = 0 ; idx < filenames_src->len ; idx++ )
    {
        LOGPRINTF("filenames_src[%d] [%s]", idx, (char*)g_ptr_array_index(filenames_src, idx) );
    }
    for ( idx = 0 ; filenames_dest && idx < filenames_dest->len ; idx++ )
    {
        LOGPRINTF("filenames_dest[%d] [%s]", idx, (char*)g_ptr_array_index(filenames_dest, idx) );
    }

    g_string_assign( s_dir_src,  dir_src  );
    g_string_assign( s_dir_dest, dir_dest );

    // copy files/folders as specified
    for ( idx = 0 ;
          idx < filenames_src->len  &&  ret == ER_OK ;
          idx++ )
    {
        // get source file/folder name
        file_src = g_ptr_array_index( filenames_src, idx );
        if (file_src == NULL)
        {
            // empty entry in list: ignore
            continue;
        }
        g_string_assign( s_file_src, file_src  );
        g_string_printf( s_path_src, "%s/%s", dir_src, file_src );

        // get target file/folder name
        if (   filenames_dest
            && idx < filenames_dest->len )
        {
            file_dest = g_ptr_array_index( filenames_dest, idx );
        }
        else
        {
            file_dest = file_src;
        }
        g_string_assign( s_file_dest, file_dest );
        g_string_printf( s_path_dest, "%s/%s", dir_dest, file_dest);
        LOGPRINTF("file_src [%s] file_dest [%s]", file_src, file_dest);

        // remove old file or folder, if present
        argc = 0;
        argv[argc++] = "rm";
        argv[argc++] = "-rf";
        argv[argc++] = s_path_dest->str;
        argv[argc++] = NULL;
        g_assert(argc < sizeof(argv)/sizeof(argv[0]));
        (void) g_spawn_sync( NULL,                   // working directory: inherit
                             (char**)argv,           // child's argument vector
                             NULL,                   // environment: inherit
                             G_SPAWN_SEARCH_PATH,    // flags
                             NULL,                   // child_setup: none
                             NULL,                   // child setup data: none
                             NULL,                   // stdout: not interested
                             NULL,                   // stderr: not interested
                             &stat,                  // exit status
                             NULL                );  // error return

        // copy file or folder
        argc = 0;
        argv[argc++] = "cp";
        argv[argc++] = "-r";
        argv[argc++] = s_path_src->str;
        argv[argc++] = s_path_dest->str;
        argv[argc++] = NULL;
        g_assert(argc < sizeof(argv)/sizeof(argv[0]));
        ok = g_spawn_sync( NULL,                   // working directory: inherit
                           (char**)argv,           // child's argument vector
                           NULL,                   // environment: inherit
                           G_SPAWN_SEARCH_PATH,    // flags
                           NULL,                   // child_setup: none
                           NULL,                   // child setup data: none
                           NULL,                   // stdout: not interested
                           NULL,                   // stderr: not interested
                           &stat,                  // exit status
                           &err                );  // error return
        if ( !ok )
        {
            ret = ER_FAIL;
            ERRORPRINTF( "g_spawn_sync error [%s] on copy [%s] to [%s]",
                                              err->message,
                                                           s_path_src->str,
                                                                   s_path_dest->str );
            g_clear_error(&err);

            // remove the new copy, in case the copy partially succeeded
            argc = 0;
            argv[argc++] = "rm";
            argv[argc++] = "-rf";
            argv[argc++] = s_path_dest->str;
            argv[argc++] = NULL;
            g_assert(argc < sizeof(argv)/sizeof(argv[0]));
            (void) g_spawn_sync( NULL,                   // working directory: inherit
                                 (char**)argv,           // child's argument vector
                                 NULL,                   // environment: inherit
                                 G_SPAWN_SEARCH_PATH,    // flags
                                 NULL,                   // child_setup: none
                                 NULL,                   // child setup data: none
                                 NULL,                   // stdout: not interested
                                 NULL,                   // stderr: not interested
                                 &stat,                  // exit status
                                 NULL                );  // error return
        }
        sync();

        // copy metadata
        if (ret == ER_OK)
        {
            (void) ermetadb_delete_document_or_folder( db_dest, s_file_dest );
            ret = ermetadb_copy_document_or_folder( db_dest, s_file_dest, s_file_src, s_dir_src );
        }
    }

    // clean up
    if (s_path_dest) { g_string_free(s_path_dest, TRUE); }
    if (s_path_src ) { g_string_free(s_path_src,  TRUE); }
    if (s_dir_dest ) { g_string_free(s_dir_dest,  TRUE); }
    if (s_file_dest) { g_string_free(s_file_dest, TRUE); }
    if (s_file_src ) { g_string_free(s_file_src,  TRUE); }

    return ret;
}


// pre-delete
int pre_delete_item ( const GString  *directory,
                      const GSList   *filelist  )
{
    g_assert(directory  &&  directory->str);
    g_assert(filelist);

    int                     ret = ER_FAIL;           // return code
    gint                    rc;
    const guint             len = g_slist_length( (GSList*) filelist );
    const filelist_entry_t  *filelist_entry = NULL;  // entry in filelist
    const GSList            *entry = NULL;
    GString                 *msg = g_string_new("");
    GtkWidget               *dialog = NULL;

    LOGPRINTF( "entry: dir [%s] filelist len [%d]", directory->str, len);

    // no shortcuts to folders deleted (yet)
    g_is_folder_shortcut_deleted = FALSE;

    // check whether deletion is permitted
    if ( strcmp(directory->str, DIR_DESKTOP_INTERNAL) == 0 )
    {
        for ( entry = filelist ; entry && msg->len == 0 ; entry = g_slist_next(entry) )
        {
            filelist_entry = entry->data;
            if (   strcmp(filelist_entry->directory_path->str, ".")            == 0
                || strcmp(filelist_entry->directory_path->str, directory->str) == 0 )
            {
                g_string_printf(msg, _("'%s' cannot be deleted."),
                                     filelist_entry->filename_display->str );
            }
        }
    }
    if ( msg->len > 0 )
    {
        // deletion not allowed
        run_error_dialog(msg->str);
    }
    else
    {
        // deletion allowed

        // build message text
        if (len == 1)
        {
            filelist_entry = filelist->data;
            if (filelist_entry->is_directory)
            {
                g_string_printf(msg, _("Deleting the folder '%s' will permanently remove the folder"
                                       " and its contents from the card.\n"
                                       "\n"
                                       "Do you wish to delete this folder?"),
                                     filelist_entry->filename_display->str );
            }
            else if ( is_shortcut_file_extension(filelist_entry->filetype->str) )
            {
                g_string_printf(msg, _("Deleting the shortcut '%s' will permanently remove it from the card.\n"
                                       "\n"
                                       "Do you wish to delete this shortcut?"),
                                     filelist_entry->filename_display->str );
            }
            else
            {
                g_string_printf(msg, _("Deleting '%s' will permanently remove it from the card.\n"
                                       "\n"
                                       "Do you wish to delete this file?"),
                                     filelist_entry->filename_display->str );
            }
        }
        else
        {
            g_string_printf( msg, _("Deleting these %d items will permanently remove them from the card.\n"
                                    "\n"
                                    "Do you wish to delete these items?"),
                                  len );
        }

        // show message, wait for user input
        dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_MESSAGE_WARNING,
                                         GTK_BUTTONS_YES_NO,
                                         msg->str );
        gtk_window_set_deletable( GTK_WINDOW(dialog), FALSE );
        rc = gtk_dialog_run( GTK_DIALOG(dialog) );
        LOGPRINTF("dialog returns [%d]", rc);
        if (   rc == GTK_RESPONSE_OK
            || rc == GTK_RESPONSE_YES )
        {
            ret = ER_OK;    // agree to delete items
        }
    }

    // clean up
    gtk_widget_destroy( dialog );
    g_string_free(msg, TRUE);

    return ret;
}


// delete
int delete_item ( const GString           *directory,
                  const filelist_entry_t  *fileinfo  )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename          && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display  && fileinfo->filename_display->str);
    g_assert(fileinfo->filetype          && fileinfo->filetype->str        );
    g_assert(fileinfo->directory_path    && fileinfo->directory_path->str  );

    int             ret = ER_OK;    // return code
    gboolean        ok  = TRUE;
    int             pass = 0;
    int             argc = 0;
    const char      *argv[10];
    gint            stat;
    char            *cp;
    const gchar     *ext;
    char            real_dir[PATH_MAX];
    gchar           *filepath   = NULL;
    const gchar     *viewer_cmd = NULL;
    gchar           *cmd_line   = NULL;
    gchar           *error_msg  = NULL;
    GError          *err        = NULL;
    erMetadb        *db         = NULL;

    const gchar     *filename         = fileinfo->filename->str;
    const gchar     *filename_display = fileinfo->filename_display->str;
    const gchar     *filetype         = fileinfo->filetype->str;
    const gchar     *directory_path   = fileinfo->directory_path->str;
    const gboolean  is_directory      = fileinfo->is_directory;

    LOGPRINTF( "entry: dir [%s] filename [%s] dirpath [%s]", directory->str, filename, directory_path );

    // not allowed to delete items from internal desktop
    cp = realpath(directory->str, real_dir);
    if (   cp == real_dir
        && strcmp(real_dir, DIR_DESKTOP_INTERNAL) == 0 )
    {
        ret = ER_FORBIDDEN;
    }

    // tell sysd to close document, then wait for a reply from sysd
    if (ret == ER_OK)
    {
        viewer_cmd = get_viewer_from_file_extension(filetype);
        filepath   = g_strdup_printf("%s/%s", directory_path, filename);
        if (viewer_cmd)
        {
            cmd_line = g_strdup_printf("\"%s\" \"%s\"", viewer_cmd, filepath);
            ipc_sys_stop_task(cmd_line);
        }
    }

    // remove file or directory
    if (ret == ER_OK)
    {
        argc = 0;
        argv[argc++] = "rm";
        argv[argc++] = "-fr";
        argv[argc++] = filepath;
        argv[argc++] = NULL;
        g_assert(argc < sizeof(argv)/sizeof(argv[0]));
        ok = g_spawn_sync( NULL,                   // working directory: inherit
                           (char**)argv,           // child's argument vector
                           NULL,                   // environment: inherit
                           G_SPAWN_SEARCH_PATH,    // flags
                           NULL,                   // child_setup: none
                           NULL,                   // child setup data: none
                           NULL,                   // stdout: not interested
                           NULL,                   // stderr: not interested
                           &stat,                  // exit status
                           &err                );  // error return
        if ( !ok )
        {
            ret = ER_FAIL;
            ERRORPRINTF("g_spawn_sync error [%s] on delete [%s]", err->message, filepath);
            g_clear_error(&err);
        }
    }

    // remove filename from database
    if (ret == ER_OK)
    {
        for (pass = 1 ; pass <= 2 ; pass++)
        {
            // select database
            switch (pass)
            {
                case 1:         // remove from local database
                    db = get_database();
                    break;
                case 2:         // remove from original database, if any
                    if ( strcmp(directory_path, ".") != 0 )
                    {
                        db = db_open( fileinfo->directory_path, FALSE );
                    }
                    else
                    {
                        db = NULL;
                    }
                    break;
                default:
                    g_assert_not_reached();
            }

            // remove filename from database
            LOGPRINTF("pass [%d] db [%p] db->directory [%s]", pass, db, db ? db->directory->str : NULL );
            if (db)
            {
                g_assert(db->directory && db->directory->str);
                if (is_directory)
                {
                    ermetadb_delete_folder( db, fileinfo->filename );
                }
                else if (   strcmp(directory_path, ".") == 0
                         || strcmp(directory_path, db->directory->str) == 0 )
                {
                    ermetadb_delete_document( db, fileinfo->filename );
                }
                else
                {
                    ermetadb_delete_document_with_dirpath( db,
                                                           fileinfo->filename,
                                                           fileinfo->directory_path );
                }
            }

            // clean up
            if (pass == 2)
            {
                ermetadb_free(db);
            }
            db = NULL;
        }
    }

    // wrap up
    if (ret == ER_OK)
    {
        // report to filemodel
        filemodel_item_deleted();

        // remember when a shortcut to folder has been deleted
        ext = g_extension_pointer(filename);
        if ( strcmp(ext, FILE_EXT_SHORTCUT_TO_DIR) == 0 )
        {
            g_is_folder_shortcut_deleted = TRUE;
        }
    }
    else
    {
        // show error message
        error_msg = g_strdup_printf( _("'%s' cannot be deleted;"
                                       " the memory card may be write-protected or damaged."
                                       " Try unlocking the card and try again."             ),
                                     filename_display                                          );
        run_error_dialog(error_msg);
    }

    // clean up
    g_free(error_msg);
    g_free(cmd_line );
    g_free(filepath );

    return ret;
}


// post-delete
int post_delete_item ( const GString  *directory,
                       const GSList   *filelist  )
{
    int     ret = ER_OK;    // return code

    LOGPRINTF("entry");

    // update symlinks to target directory of desktop shortcuts
    if (g_is_folder_shortcut_deleted)
    {
        storage_refresh_symlinks();
    }

    return ret;
}


// open database, if present
erMetadb* db_open ( const GString *dir, const gboolean do_create_database )
{
    erMetadb        *db = NULL;     // return value
    int             rc;

    g_assert(dir  &&  dir->len > 0);
    LOGPRINTF("entry: directory [%s]", dir->str);

    // open database
    db = ermetadb_new();
    if (db)
    {
        rc = ermetadb_open_database( db, dir );
        if (   rc == ER_NOT_FOUND
            && do_create_database )
        {
            rc = ermetadb_create_database( dir );
            if (rc == ER_OK)
            {
                rc = ermetadb_open_database( db, dir );
            }
        }
        if (rc != ER_OK)
        {
            ERRORPRINTF("cannot open database in [%s]", dir->str);
            ermetadb_free(db);
            db = NULL;
        }
    }

    LOGPRINTF("leave: db [%p]", db);
    return db;
}


// create shortcut
int shortcut_item ( const GString           *directory,
                    const filelist_entry_t  *fileinfo  )
{
    g_assert(directory  &&  directory->str);
    g_assert(fileinfo);
    g_assert(fileinfo->filename          && fileinfo->filename->str        );
    g_assert(fileinfo->filename_display  && fileinfo->filename_display->str);
    g_assert(fileinfo->filetype          && fileinfo->filetype->str        );
    g_assert(fileinfo->directory_path    && fileinfo->directory_path->str  );

    int             ret = ER_OK;    // return code
    int             rc;
    const gchar     *target_dir;
    gchar           *error_msg       = NULL;
    GString         *shortcut_dir    = NULL;
    GString         *shortcut_file   = g_string_new("");
    GString         *shortcut_path   = g_string_new("");
    GString         *shortcut_target = g_string_new("");
    erMetadb        *db              = NULL;
    metadata_table  *values          = NULL;

    const gchar     *filename         = fileinfo->filename->str;
    const gchar     *filename_display = fileinfo->filename_display->str;
    const gchar     *directory_path   = fileinfo->directory_path->str;

    LOGPRINTF( "entry: dir [%s] filename [%s] dirpath [%s]", directory->str, filename, directory_path );

    // no shortcut to hidden file or folder
    if ( is_hidden_filename(filename) )
    {
        ret = ER_FORBIDDEN;
    }

    // get directory to create shortcut in,
    // create this directory if not yet present
    if (ret == ER_OK)
    {
        if ( strcmp(directory_path, ".") == 0 )
        {
            target_dir = directory->str;
        }
        else
        {
            target_dir = directory_path;
        }
        shortcut_dir = storage_get_desktop_dir(target_dir);
        if (shortcut_dir == NULL)
        {
            ret = ER_INVALID_DATA;
        }
        else
        {
            // create shortcut directory
            if ( !g_file_test(shortcut_dir->str, G_FILE_TEST_IS_DIR) )
            {
                rc = g_mkdir_with_parents( shortcut_dir->str, 0755 );
                if (rc != 0)
                {
                    ERRNOPRINTF("cannot create directory [%s]", shortcut_dir->str);
                    ret = ER_FAIL;
                }
            }
        }
    }

    // create shortcut file
    if (ret == ER_OK)
    {
        ret = create_shortcut_file( target_dir,
                                    filename,
                                    filename_display,
                                    shortcut_dir->str,
                                    shortcut_file );
    }

    // copy metadata to desktop database
    if (ret == ER_OK)
    {
        db = get_database();
        if (db)
        {
            // get metadata from original database
            rc = ermetadb_get_file_metadata( db, fileinfo->filename, NULL, &values );
            if (rc == ER_OK  &&  values)
            {
                // store metadata in desktop database on this storage device
                db = db_open( shortcut_dir, TRUE );
                if (db == NULL)
                {
                    WARNPRINTF("cannot open database [%s]", shortcut_dir->str);
                    // TODO: create database
                    // ret = ER_FAIL;
                }
                else
                {
                    // insert folder or document
                    rc = ermetadb_add_document(db, shortcut_file, 0, 0);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot add folder/document, error [%d]", rc);
                        ret = rc;
                    }

                    // set file metadata
                    if (ret == ER_OK)
                    {
                        rc = ermetadb_set_file_metadata( db, shortcut_file, values );
                        if (rc != ER_OK)
                        {
                            ERRORPRINTF("cannot set file_metadata, error [%d]", rc);
                            ret = rc;
                        }
                    }

                    // close desktop database
                    rc = ermetadb_close_database(db);
                    if (rc != ER_OK)
                    {
                        ERRORPRINTF("cannot close database [%s], error [%d]", shortcut_dir->str, rc);
                        if (ret == ER_OK)
                        {
                            ret = rc;
                        }
                    }

                    // clean up
                    ermetadb_free(db);
                    db = NULL;
                }
            }
            db = NULL;
        }
    }

    // update symlinks to target directory of desktop shortcuts
    storage_refresh_symlinks();

    // show error message, if needed
    if (ret != ER_OK)
    {
        if (ret == ER_FORBIDDEN)
        {
            ERRORPRINTF("not allowed to create a shortcut to [%s]", filename);
        }
        else
        {
            error_msg = g_strdup_printf( _("'%s' cannot be opened and saved;"
                                           " the memory card may be write-protected or damaged."
                                           " Try unlocking the card and try again."),
                                         shortcut_file->str );
            run_error_dialog(error_msg);
        }
    }

    // clean up
    metadata_table_free(values);
    if (shortcut_target) { g_string_free(shortcut_target, TRUE); }
    if (shortcut_path)   { g_string_free(shortcut_path,   TRUE); }
    if (shortcut_file)   { g_string_free(shortcut_file,   TRUE); }
    if (shortcut_dir)    { g_string_free(shortcut_dir,    TRUE); }
    g_free(error_msg);

    return ret;
}
