----------------------------------------------------------------------
-- filebrowser: file explorer
--
-- Copyright (C) 2009  Iñigo Serna <inigoserna@gmail.com>
-- Time-stamp: <2009-11-22 22:16:36 inigo>


require("os")

require("lfs")

require("lgob.gtk")
require("lgob.gdk")
require("lgob.pango")


---------- Vars -------------------------------------------------------

-- constants
local SORTBY_NONE, SORTBY_NAME, SORTBY_NAME_REV, SORTBY_DATE, SORTBY_DATE_REV,
      SORTBY_SIZE, SORTBY_SIZE_REV = 0, 1, 2, 3, 4, 5, 6
local TV_COL_SELECTED, TV_COL_NAME, TV_COL_TYPE, TV_COL_SIZE, TV_COL_MTIME,
      TV_COL_PATH, TV_COL_SIZE_NUM = 0, 1, 2, 3, 4, 5, 6
local TV_COL_W = { SELECTED=25, NAME=365, TYPE=60, SIZE=100, MTIME=155 }
local CONFIG_FILE = "_files/filebrowser.ini"
local CONFIG_FILE_HEADER = "-- Iliad Toolbox: File Explorer configuration file\n" ..
   "-- this file is automatically generated, please don't edit by hand"
local DEFAULT_STYLE = [[
      style "smallfont-style" { font_name = "sans 8" }
      widget_class "*.GtkComboBox.*" style "smallfont-style"
      widget_class "*.GtkMenu.*" style "smallfont-style"
      widget_class "*.GtkButton.*" style "smallfont-style"
      widget_class "*.GtkRadioButton.*" style "smallfont-style"
      widget_class "*.GtkTreeView.*" style "smallfont-style"
]]

-- variables
local opts = { SORT = SORTBY_NAME,
               SORT_MIX_CASES = true,
               SORT_MIX_DIRS_AND_FILES = false,
               INITIAL_PATH = "/mnt/free/",
               BOOKMARKS = {"/", "/", "/mnt/free/", "/mnt/free/books/",
                            "/mnt/free/documents/", "/mnt/free/newspapers/",
                            "/mnt/free/notes/", "/media/card/",
                            "/media/cf/", "/mnt/usb/"},
               SELECT_VALID_PATHS = {"/mnt/free/", "/media/card/", "/media/cf/", "/mnt/usb/"},
               TREEVIEW = true,
               TREE_DEPTH = 3 }
local path = opts.INITIAL_PATH
local selected_files = {} -- {filename={size=s, mode=m}}: m={0(just added), 1(copy), 2(cut)}

-- ui
local pix_selected = gdk.Pixbuf.new_from_file("_files/data/selected.png")
local pix_noselected = gdk.Pixbuf.new_from_file("_files/data/noselected.png")
local pix_cut = gdk.Pixbuf.new_from_file("_files/data/cut.png")
local pix_copied = gdk.Pixbuf.new_from_file("_files/data/copied.png")
local img_open = gtk.Image.new_from_file("_files/data/open.png")
local win, lbl_path, lbl_free, view, model
local st_selected, cmb_actions, cmb_others, cmb_bookmarks

-- common used functions
local fmt = string.format


---------- Utils ------------------------------------------------------

-- check_valid_dir: check for valid directory
local function check_valid_dir(fname)
   local attr = lfs.symlinkattributes(fname)
   if attr and attr.mode == "directory" then
      return true
   else
      return false
   end
end


-- get_abspath: return absolute path
local function get_abspath(fname)
   return fname:sub(1, 1)=='/' and fname or lfs.currentdir() .. '/' .. fname
end


-- get_parentdir:
local function get_parentdir(fname)
   local dir = '/'
   fname:gsub('([^/]+)/', function(p) dir = dir .. p .. '/' end)
   return dir
end


-- get_fs_free_space: get free space in current file system, in bytes
local function get_fs_free_space(thispath)
   local buf = io.popen('df -k "'..thispath..'"', 'r'):read('*all')
   local _, _, free =  buf:find('[%w%d%/]+%s+[%d.]+%s+[%d.]+%s+([%d.]+)%s+[%d%%]+%s+[%w%d%/]+')
   return tonumber(free*1024 or 0)
end


-- format_datetime: return a formated string of date
local function format_datetime(timestamp)
   return os.date('%d/%m/%Y %H:%M', timestamp)
end


-- format_size: return a formated string of size
local function format_size(size)
   if size >= 1000000000 then
      return fmt('%.2f', size/(1024*1024*1024)) .. 'GB'
   elseif size >= 1000000 then
      return fmt('%.2f', size/(1024*1024)) .. 'MB'
   elseif size >= 1000 then
      return fmt('%.2f', size/1024) .. 'KB'
   else
      return tostring(size) .. 'B'
   end
end


-- get_filetype: return file extension
local function get_filetype(fname, mode)
   if mode == 'directory' then
      return '<dir>'
   elseif mode == 'link' then
      return '<link>'
   else
      local s, e, ext = string.find(fname, "%.(%w+)$")
      return ext or 'file'
   end
end


-- get_files
local function get_files(path)
   local entries = {}
   local fullpath
   local curdir = lfs.currentdir()
   local ret = lfs.chdir(path)
   lfs.chdir(curdir)
   if not ret then return nil end
   for f in lfs.dir(path) do
      if f ~= "." and f ~= ".." then
         local file = {}
         fullpath = path .. (path:sub(-1)=='/' and '' or '/') .. f
         local attr = lfs.symlinkattributes(fullpath)
         file.name = f
         file.type = get_filetype(fullpath, attr.mode) or 'unknown'
         file.size = attr.size or 0
         file.mtime = attr.modification or 0
         if attr.mode == 'directory' then
            if fullpath:sub(-1) ~= '/' then fullpath = fullpath .. '/' end
         end
         file.path = fullpath
         entries[f] = file
      end
   end
   table.sort(entries)
   return entries
end


-- sort_files: return an iterator with files ordered
local function sort_files(t)
   -- sort functions
   local sortfunc
   if opts.SORT == SORTBY_NAME then
      if opts.SORT_MIX_CASES then
         sortfunc = function(s1, s2) return (string.lower(s1) < string.lower(s2)) end
      else
         sortfunc = function(s1, s2) return (s1 < s2) end
      end
   elseif opts.SORT == SORTBY_NAME_REV then
      if opts.SORT_MIX_CASES then
         sortfunc = function(s1, s2) return (string.lower(s1) > string.lower(s2)) end
      else
         sortfunc = function(s1, s2) return (s1 > s2) end
      end
   elseif opts.SORT == SORTBY_DATE then
      sortfunc = function(s1, s2) return (t[s1].mtime < t[s2].mtime) end
   elseif opts.SORT == SORTBY_DATE_REV then
      sortfunc = function(s1, s2) return (t[s1].mtime > t[s2].mtime) end
   elseif opts.SORT == SORTBY_SIZE then
      sortfunc = function(s1, s2) return (t[s1].size < t[s2].size) end
   elseif opts.SORT == SORTBY_SIZE_REV then
      sortfunc = function(s1, s2) return (t[s1].size > t[s2].size) end
   end
   -- mix dirs and files
   local a = {}
   if opts.SORT_MIX_DIRS_AND_FILES then
      for n in pairs(t) do table.insert(a, n) end
      table.sort(a, sortfunc)
   else
      local b = {}
      for n in pairs(t) do
         if t[n].type=='<dir>' then
            table.insert(a, n)
         else
            table.insert(b, n)
         end
      end
      table.sort(a, sortfunc)
      table.sort(b, sortfunc)
      for _, n in ipairs(b) do table.insert(a, n) end
   end
   local function iter()
       for _, el in ipairs(a) do coroutine.yield(el, t[el]) end
   end
   return coroutine.wrap(iter)
end


-- parse_command_line: parse command line arguments
local function parse_command_line(arg)
   path = opts.INITIAL_PATH
   for _, v in ipairs(arg) do
      if v == '-t' or v == "--treeview" then
         opts.TREEVIEW = true
      elseif v == '-f' or v == "--flatview" then
         opts.TREEVIEW = false
      else
         path = get_abspath(v)
      end
   end
   if not check_valid_dir(path) then path = lfs.currentdir() end
   if path:sub(-1) ~= "/" then path = path .. "/" end
end



---------- Config -----------------------------------------------------

-- load config
local function config_load(filename)
   local f, err, errno = io.open(filename, "r")
   if not f then return err, errno end
   local buf = f:read("*a")
   f:close()
   if buf:sub(1, CONFIG_FILE_HEADER:len()) ~= CONFIG_FILE_HEADER then return "Bad format", 0 end
   local txt = "{\n"
   for k, v in buf:gfind("([^%s]+)%s*=%s*([^\n]+)") do txt = txt .. k .. " = " .. v .. ",\n" end
   txt = txt:sub(1, -3) .. "\n}"
   r = assert(loadstring("return " .. txt))
   if not r then return "Bad format", 0 end
   local newopts = r()
   for k, v in pairs(newopts) do opts[k] = v end
end


-- save config
local function config_save(filename)
   local f, err, errno = io.open(filename, "w")
   if not f then return err, errno end
   f:write(CONFIG_FILE_HEADER .. "\n\n")
   local buf
   for k, val in pairs(opts) do
      if type(val) == "table" then
         buf = "{ "
         for i= 1,#val-1 do
            buf = buf .. "\"" .. val[i] .. "\", "
         end
         buf = buf .. "\"" .. val[#val] .. "\" }"
      elseif type(val) == "string" then
         buf = '"' .. val .. '"'
      elseif type(val) == "boolean" then
         buf = val and "true" or "false"
      else
         buf = val
      end
      f:write(fmt("%s = %s\n", k, buf))
   end
   f:close()
end


function config_print()
   local buf
   for k, val in pairs(opts) do
      if type(val) == "table" then
         buf = "{ "
         for i= 1,#val-1 do
            buf = buf .. "\"" .. val[i] .. "\", "
         end
         buf = buf .. "\"" .. val[#val] .. "\" }"
      elseif type(val) == "boolean" then
         buf = val and "true" or "false"
      else
         buf = val
      end
      print(fmt("%s = %s\n", k, buf))
   end
end


---------- MYGTK ------------------------------------------------------

--------------------------------------------------
-- helper functions
local mygtk = {
   -- create a label with markup
   Markup_new = function(txt)
                   local label = gtk.Label.new()
                   label:set_markup(txt)
                   return label
                end,
   -- add alignment and padding to a widget
   align = function(widget, xalign, pad)
              xalign = xalign or 0
              pad = pad or {0, 0, 0, 0}
              local alignment = gtk.Alignment.new()
              alignment:set("xalign", xalign)
              alignment:set("yalign", 0.5)
              alignment:set_padding(pad[1], pad[2], pad[3], pad[4])
              alignment:add(widget)
              return alignment
           end,
   -- close button
   ButtonClose = function(dialog)
                    local btn = gtk.Button.new_from_stock("gtk-close")
                    btn:connect("clicked", function() dialog:hide(); dialog:destroy() end)
                    return btn
                 end,
   -- combo creation helper
   Combo_new = function(data, sensitive, callback)
                  local mod = gtk.ListStore.new("gchararray", "gchararray")
                  local cmb = gtk.ComboBox.new_with_model(mod)
                  cmb:set("sensitive", sensitive)
                  local iter = gtk.TreeIter.new()
                  local r1, r2 = gtk.CellRendererPixbuf.new(), gtk.CellRendererText.new()
                  for _, val in pairs(data) do
                     mod:append(iter)
                     mod:seto(iter, val[1], val[2])
                  end
                  cmb:pack_start(r1, false)
                  cmb:pack_start(r2, false)
                  cmb:set_cell_data_func(r2, function(rend, iter)
                                                rend:set("text", " " .. mod:get(iter, 1))
                                             end, r2)
                  cmb:add_attribute(r1, "stock-id", 0, "sensitive", 1)
                  cmb:add_attribute(r2, "text", 0, "sensitive", 1)
                  cmb:set_size_request(36, -1)
                  cmb:set("active", #data-1)  
                  cmb:connect("changed", callback, cmb)
                  return cmb
               end
}


--------------------------------------------------
-- PathLabel

-- path_split: split /path/to/file 
local function path_split(pl, x)
   local pstart, pend
   for i, val in ipairs(pl.path_tbl) do
      if x < val[1] then pstart = val[2]; break end
   end
   if not pstart then -- if file select last directory
      pstart = pl.path_tbl[#pl.path_tbl][2]
   end
   pend = pstart=='/' and pl.path:sub(2, -1) or pl.path:gsub(pstart, "")
   return pstart, pend
end

-- PathLabel
mygtk.PathLabel = {}
function mygtk.PathLabel.new(txt_attrs, bg_color)
   local self = {}
   setmetatable(self, {__index = mygtk.PathLabel})
   self.path, self.path_tbl = "", {}
   self.txt_attrs = txt_attrs or "size='medium' weight='bold'"
   self.cb_clicked = nil
   self.evbox = gtk.EventBox.new()
   if bg_color then
      self.evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse(bg_color))
   end
   self.evbox:set_events(gdk.LEAVE_NOTIFY_MASK + gdk.POINTER_MOTION_MASK +
                         gdk.POINTER_MOTION_HINT_MASK + gdk.BUTTON_PRESS_MASK)
   self.evbox:connect("leave-notify-event", mygtk.PathLabel.leave, self)
   self.evbox:connect("motion-notify-event", mygtk.PathLabel.motion, self)
   self.evbox:connect("button-press-event", mygtk.PathLabel.click, self)
   self.label = gtk.Label.new()
   -- self.label:set("ellipsize", pango.ELLIPSIZE_END)
   self.evbox:add(self.label)
   self.evbox:show_all()
   return self
end

-- connect clicked signal
function mygtk.PathLabel:connect_click(fn)
   self.cb_clicked = fn
end

-- set path and rebuild path table
function mygtk.PathLabel:set_path(ptxt)
   self.path = ptxt
   local ps = { '/' }
   ptxt:gsub('([^/]+)/', function(p) table.insert(ps, p..'/') end)
   self.path_tbl = {}
   local w, fp, l = 0, "", gtk.Label.new()
   for _, p in ipairs(ps) do
      l:set_markup(fmt("<span %s>%s</span>", self.txt_attrs, p))
      w = w + l:get_layout():get_pixel_size()
      fp = fp .. p
      table.insert(self.path_tbl, {w, fp})
   end
   self.label:set_markup(fmt("<span %s>%s</span>", self.txt_attrs, self.path))
end

-- callback: leave PathLabel
function mygtk.PathLabel.leave(pl)
   pl.label:set_markup(fmt("<span %s>%s</span>", pl.txt_attrs, pl.path))
end

-- callback: motion into PathLabel
function mygtk.PathLabel.motion(pl, ev)
   local _, x, y = gdk.Event.get_coords(ev)
   local pstart, pend = path_split(pl, x)
   local buf = fmt("<span %s><u>%s</u></span>", pl.txt_attrs, pstart)
   buf = buf .. fmt("<span %s>%s</span>", pl.txt_attrs, pend)
   pl.label:set_markup(buf)
end

-- callback: clicked into PathLabel
function mygtk.PathLabel.click(pl, ev)
   local _, x, y = gdk.Event.get_coords(ev)
   local pstart, pend = path_split(pl, x)
   if pl.cb_clicked then pl.cb_clicked(pstart) end
end
  

---------- DIALOGS ----------------------------------------------------

--------------------------------------------------
-- splash
local function ui_splash(title, subtitle)
   local splash = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   splash:set("window-position", gtk.WIN_POS_CENTER, "decorated", false,
              "width-request", 550, "height-request", 150,
              "resizable", false, "border-width", 10)
   local frame = gtk.Frame.new()
   local vbox = gtk.VBox.new(true, 0)
   vbox:set("border-width", 10)
   local line1 = mygtk.Markup_new(title)
   vbox:pack_start(line1, true, true, 10)
   local line2 = mygtk.Markup_new(subtitle)
   line2:set("ellipsize", pango.ELLIPSIZE_START)
   vbox:pack_start(line2, true, true, 10)
   frame:add(vbox)
   splash:add(frame)
   splash:show_all()
   while gtk.events_pending() do
      gtk.main_iteration()
   end
   return splash
end


--------------------------------------------------
-- dialog
local function ui_dialog(type, title, text)
   local h = 190 + #text * 15
   local dialog = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   dialog:set("window-position", gtk.WIN_POS_CENTER, "decorated", false, "modal", true,
              "height-request", h, -- "width-request", 550,
              "resizable", false, "border-width", 10)
   local frame = gtk.Frame.new()
   local vbox = gtk.VBox.new(false, 10)
   vbox:set("border-width", 10)
   -- title
   local lbl = mygtk.Markup_new("<span size='large' weight='bold'>" .. title .. "</span>")
   vbox:pack_start(lbl, false, false, 0)
   -- body
   local hbox = gtk.HBox.new(false, 0)
   local img
   if type == "error" then
      img = gtk.Image.new_from_stock("gtk-dialog-error", gtk.ICON_SIZE_DIALOG)
   elseif type == "warning" then
      img = gtk.Image.new_from_stock("gtk-dialog-warning", gtk.ICON_SIZE_DIALOG)
   else
      img = gtk.Image.new_from_stock("gtk-dialog-info", gtk.ICON_SIZE_DIALOG)
   end
   hbox:pack_start(img, false, false, 10)
   local buf = ""
   for i, line in ipairs(text) do
      buf = buf .. "\n<span size='small'>" .. line .. "</span>"
   end
   hbox:pack_start(mygtk.align(mygtk.Markup_new(buf)), true, true, 10)
   vbox:pack_start(hbox, true, false, 0)
   -- buttons
   vbox:pack_start(gtk.Label.new(), true, true, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 5)
   vbox:pack_start(mygtk.ButtonClose(dialog), false, false, 0)
   -- packaging
   frame:add(vbox)
   dialog:add(frame)
   dialog:show_all()
end


--------------------------------------------------
-- clipboard view
local function ui_clipboard_view()
   local lbl, tbl2, buf
   -- get clipboard info
   local tbl = {["select"]={}, ["copy"]={}, ["cut"]={}}
   local size = {["select"]=0, ["copy"]=0, ["cut"]=0} 
   for p, c in pairs(selected_files) do
      if c.mode == 0 then
         table.insert(tbl.select, p)
         size.select = size.select + c.size
      elseif c.mode == 1 then
         table.insert(tbl.copy, p)
         size.copy = size.copy + c.size
      elseif c.mode == 2 then
         table.insert(tbl.cut, p)
         size.cut = size.cut + c.size
      end
   end
   -- get file system info
   buf = io.popen('df -k "'..path..'"', 'r'):read('*all')
   local _, _, dev, total, used, available, pct, mountpoint =
      buf:find('([%w%d%/]+)%s+([%d.]+)%s+([%d.]+)%s+([%d.]+)%s+([%d%%]+)%s+([%w%d%/]+)')
   dev = dev or "/dev/tffsa1"
   mountpoint = mountpoint or "/"
   total = format_size(tonumber(total or 0) * 1024)
   available = format_size(tonumber(available or 0) * 1024)
   -- UI
   local dialog = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   dialog:set("border-width", 15)
   local frame = gtk.Frame.new()
   local vbox = gtk.VBox.new(false, 10)
   vbox:set("border-width", 10)
   --    title
   lbl = mygtk.Markup_new("<span size='xx-large'><b>Selected files</b></span>")
   vbox:pack_start(lbl, false, false, 0)
   --    destination
   vbox:pack_start(mygtk.align(mygtk.Markup_new("<b>Destination:</b>"), 0, {10, 0, 10, 0}),
                   false, false, 0)
   buf = fmt("<span size='small'><i>%s</i>\non %s (%s), %s / %s free</span>",
             path, mountpoint, dev, available, total)
   vbox:pack_start(mygtk.align(mygtk.Markup_new(buf), 0, {0, 0, 20, 0}), false, false, 0)
   --    selected items
   for _, t in ipairs({"select", "copy", "cut"}) do
      table.sort(tbl[t])
      local section = t=="select" and "Selected" or "To "..t
      buf = fmt("<b>%s:  %d files (%s)</b>", section, #tbl[t], format_size(size[t]))
      vbox:pack_start(mygtk.align(mygtk.Markup_new(buf), 0, {10, 0, 10, 0}),
                      false, false, 0)
      if #tbl[t] > 0 then
         tbl2 = gtk.Table.new(#tbl[t]+1, 2, false)
         lbl = mygtk.Markup_new("<span size='small'><b><u>Filename</u></b></span>")
         tbl2:attach_defaults(mygtk.align(lbl, 0, {0, 0, 20, 0}), 0, 1, 0, 1)
         lbl = mygtk.Markup_new("<span size='small'><b><u>Size</u></b></span>")
         tbl2:attach(mygtk.align(lbl), 1, 2, 0, 1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 20, 0)
         for i, p in ipairs(tbl[t]) do
            lbl = mygtk.PathLabel.new("size='x-small'")
            lbl:set_path(p)
            lbl:connect_click(function(txt)
                                 dialog:hide(); dialog:destroy()
                                 path = txt; ui_refresh(path)
                              end)
            tbl2:attach_defaults(mygtk.align(lbl.evbox, 0, {0, 0, 20, 0}), 0, 1, i, i+1)
            lbl = mygtk.Markup_new("<span size='x-small'>" ..
                                   format_size(selected_files[p].size) .. "</span>")
            tbl2:attach(mygtk.align(lbl), 1, 2, i, i+1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 20, 0)
         end
         vbox:pack_start(tbl2, false, false, 10)
      end
   end
   --    button
   vbox:pack_start(gtk.Label.new(), true, true, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 5)
   vbox:pack_start(mygtk.ButtonClose(dialog), false, false, 0)
   -- packaging
   frame:add(vbox)
   dialog:add(frame)
   dialog:fullscreen()
   dialog:show_all()
end


--------------------------------------------------
-- updating UI
local function ui_update_path()
   lbl_path:set_path(path)
   local free = format_size(get_fs_free_space(path))
   lbl_free:set_markup("<span size='xx-small'> " .. free .. " </span>")
end


local function ui_update_status_selected()
   local num, size = 0, 0
   for f, c in pairs(selected_files) do num = num+1; size = size+c.size end
   if num == 0 then
      buf = "<span size='small'> No files selected </span>"
      cmb_actions:set("sensitive", false)
   else
      buf = fmt("<span size='small'> %d file(s) selected, %s </span>",
                num, format_size(size))
      cmb_actions:set("sensitive", true)
   end
   st_selected:set_markup(buf)
end


local function ui_update_files_tv(path, dad, depth)
   local files = get_files(path)
   if not files then return end
   local myiter = gtk.TreeIter.new()
   for f, attr in sort_files(files) do
      local selfile, pix = selected_files[attr.path], pix_noselected
      if selfile then
         if selfile.mode == 0 then pix = pix_selected 
         elseif selfile.mode == 1 then pix = pix_copied
         elseif selfile.mode == 2 then pix = pix_cut
         else pix = pix_noselected end
      end
      model:append(myiter, dad)
      -- {pix_selected?, filename, type, size, mtime, fullpath, sizenum}
      model:set(myiter, TV_COL_SELECTED, pix,
                TV_COL_NAME, attr.name, TV_COL_TYPE, attr.type,
                TV_COL_SIZE, format_size(attr.size),
                TV_COL_MTIME, format_datetime(attr.mtime),
                TV_COL_PATH, attr.path, TV_COL_SIZE_NUM, attr.size)
      if attr.type == '<dir>' then
         if depth < opts.TREE_DEPTH then
            ui_update_files_tv(attr.path, myiter, depth+1)
         else
            local myiter2 = gtk.TreeIter.new()
            model:append(myiter2, myiter)
            model:set(myiter2, TV_COL_SELECTED, pix_noselected,
                      TV_COL_NAME, "[more]", TV_COL_TYPE, "[more]",
                      TV_COL_SIZE, "0.00KB",
                      TV_COL_MTIME, format_datetime(attr.mtime),
                      TV_COL_PATH, "[more]", TV_COL_SIZE_NUM, 0)
         end
      end
   end
end


local function ui_update_files_lv(path)
   local files = get_files(path)
   if not files then return end
   local myiter = gtk.TreeIter.new()
   for f, attr in sort_files(files) do
      local selfile, pix = selected_files[attr.path], pix_noselected
      if selfile then
         if selfile.mode == 0 then pix = pix_selected 
         elseif selfile.mode == 1 then pix = pix_copied
         elseif selfile.mode == 2 then pix = pix_cut
         else pix = pix_noselected end
      end
      model:append(myiter)
      -- {pix_selected?, filename, type, size, mtime, fullpath, size_num}
      model:set(myiter,TV_COL_SELECTED, pix,
                TV_COL_NAME, attr.name, TV_COL_TYPE, attr.type,
                TV_COL_SIZE, format_size(attr.size),
                TV_COL_MTIME, format_datetime(attr.mtime),
                TV_COL_PATH, attr.path, TV_COL_SIZE_NUM, attr.size)
   end
end


function ui_refresh(path)
   local splash = ui_splash("Please, wait while loading directory",
                            "<i>" .. path .. "</i>")
   ui_update_path() 
   model:clear()
   if opts.TREEVIEW then
      ui_update_files_tv(path, nil, 0)
   else
      ui_update_files_lv(path)
   end
   ui_update_status_selected()
   cmb_others:set("sensitive", opts.TREEVIEW)
   splash:hide()
   splash:destroy()
end


--------------------------------------------------
-- selecting files
local function check_file_for_select(fullpath)
   -- check if parent directory already in list
   for pf, _ in pairs(selected_files) do
      if fullpath:find(pf) == 1 and pf ~= fullpath then
         return false, "in parent dir", pf
      end
   end
   -- check if a child is already selected
   for pf, _ in pairs(selected_files) do
      if pf:find(fullpath) == 1 and pf ~= fullpath then
         return true, "child selected", pf
      end
   end
   -- check if user directories
   for _, vp in ipairs(opts.SELECT_VALID_PATHS) do
      if fullpath:find(vp) == 1 then
         return true, "", ""
      end
   end
   -- else system file
   return false, "system file", ""
end


local function select_file(iter)
   local fullpath = model:get(iter, TV_COL_PATH)
   local valid, reason, extra = check_file_for_select(fullpath)
   if not valid then
      local text, subtitle
      if reason == "system file" then
         title = "ERROR: can't select file"
         text = { "<i>" .. fullpath .. "</i>",
                  "This is a system file or directory. Moving or deleting it",
                  "could damage your Iliad, thus it would be a very bad idea." }
      elseif reason == "in parent dir" then
         title = "ERROR: can't select file"
         text = { "<i>" .. fullpath .. "</i>",
                  "because a parent directory is already selected",
                  "<i>" .. extra .. "</i>" }
      else
         title = "ERROR: unknown error"
         text = {"An unknown error was produced selecting file",
                  "<i>" .. fullpath .. "</i>" }
      end
      ui_dialog("error", title, text)
   else
      if model:get(iter, TV_COL_SELECTED) == pix_noselected then
         if reason == "child selected" then
            title = "WARNING: one child already selected"
            text = { "<i>" .. extra .. "</i>",
                     "so we'll remove it and add the parent to the selection list",
                     "<i>" .. fullpath .. "</i>" }
            ui_dialog("warning", title, text)
            -- remove all children from list and update model
            local to_delete = {}
            for fp, _ in pairs(selected_files) do
               if fp:find(fullpath) == 1 then
                  selected_files[fp] = nil
                  to_delete[fp] = true
               end
            end
            model:foreach(function(m, p, iter)
                             if to_delete[model:get(iter, TV_COL_PATH)] then
                                model:set(iter, TV_COL_SELECTED, pix_noselected)
                             end
                          end)
         end
         -- add file to list
         local size
         if model:get(iter, TV_COL_TYPE) == "<dir>" then
            -- Iliad "du" doesn't accept 'b' flag for bytes, so use "k" and convert
            local buf = io.popen("du -sk \"" .. fullpath .. "\""):read("*all")
            size = tonumber(buf:sub(buf:find("%d+"))) * 1024 or 0
         else
            size = model:get(iter, TV_COL_SIZE_NUM) or 0
         end
         selected_files[fullpath] = {["size"]=size, ["mode"]=0}
         model:set(iter, TV_COL_SELECTED, pix_selected)
      elseif model:get(iter, TV_COL_SELECTED) == pix_selected then
         selected_files[fullpath].mode = 1
         model:set(iter, TV_COL_SELECTED, pix_copied)
      elseif model:get(iter, TV_COL_SELECTED) == pix_copied then
         selected_files[fullpath].mode = 2
         model:set(iter, TV_COL_SELECTED, pix_cut)
      else
         -- remove from list
         selected_files[fullpath] = nil
         model:set(iter, TV_COL_SELECTED, pix_noselected)
      end
      ui_update_status_selected()
   end
end


--------------------------------------------------
-- treeview callbacks
local function cb_treeview_click(widget, ev)
   local _, x, y = gdk.Event.get_coords(ev)
   local _, iterpath = view:get_path_at_pos(x, y)
   if not iterpath then return true end
   local iter = gtk.TreeIter.new()
   model:get_iter_from_string(iter, iterpath)
   -- select file only if clicked over icon
   if x < TV_COL_W.SELECTED then
      select_file(iter)
      return true -- don't propagate signal, stop here
   end
   -- change dir
   local type = model:get(iter, TV_COL_TYPE)
   local posx_type = TV_COL_W.SELECTED + TV_COL_W.NAME
   local posx_size = posx_type + TV_COL_W.TYPE
   if type == "<dir>" and (x>posx_type and x<posx_size) then -- FIXME: link2dir
      path = model:get(iter, TV_COL_PATH)
      ui_refresh(path)
      return true -- don't propagate signal, stop here
   end
   -- expand or collapse row
   if opts.TREEVIEW then
      local pathiter = gtk.TreePath.new_from_string(model:get_string_from_iter(iter))
      if view:row_expanded(pathiter) then
         view:collapse_row(pathiter)
      else
         view:expand_row(pathiter, false)
      end
   end
   return true -- don't propagate signal, stop here
end


--------------------------------------------------
-- other callbacks
local function cb_goto_bookmark(combo)
   local bk = combo:get_active_text()
   if bk == "Bookmarks" or bk == path then return end
   combo:set("active", #opts.BOOKMARKS)
   if bk:sub(-1) ~= "/" then bk = bk .. "/" end
   if not check_valid_dir(bk) then
      ui_dialog("error", "Goto bookmark", {"ERROR: not a valid directory", bk})
      return
   end
   path = bk
   ui_refresh(path)
end


local function mark_copycut(mode)
   local txt = mode==1 and "copied" or "cut"
   local pix = mode==1 and pix_copied or pix_cut
   local tbl = {n=0}
   for p, c in pairs(selected_files) do
      if c.mode == 0 then tbl[p]=true; c.mode=mode; tbl.n=tbl.n+1 end
   end
   model:foreach(function(m, p, iter)
                    if tbl[model:get(iter, TV_COL_PATH)] then
                       model:set(iter, TV_COL_SELECTED, pix)
                    end
                 end)
   -- ui_dialog("info", "Files selection",
   --           {tbl.n .. " files " .. txt .. " to clipboard"})
end


local function do_copymove(mode, f, i, total)
   local splash, size, free
   local action = mode=="copy" and "Copying" or "Moving"
   local cmd = mode=="copy" and "cp -af" or "mv -f"
   splash = ui_splash(fmt(action.." file %d/%d", i, total), "<i>" .. f .. "</i>")
   if path ~= get_parentdir(f) then
      size = selected_files[f].size
      free = get_fs_free_space(path)
      if free < size then
         ui_dialog("error", "ERROR: not enough space",
                   {fmt("Needed %s, free %s", format_size(size), format_size(free))})
      else
         os.execute(cmd .. ' "' .. f .. '" "' .. path .. '"')
         selected_files[f] = nil
      end
   else
      selected_files[f] = nil
   end
   splash:hide()
   splash:destroy()
end


local function cb_actions_paste()
   local to_copy, size_copy, to_move, size_move = {}, 0, {}, 0
   for p, c in pairs(selected_files) do
      if c.mode == 1 then
         table.insert(to_copy, p)
         size_copy = size_copy + c.size
      elseif c.mode == 2 then
         table.insert(to_move, p)
         size_move = size_move + c.size
      end
   end
   if #to_copy == 0 and #to_move == 0 then
      local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_WARNING,
                                        gtk.BUTTONS_OK, "Paste files")
      dlg:set_markup("No selected files to paste")
      dlg:run()
      dlg:destroy()
      return
   end
   local dialog, frame, vbox, lbl, img, tbl, hbox, btn
   dialog = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   dialog:set("border-width", 15)
   frame = gtk.Frame.new()
   vbox = gtk.VBox.new(false, 10)
   vbox:set("border-width", 10)
   -- title
   lbl = mygtk.Markup_new("<span size='xx-large'><b>Paste files</b></span>")
   vbox:pack_start(lbl, false, false, 0)
   -- contents
   hbox = gtk.HBox.new(false, 50)
   img = gtk.Image.new_from_stock("gtk-dialog-question", gtk.ICON_SIZE_DIALOG)
   hbox:pack_start(img, false, false, 0)
   lbl = mygtk.Markup_new("You are going to copy " .. #to_copy .. " files (" ..
                          format_size(size_copy) .. ")\nand move " .. #to_move ..
                          " files (" .. format_size(size_move) .. ").\n" ..
                          "Please confirm.")
   hbox:pack_start(lbl, true, true, 0)
   vbox:pack_start(mygtk.align(hbox, 0, {0, 0, 10, 0}), false, false, 20)

   if #to_copy > 0 then
      lbl = mygtk.Markup_new("<span size='large'><b>Files to copy:</b></span>")
      vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 5)
      tbl = gtk.Table.new(#to_copy+1, 2, false)
      lbl = mygtk.Markup_new("<span size='small'><b><u>Filename</u></b></span>")
      tbl:attach_defaults(mygtk.align(lbl, 0.5, {0, 0, 10, 0}), 0, 1, 0, 1)
      lbl = mygtk.Markup_new("<span size='small'><b><u>Size</u></b></span>")
      tbl:attach(mygtk.align(lbl), 1, 2, 0, 1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
      local i = 1
      for p, c in pairs(selected_files) do
         if c.mode == 1 then
            lbl = mygtk.Markup_new("<span size='small'>" .. p .. "</span>")
            tbl:attach_defaults(mygtk.align(lbl, 0, {0, 0, 10, 0}), 0, 1, i, i+1)
            lbl = mygtk.Markup_new("<span size='small'>" .. format_size(c.size) .. "</span>")
            tbl:attach(mygtk.align(lbl), 1, 2, i, i+1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
            i = i + 1
         end
      end
      vbox:pack_start(tbl, false, false, 0)
   end
   if #to_move > 0 then
      lbl = mygtk.Markup_new("<span size='large'><b>Files to move:</b></span>")
      vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 5)
      tbl = gtk.Table.new(#to_move+1, 2, false)
      lbl = mygtk.Markup_new("<span size='small'><b><u>Filename</u></b></span>")
      tbl:attach_defaults(mygtk.align(lbl, 0.5, {0, 0, 10, 0}), 0, 1, 0, 1)
      lbl = mygtk.Markup_new("<span size='small'><b><u>Size</u></b></span>")
      tbl:attach(mygtk.align(lbl), 1, 2, 0, 1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
      local i = 1
      for p, c in pairs(selected_files) do
         if c.mode == 2 then
            lbl = mygtk.Markup_new("<span size='small'>" .. p .. "</span>")
            tbl:attach_defaults(mygtk.align(lbl, 0, {0, 0, 10, 0}), 0, 1, i, i+1)
            lbl = mygtk.Markup_new("<span size='small'>" .. format_size(c.size) .. "</span>")
            tbl:attach(mygtk.align(lbl), 1, 2, i, i+1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
            i = i + 1
         end
      end
      vbox:pack_start(tbl, false, false, 0)
   end

   lbl = mygtk.Markup_new("<span size='large'><b>Destination:</b></span>")
   vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 10)
   -- lbl = mygtk.Markup_new("<i>" .. path .. "</i>")
   lbl = mygtk.PathLabel.new("size='small'")
   lbl:set_path(path)
   lbl:connect_click(function(txt)
                                 dialog:hide(); dialog:destroy()
                                 path = txt; ui_refresh(path)
                              end)
   vbox:pack_start(mygtk.align(lbl.evbox, 0, {0, 0, 10, 0}), false, false, 0)

   -- buttons
   vbox:pack_start(gtk.Label.new(), true, true, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 5)
   hbox = gtk.HBox.new(true, 150)
   btn = gtk.Button.new_from_stock("gtk-yes")
   btn:connect("clicked", function()
                             if #to_copy > 0 then
                                for i, f in ipairs(to_copy) do
                                   do_copymove("copy", f, i, #to_copy)
                                end
                             end
                             if #to_move > 0 then
                                for i, f in ipairs(to_move) do
                                   do_copymove("move", f, i, #to_move)
                                end
                             end
                             dialog:hide(); dialog:destroy()
                             ui_refresh(path)
                          end)
   hbox:pack_start(btn, true, true, 0)
   btn = gtk.Button.new_from_stock("gtk-no")
   btn:connect("clicked", function() dialog:hide(); dialog:destroy() end)
   hbox:pack_start(btn, true, true, 0)
   vbox:pack_start(hbox, false, false, 0)
   -- packaging
   frame:add(vbox)
   dialog:add(frame)
   dialog:fullscreen()
   dialog:show_all()
end


local function do_delete(files)
   local splash
   for i, f in ipairs(files) do
      splash = ui_splash(fmt("Deleting file %d/%d", i, #files),
                         "<i>" .. f .. "</i>")
      os.execute('rm -rf "' .. f .. '"')
      selected_files[f] = nil
      splash:hide()
      splash:destroy()
   end
end


local function cb_actions_delete()
   local to_delete, size = {}, 0
   for p, c in pairs(selected_files) do
      if c.mode == 0 then
         table.insert(to_delete, p)
         size = size + c.size
      end
   end
   if #to_delete == 0 then
      local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_WARNING,
                                        gtk.BUTTONS_OK, "Delete files")
      dlg:set_markup("No selected files to delete")
      dlg:run()
      dlg:destroy()
      return
   end
   local dialog, frame, vbox, lbl, img, tbl, hbox, btn
   dialog = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   dialog:set("border-width", 15)
   frame = gtk.Frame.new()
   vbox = gtk.VBox.new(false, 10)
   vbox:set("border-width", 10)
   -- title
   lbl = mygtk.Markup_new("<span size='xx-large'><b>Delete files</b></span>")
   vbox:pack_start(lbl, false, false, 0)
   -- contents
   hbox = gtk.HBox.new(false, 50)
   img = gtk.Image.new_from_stock("gtk-dialog-question", gtk.ICON_SIZE_DIALOG)
   hbox:pack_start(img, false, false, 0)
   lbl = mygtk.Markup_new("You are going to pemanently delete " .. #to_delete ..
                          " files (" .. format_size(size) .. ").\nPlease confirm.")
   hbox:pack_start(lbl, true, true, 0)
   vbox:pack_start(mygtk.align(hbox, 0, {0, 0, 10, 0}), false, false, 10)
   tbl = gtk.Table.new(#to_delete+1, 2, false)
   lbl = mygtk.Markup_new("<b><u>Filename</u></b>")
   tbl:attach_defaults(mygtk.align(lbl, 0.5, {0, 0, 10, 0}), 0, 1, 0, 1)
   lbl = mygtk.Markup_new("<b><u>Size</u></b>")
   tbl:attach(mygtk.align(lbl), 1, 2, 0, 1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
   local i = 1
   for p, c in pairs(selected_files) do
      if c.mode == 0 then
         lbl = mygtk.Markup_new("<span size='small'>" .. p .. "</span>")
         tbl:attach_defaults(mygtk.align(lbl, 0, {0, 0, 10, 0}), 0, 1, i, i+1)
         lbl = mygtk.Markup_new("<span size='small'>" .. format_size(c.size) .. "</span>")
         tbl:attach(mygtk.align(lbl), 1, 2, i, i+1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
         i = i + 1
      end
   end
   vbox:pack_start(tbl, false, false, 10)
   -- buttons
   vbox:pack_start(gtk.Label.new(), true, true, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 5)
   hbox = gtk.HBox.new(true, 150)
   btn = gtk.Button.new_from_stock("gtk-yes")
   btn:connect("clicked", function(files)
                             do_delete(files)
                             dialog:hide(); dialog:destroy()
                             ui_refresh(path)
                          end, to_delete)
   hbox:pack_start(btn, true, true, 0)
   btn = gtk.Button.new_from_stock("gtk-no")
   btn:connect("clicked", function() dialog:hide(); dialog:destroy() end)
   hbox:pack_start(btn, true, true, 0)
   vbox:pack_start(hbox, false, false, 0)
   -- packaging
   frame:add(vbox)
   dialog:add(frame)
   dialog:fullscreen()
   dialog:show_all()
end


local function cb_actions(combo)
   local COPY, CUT, PASTE, DELETE, CLEAR, VIEW, LABEL = 0, 1, 2, 3, 4, 5, 6
   local opt = combo:get_active()
   if opt == LABEL then
      return true
   elseif opt == COPY then
      mark_copycut(1)
   elseif opt == CUT then
      mark_copycut(2)
   elseif opt == PASTE then
      cb_actions_paste()
   elseif opt == DELETE then
      cb_actions_delete()
   elseif opt == CLEAR then
      selected_files = {}
      model:foreach(function(m, p, iter)
                       model:set(iter, TV_COL_SELECTED, pix_noselected)
                    end)
      ui_update_status_selected()
   elseif opt == VIEW then
      ui_clipboard_view()
   end
   combo:set_active(LABEL)
   return true
end


local function cb_others_actions(combo)
   local EXPAND_ALL, EXPAND_1, COLLAPSE_ALL, SELECT_ALL, SELECT_ALL_COPY,
         SELECT_ALL_CUT, LABEL = 0, 1, 2, 3, 4, 5, 6
   local opt = combo:get_active()
   if opt == LABEL then
      return true
   elseif opt == EXPAND_ALL then
      view:expand_all()
      model:foreach(function(m, p, iter)
                       local pathstr = model:get_string_from_iter(iter)
                       local pathiter = gtk.TreePath.new_from_string(pathstr)
                       local depth = gtk.TreePath.get_depth(pathiter)
                       if depth >= opts.TREE_DEPTH + 1 then
                          view:collapse_row(pathiter)
                       end
                    end)
   elseif opt == EXPAND_1 then
      view:collapse_all()
      local iter = gtk.TreeIter.new()
      model:get_iter_first(iter)
      while iter do
         local pathstr = model:get_string_from_iter(iter)
         view:expand_row(gtk.TreePath.new_from_string(pathstr))
         if not model:iter_next(iter) then break end
      end
   elseif opt == COLLAPSE_ALL then
      view:collapse_all()
   elseif opt == SELECT_ALL then
      local iter = gtk.TreeIter.new()
      model:get_iter_first(iter)
      while iter do
         select_file(iter)
         if not model:iter_next(iter) then break end
      end
   elseif opt == SELECT_ALL_COPY then
      local iter = gtk.TreeIter.new()
      model:get_iter_first(iter)
      while iter do
         select_file(iter)
         if not model:iter_next(iter) then break end
      end
      mark_copycut(1)
   elseif opt == SELECT_ALL_CUT then
      local iter = gtk.TreeIter.new()
      model:get_iter_first(iter)
      while iter do
         select_file(iter)
         if not model:iter_next(iter) then break end
      end
      mark_copycut(2)
   end
   combo:set_active(LABEL)
   return true
end


--------------------------------------------------
-- preferences

-- callback: view mode
local function cb_prefs_viewmode(mode)
   if mode == 0 then
      opts.TREEVIEW = false
   else
      opts.TREEVIEW = true
      opts.TREE_DEPTH = mode
   end
   ui_refresh(path)
end


-- callback: click on path
local function cb_prefs_clickpath(dialog)
   path = txt
   ui_refresh(path)
   dialog:hide()
   dialog:destroy()
end


-- preferences
local function ui_preferences()
   local lbl, lbl2, btn, tbl
   -- window
   local dialog = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   dialog:set("border-width", 15)
   local frame = gtk.Frame.new()
   local scrollwin = gtk.ScrolledWindow.new()
   scrollwin:set("hscrollbar-policy", gtk.POLICY_NEVER,
                 "vscrollbar-policy", gtk.POLICY_AUTOMATIC)
   local vbox = gtk.VBox.new(false, 0)
   vbox:set("border-width", 10)
   --   title
   lbl = mygtk.Markup_new("<span size='xx-large' weight='bold'>Preferences</span>")
   vbox:pack_start(lbl, false, false, 10)
   -- flat vs. tree view mode
   lbl = mygtk.Markup_new("<span size='large'><b>View mode:</b></span>")
   vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 5)
   local vbox2 = gtk.VBox.new(true, 0)
   local rbtn = {}
   rbtn[0] = gtk.RadioButton.new_with_label(nil, "Flat view")
   if opts.TREEVIEW == false then rbtn[0]:set_active(true) end
   rbtn[0]:connect("pressed", cb_prefs_viewmode, 0)
   vbox2:pack_start(rbtn[0], false, false, 0)
   for i = 1, 5 do
      local buf = "Tree view, "..i.." level max"
      if i == 1 then buf = buf .. " (fastest)" 
      elseif i == 3 then buf = buf .. " (good enough)"
      elseif i == 5 then buf = buf .. " (slowest)" end
      rbtn[i] = gtk.RadioButton.new_with_label_from_widget(rbtn[i-1], buf)
      if opts.TREEVIEW and opts.TREE_DEPTH == i then rbtn[i]:set_active(true) end
      rbtn[i]:connect("pressed", cb_prefs_viewmode, i)
      vbox2:pack_start(rbtn[i], false, false, 0)
   end
   vbox:pack_start(mygtk.align(vbox2, 0, {0, 0, 20, 0}), false, false, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 10)
   -- initial path
   lbl = mygtk.Markup_new("<span size='large'><b>Initial path:</b></span>")
   vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 10)
   tbl = gtk.Table.new(1, 2, false)
   local lbl2 = mygtk.PathLabel.new("size='small'")
   lbl2:set_path(opts.INITIAL_PATH)
   lbl2:connect_click(function(txt)
                        dialog:hide(); dialog:destroy()
                        path = txt; ui_refresh(path)
                     end)
   tbl:attach_defaults(mygtk.align(lbl2.evbox, 0, {0, 0, 10, 0}), 0, 1, 0, 1)
   btn = gtk.Button.new_from_stock("gtk-edit")
   btn:connect("clicked", function()
                             opts.INITIAL_PATH = path
                             lbl2:set_path(path)
                          end)
   tbl:attach(btn, 1, 2, 0, 1, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 0, 0)
   vbox:pack_start(tbl, false, false, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 10)
   -- bookmarks
   lbl = mygtk.Markup_new("<span size='large'><b>Bookmarks:</b></span>")
   vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 5)
   tbl = gtk.Table.new(#opts.BOOKMARKS, 3, false)
   for i, bk in ipairs(opts.BOOKMARKS) do
      lbl = mygtk.Markup_new("<span size='small'>" .. i ..  ".</span>")
      tbl:attach(lbl, 0, 1, i-1, i, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 10, 0)
      lbl = mygtk.PathLabel.new("size='small'")
      lbl:set_path(bk)
      lbl:connect_click(function(txt)
                           dialog:hide(); dialog:destroy()
                           path = txt; ui_refresh(path)
                        end)
      tbl:attach_defaults(mygtk.align(lbl.evbox), 1, 2, i-1, i)
      btn = gtk.Button.new_from_stock("gtk-edit")
      btn:connect("clicked", function(args)
                                local i, lbl = args[1], args[2]
                                opts.BOOKMARKS[i] = path
                                lbl:set_path(path)
                                -- modify bookmarks combo
                                cmb_bookmarks:remove_text(i-1)
                                cmb_bookmarks:insert_text(i-1, path)
                             end, {i, lbl})
      tbl:attach(btn, 2, 3, i-1, i, gtk.GTK_SHRINK, gtk.GTK_SHRINK, 0, 0)
   end
   vbox:pack_start(tbl, false, false, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 10)
   -- current path
   lbl = mygtk.Markup_new("You can change the initial path or any of the bookmarks\n" ..
                          "with the current folder in use:")
   vbox:pack_start(mygtk.align(lbl, 0, {0, 0, 10, 0}), false, false, 10)
   lbl = mygtk.PathLabel.new("size='small' style='italic'")
   lbl:set_path(path)
   lbl:connect_click(function(txt)
                        path = txt; ui_refresh(path)
                        dialog:hide(); dialog:destroy()
                     end)
   vbox:pack_start(mygtk.align(lbl.evbox, 0, {0, 0, 10, 0}), false, false, 0)
   vbox:pack_start(gtk.Label.new(), true, true, 0)
   vbox:pack_start(gtk.HSeparator.new(), false, false, 10)
   -- buttons
   vbox:pack_start(mygtk.ButtonClose(dialog), false, false, 0)
   -- packaging
   scrollwin:add_with_viewport(vbox)
   frame:add(scrollwin)
   dialog:add(frame)
   dialog:fullscreen()
   dialog:show_all()
end


--------------------------------------------------
-- init UI
local function cb_quit()
   local err, errno = config_save(CONFIG_FILE)
   if err then
      local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                                        gtk.BUTTONS_OK, "Error saving configuration")
      dlg:set_markup("An error was produced when saving the configuration.\n\n" ..
                     fmt("Error %d: %s", errno, err))
      dlg:run()
      dlg:destroy()
   end
   win:hide(); win:destroy()
end


local function ui_init()
   gtk.rc_parse_string(DEFAULT_STYLE)
   win = gtk.Window.new()
   win:set("title", "File Explorer", "border-width", 20)
   win:connect("delete-event", gtk.main_quit)

   -- header: title and quit button
   local hbox = gtk.HBox.new(false, 10)
   local evbox = gtk.EventBox.new()
   evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("black"))
   local title = mygtk.Markup_new("<span size='xx-large' color='white' weight='bold'>Iliad File Explorer</span>")
   evbox:add(title)
   hbox:pack_start(evbox, true, true, 0)
   local btn_quit = gtk.Button.new()
   local img = gtk.Image.new_from_file("_files/data/quit.png")
   btn_quit:connect("clicked", cb_quit)
   btn_quit:add(img)
   hbox:pack_end(btn_quit, false, false, 0)

   -- path
   local hb_path = gtk.HBox.new(false, 10)
   lbl_path = mygtk.PathLabel.new("size='medium' weight='bold'", "grey75")
   lbl_path:connect_click(function(txt) path = txt; ui_refresh(path) end)
   hb_path:pack_start(lbl_path.evbox, false, false, 0)
   hb_path:pack_start(gtk.Label.new(""), true, true, 0)
   local ev_free = gtk.EventBox.new()
   ev_free:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("grey75"))
   lbl_free = gtk.Label.new()
   ev_free:add(lbl_free)
   hb_path:pack_end(ev_free, false, false, 0)

   -- renderers for treeview
   local rend_select = gtk.CellRendererPixbuf.new()
   local renderer_name = gtk.CellRendererText.new()
   renderer_name:set("scale-set", true, "scale", 0.75, "xpad", 5, "xalign", 0,
                     "ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_MIDDLE)
   local renderer_center = gtk.CellRendererText.new()
   renderer_center:set("scale-set", true, "scale", 0.75, "xpad", 5, "xalign", 0.5)
   local renderer_right = gtk.CellRendererText.new()
   renderer_right:set("scale-set", true, "scale", 0.75, "xpad", 5, "xalign", 1)
   
   -- treeview: {pix_selected?, filename, type, size, mtime, fullpath, size_num}
   model = gtk.TreeStore.new("GdkPixbuf", "gchararray", "gchararray", "gchararray",
                             "gchararray", "gchararray", "glong")
   view = gtk.TreeView.new()
   view:set("rules-hint", true, "model", model)
   view:connect("button-press-event", cb_treeview_click)
   local scrollwin = gtk.ScrolledWindow.new()
   scrollwin:set("hscrollbar-policy", gtk.POLICY_NEVER,
                 "vscrollbar-policy", gtk.POLICY_AUTOMATIC)
   scrollwin:add(view)

   -- columns
   local col_select = gtk.TreeViewColumn.new_with_attributes(" ", rend_select,
                                                             "pixbuf", TV_COL_SELECTED)
   col_select:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", TV_COL_W.SELECTED)
   view:append_column(col_select)
   local cols = { {"Filename", renderer_name, TV_COL_W.NAME},
                  {"Type", renderer_center, TV_COL_W.TYPE},
                  {"Size", renderer_right, TV_COL_W.SIZE},
                  {"MTime", renderer_right, TV_COL_W.TYPE} }
   for i, coldef in ipairs(cols) do
      local col = gtk.TreeViewColumn.new_with_attributes(coldef[1], coldef[2], "text", i)
      col:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", coldef[3])
      view:append_column(col)
      if i == 1 then view:set_expander_column(col) end -- filename
   end

   -- statusbar
   local hb_status = gtk.HBox.new(false, 0)

   -- statusbar: selected files
   local frame_sel = gtk.Frame.new()
   frame_sel:set("shadow-type", gtk.SHADOW_ETCHED_OUT)
   st_selected = gtk.Label.new(" No files selected ")
   frame_sel:add(st_selected)
   hb_status:pack_start(frame_sel, true, true, 0)

   -- statusbar: actions
   local data = { {"gtk-copy", "Copy to clipboard"},
                  {"gtk-cut", "Cut to clipboard"},
                  {"gtk-paste", "Paste"},
                  {"gtk-delete", "Delete"},
                  {"gtk-refresh", "Clear selected files"},
                  {"gtk-edit", "View selected files"},
                  {"gtk-execute", "Select the action"} }
   cmb_actions = mygtk.Combo_new(data, false, cb_actions)
   hb_status:pack_start(cmb_actions, false, false, 0)

   -- statusbar: padding
   hb_status:pack_start(gtk.Label.new(""), false, false, 10)

   -- statusbar: expand / collapse rows
   local data = { {"gtk-indent", "Expand all rows"},
                  {"gtk-indent", "Expand 1 level"},
                  {"gtk-unindent", "Collapse all rows"},
                  {"gtk-dnd-multiple", "Select all"},
                  {"gtk-dnd-multiple", "Select all & Copy"},
                  {"gtk-dnd-multiple", "Select all & Cut"},
                  {"gtk-index", "Select the action"} }
   cmb_others = mygtk.Combo_new(data, opts.TREEVIEW, cb_others_actions)
   hb_status:pack_start(cmb_others, false, false, 0)

   -- statusbar: padding
   hb_status:pack_start(gtk.Label.new(""), false, false, 10)

   -- statusbar: preferences
   local btn_preferences = gtk.Button.new_from_stock("gtk-preferences")
   btn_preferences:connect("clicked", ui_preferences)
   hb_status:pack_start(btn_preferences, false, false, 10)

   -- statusbar: padding
   hb_status:pack_start(gtk.Label.new(""), false, false, 10)

   -- statusbar: bookmarks
   cmb_bookmarks = gtk.ComboBox.new_text()
   for _, txt in ipairs(opts.BOOKMARKS) do cmb_bookmarks:append_text(txt) end
   cmb_bookmarks:append_text("Bookmarks")
   cmb_bookmarks:set_size_request(125, -1)
   cmb_bookmarks:set("active", #opts.BOOKMARKS)
   cmb_bookmarks:connect("changed", cb_goto_bookmark, cmb_bookmarks)
   hb_status:pack_start(cmb_bookmarks, false, false, 0)
   
   -- widgets packaging and show
   local vbox = gtk.VBox.new(false, 20)
   vbox:pack_start(hbox, false, false, 10)
   vbox:pack_start(hb_path, false, false, 0)
   vbox:pack_start(scrollwin, true, true, 0)
   vbox:pack_start(hb_status, false, false, 0)
   win:add(vbox)
   win:fullscreen()
   win:show_all()
end


---------- Main -------------------------------------------------------
local err, errno = config_load(CONFIG_FILE)
if err then
   local dlg = gtk.MessageDialog.new(win, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                                     gtk.BUTTONS_OK, "Error loading configuration")
   dlg:set_markup("An error was produced when loading the configuration.\n\n" ..
                  fmt("Error %d: %s\nWe will build a new one with default values", errno, err))
   dlg:run()
   dlg:destroy()
end
parse_command_line(arg)
ui_init()
ui_refresh(path)

gtk.main()


----------------------------------------------------------------------
