----------------------------------------------------------------------
-- sysinfo: Show Iliad system information
--
-- Copyright (C) 2009  Iñigo Serna <inigoserna@gmail.com>
-- Time-stamp: <2009-11-22 14:19:15 inigo>


require("os")

require("alien")
require("bit")


---------- Vars -------------------------------------------------------
-- values
local refresh_its = { ["No refresh"] = 0, ["Refresh: 5 sec"] = 5,
                      ["Refresh: 10 sec"] = 10 , ["Refresh: 30 sec"] = 30,
                      ["Refresh: 1 min"] = 60, ["Refresh: 5 min"] = 300,
                      ["Refresh: 10 min"] = 600, ["Refresh: 15 min"] = 900 }
local refresh_opts = { "No refresh", "Refresh: 5 sec", "Refresh: 10 sec",
                       "Refresh: 30 sec", "Refresh: 1 min", "Refresh: 5 min",
                       "Refresh: 10 min", "Refresh: 15 min" }
local TIMEOUT = 60
local refresh_id

local bor, band, lshift = bit.bor, bit.band, bit.lshift

-- system data
local sys_datetime
local sys_name, sys_cpu, sys_cpu_bogomips, sys_version, sys_uptime, sys_battery
local storage
local mem_total, mem_used, mem_free, mem_shared, mem_buffer, mem_cached
local cpu_usr, cpu_sys, cpu_nice, cpu_idle, cpu_io, cpu_irq, cpu_softirq
local load_avg_1, load_avg_5m, load_avg_15m
local procs

-- ui
local combo, contents, dmodel, pmodel


---------- Utils ------------------------------------------------------

-- Date and time
local function get_datetime_info()
   sys_datetime = os.date("%A, %d %B %Y    %H:%M:%S")
end


-- System information
local function get_system_info()
   -- parse /proc/cpuinfo
   local f = io.open('/proc/cpuinfo', 'r')
   if (f == nil) then return nil end
   local buf = f:read('*all')
   f:close()
   _, _, sys_name = buf:find('Hardware%s+:%s+([%w%d ]+)')
   _, _, sys_cpu = buf:find('Processor%s+:%s+([%w%d-%(%) ]+)')
   _, _, sys_cpu_bogomips = buf:find('BogoMIPS%s+:%s+([%d.]+)')
   sys_name = sys_name or 'System name'
   sys_cpu = sys_cpu or 'cpu'
   sys_cpu_bogomips = sys_cpu_bogomips or '0.0'
   
   -- parse /proc/version
   local f = io.open('/proc/version', 'r')
   if (f == nil) then return nil end
   local buf = f:read('*all')
   f:close()
   sys_version= buf:sub(1, 36) or ''
end


-- Uptime
local function get_uptime_info()
   local f = io.open('/proc/uptime', 'r')
   if (f == nil) then return nil end
   local buf = f:read('*all')
   f:close() 
   _, _, sys_uptime, sys_uptime_idle = buf:find('([%d.]+)%s+([%d.]+)')
   local h, m, s
   s = tonumber(sys_uptime) or 0
   m = math.floor(s/60)
   h = math.floor(m/60)
   m = m % 60
   s = s % 60
   if h == 0 then
      sys_uptime = string.format('%d mins %2d secs', m, s)
   else
      sys_uptime = string.format('%d hours %2d mins %2d secs', h, m, s)
   end
end


-- Battery
local function _IOR(x,y,t)
   local IOCPARM_MASK = 0x7f
   local IOC_READ = 0x80000000
   local a = lshift(string.byte(x),8)
   local b = lshift(band(t, IOCPARM_MASK), 16)
   return bor(IOC_READ, a, b, y)
end

local function get_battery_info()
   -- open the library (the functions we want are in the default libs)
   local lib = alien.default
   -- get the functions we need
   local open, close, ioctl = lib.open, lib.close, lib.ioctl
   -- specify the API
   open:types  {ret='int', 'string', 'int'}
   close:types {ret='int', 'int'}
   ioctl:types {ret='int', 'int', 'int', 'ref int'}

   local ret, fd, bat_pct, bat_time
   local READ_CHARGE = _IOR('b', 1, 4)
   local READ_TIME   = _IOR('b', 2, 4)

   fd = open("/dev/battery", 0)
   ret, bat_pct = ioctl(fd, READ_CHARGE, 0)
   ret, bat_time = ioctl(fd, READ_TIME, 0)
   if bat_time == 0xffff then
      sys_battery = string.format('%d%%, now charging', bat_pct)
   else
      sys_battery = string.format("%d%%, %d:%2d hours remaining",
                                  bat_pct, math.floor(bat_time/60), bat_time%60)
   end
   close(fd)
   sys_battery = sys_battery or "no battery information"
end


-- Storage
local function get_storage_info()
   local f = io.popen('df -h', 'r')
   if (f == nil) then return nil end
   local buf = f:read('*all')
   f:close()
   storage = {}
   for dev, size, used, available, pct, mountpoint in
      buf:gmatch('([%w%d%/]+)%s+([%d%w.]+)%s+([%d%w.]+)%s+([%d%w.]+)%s+([%d%%]+)%s+([%w%d%/]+)') do
      table.insert(storage, {dev, size, used, available, pct, mountpoint})
   end
end


-- Parse top command info
local function get_top_info()
   local f = io.popen('top -b -n 1 -d 0', 'r')
   if (f == nil) then return nil end
   local buf = f:read('*all')
   f:close()

   -- memory
   _, _, mem_used, mem_free, mem_shared, mem_buffer, mem_cached =
      buf:find('Mem:%s+(%d+)K used,%s+(%d+)K free,%s+(%d+)K shrd,%s+(%d+)K buff,%s+(%d+)K cached')
   mem_total = mem_free + mem_used

   -- cpu
   _, _, cpu_usr, cpu_sys, cpu_nice, cpu_idle, cpu_io, cpu_irq, cpu_softirq =
      buf:find('CPU:%s+(%d+)%% usr%s+(%d+)%% sys%s+(%d+)%% nice%s+(%d+)%% idle%s+(%d+)%% io%s+(%d+)%% irq%s+(%d+)%% softirq')

   -- load
   _, _, load_avg_1, load_avg_5m, load_avg_15m =
      buf:find('Load average:%s+([%d.]+)%s+([%d.]+)%s+([%d.]+)')

   -- procesess
   procs = {}
   for pid, ppid, user, stat, vsz, mem, cpu, cmd in 
      buf:gmatch('%s+(%d+)%s+(%d+)%s+(%w+)%s+(%w+)%s+(%d+)%s+(%d+)%%%s+(%d+)%%%s+([%w%d_ .%/%-%[%]]+)') do
      table.insert(procs, {tonumber(pid), tonumber(ppid), user, stat, tonumber(vsz),
                           tonumber(mem), tonumber(cpu), cmd})
   end

end


local function update_data()
   get_system_info()
   get_datetime_info()
   get_uptime_info()
   get_battery_info()
   get_storage_info()
   get_top_info()
end


---------- UI ---------------------------------------------------------
local function cb_refresh_changed()
   TIMEOUT = refresh_its[combo:get_active_text()]
   if refresh_id ~= nil then 
      glib.Timeout.remove(refresh_id)
   end
   if TIMEOUT == 0 then
      refresh_id = nil
   else
      refresh_id = glib.Timeout.add(glib.PRIORITY_DEFAULT, TIMEOUT*1000, ui_refresh)
   end
end


local function ui_init()
   local win = gtk.Window.new()
   win:set("title", "System Information", "border-width", 20)
   win:connect("delete-event", gtk.main_quit)

   -- header: title, refresh combo and quit button
   local hbox = gtk.HBox.new(false, 10)
   local evbox = gtk.EventBox.new()
   evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("black"))
   local title = gtk.Label.new()
   title:set_markup("<span size='xx-large' color='white' weight='bold'>System Information</span>")
   evbox:add(title)
   hbox:pack_start(evbox, true, true, 0)

   combo = gtk.ComboBox.new_text()
   for _, txt in ipairs(refresh_opts) do combo:append_text(txt) end
   combo:set("active", 4)
   combo:connect("changed", cb_refresh_changed)
   hbox:pack_start(combo, false, false, 0)

   local btn_quit = gtk.Button.new()
   local img = gtk.Image.new_from_file("_files/data/quit.png")
   btn_quit:connect("clicked", function () win:hide(); win:destroy() end)
   btn_quit:add(img)
   hbox:pack_start(btn_quit, false, false, 0)

   -- renderers for treeview
   local renderer_left = gtk.CellRendererText.new()
   renderer_left:set("scale-set", true, "scale", 0.8, "xpad", 5, "xalign", 0,
                     "ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_END)
   local renderer_right = gtk.CellRendererText.new()
   renderer_right:set("scale-set", true, "scale", 0.8, "xpad", 5, "xalign", 1,
                      "ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_END)

   -- page 1: summary
   contents = gtk.Label.new()

   dmodel = gtk.ListStore.new("gchararray", "gchararray", "gchararray", "gchararray",
                              "gchararray", "gchararray")
   local view2 = gtk.TreeView.new()
   view2:set("rules-hint", true, "model", dmodel)
   local cols = { {"Device", renderer_left}, {"Size", renderer_right},
                  {"Used", renderer_right}, {"Available", renderer_right},
                  {"Use%", renderer_right}, {"Mounted on", renderer_left} }
   for i, coldef in ipairs(cols) do
      local col = gtk.TreeViewColumn.new_with_attributes(coldef[1], coldef[2],
                                                         "text", i-1)
      col:set("expand", true)
      view2:append_column(col)
   end

   local vbox2 = gtk.VBox.new(false, 0)
   vbox2:pack_start(contents, false, false, 0)
   vbox2:pack_start(view2, false, false, 0)
   
   -- page 2: processes
   pmodel = gtk.ListStore.new("gint", "gint", "gchararray", "gchararray",
                              "gchararray", "gchararray")
   local view = gtk.TreeView.new()
   view:set("rules-hint", true, "model", pmodel)
   local cols = { {" PID ", renderer_right}, {" PPID", renderer_right},
                  {"Mem", renderer_right}, {"CPU", renderer_right},
                  {"State", renderer_right}, {"Command", renderer_left} }
   for i, coldef in ipairs(cols) do
      local col = gtk.TreeViewColumn.new_with_attributes(coldef[1], coldef[2],
                                                         "text", i-1)
      col:set_sort_column_id(i-1)
      view:append_column(col)
   end

   local scrollwin = gtk.ScrolledWindow.new()
   scrollwin:set("hscrollbar-policy", gtk.POLICY_NEVER,
                 "vscrollbar-policy", gtk.POLICY_AUTOMATIC)
   scrollwin:add(view)

   -- notebook
   local nb = gtk.Notebook.new()
   nb:append_page(vbox2, gtk.Label.new(" Summary "))
   nb:append_page(scrollwin, gtk.Label.new(" Processes "))

   -- widgets packaging
   local vbox = gtk.VBox.new(false, 5)
   vbox:pack_start(hbox, false, false, 10)
   vbox:pack_start(nb, true, true, 10)
   win:add(vbox)

   -- show
   win:fullscreen()
   win:show_all()
end


local function ui_redraw_summary()
   local buf = "\n<span size='large'>%s</span>\n" ..
      "%s\n\n" ..
      "<span size='x-large'><b>Hardware</b></span>\n\n" ..
      "    <u>Processor:</u> %s, %s BogoMIPS\n" ..
      "    <u>Kernel:</u> %s\n\n\n" ..
      "<span size='x-large'><b>Resources</b></span>\n\n" ..
      "    <u>Uptime:</u> %s\n" ..
      "    <u>Battery:</u> %s\n" ..
      "    <u>Load Average:</u>    1 min: %.2f    5 min: %.2f    15 min: %.2f\n" ..
      "    <u>CPU:</u>    %d%% user    %d%% sys    %d%% idle    %d%% io\n" ..
      "    <u>Memory:</u>    %s KB total    %s KB free    %s KB used\n\n\n" ..
      "<span size='x-large'><b>Storage</b></span>\n"
   contents:set_markup(string.format(buf, sys_name, sys_datetime,
                                     sys_cpu, sys_cpu_bogomips, sys_version,
                                     sys_uptime, sys_battery,
                                     load_avg_1, load_avg_5m, load_avg_15m,
                                     cpu_usr, cpu_sys, cpu_idle, cpu_io,
                                     mem_total, mem_free, mem_used))

   local iter = gtk.TreeIter.new()
   dmodel:clear()
   for _, val in ipairs(storage) do
      dmodel:append(iter)
      dmodel:set(iter, 0, val[1], 1, val[2], 2, val[3], 3, val[4],
                 4, val[5], 5, val[6])
   end

end


local function sort_percent(col, iter1, iter2)
   local el1, el2 = pmodel:get(iter1, col), pmodel:get(iter2, col)
   if el1 == nil or el2 == nil then return 0 end
   el1 = tonumber(string.sub(el1, 1, -2))
   el2 = tonumber(string.sub(el2, 1, -2))
   if el1 < el2 then return -1
   elseif el1 > el2 then return 1
   else return 0 end
end


local function ui_update_processes()
   local iter = gtk.TreeIter.new()
   pmodel:clear()
   for _, val in ipairs(procs) do
      pmodel:append(iter)
      -- {pid, ppid, user, stat, vsz, mem, cpu, cmd} -> (pid, ppid, mem, cpu, stat, cmd)
      pmodel:set(iter, 0, val[1], 1, val[2], 2, val[6]..'%', 3, val[7]..'%',
                 4, val[4], 5, val[8])
   end
   pmodel:set_sort_column_id(0, gtk.SORT_ASCENDING)
   pmodel:set_sort_func(2, sort_percent, 2)
   pmodel:set_sort_func(3, sort_percent, 3)
end


function ui_refresh()
   update_data()
   ui_redraw_summary()
   ui_update_processes()
   return true
end


---------- Main -------------------------------------------------------
ui_init()
ui_refresh()
refresh_id = glib.timeout_add(glib.PRIORITY_DEFAULT, TIMEOUT*1000, ui_refresh)


----------------------------------------------------------------------
