----------------------------------------------------------------------
-- recentfiles: clean recent files list
--
-- Copyright (C) 2009  Iñigo Serna <inigoserna@gmail.com>
-- Time-stamp: <2009-11-22 14:28:07 inigo>


require("lfs")
require("posix")


---------- Vars -------------------------------------------------------
-- globals
local RECENT_DIR = "/mnt/settings/recentdocuments"

local files = {}
local files_len = 0


---------- Utils ------------------------------------------------------
local function get_recent_items(path)
   local fs = {}
   for f in lfs.dir(path) do
      if f ~= "." and f ~= ".." then
         local xxx = {}
         local linkpath = path .. '/' .. f
         local attr = lfs.attributes(linkpath)
         local fullpath = posix.readlink(linkpath)
         if fullpath ~= nil then
            _, _, xxx.path, xxx.file = string.find(fullpath, "(.+)/(.+)")
            xxx.mtime = attr.modification or 0
            fs[f] = xxx
         end
      end
   end
   return fs
end

local function get_table_size(t)
   local num = 0
   for k, v in pairs(t) do
      num = num + 1
   end
   return num
end

-- format_datetime: return a formated string of date
local function format_datetime(timestamp)
   return os.date('%d/%m/%Y %H:%M:%S', timestamp)
end

-- format_size: return a formated string of size
local function format_size(size)
   if size >= 1000000 then
      return string.format('%.2f', size/(1024*1024)) .. ' MB'
   elseif size >= 1000 then
      return string.format('%.2f', size/1024) .. ' KB'
   else
      return tostring(size) .. ' Bytes'
   end
end

local function print_files(files)
   for fname, attr in pairs(files) do
      io.write(fname .. ' -> ' .. attr.path .. ' / ' .. attr.file .. '\t' .. attr.mtime .. '\n')
   end
end


---------- UI ---------------------------------------------------------
local win = gtk.Window.new()
win:set("title", "Recent files", "border-width", 20)
win:connect("delete-event", gtk.main_quit)

-- header: title and quit button
local hbox1 = gtk.HBox.new(false, 10)
local evbox = gtk.EventBox.new()
evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("black"))
local title = gtk.Label.new()
title:set_markup("<span size='xx-large' color='white' weight='bold'>Recent files</span>")
evbox:add(title)
hbox1:pack_start(evbox, true, true, 0)

local btn_quit = gtk.Button.new()
local img = gtk.Image.new_from_file("_files/data/quit.png")
btn_quit:connect("clicked", function () win:hide(); win:destroy() end)
btn_quit:add(img)
hbox1:pack_start(btn_quit, false, false, 0)

-- model and treeview
local model = gtk.ListStore.new("gchararray", "gboolean", "gchararray")
local view = gtk.TreeView.new()
local iter = gtk.TreeIter.new()
local rend_file = gtk.CellRendererText.new()
rend_file:set("scale-set", true, "scale", 0.85,
              "ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_MIDDLE)
local col_file = gtk.TreeViewColumn.new_with_attributes("Filename", rend_file, "text", 0)
col_file:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", 615, "alignment", 0.5)
view:append_column(col_file)
local rend_delete = gtk.CellRendererToggle.new()
rend_delete:connect("toggled",
    function(data, path)
       model:get_iter_from_string(iter, path)
       model:set(iter, 1, not model:get(iter, 1))
    end)
local col_delete = gtk.TreeViewColumn.new_with_attributes("Remove?", rend_delete, "active", 1)
col_delete:set("alignment", 0.5)
view:append_column(col_delete)
view:set("rules-hint", true, "model", model)

-- fill model
function populate_model()
   files = get_recent_items(RECENT_DIR)
   files_len = get_table_size(files)
   -- print_files(files)
   local ff = {}
   for f, a in pairs(files) do table.insert(ff, f) end
   table.sort(ff, function(s1, s2) return s1 > s2 end)
   model:clear()
   for _, f  in ipairs(ff) do
      model:append(iter)
      model:set(iter, 0, files[f].file, 1, false, 2, f)
   end
end

-- description
local label = gtk.Label.new()
label:set_markup("<i>Select the files you want to remove</i>")

-- buttons
local hbox2 = gtk.HBox.new(true, 25)
local btn_markall = gtk.Button.new_with_mnemonic("Mark all")
btn_markall:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          model:set(iter, 1, true)
       end
    end)
hbox2:add(btn_markall)
local btn_clearall = gtk.Button.new_with_mnemonic("Clear all")
btn_clearall:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          model:set(iter, 1, false)
       end
    end)
hbox2:add(btn_clearall)
local btn_remove = gtk.Button.new_with_mnemonic("Remove")
btn_remove:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          if model:get(iter, 1) == true then
             local fullpath = RECENT_DIR .. '/' .. model:get(iter, 2)
             -- print(fullpath)
             os.remove(fullpath)
          end
       end
       populate_model()
    end)
hbox2:add(btn_remove)

-- widgets packaging
local vbox = gtk.VBox.new(false, 5)
vbox:pack_start(hbox1, false, false, 10)
vbox:pack_start(view, true, true, 0)
vbox:pack_start(label, false, false, 10)
vbox:pack_start(hbox2, false, false, 10)
win:add(vbox)

-- main loop
populate_model()
win:fullscreen()
win:show_all()


----------------------------------------------------------------------
