import os
from calibre.customize.conversion import InputFormatPlugin, OptionRecommendation

import errno
import os
import sys
import subprocess
from functools import partial
from calibre.ebooks import ConversionError, DRMError
from calibre import isosx, iswindows, islinux
from calibre import CurrentDir
from calibre.ebooks.metadata.opf2 import OPFCreator

IMPTOHTML = 'C:\\ConvertIMPGUI\\ConvertIMPGUI.exe'
popen = subprocess.Popen
if iswindows and hasattr(sys, 'frozen'):
    popen = partial(subprocess.Popen, creationflags=0x08) # CREATE_NO_WINDOW=0x08 so that no ugly console is popped up

def imptohtml(output_dir, pdf_path):
    '''
    Convert the pdf into html using the pdftohtml app.
    This will write the html as index.html into output_dir.
    It will also wirte all extracted images to the output_dir
    '''

    if isinstance(pdf_path, unicode):
        pdf_path = pdf_path.encode(sys.getfilesystemencoding())
    if not os.access(pdf_path, os.R_OK):
        raise ConversionError('Cannot read from ' + pdf_path)

    with CurrentDir(output_dir):
        index = os.path.join(os.getcwd(), 'index.html')
        # This is neccessary as pdftohtml doesn't always (linux) respect absolute paths
        pdf_path = os.path.abspath(pdf_path)
        cmd = [IMPTOHTML, pdf_path, index]

        try:
            p = popen(cmd, stderr=subprocess.PIPE)
        except OSError, err:
            if err.errno == 2:
                raise ConversionError(_('Could not find '+IMPTOHTML+', check it is in your PATH'))
            else:
                raise

        while True:
            try:
                ret = p.wait()
                break
            except OSError, e:
                if e.errno == errno.EINTR:
                    continue
                else:
                    raise

        if ret != 0:
            err = p.stderr.read()
            raise ConversionError(err)
        if not os.path.exists(index) or os.stat(index).st_size < 100:
            raise DRMError()

        with open(index, 'r+b') as i:
            raw = i.read()
            raw = '<!-- created by calibre\'s pdftohtml -->\n' + raw
            i.seek(0)
            i.truncate()
            i.write(raw)



class IMPXInput(InputFormatPlugin):

  name                = 'IMP Input Conversion plugin' # Name of the plugin
  description         = 'Convert IMP to HTML'
  author              = 'Jason Cassidy' # The author of this plugin
  version             = (0, 0, 1)   # The version number of this plugin
  file_types          = set(['imp']) # The file types that this plugin will be applied to

  def convert(self, stream, options, file_ext, log, accelerators):
    log.debug('Converting file to html...')
    # The main html file will be named index.html
    imptohtml(os.getcwd(), stream.name)

    from calibre.ebooks.metadata.meta import get_metadata
    log.debug('Retrieving document metadata...')
    mi = get_metadata(stream, 'imp')
    opf = OPFCreator(os.getcwd(), mi)

    manifest = [('index.html', None)]

    images = os.listdir(os.getcwd())
    images.remove('index.html')
    for i in images:
        manifest.append((i, None))
    log.debug('Generating manifest...')
    opf.create_manifest(manifest)

    opf.create_spine(['index.html'])
    log.debug('Rendering manifest...')
    with open('metadata.opf', 'wb') as opffile:
        opf.render(opffile)

    return os.path.join(os.getcwd(), 'metadata.opf')