#!/usr/bin/env lua
 
require ("lfs")
local gtk = require("lgob.gtk")             -- Binding for GTK+
local gdk = require("lgob.gdk")             -- Binding for GDK
require("print_r")              -- debug stuff
 
module(..., package.seeall)
 
local builder = require 'gui'     -- This is the generate GUI implementation
local sf = string.format          -- Shortcut
 
-- callbacks: the 'o' parameter is the app object
function date(o) 
  local date = os.date('%a %d %b %Y, %H:%M ')
  o.buffer:insert_at_cursor(date, string.len(date))
end
function delete(o)
  o.buffer:delete_selection(true, true)
end
function copy(o)
  o.buffer:copy_clipboard(o.clipboard)
end
function cut(o)
  o.buffer:cut_clipboard(o.clipboard, true)
  o.text:set_cursor_visible(true)
end
function paste(o)
  o.buffer:paste_clipboard(o.clipboard, nil, true)
end
 
local conv = {
  Enter = '\n',
  Tab = '\t',
}
 
function mod_set(o, k)
  local table =  o.modTable[k]
  if table then
    o.modKey = k
    for i, x in ipairs(table) do
      local q = o.modKeys[i]            -- shortcut`
      q.b:set_label(x)                  -- new label
      -- remove old handler, add and remember new one
      q.b:disconnect(q.h)               
      q.h = q.b:connect('clicked', key, {o,x})
    end
    o.mod = 2             -- ready for alternative key
  else
    mod_end(o)            -- no alternatives for this key
  end

end

function mod_end(o)
  if o.mod == 3 then 
    o.mod = 0; 
    return 
  end

  if o.mod == 2   then 

    -- restore original keys
    local table =  o.modTable[o.modKey]
    for i = 1, #table do
      local q = o.modKeys[i]            -- shortcut
      q.b:set_label(q.k)                -- original label
      -- remove current callback, and resore old one
      q.b:disconnect(q.h)
      q.h = q.b:connect('clicked', key, {o,q.k})
    end

  end

  o.mod = 3
  o.Mod:set_active(false)       -- triggers key callback!
  o.mod = 0

end

local upper = {
  ['à']='À', ['â']='Â', ['ã']='Ã',
  ['ç']='Ç',
  ['é']='É', ['è']='É', ['ê']='Ê', ['ë']='Ë',
  ['î']='Î', ['ï']='Î',
  ['ó']='Ó', ['ô']='Ô', ['ö']='Ö', ['õ']='Õ', ['œ']='Œ',
  ['ù']='Ù', ['û']='Û', ['ü']='Ü',
}

function key(d) 
  local o, k = d[1], d[2]
  k   = conv[k] or k
  local buf = o.buffer
  local b,e = gtk.TextIter:new(), gtk.TextIter:new()
  local sel = buf:get_selection_bounds(b,e)
  
  ------ normal characters -------------------------------
  if not o.special[k] then
    
    if (o.mod == 1) then
      mod_set(o, k)
    else
      if (o.shift and o.caps) or (not o.shift and not o.caps) then 
        k = string.lower(k) 
      else
        k=upper[k] or k
      end
      -- set_active causes the button handler to be called directly,
      -- interupting this one !?!?! So we must be 're-entrant'
      if sel then buf:delete(b,e) end     -- typing deletes selected text, if any
      buf:insert_at_cursor(k, string.len(k)) 
      if o.shift then o.Shift:set_active(false) end    -- o.shift is reset by the handler
      if o.mod ~= 0  then mod_end(o) end  
    end

  ------ Special keys -------------------------------
  elseif k == 'Shift' then
    o.shift = not o.shift
  elseif k == 'Caps' then
    o.caps = not o.caps
  elseif k == 'Mod' then
    if o.mod == 0 then
      o.mod = 1
    else
      mod_end(o)
    end
  elseif k == 'BS' or k == 'Del' then
    -- sel, b, e = buf:get_selection_bounds() would be nice
    if not sel then
      -- probably problem in lgob: b:backward_char() does not work
      if k == 'BS' then gtk.TextIter.backward_char(b) end
      if k == 'Del' then gtk.TextIter.forward_char(e) end
    end
    buf:delete(b,e)     
  end
end
 
function show(o) 
  o.main:show()
  o.small:hide()
  o.main:set_focus(o.text)      -- focus on text widget to show the cursor
end
function hide(o) 
  o.main:hide()
  o.small:show_all()
  local w,h = o.small:get_size()
  o.small:move(gdk.screen_width()-w, 1)      -- 1, because 0 does not work !?!?!?
end
 
function quit(o) 
  local from = gtk.TextIter.new()
  local to = gtk.TextIter.new()
  o.buffer:get_bounds(from, to)
  local text = o.buffer:get_text(from, to)
  o.dataFile:write(text)
  o.dataFile:close()
  gtk.main_quit()
end
 
local kx, ky = 0, 0
 
local function addKey(o, w, k, special) 
  local b
  if k then
    if special then o.special[k]=true end
    if k=='Shift' or k=='Caps' or k=='Mod' then
      b = gtk.ToggleButton:new()
      o[k] = b  -- remember widget to be able to set it's state
    else
      b = gtk.Button:new()
    end
    b:set_label(k)
    b:set_can_focus(false)      -- avoids removing the focus from the TextView
    local h = b:connect('clicked', key, {o,k})
    if ky == 0 then       -- top row?
      o.modKeys[#o.modKeys+1] = {b=b, h=h, k=k}      -- remember widget, handler and label
    end
  else
    b = gtk.Label:new()
  end
 

  o.keyboard:attach_defaults(b,kx, kx+w, ky, ky+1)
  kx = kx + w
end
 
local function setKeys(o) 
 
  addKey(o, 2, '~')
  addKey(o, 2, '!')
  addKey(o, 2, '@')
  addKey(o, 2, '#')
  addKey(o, 2, '$')
  addKey(o, 2, '%')
  addKey(o, 2, '^')
  addKey(o, 2, '&')
  addKey(o, 2, '*')
  addKey(o, 2, '(')
  addKey(o, 2, ')')
  addKey(o, 2, '_')
  addKey(o, 2, '+')
  addKey(o, 4, 'Del', true)
 
  ky = ky+1; kx=0       -- next row
  addKey(o, 2, '`')
  addKey(o, 2, '1')
  addKey(o, 2, '2')
  addKey(o, 2, '3')
  addKey(o, 2, '4')
  addKey(o, 2, '5')
  addKey(o, 2, '6')
  addKey(o, 2, '7')
  addKey(o, 2, '8')
  addKey(o, 2, '9')
  addKey(o, 2, '0')
  addKey(o, 2, '-')
  addKey(o, 2, '=')
  addKey(o, 4, 'BS', true)
  ky = ky+1; kx=0       -- next row
 
  addKey(o, 3, 'Tab')     -- tab
  addKey(o, 2, 'Q')
  addKey(o, 2, 'W')
  addKey(o, 2, 'E')
  addKey(o, 2, 'R')
  addKey(o, 2, 'T')
  addKey(o, 2, 'Y')
  addKey(o, 2, 'U')
  addKey(o, 2, 'I')
  addKey(o, 2, 'O')
  addKey(o, 2, 'P')
  addKey(o, 2, '[')
  addKey(o, 2, ']')
  addKey(o, 3, '\\')
  ky = ky+1; kx=0       -- next row
 
  addKey(o, 4, 'Caps', true)     -- caps lock
  addKey(o, 2, 'A')
  addKey(o, 2, 'S')
  addKey(o, 2, 'D')
  addKey(o, 2, 'F')
  addKey(o, 2, 'G')
  addKey(o, 2, 'H')
  addKey(o, 2, 'J')
  addKey(o, 2, 'K')
  addKey(o, 2, 'L')
  addKey(o, 2, ';')
  addKey(o, 2, "'")
  addKey(o, 4, 'Enter', true)
  ky = ky+1; kx=0       -- next row
 
  addKey(o, 5, "Shift", true)     --shift
  addKey(o, 2, 'Z')
  addKey(o, 2, 'X')
  addKey(o, 2, 'C')
  addKey(o, 2, 'V')
  addKey(o, 2, 'B')
  addKey(o, 2, 'N')
  addKey(o, 2, 'M')
  addKey(o, 2, '<')
  addKey(o, 2, '>')
  addKey(o, 2, '?')
  addKey(o, 5, 'Mod', true)     --shift
 
  ky = ky+1; kx=0       -- next row
  addKey(o, 6, nil)
  addKey(o, 12, ' ')
  addKey(o, 1, nil)
  addKey(o, 2, ',')
  addKey(o, 2, '.')
  addKey(o, 2, '/')
  addKey(o, 5, nil)
end
 
function new(self,o) 
  o = o or {}
  setmetatable(o, self)
  self.__index = self
 
  -- Callback table: these function are called by GUI events
  local CB_mt = {}
  function CB_mt.__index(t, key)
    return function(a,b,c)
      print ("undefined callback: "..key, a, b, c)
    end
  end
  o.cb = { }
  setmetatable(o.cb, CB_mt)
 
  gtk.rc_parse('myrc')            
 
  -- attach calbacks to actions entered in Glade
  o.cb['quit'] = quit
  o.cb['show'] = show
  o.cb['hide'] = hide
  o.cb['date'] = date
  o.cb['delete'] = delete
  o.cb['paste'] = paste
  o.cb['cut'] = cut
  o.cb['copy'] = copy
  o.tree = builder(o.cb, o)         -- Contruct the GUI 
pr(o.tree)
  o.main  = o.tree['main']
  o.keyboard = gtk.Table:new(5, 30, true)
  o.tree['main.keyframe']:remove(o.tree['main.keyframe.keyboard'])
  o.tree['main.keyframe']:add(o.keyboard)
 
  o.clipboard = gtk.clipboard_get(0)
  o.small   = o.tree['small']
  o.text    = o.tree['main.text']
  o.buffer  = o.text:get_buffer()

  -- special keys
  o.shift   = false
  o.caps    = false

  o.mod     = 0         -- 0=not active, 1=wait for key, 2=active
  o.modKey  = nil
  o.modKeys = {}      

  o.modTable = {
    A = {'á','à','â','ã'},
    C = {'ç',},
    E = {'é','è','ê','ë'},
    O = {'ó','ô','ö','õ','œ'},
    U = {'ú','ù','û','ü',},
    I = {'ì','î','ï',},
    N = {'ñ',},
    S = {'ß',},
    ['!'] = {'¡',},
    ['?'] = {'¿',},
  }      
 
  o.special = {}
  setKeys(o)

  local text=nil
  o.dataFile = io.open('MiniPad.txt', 'r')
  if o.dataFile then
    text  = o.dataFile:read('*all')
    o.dataFile:close()
  end
  o.dataFile = io.open('MiniPad.txt', 'w')
  assert(lfs.lock(o.dataFile, 'w'))     -- allow only one instance to run
 
  text = text or '<Your text here>'
  o.buffer:set_text(text,string.len(text))
 
  return o
end
 
function timer(o) 
  o.tree['small.Show']:set_label('Notes')       -- cause refresh
  return true
end

function run(o) 
  o.small:set_position(gtk.WIN_POS_NONE)
  --o.main:fullscreen(true)
  o.main:show_all()
  o:show()
  glib.Timeout.add(glib.PRIORITY_DEFAULT, 4000, timer, o)
  gtk.main()
end
