"""
    chapterbook.py -- a package to simplify usage of pylrs and streamline
               	      creation of LRF e-Books for the Sony PRS-500.
"""

#
# Acknowledgements:
#   This software would not have been possible without the efforts
#   of Mike Higgins (Falstaff), author of pylrs.
#   
# Copyright (c) 2007 EatingPie
#
#	You know the drill.  If you have any inkling of using this, it's
#   provided as-is and I am not responsible for an damage caused by
#   the use of the software.  This includes, but is not limited to,
#   the downfall of governments, the destruction of economies, the
#   collapse of infrastructure, etc.
#
# VERSIONS
#
#	2.0.3 - Add <blockquote> tag and offset of 40+20 by default
#
#   2.0.2 - Added function to remove all HTML Tags that are not
#           handled.
#           Fixed explanation of baselineskip (it's line spacing not
#           paragraph spacing).
#
#   2.0.1  - Fixed indentation error in text_to_unicode()
#
#

from sys				import *
from pylrs.pylrs		import *
from textconvert		import *

#
# Verson Number
#
PIELRF_LIB_VERSION = "2.0.3"

#############################################################
# class PageStyles():										#
#															#
#		Encapsulation for TextStyles and PageStyles.		#
#															#
#############################################################
class PageStyles():

	blockStyle    = None
	quoteStyle    = None

	paragraph     = None
	centered      = None
	paragraphLJ   = None
	paragraphH1   = None
	centeredH1    = None
	chapterHead   = None
	tocEntry      = None
	tocHead       = None
	header        = None

# endclass PageStyles()

#############################################################
#############################################################

#############################################################
# class ChapterBook											#
#															#
#		An encapsulation around Book(), taking options		#
#		portion of cmdopts structure as parameters.			#
#															#
#############################################################
class ChapterBook(Book):

	#
	# Encapsulating PageStyles is the main function of ChapterBook
	#
	# Required Parameters
	#
	# Screen/Margins
	#	cmdopts.textheight
	#	cmdopts.textwidth
	#	cmdopts.topmargin
	#	cmdopts.sidemargin
	#
	# Font Size / Weight
    #	cmdopts.headerheight
	#	cmdopts.textsize
	#	cmdopts.textweight
	#	cmdopts.headersize
    #	cmdopts.headerweight
	#	cmdopts.chaptersize
	#	cmdopts.chapterweight
	#
	# Paragraph Indent / Line Spacing
	#	cmdopts.parindent
    #	cmdopts.baselineskip
	#

	styles = None

	#############################################################
	# __init__()												#
	#############################################################
	def __init__(self, options, **settings) :

		ps                = dict(textwidth=options.textwidth,
								 textheight=options.textheight)
		ps['topmargin']   = options.topmargin
		ps['headheight']  = options.headerheight

		# Sweeten up the title
		options.title     = convert_title_text(options.title)

		quoteoffset       = options.sidemargin + options.quoteoffset
		#
		# Title and author sort options are ignored for now, since
		# they don't seem to have any effect, or serve a different
		# purpose than I understood.
		#
		Book.__init__(self,
					  pagestyledefault= ps,
					  title=            options.title,
					  author=           options.author,
					  category=         options.category,
					  isbn=             options.isbn,
					  bookid=           options.bookid,
					  **settings)

		s = PageStyles()
		s.blockStyle   = BlockStyle(sidemargin=  options.sidemargin)
		s.quoteStyle   = BlockStyle(sidemargin=  quoteoffset)

		s.paragraph    = TextStyle(parindent=    options.parindent,
								   baselineskip= options.baselineskip,
								   fontsize=     options.textsize,
								   fontweight=   options.textweight)
		s.centered     = TextStyle(align="center",
								   baselineskip= options.baselineskip,
								   fontsize=     options.textsize,
								   fontweight=   options.textweight)
		s.paragraphLJ  = TextStyle(baselineskip= options.baselineskip,
								   fontsize=     options.textsize,
								   fontweight=   options.textweight)

		s.paragraphH1  = TextStyle(parindent=    options.parindent,
								   baselineskip= options.baselineskip,
								   fontsize=     options.chaptersize,
								   fontweight=   options.chapterweight)
		s.centeredH1   = TextStyle(align="center",
								   fontsize=     options.chaptersize,
								   fontweight=   options.chapterweight)

		s.chapterHead  = TextStyle(align="center",
								   fontsize=     options.chaptersize,
								   fontweight=   options.chapterweight)

		s.header       = TextStyle(align='foot',
								   fontsize=     options.headersize,
								   fontweight=   options.headerweight)

		s.tocEntry     = TextStyle(fontsize=     options.textsize,
								   fontweight=   options.textweight)

		s.tocHead      = s.chapterHead

		self.styles = s

	#enddef __init__()

	#############################################################
	# def make_title_sort():									#
	#															#
	#############################################################
	def make_title_sort(self, title):

		titlesort  = title
		titlesplit = title.split(" ")

		# Title with one name is just returned
		if len(titlesplit) <= 1 :
			return title

		if titlesplit[0] == "The" or titlesplit[0] == "the" or \
		   titlesplit[0] == "A"   or titlesplit[0] == "a" :
			del titlesplit[0]
			titlesort = " ".join(titlesplit)
		#endif

		return titlesort

	# enddef make_title_sort()

	#############################################################
	# def make_author_sort():									#
	#															#
	#############################################################
	def make_author_sort(self, author):

		authorsplit  = author.split(" ")
		authorlast   = len(authorsplit)-1
		# One-name author
		if authorlast <= 0 :
			return author

		lastname     = authorsplit.pop(authorlast)
		authorsort   = lastname+", "+" ".join(authorsplit)

		return authorsort

	# enddef make_author_sort()

	#############################################################
	# def TocPage():											#
	#															#
	#############################################################
	def TocPage(self, headertext=None, **settings):
		p = TocPage(self, headertext, **settings)
		self.append(p)
		return p
	# enddef TocPage()

	#############################################################
	# def Chapter():											#
	#															#
	#############################################################
	def Chapter(self, headertext=None, **settings):
		p = Chapter(self, headertext, **settings)
		self.append(p)
		return p
	# enddef Chapter()

#endclass ChapterBook()


#############################################################
#############################################################


#############################################################
# class Chapter												#
#############################################################
class Chapter(Page):

	#
	# Constants - Bit Vectors
	#
	# Text Format/Style (not used as a vector currently)
	#
	ROMAN        = 0
	ITALIC       = 1
	BOLD         = 2
	SUB          = 4
	SUP          = 8
	QUOTE        = 16
	#
	# Layout
	#
	STANDARD     = 0
	CENTERED     = 1
	HEADER       = 2

	#
	# Instance Variables
	#
	first        = True
	layout       = STANDARD
	style        = ROMAN

	book         = None
	textBlock    = None
	chapterBlock = None
	TargetBlock  = None


	#############################################################
	# __init__()												#
	#############################################################
	def __init__(self, book, headertext=None, **settings) :
		
		#
		# Initialization is somewhat redundant, but declaring
		# the things above makes things a bit clearer
		#
		self.book         = book
		self.first        = True
		self.textBlock    = None
		self.chapterBlock = None
		self.layout       = self.STANDARD
		self.style        = self.ROMAN

		styles = book.styles
		if headertext :
			#
			# Make the Header, since the text was provided
			# It requires a style applied to the whole Page
			#
			header = Paragraph().append(headertext)
			bs     = styles.blockStyle
			hs     = styles.header
			hb     = TextBlock(hs, bs).append(header)
			hdr    = Header()
			hdr.PutObj(hb)
			ps     = self.book.PageStyle(header=hdr, **settings)
		else:
			ps     = book.PageStyle(**settings)
		#endif

		Page.__init__(self, ps, **settings)

		#
		# This is essentially an empty textblock at the top of
		# the page, which acts as a target for jump buttons.
		# Basically, when you select a TOC entry, it jumps to
		# this TextBlock -- which happens to be the very very
		# beginning of the chapter.
		#
		bs = styles.blockStyle
		ts = styles.centered
		self.TargetBlock = self.TextBlock(ts, bs);

	# enddef  __init__


	#############################################################
	# def strip_heading():										#
	#															#
	#	Even though everything has been converted, chapter		#
	#	titles must be stripped of lots of excess.				#
	#															#
	#############################################################
	def strip_heading(self, line):

		line = line.replace("\r\n"," ")
		line = line.replace("\n"," ")
		spaceline = line.replace("  ", " ")
		while spaceline != line :
			line      = spaceline
			spaceline = line.replace("  ", " ")
		return line

	# enddef strip_heading()

	#############################################################
	# def next_layout():										#
	#############################################################
	def next_layout(self, new_layout, isChapter=False) :

		if isChapter :
			new_layout |= self.CENTERED | self.HEADER

		#
		# the whole point of this is to NOT generate a new textblock
		# unless we absolutely need to!
		#
		if (not self.first) and \
			(self.textBlock != None) and \
			(new_layout     == self.layout) :
			return self.textBlock.Paragraph()

		#
		# Set Up paragraph based on flags
		# Initial values will be not centered/not header.
		#
		styles   = self.book.styles
		newStyle = styles.paragraph
		newBlock = styles.blockStyle
		if new_layout & self.QUOTE :
			newBlock = styles.quoteStyle
		if new_layout & self.CENTERED :
			if new_layout & self.HEADER :
				newStyle = styles.centeredH1
			else :
				newStyle = styles.centered
		else :
			if new_layout & self.HEADER :
				newStyle = styles.paragraphH1
			else :
				newStyle = styles.paragraph

		self.textBlock = self.TextBlock(newStyle, newBlock)
		self.first     = False
		self.layout    = new_layout;

		return self.textBlock.Paragraph()

	# enddef next_layout()

	#############################################################
	#############################################################

	#############################################################
	# def Heading():											#
	#															#
	#############################################################
	def Heading(self, line):

		line = self.strip_heading(line)

		self.Body(line, isChapter=True)

		#
		# Save the TextBlock for return, in case a TOC wants to
		# point here
		#
		headingTarget = self.textBlock

		#
		# This will insert a standard sized line-break
		# between the chapter heading and the following text.
		# Just generating / inserting a blank paragraph does this
		# 
		nextLayout    = self.STANDARD
		paragraph     = self.next_layout(nextLayout, isChapter=False)

		return headingTarget

	# enddef Heading()


	#############################################################
	# def Body():												#
	#############################################################
	def Body(self, line, isChapter=False):

		italic_bgn    = re.compile("i>",           re.I|re.M)
		italic_end    = re.compile("/i>",          re.I|re.M)
		bold_bgn      = re.compile("b>",           re.I|re.M)
		bold_end      = re.compile("/b>",          re.I|re.M)
		em_bgn        = re.compile("em>|cite>",    re.I|re.M)
		em_end        = re.compile("/em>|/cite>",  re.I|re.M)
		#
		sub_bgn       = re.compile("sub>",         re.I|re.M)
		sub_end       = re.compile("/sub>",        re.I|re.M)
		sup_bgn       = re.compile("sup>",         re.I|re.M)
		sup_end       = re.compile("/sup>",        re.I|re.M)
		#
		hdr_bgn       = re.compile("h.>",          re.I|re.M)
		hdr_end       = re.compile("/h.>",         re.I|re.M)
		#
		center_bgn    = re.compile("center>",      re.I|re.M)
		center_end    = re.compile("/center>",     re.I|re.M)
		#
		quote_bgn     = re.compile("blockquote>",  re.I|re.M)
		quote_end     = re.compile("/blockquote>", re.I|re.M)
		#
		br_bgn        = re.compile("br>",          re.I|re.M)
		br_end        = re.compile("/br>",         re.I|re.M)

		nextLayout    = self.layout
		if len(line) <= 0 :
			return

		#
		# Set Up Initial paragraph based on previous tags
		# The bodyTarget is the "topmost" TextBlock in the
		# paragraph (there may be multiple TextBlocks due to
		# formatting, so we need to save the first)
		#
		paragraph  = self.next_layout(nextLayout, isChapter)
		bodyTarget = self.textBlock

		#
		# Split at html open, a "<"
		#
		first_char = line[0]
		formatting = line.split("<")
		length     = len(formatting)
		start      = 0

		#
		# If the open of the paragraph was NOT a tag
		#
		initial_lt = True
		if first_char != '<' and len(formatting[0]) >= 1 :
			initial_lt = False
		#endif

		#
		# Each line in split is broken at a potential html format tag
		#
		first = True
		i     = start
		while i < length :
			if len(formatting[i]) <= 0 :
				i += 1
				continue
			#endif

			f_str      = formatting[i]
			if first :
				needs_lt = initial_lt
				first    = False
			else :
				needs_lt = True

			#
			# First character should tell us what kind of
			# formatting we are doing here
			#

			#
			# START FORMATTING
			#

			#
			# <I> - ITALICS
			# For some reason find range 0,1 doesn't work, but 0,2 does
			#
			if italic_bgn.match(f_str, 0, 2) != None :
				self.style = self.ITALIC
				f_str      = italic_bgn.sub("", f_str, 1)
				paragraph.append(Italic(f_str))
				i += 1
				continue
			#endif
			#
			# <EM>|<CITE> - Same is Italics
			#
			if em_bgn.match(f_str, 0, 5) != None :
				self.style = self.ITALIC
				f_str      = em_bgn.sub("", f_str, 1)
				paragraph.append(Italic(f_str))
				i += 1
				continue
			#endif

			#
			# <B> - BOLD
			#
			if bold_bgn.match(f_str, 0, 2) != None :
				self.style = self.BOLD
				f_str      = bold_bgn.sub("", f_str, 1)
				paragraph.append(Bold(f_str))
				i += 1
				continue
			#endif

			#
			# SUB
			# <SUB> - SUBSCRIPT
			#
			if sub_bgn.match(f_str, 0, 4) != None :
				self.style = self.SUB
				f_str      = sub_bgn.sub("", f_str, 1)
				paragraph.append(Sub(f_str))
				i += 1
				continue
			#endif

			#
			# SUPER
			# <SUP> - SUPERSCRIPT
			#
			if sup_bgn.match(f_str, 0, 4) != None :
				self.style = self.SUP
				f_str      = sup_bgn.sub("", f_str, 1)
				paragraph.append(Sup(f_str))
				i += 1
				continue
			#endif

			#
			# END FORMATTING
			#

			#
			# </I> - END ITALICS
			#
			if italic_end.match(f_str, 0, 3) != None :
				self.style = self.ROMAN
				f_str      = italic_end.sub("", f_str, 1)
				paragraph.append(f_str)
				i += 1
				continue
			#endif
			#
			# </EM>|</CITE> - END ITALICS
			#
			if em_end.match(f_str, 0, 6) != None :
				self.style = self.ROMAN
				f_str      = em_end.sub("", f_str, 1)
				paragraph.append(f_str)
				i += 1
				continue
			#endif

			#
			# </B> - END BOLD
			#
			if bold_end.match(f_str, 0, 3) != None :
				self.style = self.ROMAN
				f_str      = bold_end.sub("", f_str, 1)
				paragraph.append(f_str)
				i += 1
				continue
			#endif

			#
			# SUPER
			# </SUB> - END SUBSCRIPT
			#
			if sub_end.match(f_str, 0, 5) != None :
				self.style = self.ROMAN
				f_str      = sub_end.sub("", f_str, 1)
				paragraph.append(f_str)
				i += 1
				continue
			#endif

			#
			# SUB
			# </SUP> - END SUPERSCRIPT
			#
			if sup_end.match(f_str, 0, 5) != None :
				self.style = self.ROMAN
				f_str      = sup_end.sub("", f_str, 1)
				paragraph.append(f_str)
				i += 1
				continue
			#endif

			#
			# Fall-Thru Formatting
			#
			# Tags that apply breaks and centering may have text
			# afterward that keeps formatting, so they fall-thru to
			# printable case.
			#
			# These ALL generate new paragraphs, so we need to strip
			# any leading whitespaces, including "/r"
			#

			#
			# <H1>|<H2>|<H3>  - HEADER
			#
			# Set paragrah element to be a header.
			#
			if hdr_bgn.match(f_str, 0, 4) != None :
				nextLayout |= self.HEADER
				f_str       = hdr_bgn.sub("", f_str, 1)
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# <CENTER>  - CENTERED TEXT
			#
			# Set paragrah element to be centered.
			#
			if center_bgn.match(f_str, 0, 7) != None :
				nextLayout |= self.CENTERED
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = center_bgn.sub("", f_str, 1)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# <BLOCKQUOTE>  - QUOTED TEXT
			#
			# Set quote element to be a quote.
			#
			if quote_bgn.match(f_str, 0, 11) != None :
				nextLayout |= self.QUOTE
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = quote_bgn.sub("", f_str, 1)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# </H1>|</H2>|</H3>  - END HEADER
			#
			if hdr_end.match(f_str, 0, 5) != None :
				nextLayout &= ~self.HEADER
				f_str       = hdr_end.sub("", f_str, 1)
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# </CENTER> - END CENTERED (JUSTIFIED TEXT)
			#
			if center_end.match(f_str, 0, 8) != None :
				nextLayout &= ~self.CENTERED
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = center_end.sub("", f_str, 1)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# </BLOCKQUOTE> - END QUOTE
			#
			if quote_end.match(f_str, 0, 12) != None :
				nextLayout &= ~self.QUOTE
				paragraph   = self.next_layout(nextLayout, isChapter)
				f_str       = quote_end.sub("", f_str, 1)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				if len(f_str) <= 0 or f_str.isspace():
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# <BR> - Break
			#
			if br_bgn.match(f_str, 0, 3) != None :
				f_str       = br_bgn.sub("", f_str, 1)
				f_str       = f_str.lstrip().lstrip("/r").lstrip("/n")
				is_empty    = False
				if len(f_str) <= 0 or f_str.isspace() :
					is_empty = True

				#
				# BR simply generates a new paragraph
				#
				paragraph = self.next_layout(nextLayout, isChapter)

				if is_empty :
					i += 1
					continue
				needs_lt = False
			#endif

			#
			# The split happened on an actual "<" sign, so we
			# need to put that back and print the string, unless
			# this is a fall-thru case.
			#
			s = f_str
			if needs_lt :
				s = "<" + f_str
			if self.style == self.ROMAN :
				paragraph.append(s)
			elif self.style == self.ITALIC :
				paragraph.append(Italic(s))
			elif self.style == self.BOLD :
				paragraph.append(Bold(s))
			elif self.style == self.SUB :
				paragraph.append(Sub(s))
			elif self.style == self.SUP :
				paragraph.append(Sup(s))

			#increment
			i += 1
		#endfor

		return bodyTarget

	# enddef Body()

# endclass Chapter()

#############################################################
#############################################################

#############################################################
# class TocPage():											#
#															#
#############################################################
class TocPage(Chapter):

	tocHead    = None
	tocEntries = None

	#############################################################
	# def Heading():											#
	#############################################################
	def Heading(self, headingText):

		styles   = self.book.styles
		bs       = styles.blockStyle
		headTs   = styles.chapterHead

		self.tocHead  = self.TextBlock(headTs, bs)

		self.tocHead.Paragraph(headingText)
		self.tocHead.Paragraph()

		return self.tocHead

	#enddef Heading

	#############################################################
	# def TocEntry():											#
	#############################################################
	def TocEntry(self, text=None, page=None):

		if not self.tocEntries :
			styles          = self.book.styles
			bs              = styles.blockStyle
			tocTs           = styles.tocEntry
			self.tocEntries = self.TextBlock(tocTs, bs)
		#endif

		#
		# Can Add
		# 	1) a blank spacer
		# 	2) Text Entry
		# 	3) Toc Button
		#
		if not text :
			self.tocEntries.Paragraph()
		elif not page :
			self.tocEntries.Paragraph(text)
		else :
			target = page.TargetBlock
			button = CharButton(JumpButton(target), text)
			self.tocEntries.Paragraph(button)
		#endif

		return self.tocEntries

	# enddef TocEntry()

# endclass TocPage()

#############################################################
# def add_book_options():									#
#															#
#	These are the required options for ChapterBook()		#
#	Instantiation.  A shortcut way to fill the options.		#
#															#
#############################################################
def add_book_options(cmdline):

	#
	# Title / Author
	#
	cmdline.add_option("-t", "--title", dest="title",default="Unknown Title",
					   action="store",  type="string",
					   help="Specify Book Title, use quotes")
	cmdline.add_option("-a", "--author",dest="author",default="Unknown Author",
					   action="store",  type="string",
					   help="Specify Book Author, use quotes.")
	cmdline.add_option("--category", dest="category", default="Fiction",
					   action="store",  type="string",
					   help="Specify Book Category. DEFAULT: Fiction")
	cmdline.add_option("--bookid", dest="bookid", default="FB0123456",
					   action="store",  type="string",
					   help="Specify eReader Book ID (unnecessary).")
	cmdline.add_option("--isbn", dest="isbn", default="123-0-1234-5678-9",
					   action="store",  type="string",
					   help="Specify Book ISBN (unnecessary).")

	#
	# Screen Dimensions
	#
	cmdline.add_option("--screenheight", dest="textheight", default=690,
					   action="store",  type="int", metavar="HEIGHT (pixels)",
					   help="Height of text \"screen\" area (pixels). "
					   "Note this is NOT the actual physical screen height "
					   "but the area text can be displayed within. "
					   "Using 690x575 for the 800x600 Sony Reader. "
					   "DEFAULT: 690")
	cmdline.add_option("--screenwidth", dest="textwidth", default=575,
					   action="store",  type="int", metavar="WIDTH (pixels)",
					   help="Width of text \"screen\" area (pixels). "
					   "DEFAULT: 575")

	#
	# Margins
	#
	cmdline.add_option("--headerheight", dest="headerheight", default=55,
					   action="store",  type="int",
					   help="Vertical height of Header (pixels). DEFAULT: 55")
	cmdline.add_option("--topmargin", dest="topmargin", default=15,
					   action="store",  type="int",
					   help="Height of Top Margin (pixels). DEFAULT: 15")
	cmdline.add_option("--sidemargin", dest="sidemargin", default=20,
					   action="store",  type="int",
					   help="Width of Side Margins (pixels). DEFAULT: 20")
	cmdline.add_option("--quoteoffset", dest="quoteoffset", default=40,
					   action="store",  type="int",
					   help="Additional Margin Depth for a Quote "
					   		"(margin=sidemargin+quoteoffset (pixels). "
					   		"DEFAULT: 40")

	#
	# Paragraph Settings
	#
	cmdline.add_option("--parindent", dest="parindent", default=200,
					   action="store",  type="int",
					   help="How far to Indent Paragraph's first line. "
					   "DEFAULT: 200")
	cmdline.add_option("--baselineskip", dest="baselineskip", default=100,
					   action="store",  type="int",
					   help="Line spacing."
					   "DEFAULT: 100, for double-spaced lines use 240.")

	#
	# Font size and boldness
	#
	cmdline.add_option("--fontsize", dest="textsize", default=95,
					   action="store",  type="int",
					   metavar="FONTSIZE (points)",
					   help="Text Font size (points) - DEFAULT: 95")
	cmdline.add_option("--fontweight", dest="textweight", default=400,
					   action="store",  type="int",
					   metavar="FONTWEIGHT (strength)",
					   help="Text Font weight - DEFAULT 400, Bold is 800.")
	#
	cmdline.add_option("--headerfontsize", dest="headersize", default=80,
					   action="store",  type="int",
					   metavar="FONTSIZE (points)",
					   help="Header Font size (points) - DEFAULT: 80")
	cmdline.add_option("--headerfontweight", dest="headerweight", default=400,
					   action="store",  type="int",
					   metavar="FONTWEIGHT (strength)",
					   help="Header Font weight - DEFAULT 400, Bold is 800.")
	#
	cmdline.add_option("--chapterfontsize", dest="chaptersize", default=120,
					   action="store",  type="int",
					   metavar="FONTSIZE (points)",
					   help="Chapter Font size (points) - DEFAULT: 120")
	cmdline.add_option("--chapterfontweight", dest="chapterweight",default=800,
					   action="store",  type="int",
					   metavar="FONTWEIGHT (strength)",
					   help="Chapter Font weight - DEFAULT 800, Roman is 400.")

# enddef add_book_options()
