/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
package java.awt;

import java.awt.AWTEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.TextEvent;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.EventListener;

public class TextField extends TextComponent {

    int columns;

    char echoChar;

    transient ActionListener actionListener;

    private static final String base = "textfield";
    private static int nameCounter = 0;

    private static final long serialVersionUID = -2966288784432217853L;

    public TextField() {
	this("", 0);
    }

    public TextField(String text) {
        this(text, (text != null) ? text.length() : 0);
    }

    public TextField(int columns) {
	this("", columns);
    }

    public TextField(String text, int columns) {
	super(text);
        this.columns = (columns >= 0) ? columns : 0;
	setName( base + nameCounter++);
	setFont(new Font("Arial", 1, 12));
	setForeground(Color.black);
	enableEvents(AWTEvent.MOUSE_EVENT_MASK +
		     AWTEvent.KEY_EVENT_MASK +
		     AWTEvent.TEXT_EVENT_MASK +
		     AWTEvent.ACTION_EVENT_MASK);
    }

    public char getEchoChar() {
	return echoChar;
    }

    public void setEchoChar(char c) {
        if (echoChar != c) {
            echoChar = c;
        }

	if (valid) {
	    invalidate();
	}
    }

    public synchronized void setEchoCharacter(char c) {
	setEchoChar(c);
    }

    public void setText(String t) {
        super.setText(t);

	if (valid) {
	    invalidate();
	}
    }

    public boolean echoCharIsSet() {
	return echoChar != 0;
    }

    public int getColumns() {
	return columns;
    }

    public synchronized void setColumns(int columns) {
	int oldVal = this.columns;
	if (columns < 0) {
	    throw new IllegalArgumentException("columns less than zero.");
	}
	if (columns != oldVal) {
	    this.columns = columns;
	    invalidate();
	}
    }

    public Dimension preferredSize(int aColumns) {
    	return getPreferredSize(aColumns);
    }

    public Dimension getPreferredSize() {
    	return getPreferredSize(columns);
    }

    public Dimension preferredSize() {
    	return getPreferredSize(columns);
    }

    public Dimension getMinimumSize(int columns) {
    	return minimumSize(columns);
    }

    public Dimension minimumSize(int columns) {
	return super.minimumSize();
    }

    public Dimension getMinimumSize() {
    	return minimumSize();
    }

    public Dimension minimumSize() {
	return (columns > 0) ?
	    minimumSize(columns) :
	    super.minimumSize();
    }

    public synchronized void addActionListener(ActionListener l) {
	if (l == null) {
	    return;
	}
	actionListener = AWTEventMulticaster.add(actionListener, l);
        newEventsOnly = true;
    }

    public synchronized void removeActionListener(ActionListener l) {
	if (l == null) {
	    return;
	}
	actionListener = AWTEventMulticaster.remove(actionListener, l);
    }

    public synchronized ActionListener[] getActionListeners() {
        return (ActionListener[])(getListeners(ActionListener.class));
    }

    public EventListener[] getListeners(Class listenerType) {
	EventListener l = null; 
	if  (listenerType == ActionListener.class) { 
	    l = actionListener;
	} else {
	    return super.getListeners(listenerType);
	}
	return AWTEventMulticaster.getListeners(l, listenerType);
    }

    protected void processEvent(AWTEvent e) {
	System.out.println(getName() + " processEvent: " + e);
        if (e instanceof KeyEvent) {
	    if (e.id == KeyEvent.KEY_TYPED) {
		char key = ((KeyEvent)e).getKeyChar();
		//System.out.println(getName() + " KEY_TYPED: " + key + " decimal: " + ((int)key));
		if (key == 8) {
		    // Backspace
		    String newText = getText();
		    if (newText.length() > 1) {
			newText = newText.substring(0, newText.length() - 1);
			setText(newText);
		    } else {
			setText("");
		    }
		} else {
		    setText(getText() + key);
		}
	    }
	} else if (e instanceof ActionEvent) {
            processActionEvent((ActionEvent)e);
        } else {
	    super.processEvent(e);
	}
    }

    protected void processActionEvent(ActionEvent e) {
	System.out.println(getName() + " processActionEvent: " + e);
        ActionListener listener = actionListener;
        if (listener != null) {
            listener.actionPerformed(e);
        }
    }

    protected String paramString() {
	String str = super.paramString();
	if (echoChar != 0) {
	    str += ",echo=" + echoChar;
	}
	return str;
    }


    private int textFieldSerializedDataVersion = 1;

    private void writeObject(ObjectOutputStream s)
      throws IOException
    {
        s.defaultWriteObject();

        AWTEventMulticaster.save(s, actionListenerK, actionListener);
        s.writeObject(null);
    }

    private void readObject(ObjectInputStream s)
      throws ClassNotFoundException, IOException
    {
        s.defaultReadObject();

        if (columns < 0) {
            columns = 0;
        }

        Object keyOrNull;
        while(null != (keyOrNull = s.readObject())) {
	    String key = ((String)keyOrNull).intern();

	    if (actionListenerK == key) {
                addActionListener((ActionListener)(s.readObject()));
            } else { 
                s.readObject();
            }
        }
    }

    public Dimension getPreferredSize(int aColumns) {
	FontMetrics fm = getFontMetrics(font);

        int w = (fm.stringWidth("M") * aColumns) + 10;
        int h = fm.getHeight() + 6;

	return new Dimension(w, h);
    }

    public void paint(Graphics g) {
	g.setColor(getForeground());
	g.fill3DRect(0, 0, width, height, false);

	g.setColor(getBackground());
	g.fillRect(3, 3, width-6, height-6);

	g.setColor(getForeground());

        FontMetrics fontmetrics = g.getFontMetrics(font);
        int baseLine = fontmetrics.getAscent() +
	    ((height - fontmetrics.getHeight()) / 2);

	String curText = getText();
	if (echoCharIsSet()) {
	    char[] ecs = new char[curText.length()];
	    char ec = getEchoChar();
	    for (int i = 0; i < ecs.length; i++) {
		ecs[i] = ec;
	    }
	    curText = new String(ecs);
	}

	g.drawString(curText, 5, baseLine);
    }

}
