/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
package java.awt;

import java.awt.event.MouseEvent;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.EventListener;

public class TextComponent extends Component {

    protected String text;

    protected int selectionStart;
    protected int selectionEnd;

    protected boolean editable = true;

    transient protected TextListener textListener;

    private static final long serialVersionUID = -2214773872412987419L;
    private int textComponentSerializedDataVersion = 1;

    TextComponent(String text) {
	this.text = (text != null) ? text : "";
	setCursor(Cursor.getPredefinedCursor(Cursor.TEXT_CURSOR));
    }

    public synchronized void setText(String t) {
	if (t == null) {
	    t = "";
	}
	if (!text.equals(t)) {
	    text = t;
	    if (textListener != null) {
		processTextEvent( new TextEvent(this, TextEvent.TEXT_VALUE_CHANGED));
	    }
	}
    }

    public synchronized String getText() {
	return text;
    }

    public synchronized String getSelectedText() {
	return getText().substring(getSelectionStart(), getSelectionEnd());
    }

    public boolean isEditable() {
	return editable;
    }

    public synchronized void setEditable(boolean b) {
        if (editable != b) {
	    editable = b;
	}
    }

    public synchronized int getSelectionStart() {
	return selectionStart;
    }

    public synchronized void setSelectionStart(int selectionStart) {
	select(selectionStart, getSelectionEnd());
    }

    public synchronized int getSelectionEnd() {
	return selectionEnd;
    }

    public synchronized void setSelectionEnd(int selectionEnd) {
	select(getSelectionStart(), selectionEnd);
    }
    
    public synchronized void select(int selectionStart, int selectionEnd) {
	int length = getText().length();
	if (selectionStart < 0) {
	    selectionStart = 0;
	}
	if (selectionStart > length) {
	    selectionStart = length;
	}
	if (selectionEnd > length) {
	    selectionEnd = length;
	}
	if (selectionEnd < selectionStart) {
	    selectionEnd = selectionStart;
	}

	this.selectionStart = selectionStart;
	this.selectionEnd = selectionEnd;
    }

    public synchronized void selectAll() {
	selectionStart = 0;
	selectionEnd = getText().length();
    }

    public synchronized void setCaretPosition(int position) {
	if (position < 0) {
	    throw new IllegalArgumentException("Position can not be less than zero.");
	}

	int length = getText().length();
	if (position > length) {
	    position = length;
	}

	select(position, position);
    }

    public synchronized int getCaretPosition() {
	int currentPosition = selectionStart;
        int currentLength = getText().length();

        if (currentPosition > currentLength) {
            return currentLength;
        } else {
	    return currentPosition;
	}
    }

    public synchronized void addTextListener(TextListener l) {
	if (l == null) {
	    return;
	}
	textListener = AWTEventMulticaster.add(textListener, l);
        newEventsOnly = true;
    }

    public synchronized void removeTextListener(TextListener l) {
	if (l == null) {
	    return;
	}
	textListener = AWTEventMulticaster.remove(textListener, l);
    }

    public synchronized TextListener[] getTextListeners() {
        return (TextListener[])(getListeners(TextListener.class));
    }

    protected void processEvent(AWTEvent e) {
	if (isEditable() && (e instanceof MouseEvent)) {
	    if (e.id == MouseEvent.MOUSE_CLICKED) {
		if (!hasFocus()) {
		    requestFocus();
		}
	    }
	} else if (e instanceof TextEvent) {
            processTextEvent((TextEvent)e);
        } else {
	    super.processEvent(e);
	}
    }

    protected void processTextEvent(TextEvent e) {
        TextListener listener = textListener;
        if ((listener != null) && (e.getID() == TextEvent.TEXT_VALUE_CHANGED)) {
	    listener.textValueChanged(e);
        }
    }

    protected String paramString() {
	String str = super.paramString() + ",text=" + getText();
	if (editable) {
	    str += ",editable";
	}
	return str + ",selection=" + getSelectionStart() + "-" + getSelectionEnd();
    }

    private void writeObject(java.io.ObjectOutputStream s)
      throws IOException 
    {
        s.defaultWriteObject();

        AWTEventMulticaster.save(s, textListenerK, textListener);
        s.writeObject(null);
    }

    private void readObject(ObjectInputStream s)
        throws ClassNotFoundException, IOException
    {
        s.defaultReadObject();

        // Make sure the state we just read in for text, 
        // selectionStart and selectionEnd has legal values
	this.text = (text != null) ? text : "";
        select(selectionStart, selectionEnd);

        Object keyOrNull;
        while(null != (keyOrNull = s.readObject())) {
	    String key = ((String)keyOrNull).intern();

	    if (textListenerK == key) {
	        addTextListener((TextListener)(s.readObject()));
            } else { 
                // skip value for unrecognized key
	        s.readObject();
            }
        }
    }

}
