/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
package java.awt;

import java.io.IOException;
import java.io.ObjectInputStream;

public class Label extends Component {

    public static final int LEFT   = 0;
    public static final int CENTER = 1;
    public static final int RIGHT  = 2;

    protected String text;

    protected int alignment;

    private static final String base = "label";
    private static int nameCounter = 0;

    private static final long serialVersionUID = 3094126758329070636L;

    public Label() {
	this("", LEFT);
    }

    public Label(String text) {
        this(text, LEFT);
    }

    public Label(String text, int alignment) {
	this.text = text;
	setAlignment(alignment);
	setName( base + nameCounter++);
	setFont(new Font("Arial", 1, 12));
	setForeground(Color.black);
    }

    public int getAlignment() {
	return alignment;
    }

    public synchronized void setAlignment(int alignment) {
	switch (alignment) {
	case LEFT:
	case CENTER:
	case RIGHT:
	    this.alignment = alignment;
	    break;
	default:
	    throw new IllegalArgumentException("Unknown alignment: " + alignment);
	}
    }

    public String getText() {
	return text;
    }

    public void setText(String text) {
	String curText = getText();

	if ((curText != text) &&
	    ((curText == null) || !curText.equals(text))) {
	    this.text = text;
	    if (valid) {
		invalidate();
	    }
	}
    }

    protected String paramString() {
	String str = ",align=";
	switch (alignment) {
	  case LEFT:   str += "left"; break;
	  case CENTER: str += "center"; break;
	  case RIGHT:  str += "right"; break;
	}
	return super.paramString() + str + ",text=" + text;
    }

    public Dimension getPreferredSize() {
	FontMetrics fm = getFontMetrics(font);

        int w = fm.stringWidth(text) + 10;
        int h = fm.getHeight() + 6;

	return new Dimension(w, h);
    }

    public void paint(Graphics g) {
	g.setColor(getForeground());

        FontMetrics fontmetrics = g.getFontMetrics(font);
        int baseLine = fontmetrics.getAscent() +
	    ((height - fontmetrics.getHeight()) / 2);

	int leftMargin = 0;
	switch (alignment) {
	case CENTER:
	    leftMargin = (width - fontmetrics.stringWidth(text)) / 2;
	    break;
	case RIGHT:
	    leftMargin = width - fontmetrics.stringWidth(text);
	    break;
	}

        g.drawString(text, leftMargin, baseLine);
    }
}
