/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
package java.awt;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.EventListener;

public class Checkbox extends Component implements ItemSelectable {

    String label;
    boolean state;
    CheckboxGroup group;

    transient ItemListener itemListener;

    private static final String base = "checkbox";
    private static int nameCounter = 0;

    private static final long serialVersionUID = 7270714317450821763L;
    private int checkboxSerializedDataVersion = 1;

    public Checkbox() throws HeadlessException {
        this("", false, null);
    }

    public Checkbox(String label) throws HeadlessException {
	this(label, false, null);
    }

    public Checkbox(String label, boolean state) throws HeadlessException {
        this(label, state, null);
    }

    public Checkbox(String label, CheckboxGroup group, boolean state) {
    	this(label, state, group);
    }

    public Checkbox(String label, boolean state, CheckboxGroup group) {
	setName( base + nameCounter++);
	this.label = label;
	this.state = state;
	this.group = group;
	if ((group != null) && state) {
	    group.setSelectedCheckbox(this);
	}
	setFont(new Font("Arial", 1, 12));
	setForeground(Color.black);
	enableEvents(AWTEvent.MOUSE_EVENT_MASK +
		     AWTEvent.ACTION_EVENT_MASK);
    }

    public String getLabel() {
	return label;
    }

    public void setLabel(String label) {
	String curLabel = getLabel();

	if ((curLabel != label) &&
	    ((curLabel == null) || !curLabel.equals(label))) {
	    this.label = label;
	    if (valid) {
		invalidate();
	    }
	}
    }

    public boolean getState() {
	return state;
    }

    public void setState(boolean state) {
	if (this.state == state) {
	    return;
	}

    	CheckboxGroup curGroup = group;
	if (curGroup != null) {
	    if (state) {
		curGroup.setSelectedCheckbox(this);
	    } else if (curGroup.getSelectedCheckbox() == this) {
		state = true;
	    }
	}
	setStateInternal(state);

	if (itemListener != null) {
	    processItemEvent( new ItemEvent(this, ItemEvent.ITEM_STATE_CHANGED, this, (state ? ItemEvent.SELECTED : ItemEvent.DESELECTED)));
	}
    }

    void setStateInternal(boolean state) {
	this.state = state;
    }

    public Object[] getSelectedObjects() {
        if (state) {
            Object[] items = { label };
            return items;
        } else {
	    return null;
	}
    }

    public CheckboxGroup getCheckboxGroup() {
	return group;
    }

    public void setCheckboxGroup(CheckboxGroup g) {
        if (group == g) {
            return;
        }

        CheckboxGroup curGroup;
        boolean curState;

	synchronized (this) {
            curGroup = this.group;
            curState = getState();

	    group = g;

	    if ((group != null) && curState) {
	        if (group.getSelectedCheckbox() != null) {
	            setState(false);
	        } else {
	            group.setSelectedCheckbox(this);
	        }
	    }
	}

        if ((curGroup != null) && curState) {
            curGroup.setSelectedCheckbox(null);
        }
    }

    public synchronized void addItemListener(ItemListener l) {
	if (l == null) {
	    return;
	}
        itemListener = AWTEventMulticaster.add(itemListener, l);
        newEventsOnly = true;
    }

    public synchronized void removeItemListener(ItemListener l) {
	if (l == null) {
	    return;
	}
        itemListener = AWTEventMulticaster.remove(itemListener, l);
    }

    public EventListener[] getListeners(Class listenerType) {
	EventListener l = null; 
	if  (listenerType == ItemListener.class) { 
	    l = itemListener;
	} else {
	    return super.getListeners(listenerType);
	}
	return AWTEventMulticaster.getListeners(l, listenerType);
    }

    protected void processEvent(AWTEvent e) {
	if (e instanceof MouseEvent) {
	    if (e.id == MouseEvent.MOUSE_CLICKED) {
		System.out.println(getName() + ": state: " + state + " processEvent: " + e);
		setState(!getState());
	    }
	} else if (e instanceof ItemEvent) {
            processItemEvent((ItemEvent)e);
        } else {
	    super.processEvent(e);
	}
    }

    protected void processItemEvent(ItemEvent e) {
        ItemListener listener = itemListener;
        if (listener != null) {
            listener.itemStateChanged(e);
        }
    }

    protected String paramString() {
	String str = super.paramString();
	String label = this.label;
	if (label != null) {
	    str += ",label=" + label;
	}
	return str + ",state=" + state;
    }

    private void writeObject(ObjectOutputStream s)
	throws java.io.IOException
    {
	s.defaultWriteObject();

	AWTEventMulticaster.save(s, itemListenerK, itemListener);
	s.writeObject(null);
    }

    private void readObject(ObjectInputStream s)
	throws ClassNotFoundException, IOException
    {
	s.defaultReadObject();

	Object keyOrNull;
	while(null != (keyOrNull = s.readObject())) {
	    String key = ((String)keyOrNull).intern();

	    if (itemListenerK == key) {
		addItemListener((ItemListener)(s.readObject()));
	    } else {
		s.readObject();
	    }
	}
    }

    public Dimension getPreferredSize() {
	FontMetrics fm = getFontMetrics(font);

        int w = fm.stringWidth(label) + 30;
        int h = fm.getHeight() + 10;

	return new Dimension(w, h);
    }

    public void paint(Graphics g) {
        FontMetrics fontmetrics = g.getFontMetrics(font);
	String curLabel = label;
        int leftMargin = width - fontmetrics.stringWidth(curLabel);
        int baseLine = fontmetrics.getAscent() +
	    ((height - fontmetrics.getHeight()) / 2);

	g.setColor(getForeground());

	if (group != null) {
	    if (getState()) {
		g.fillArc(5, baseLine - 20, 19, 19, 0, 360);
	    } else {
		g.drawArc(5, baseLine - 20, 19, 19, 0, 360);
	    }
	} else {
	    if (getState()) {
		g.fillRect(5, baseLine - 20, 20, 20);
	    } else {
		g.drawRect(5, baseLine - 20, 20, 20);
	    }
	}

        g.drawString(curLabel, leftMargin, baseLine);
    }
}
