/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
package java.awt;

import java.awt.AWTEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.EventListener;

public class Button extends Component {

    String label;
    String actionCommand;
    transient ActionListener actionListener;

    private static final String base = "button";
    private static int nameCounter = 0;

    private static final long serialVersionUID = -8774683716313001058L;
    private int buttonSerializedDataVersion = 1;

    public Button() {
	this("");
    }

    public Button(String label) {
	this.label = label;
	setName( base + nameCounter++);
	setFont(new Font("Arial", 1, 12));
	setForeground(Color.black);
	enableEvents(AWTEvent.MOUSE_EVENT_MASK + AWTEvent.ACTION_EVENT_MASK);
    }

    public String getLabel() {
	return label;
    }

    public void setLabel(String label) {
	String curLabel = getLabel();

	if ((curLabel != label) &&
	    ((curLabel == null) || !curLabel.equals(label))) {
	    this.label = label;
	    if (valid) {
		invalidate();
	    }
	}
    }

    public void setActionCommand(String command) {
        actionCommand = command;
    }

    public String getActionCommand() {
        return ((actionCommand == null) ? label : actionCommand);
    }

    public synchronized void addActionListener(ActionListener l) {
	if (l == null) {
	    return;
	}
	actionListener = AWTEventMulticaster.add(actionListener, l);
        newEventsOnly = true;
    }

    public synchronized void removeActionListener(ActionListener l) {
	if (l == null) {
	    return;
	}
	actionListener = AWTEventMulticaster.remove(actionListener, l);
    }

    public synchronized ActionListener[] getActionListeners() {
        return (ActionListener[]) (getListeners(ActionListener.class));
    }

    protected void processEvent(AWTEvent e) {
	if (e instanceof MouseEvent) {
	    if ((e.id == MouseEvent.MOUSE_CLICKED) &&
		(actionListener != null) &&
		isEnabled()) {
		processActionEvent( new ActionEvent(this,
						    ActionEvent.ACTION_PERFORMED,
						    getActionCommand()));
	    }
	} else if (e instanceof ActionEvent) {
            processActionEvent((ActionEvent)e);
        } else {
	    super.processEvent(e);
	}
    }

    protected void processActionEvent(ActionEvent e) {
        ActionListener listener = actionListener;
        if (listener != null) {
            listener.actionPerformed(e);
        }
    }

    protected String paramString() {
	return super.paramString() + ",label=" + label;
    }


    private void writeObject(ObjectOutputStream s)
      throws IOException
    {
      s.defaultWriteObject();

      AWTEventMulticaster.save(s, actionListenerK, actionListener);
      s.writeObject(null);
    }

    private void readObject(ObjectInputStream s)
      throws ClassNotFoundException, IOException
    {
	s.defaultReadObject();

	Object keyOrNull;
	while(null != (keyOrNull = s.readObject())) {
	    String key = ((String)keyOrNull).intern();

	    if (actionListenerK == key) {
		addActionListener((ActionListener)(s.readObject()));
	    } else {
		s.readObject();
	    }
	}
    }

    public Dimension getPreferredSize() {
	FontMetrics fm = getFontMetrics(font);

        int w = fm.stringWidth(label) + 25;
        int h = fm.getHeight() + 15;

	return new Dimension(w, h);
    }

    public Dimension getMinimumSize() {
	return new Dimension(100, 25);
    }

    public void paint(Graphics g) {
        g.setColor(Color.darkGray);
        g.drawRoundRect(0, 0, width, height, 10, 10);

        g.setColor(Color.gray);
        g.fillRoundRect(3, 3, width - 6, height - 6, 10, 10);

	if (isEnabled()) {
	    g.setColor(getForeground());
	} else {
	    g.setColor(Color.darkGray);
	}

        FontMetrics fontmetrics = g.getFontMetrics(font);
        int leftMargin = (width - fontmetrics.stringWidth(label)) / 2;
        int baseLine = fontmetrics.getAscent() +
	    ((height - fontmetrics.getHeight()) / 2);

        g.drawString(label, leftMargin, baseLine);
    }
}
