/*
 * xepdmgr.c
 *
 * Daemon able to autonomously update the screen of the iliad
 *
 * History:
 *      23/03/2008 DRF Initial implementation. Almost works (only sees changes
 *                     to root window), ignoring that there is no iliad code yet
 *      31/08/2008 DRF Implement the communication with the clients (root window
 *                      atoms/properties).Small fix to the Damage part.
 *       6/09/2008 DRF Make the comms with the clients more reliable.
 *       7/09/2008 DRF Implementation of the signal support and timeouts.
 *                     Implement the child process support. Implement the
 *                     messages actions.
 *       1/10/2008 HSL Fix some cut&paste errors causing too much full refreshes
 *                     Add code to kill the busy led, and to enable the keyboard
 *                     button on the tool bar
 *                 DRF Add a --version command-line switch
 *                     Implement the TRACKEDAREAS related part.
 *       4/10/2008 DRF Fix the "wrong number of parametrs" msg in --help.
 *                     Misc. fixes in the TRACKEDAREAS part.
 *      20/11/2008 DRF Add a debug option; now is quiet by default.. Started the
 *                     integration of displayMgr's udp and /dev/fb parts for the
 *                     server.
 *      08/12/2008 DRF Finish the integration of displayMgr's udp commands. Use
 *                     the ioctls directly when in server mode.
 *      11/12/2008 DRF Fixed a bug in the initialization of udp socket.
 *      12/12/2008 DRF Added the patchlevel letter to the version.
 *                     Make the dummy build less intrusive (use less ifdefs).
 *      15/12/2008 DRF ContentLister IPC channel for server mode. Remove
 *                     the references to modify the toolbar.
 *      28/03/2009 DRF Add the possibilty to launch to don't launch any program
 *                     when in wrapper mode (it waits a ctrl-c). Update
 *                     DUMMY_BUILD compilation so that it works again. Fixed a
 *                     bug that prevented launching programs when using --debug.
 *      12/04/2009 DRF Fixed a core for server mode introduced in last revision.
 *                     Add the ability to exit gracefully on sigterm. Add an
 *                     heuristic to use a smaller timeout when almost all the
 *                     screen has to be updated.
 *      25/04/2009 DRF Add EpdCancel().
 *      12/09/2009 DRF Fix auto-disabling.
 *      17/09/2009 DRF Change dirty area management from tracked areas to
 *                     screen blocks (Hansel's proposal).
 *      18/09/2009 DRF Finish the screen blocks implementation.
 *
 * Authors: DRF Dario Rodriguez dario@softhome.net
 *          HSL Hansel from mobileread.com
 * (c) 2008 Dario Rodriguez
 * This program is licensed under the terms of the GNU GPL v3 or greater.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#ifndef FIONREAD
#include <sys/filio.h>        /* FIONREAD en otros S.O. */
#endif
#include <X11/Xlib.h>
#include <X11/extensions/Xdamage.h>
#include "xepdmgr_einkfb.h"
#include "liberdm/erdm.h"
#include "liberdm/erdmServer.h"
#include "liberdm/erdminternal.h"
#include "liberipc/eripcbusyd.h"
#include "liberipc/eripccontentlister.h"

#define VERSIONMAJOR 1
#define VERSIONMINOR 9
#define VERSIONPATCHLEVEL 'i'

#define SHARED_ATOM_NAME_FOR_COMM "XEPDMGR_CMD"
#define SHARED_ATOM_NAME_FOR_ID "XEPDMGR_NEXTID"
#define MAXCMDSIZE 1024

#define PIXELTHRESHOLD (1024*768/3)
#define PARTIALPIXELTHRESHOLD (256*256)

#define FASTTIMEOUTPIXELTHRESHOLD (1024*768*8/10)

#define REFRESHTIMEOUTMS 500
#define REFRESHFASTTIMEOUTMS 500 /* Usefullness of this still has to be tested*/

#define SCREENBLOCKX 16
#define SCREENBLOCKY 16

#define UDP_MAX_SIZE 1024

#define FDRD 0
#define FDWR 1


typedef struct TypeEpdmgr {
        sEink *Eink;
        Display *dpy;
        Window root;
        Atom atom;
        Atom atom_id;
        Damage damage;
        int damage_event, damage_error;
        int xfixes_event, xfixes_error;
        int screen_blocks_size;
        char *screen_blocks;
        int screen_blocks_x2,screen_blocks_y2;
        int ux1,uy1,ux2,uy2;
        int pixel_count;
        int pending_refresh;
        int auto_disabled;
        int cmd_last_id;
        int cmd_last_lines;
        int quiet;
        erClientChannel_t contentListerChannel;
} sEpdmgr;

typedef enum EnumRefreshType {
        refreshFull=0,
        refreshTyping,
        refreshAuto
} eRefreshType;

/* Global vars */
int (*default_x_error_func)(Display *dpy, XErrorEvent *err);
volatile int flag_sigint=0;
volatile int flag_sigterm=0;
volatile int flag_sigchld=0;
char *argv0=NULL;
int signal_fd=-1;

/* Epdmgr struct manipulation */
sEpdmgr *EpdmgrInit(char *Display, int EnableIoctls);
void EpdmgrFini(sEpdmgr *Epdmgr);

/* Message processing and refresh */
void ProcessMessage(sEpdmgr *Epd);
char *RefreshType2Str(eRefreshType RefreshType);
void Refresh(sEpdmgr *Epd, eRefreshType RefreshType);
void Cancel(sEpdmgr *Epd);

/* Dirty areas processing */
void DirtyAdd(sEpdmgr *Epd, int x1, int y1, int x2, int y2);
void DirtyUpdateCount(sEpdmgr *Epd);
void DirtyClear(sEpdmgr *Epd);

/* Aux funcs */
static int fatal_error(char *string, ...);
static int x_error_func(Display *dpy, XErrorEvent *err);
static int install_signal(int Num,void (*f)(int));
static void sigint(int Num);
static void sigterm(int Num);
static void sigchld(int Num);
static int ReapChildren(int Pid);

#ifdef DUMMY_BUILD
/* Setup some macros so that we can build without irex libraries */
static int dummy_var;
#define initServer(ptr_udp_fd,b) ((*(ptr_udp_fd))=0)
#define dmMessageParser(a,b) 1
#define dmDisplay(a,b) dummy_var=0
#define erIpcStartClient(a,b) ((*(b))=0)
#define clDisplayUpdated(a) dummy_var=0
#define busySetBusy(a,b) dummy_var=0
#define EinkInit() ((sEink *)"Dummy")
#define EinkFini(a) dummy_var=0
#define EinkBlank(a) dummy_var=0
#define EinkUpdatePartial(a,b,c,d,e,f,g) dummy_var=0
#define EinkUpdateFull(a,b,c) dummy_var=0
#endif

#warning TODO: server mode doesn't notify contentLister of refreshes yet (don't know why it needs them, yet).

/* Program entry point */
int
main(int argc, char *argv[], char *envp[])
{
        sEpdmgr *Epd;
        char *display_name;
        int signal_fdpair[2];
        int dpy_fd, max_fd;
        fd_set read_set;
        XEvent event;
        XDamageNotifyEvent *de;
        int child_pid;
        char Dummy;
        int Quiet=1;
        int argv_parsed;
        int udp_fd;
        int server_mode;
        struct timeval pre,post,timeout,pending;
        char udp_msg[UDP_MAX_SIZE];
        argv0=argv[0];
        for(argv_parsed=1;argv_parsed<argc;argv_parsed++) {
                if(*(argv[argv_parsed])!='-')
                        break;
                else if(strcmp(argv[argv_parsed],"--help")==0)
                        return(fatal_error("Syntax: %s [--version] [--help] [{ --debug | -d }] [displayname [program_to_exec [program_params [...]]]]\nExample for working as a program wrapper: %s :0 ./myprogram\nExample for working w/o launching program: %s :0 -\nExample for working as server for displayMgr replacement: %s -p\n",argv[0],argv[0],argv[0],argv[0]));
                else if(strcmp(argv[argv_parsed],"--version")==0)
                        return(fatal_error("%s: version %i.%i%c\n",argv0,VERSIONMAJOR,VERSIONMINOR,VERSIONPATCHLEVEL));
                else if(strcmp(argv[argv_parsed],"--debug")==0 || strcmp(argv[argv_parsed],"-d")==0)
                        Quiet=0;
                else if(strcmp(argv[argv_parsed],"-p")==0)
                        ; /* Ignored */
                else
                        return(fatal_error("%s: Argument not understood: \"%s\"\n",argv0,argv[argv_parsed]));
        }
        if(argv_parsed==argc)
                display_name=":0";
        else
                display_name=argv[argv_parsed++];
        server_mode=(argc>argv_parsed)?0:1;
        Epd=NULL;
        udp_fd=-1;
        pipe(signal_fdpair);
        signal_fd=signal_fdpair[FDWR];
        install_signal(SIGINT,sigint);
        install_signal(SIGTERM,sigterm);
        if(!server_mode) {
                /* wrapper mode */
                if((Epd=EpdmgrInit(display_name,0))==NULL)
                        return(1);
                Epd->quiet=Quiet;
                Epd->pending_refresh=0;
                /* Start child process */
                install_signal(SIGCHLD,sigchld);
                child_pid=-1;
                if((argv_parsed<argc && strcmp(argv[argv_parsed],"-")!=0) && (child_pid=fork())==0) {
                        int fd;
                        erClientChannel_t erbusyChannel;
                        fprintf(stderr,"%s: client mode, child %s\n",argv[0],argv[argv_parsed]);
                        erIpcStartClient(ER_BUSYD_CHANNEL, &erbusyChannel);
                        busySetBusy(erbusyChannel, ccBusyState_Off);
                        /* Close unneeded fds */
                        for(fd=3;fd<1024;fd++)
                                close(fd);
                        execve(argv[argv_parsed],argv+argv_parsed,envp);
                        return(fatal_error("%s: error launching child process\n",argv[0]));
                } else if(child_pid==-1)
                        fprintf(stderr,"%s: client mode, without child\n",argv[0]);
        } else {
                /* server mode */
                child_pid=-1;
                udp_fd=-1;
                fprintf(stderr,"%s: server mode\n",argv[0]);
                if(initServer(&udp_fd,1)!=0)
                        return(fatal_error("%s: error binding the server udp socket (is there other displayMgr running?)\n",argv[0]));
                /* Note: we cannot initialize the display here as the Xserver */
                /*       is not started yet; we will initialize it when the first*/
                /*       udp message arrives */
        }
        while(!flag_sigint && !flag_sigterm) {
                if(Epd!=NULL) {
                        while(XPending(Epd->dpy)) {
                                XNextEvent(Epd->dpy,&event);
                                if(event.type==PropertyNotify && event.xproperty.window==Epd->root && event.xproperty.atom==Epd->atom) {
                                        ProcessMessage(Epd);
                                } else if(event.type==Epd->damage_event+XDamageNotify) {
                                        de=(XDamageNotifyEvent *) &event;
                                        if(!Epd->quiet)
                                                printf("Damaged: areaxy(%i,%i) areawh(%i,%i)\n",de->area.x,de->area.y,de->area.width,de->area.height);
                                        DirtyAdd(Epd,de->area.x,de->area.y,de->area.x+de->area.width-1,de->area.y+de->area.height-1);
                                        Epd->pending_refresh=1;
                                        /* Reset the timeout */
                                        if(Epd->pixel_count<FASTTIMEOUTPIXELTHRESHOLD)
                                                pending.tv_sec=REFRESHTIMEOUTMS/1000,pending.tv_usec=(REFRESHTIMEOUTMS%1000)*1000;
                                        else
                                                pending.tv_sec=REFRESHFASTTIMEOUTMS/1000,pending.tv_usec=(REFRESHFASTTIMEOUTMS%1000)*1000;
                                }
                        }
                        dpy_fd=ConnectionNumber(Epd->dpy);
                        FD_ZERO(&read_set);
                        FD_SET(dpy_fd,&read_set);
                        max_fd=dpy_fd;
                        FD_SET(signal_fdpair[FDRD],&read_set);
                        max_fd=(signal_fdpair[FDRD]>max_fd)?signal_fdpair[FDRD]:max_fd;
                        if(Epd->pending_refresh && !Epd->auto_disabled) {
                                gettimeofday(&pre,NULL);
                                timeout.tv_sec=pending.tv_sec;
                                timeout.tv_usec=pending.tv_usec;
                        }
                } else {
                        FD_ZERO(&read_set);
                        FD_SET(signal_fdpair[FDRD],&read_set);
                        max_fd=signal_fdpair[FDRD]>max_fd;
                }
                if(udp_fd!=-1) {
                        FD_SET(udp_fd,&read_set);
                        max_fd=(udp_fd>max_fd)?udp_fd:max_fd;
                }
                if(select(max_fd+1,&read_set,NULL,NULL,(Epd!=NULL && Epd->pending_refresh && !Epd->auto_disabled)?&timeout:(struct timeval *)NULL)<0)
                        continue;
                /* Check for signal "writes" */
                if(FD_ISSET(signal_fdpair[FDRD],&read_set))
                        read(signal_fdpair[FDRD],&Dummy,1);
                if(Epd!=NULL && Epd->pending_refresh && !Epd->auto_disabled) {
                        gettimeofday(&post,NULL);
                        /* Calc. the delta */
                        if(post.tv_usec>=pre.tv_usec) {
                                timeout.tv_sec=post.tv_sec-pre.tv_sec;
                                timeout.tv_usec=post.tv_usec-pre.tv_usec;
                        } else {
                                timeout.tv_sec=post.tv_sec-pre.tv_sec-1;
                                timeout.tv_usec=(1000000L+post.tv_usec)-pre.tv_usec;
                        }
                        /* decrement the timeout */
                        pending.tv_sec-=timeout.tv_sec;
                        pending.tv_usec-=timeout.tv_usec;
                        if(pending.tv_sec>0 && pending.tv_usec<0) {
                                pending.tv_sec--;
                                pending.tv_usec+=1000000L;
                        }
                        /* Process timeout expired */
                        if(!Epd->auto_disabled && (pending.tv_sec<0 || (pending.tv_sec==0 && pending.tv_usec<0)))
                                Refresh(Epd,refreshAuto);
                }
                /* Check for UDP action */
                if(udp_fd!=-1 && FD_ISSET(udp_fd,&read_set)) {
                        int Pending,n;
                        struct sockaddr_in clientAdr;
                        socklen_t len;
                        if(Epd==NULL) {
                                /* Initialize Epd */
                                if((Epd=EpdmgrInit(display_name,1))==NULL)
                                        return(fatal_error("%s: could not initialize display or /dev/fb0\n",argv[0]));
                                Epd->quiet=Quiet;
                                Epd->pending_refresh=0;
                                /* Blank the display prior to subsequent commands */
                                EinkBlank(Epd->Eink);
                        }
                        while((Pending=0)==0 && ioctl(udp_fd,FIONREAD,&Pending)==0 && Pending>0) {
                                uDmCommand cmd;
                                len=sizeof(clientAdr);
                                n=recvfrom(udp_fd, udp_msg, UDP_MAX_SIZE, 0, (struct sockaddr *) &clientAdr, &len);
                                if(n<=0)
                                        continue;
                                udp_msg[n] = '\0';
                                if(dmMessageParser(udp_msg,&cmd)!=0)
                                        continue;
                                if(cmd.dmCmdDisplay.cmd==dmCcDisplay || cmd.dmCmdDisplay.cmd==dmCcDisplayPartial) {
                                        if(Epd->pending_refresh)
                                                Refresh(Epd,(cmd.dmCmdDisplay.qual==dmQFull)?refreshFull:refreshTyping);
                                } else if(cmd.dmCmdDisplay.cmd==dmCcEraseToWhite) {
                                        EinkBlank(Epd->Eink);
                                } else
                                        continue; /* Command not supported */
                                /* Note to self: liberdm/src/erdmServer.c for dmParseCommand() -- called from dmMessageParser() */
                        }
                }
                /* Check if child exited */
                if(flag_sigchld) {
                        if(ReapChildren(child_pid)!=-1) {
                                child_pid=-1;
                                ProcessMessage(Epd);
                                break;
                        }
                }
        }
        if(child_pid!=-1) {
                kill(child_pid,SIGQUIT);
                ReapChildren(child_pid);
                child_pid=-1;
        }
        if(Epd!=NULL)
                EpdmgrFini(Epd);
        signal_fd=-1;
        close(signal_fdpair[FDRD]);
        close(signal_fdpair[FDWR]);
        return(0);
}

/* Epdmgr struct manipulation */
sEpdmgr *
EpdmgrInit(char *Display, int EnableIoctls)
{
        sEpdmgr *Epd;
        if((Epd=malloc(sizeof(sEpdmgr)))==NULL)
                return(NULL); /* Insufficient memory */
        memset(Epd,0,sizeof(sEpdmgr));
        if(EnableIoctls) {
                if((Epd->Eink=EinkInit())==NULL) {
                        fatal_error("%s: couldn't open /dev/fb0\n",argv0);
                        EpdmgrFini(Epd);
                        return(NULL); /* Error opening /dev/fb0 */
                }
                if(erIpcStartClient(ER_CONTENTLISTER_CHANNEL, &(Epd->contentListerChannel))!=0) {
                        fatal_error("%s: couldn't initiatize the channel to contentlister\n",argv0);
                        EpdmgrFini(Epd);
                        return(NULL); /* Error calling erIpcStartClient */
                }
        } else
                Epd->Eink=NULL;
        if((Epd->dpy=XOpenDisplay(Display))==NULL) {
                fatal_error("%s: couldn't open display %s\n",argv0,Display);
                EpdmgrFini(Epd);
                return(NULL); /* Error connecting to display */
        }
        default_x_error_func=XSetErrorHandler(x_error_func);
        if(!XDamageQueryExtension(Epd->dpy, &Epd->damage_event, &Epd->damage_error)) {
                fatal_error("%s: no damage extension\n",argv0);
                EpdmgrFini(Epd);
                return(NULL); /* Error: no XDamage extension */
        }
        if(!XFixesQueryExtension(Epd->dpy, &Epd->xfixes_event, &Epd->xfixes_error)) {
                fatal_error("%s: no damage extension\n",argv0);
                EpdmgrFini(Epd);
                return(NULL); /* Error: no XFixes extension */
        }
        Epd->root=RootWindow(Epd->dpy,DefaultScreen(Epd->dpy));
        XSelectInput (Epd->dpy, Epd->root,
                      SubstructureNotifyMask|
                      ExposureMask|
                      StructureNotifyMask|
                      PropertyChangeMask);
        // Note to self: XDamageDeltaRectangles would also be useful for scribble support
        Epd->damage = XDamageCreate (Epd->dpy, Epd->root, XDamageReportDeltaRectangles);
        Epd->atom=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_COMM, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom, Epd->root, CurrentTime);
        Epd->atom_id=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_ID, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom_id, Epd->root, CurrentTime);
        Epd->cmd_last_id=0;
        Epd->cmd_last_lines=0;
        // Clear the command atom
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom, Epd->atom, 8,PropModeReplace, (unsigned char *) "", 1);
        // Set the initial id
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom_id, Epd->atom_id, 8,PropModeReplace, (unsigned char *) ((Epd->cmd_last_id==0)?"1":"0"), 2);
        // Initialize the screen blocks (DirtyAdd) part
        Epd->screen_blocks_size=0;
        Epd->screen_blocks=NULL;
        Epd->screen_blocks_x2=Epd->screen_blocks_y2=-1;
        DirtyClear(Epd);
        // Return successfully
        return(Epd);
}

void
EpdmgrFini(sEpdmgr *Epd)
{
        if(Epd==NULL)
                return;
        if(Epd->dpy!=NULL) {
                XCloseDisplay(Epd->dpy);
                Epd->dpy=NULL;
        }
        if(Epd->Eink!=NULL)
                EinkFini(Epd->Eink),Epd->Eink=NULL;
        free(Epd);
}


/* Message processing */
void
ProcessMessage(sEpdmgr *Epd)
{
        char *cmd,*ptr;
        Atom type;
        int format;
        int cur_id;
        int line_no;
        unsigned long size, remain;
        char *start,*end;
        if(XGetWindowProperty(Epd->dpy, Epd->root,Epd->atom,0,MAXCMDSIZE,True,Epd->atom,
             &type,&format,&size,&remain,(unsigned char **)&cmd)!=Success || cmd==NULL)
             return;
#ifdef DEBUG_MESSAGES
        {
                int i;
                printf("Received: \"");
                for(i=0;cmd[i]!='\0';i++) {
                        if(cmd[i]<' ' || cmd[i]=='\"') {
                                printf("\\x%02x",cmd[i]);
                        } else
                                printf("%c",cmd[i]);
                }
                printf("\"\n");
        }
#endif
        // check the id
        ptr=cmd;
        if((cmd[0]=='0' || cmd[0]=='1' || cmd[0]=='?') && cmd[1]=='\n') {
                cur_id=(*cmd=='?')?-1:*cmd-'0';
                ptr+=2;
        } else
                return; /* incorrect format */
        // parse cmd, do action, etc
        for(start=ptr,line_no=0;(end=strchr(start,'\n'))!=NULL;start=end+1,line_no++) {
                if(cur_id==Epd->cmd_last_id && line_no<Epd->cmd_last_lines)
                        continue; /* This line was processed earlier */
                *end='\0';
#ifdef DEBUG_MESSAGES
                printf("                                 Command: \"%s\"\n",start);
#endif
                if(strcmp(start,"Flush")==0) {
                        Refresh(Epd,refreshAuto);
                } else if(strcmp(start,"Cancel")==0) {
                        Cancel(Epd);
                } else if(strcmp(start,"RefreshFull")==0) {
                        Refresh(Epd,refreshFull);
                } else if(strcmp(start,"RefreshTyping")==0) {
                        Refresh(Epd,refreshTyping);
                } else if(strcmp(start,"Auto(1)")==0) {
                        Epd->auto_disabled=0;
                } else if(strcmp(start,"Auto(0)")==0) {
                        Epd->auto_disabled=1;
                } else {
                        ; /* Ignore the message */
                }
        }
        // Save the just-processed id and line count
        Epd->cmd_last_id=(cur_id!=-1)?cur_id:0;
        Epd->cmd_last_lines=line_no;
        // Set the next id
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom_id, Epd->atom_id, 8,PropModeReplace, (unsigned char *) ((Epd->cmd_last_id==0)?"1":"0"), 2);
        XFree((XPointer)cmd);
}

/* Screen refresh */
char *
RefreshType2Str(eRefreshType RefreshType)
{
        return((RefreshType==refreshAuto)?"Auto":
               (RefreshType==refreshFull)?"Full":
               (RefreshType==refreshTyping)?"Typing":"Unknown");
}
void
Refresh(sEpdmgr *Epd, eRefreshType RefreshType)
{
        XserverRegion parts;
        eRefreshType final_type;
        DirtyUpdateCount(Epd);
        if(RefreshType==refreshAuto) {
                if(Epd->pixel_count>=PIXELTHRESHOLD)
                        final_type=refreshFull;
                else
                        final_type=refreshTyping;
        } else
                final_type=RefreshType;
        if(!Epd->quiet)
                printf("Refresh called; pixel_count=%i, area=(%i,%i)-(%i,%i)\n",Epd->pixel_count,Epd->ux1,Epd->uy1,Epd->ux2,Epd->uy2);
        if(Epd->Eink==NULL) { /* Client mode */
                dmDisplay(dmCmdPriorUrgent,(final_type==refreshFull)?dmQFull:dmQTyping);
        } else { /* Server mode */
                if(Epd->pixel_count<=PARTIALPIXELTHRESHOLD &&
                   Epd->ux1>=4 && Epd->ux2<=(768-4) &&
                   Epd->uy1>=4 && Epd->uy2<=(1024-4)) {
                        EinkUpdatePartial(Epd->Eink,Epd->uy1,Epd->ux1,Epd->uy2,Epd->ux2,(final_type==refreshFull)?dmQFull:dmQTyping,0);
                        if(!Epd->quiet)
                                printf("Doing partial update %i,%i,%i,%i,%s\n",Epd->uy1,Epd->ux1,Epd->uy2,Epd->ux2,RefreshType2Str(final_type));
                } else {
                        EinkUpdateFull(Epd->Eink,(final_type==refreshFull)?dmQFull:dmQTyping,0);
                        if(!Epd->quiet)
                                printf("Doing full update %s\n",RefreshType2Str(final_type));
                }
                clDisplayUpdated(Epd->contentListerChannel);
        }
        parts=XFixesCreateRegion(Epd->dpy, 0, 0);
        XDamageSubtract(Epd->dpy, Epd->damage, None, parts);
        Epd->pending_refresh=0;
        DirtyClear(Epd);
}

void
Cancel(sEpdmgr *Epd)
{
        XserverRegion parts;
        parts=XFixesCreateRegion(Epd->dpy, 0, 0);
        XDamageSubtract(Epd->dpy, Epd->damage, None, parts);
        Epd->pending_refresh=0;
        DirtyClear(Epd);
}

/* Dirty areas / screen blocks processing */
void
DirtyAdd(sEpdmgr *Epd, int x1, int y1, int x2, int y2)
{
        int px1,px2,py1,py2;
        int py,offset;
        /* Update (ux1,uy1)-(ux2,uy2) */
        if(Epd->ux1==-1) { /* case: not initialized yet */
                Epd->ux1=x1,Epd->ux2=x1,Epd->uy1=y1,Epd->uy2=y2;
        } else { /* case: extend the area */
                Epd->ux1=(Epd->ux1<x1)?Epd->ux1:x1;
                Epd->uy1=(Epd->uy1<y1)?Epd->uy1:y1;
                Epd->ux2=(Epd->ux2>x2)?Epd->ux2:x2;
                Epd->uy2=(Epd->uy2>y2)?Epd->uy2:y2;
        }
        /* Ensure values are positive */
        x1=(x1<0)?0:x1;
        y1=(y1<0)?0:y1;
        x2=(x2<0)?0:x2;
        y2=(y2<0)?0:y2;
        px1=x1/SCREENBLOCKX;
        py1=y1/SCREENBLOCKY;
        px2=x2/SCREENBLOCKX;
        py2=y2/SCREENBLOCKY;
        /* Make room for the information */
        if(px2>Epd->screen_blocks_x2 || py2>Epd->screen_blocks_y2) {
                int nx2,ny2;
                char *tmp;
                nx2=(px2>Epd->screen_blocks_x2)?px2:Epd->screen_blocks_x2;
                ny2=(py2>Epd->screen_blocks_y2)?py2:Epd->screen_blocks_y2;
                /* nx2,ny2 are blocks coordinates indexed from 0; the space needed is (x2+1)*(y2+1) */
                if((tmp=malloc((nx2+1)*(ny2+1)))!=NULL) {
                        int y;
                        memset(tmp,0,(nx2+1)*(ny2+1));
                        for(y=0;y<=Epd->screen_blocks_y2;y++)
                                memcpy(tmp+(nx2+1)*y,Epd->screen_blocks+(Epd->screen_blocks_x2+1)*y,(Epd->screen_blocks_x2+1));
                        Epd->screen_blocks_x2=nx2;
                        Epd->screen_blocks_y2=ny2;
                        free(Epd->screen_blocks);
                        Epd->screen_blocks=tmp;
                        Epd->screen_blocks_size=(nx2+1)*(ny2+1);
                }
        }
        /* if no space, exit */
        if(Epd->screen_blocks_size==0)
                return;
        /* Update the info */
        px1=(px1>Epd->screen_blocks_x2)?Epd->screen_blocks_x2:px1;
        py1=(py1>Epd->screen_blocks_y2)?Epd->screen_blocks_y2:py1;
        px2=(px2>Epd->screen_blocks_x2)?Epd->screen_blocks_x2:px2;
        py2=(py2>Epd->screen_blocks_y2)?Epd->screen_blocks_y2:py2;
        for(py=py1,offset=0;py<=py2;py++,offset+=Epd->screen_blocks_x2+1)
                memset(Epd->screen_blocks+offset+px1,1,px2-px1+1);
}

void
DirtyUpdateCount(sEpdmgr *Epd)
{
        int px,py,offset;
        int amount,count,dirty;
        Epd->pixel_count=0;
        if(Epd->screen_blocks==NULL)
                return;
        amount=SCREENBLOCKX*SCREENBLOCKY;
        count=0;
        for(py=0,offset=0,dirty=0;py<=Epd->screen_blocks_y2;py++) {
                for(px=0;px<=Epd->screen_blocks_x2;px++,offset++) {
                        if(Epd->screen_blocks[offset])
                                count+=amount,dirty++;
                }
        }
        Epd->pixel_count=count;
        if(!Epd->quiet)
                printf("dirty count: %i/%i\n",dirty,offset);
}

void
DirtyClear(sEpdmgr *Epd)
{
        Epd->ux1=Epd->uy1=Epd->ux2=Epd->uy2=-1;
        Epd->pixel_count=0;
        memset(Epd->screen_blocks,0,Epd->screen_blocks_size);
}

/* Aux funcs implementations*/
static int
fatal_error(char *string, ...)
{
        va_list arg_list;
        va_start(arg_list,string);
        vfprintf(stderr,string,arg_list);
        va_end(arg_list);
        fflush(stderr);
        return(1);
}


static int
x_error_func(Display *dpy, XErrorEvent *err)
{
        if(err->error_code==BadWindow) {
                return(0);
        } else if(err->error_code==BadDrawable) {
                return(0);
        }
        return(default_x_error_func(dpy,err));
}

static int
install_signal(int Num,void (*f)(int))
{
        struct sigaction sact;
        sact.sa_handler=f;
        sigemptyset(&sact.sa_mask);
        sact.sa_flags=0;
        return(sigaction(Num,&sact,0));
}

static void
sigint(int Num)
{
        flag_sigint++;
        if(signal_fd!=-1)
                write(signal_fd,".",1);
}

static void
sigchld(int Num)
{
        flag_sigchld++;
        if(signal_fd!=-1)
                write(signal_fd,".",1);
}

static void
sigterm(int Num)
{
        flag_sigterm++;
        if(signal_fd!=-1)
                write(signal_fd,".",1);
}

static
int ReapChildren(int Pid)
{
        int reaped_pid,status;
        while(1) {
                reaped_pid=waitpid(-1,&status,WNOHANG|WUNTRACED);
                if(reaped_pid<=0)
                        break;
                if(reaped_pid==Pid)
                        return(0);
        }
        return(-1);
}

