#!/usr/local/bin/lua

-- 
-- Copyright (C) 2009  Iñigo Serna <mailto:inigoserna@gmail.com>
-- Time-stamp: <2009-09-17 20:33:17 inigo>
-- 
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--


require("lfs")
require("posix")

require("lgob.gtk")
require("lgob.pango")


---------- Vars -------------------------------------------------------
-- globals
local RECENT_DIR = "/mnt/settings/recentdocuments"
--local RECENT_DIR = "/home/inigo/tmp/1"

local files = {}
local files_len = 0


---------- Utils ------------------------------------------------------
local function get_recent_items(path)
   local fs = {}
   for f in lfs.dir(path) do
      if f ~= "." and f ~= ".." then
         local xxx = {}
         local linkpath = path .. '/' .. f
         local attr = lfs.attributes(linkpath)
         local fullpath = posix.readlink(linkpath)
         if fullpath ~= nil then
            _, _, xxx.path, xxx.file = string.find(fullpath, "(.+)/(.+)")
            xxx.mtime = attr.modification or 0
            fs[f] = xxx
         end
      end
   end
   return fs
end

local function get_table_size(t)
   local num = 0
   for k, v in pairs(t) do
      num = num + 1
   end
   return num
end

-- format_datetime: return a formated string of date
local function format_datetime(timestamp)
   return os.date('%d/%m/%Y %H:%M:%S', timestamp)
end

-- format_size: return a formated string of size
local function format_size(size)
   if size >= 1000000 then
      return string.format('%.2f', size/(1024*1024)) .. ' MB'
   elseif size >= 1000 then
      return string.format('%.2f', size/1024) .. ' KB'
   else
      return tostring(size) .. ' Bytes'
   end
end

local function print_files(files)
   for fname, attr in pairs(files) do
      io.write(fname .. ' -> ' .. attr.path .. ' / ' .. attr.file .. '\t' .. attr.mtime .. '\n')
   end
end



---------- UI ---------------------------------------------------------
local win = gtk.Window.new()
win:set("title", "Recent files", "border-width", 20)
win:connect("delete-event", gtk.main_quit)

-- title
local title = gtk.Label.new()
title:set_markup("<span size='xx-large'><b>Recent files</b></span>")

-- model and treeview
local model = gtk.ListStore.new("gchararray", "gboolean", "gchararray")
local view = gtk.TreeView.new()
local iter = gtk.TreeIter.new()
local rend_file = gtk.CellRendererText.new()
rend_file:set("scale-set", true, "scale", 0.85,
              "ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_MIDDLE)
local col_file = gtk.TreeViewColumn.new_with_attributes("Filename", rend_file, "text", 0)
col_file:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", 615, "alignment", 0.5)
view:append_column(col_file)
local rend_delete = gtk.CellRendererToggle.new()
rend_delete:connect("toggled",
    function(data, path)
       model:get_iter_from_string(iter, path)
       model:set(iter, 1, not model:get(iter, 1))
    end)
local col_delete = gtk.TreeViewColumn.new_with_attributes("Remove?", rend_delete, "active", 1)
col_delete:set("alignment", 0.5)
view:append_column(col_delete)
view:set("rules-hint", true, "model", model)

-- fill model
function populate_model()
   files = get_recent_items(RECENT_DIR)
   files_len = get_table_size(files)
   -- print_files(files)
   local ff = {}
   for f, a in pairs(files) do table.insert(ff, f) end
   table.sort(ff, function(s1, s2) return s1 > s2 end)
   model:clear()
   for _, f  in ipairs(ff) do
      model:append(iter)
      model:set(iter, 0, files[f].file, 1, false, 2, f)
   end
end

-- description
local label = gtk.Label.new()
label:set_markup("<i>Select the files you want to remove</i>")

-- buttons
local hbox = gtk.HBox.new(true, 25)
local btn_markall = gtk.Button.new_with_mnemonic("Mark all")
btn_markall:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          model:set(iter, 1, true)
       end
    end)
hbox:add(btn_markall)
local btn_clearall = gtk.Button.new_with_mnemonic("Clear all")
btn_clearall:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          model:set(iter, 1, false)
       end
    end)
hbox:add(btn_clearall)
local btn_remove = gtk.Button.new_with_mnemonic("Remove")
btn_remove:connect("clicked",
    function()
       for i = 1, files_len do
          model:get_iter_from_string(iter, tostring(i-1))
          if model:get(iter, 1) == true then
             local fullpath = RECENT_DIR .. '/' .. model:get(iter, 2)
             print(fullpath)
             os.remove(fullpath)
          end
       end
       populate_model()
    end)
hbox:add(btn_remove)
local btn_quit = gtk.Button.new_with_mnemonic("Quit")
btn_quit:connect("clicked", gtk.main_quit)

-- widgets packaging
local vbox = gtk.VBox.new(false, 5)
vbox:pack_start(title, false, false, 10)
vbox:pack_start(view, true, true, 0)
vbox:pack_start(label, false, false, 10)
vbox:pack_start(hbox, false, false, 0)
vbox:pack_start(btn_quit, false, false, 5)
win:add(vbox)

-- main loop
populate_model()
win:fullscreen()
win:show_all()
gtk.main()
