/*
 * File Name: scribble.c
 */

/*
 * scribble is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * scribble is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

#include <glib/gi18n.h>

#include "scribble.h"
// delta.h is copied from display component of iRex
#include "delta.h"

#include "xournal.h"

#include <gdk/gdk.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <pthread.h>

// using a GtkLabel to force a display update from the display driver in sysd. The
// current algorithm (R1.5/1.6/1.7) still seems to react on setting a label in the
// same way. NOTE that this might change in future versions of the DR firmware.
extern GtkWidget *g_label;

// using a mutex to protect against crashes. Crashes seem to be caused by simultatious
// update of scribbles and full screen refresh. Looking at the uds they seem to have 
// protected this somehow. Maybe also the code in sysd is adjusted to make scribbles
// work more reliable. Should find a mechanism to see if it is safe to do updates of
// scribbles or prevent any updates from xournal that cause the display to be updated!
// Unfortunately xournal is using gnome_canvas so it will not be easy to stop updates
// while updating scribbles ...
// (mutex doesn't help...)
pthread_mutex_t mutex_scribbles;

// Using a timeout to protect expose events.
// Theory is that update event at same time as scribble leads to a crash. So 
// protect scribbles after expose event. Problem is that we do not know when
// a full refresh is started. Best was if we could align with the code of
// display.c in sysd. This component remembers when it has done a DELTA-updates
// and does not do an update the next 900ms.
// Best would be if the FBIO-driver would protect itself...
static int g_timeout = 0;
static int g_protect_scribble = 0;

//==============================================================================
// Functions Implementation
//==============================================================================

static int fbDev = -1;

void scribble_init(void)
{
	// Always enable low level driver, but only use when MODE_DRIVER is enabled
	//LOGPRINTF("enter");
	erscribble_driver_init();
}

void scribble_ll_init(void)
{
	// USE_LL is not used it does not add anything to the LABEL and SCRIBBLE modes
#if USE_LL
	if (fbDev == -1)
	{
		fbDev = open ("/dev/fb0", O_RDWR);
		if (fbDev == -1)   
		{
			printf("Error opening framebufferdevice. # mknod /dev/fb0 c 29 0\n");
			exit(-1);
		}
	}
#endif
}

void scribble_close(void)
{
	//LOGPRINTF("enter");
	erscribble_driver_close();
}

void scribble_ll_close(void)
{
#if USE_LL
	close(fbDev);
	
	fbDev = -1;
#endif
}

void scibble_ll_update_display(void)
{
#if USE_LL
	struct display_update_info info;
    int ret;

    info.waveform = 1;
    info.color = 0;

	/* tell the display to update itself */
    if ((ret = ioctl(fbDev, FBIO_DELTA_UPDATE_DISPLAY, &info)))
    {
        printf("Error sending FBIO_DISPLAY: %x\n", ret);
    }
#endif
}

void scribble_driver_new(void)
{
	if (ui.scribble_update_mode & SCRIBBLE_MODE_DRIVER)
	{
		//LOGPRINTF("enter");
		// reset
		erscribble_driver_draw_reset_context();
	}
}

void scribble_driver_draw(int posx, int posy, int size, guint color_rgba, gboolean is_last_point)
{
    ScbDevPoint point;
	unsigned char color;
	
	if (ui.scribble_update_mode & SCRIBBLE_MODE_DRIVER)
	{
		point.point.x = posx;
		point.point.y = posy;

		// limit size (TODO: determine what limitations are)
		if (size < 1) size = 1;
		if (size > 5) size = 5;
		// TODO: The problem with size is that the size on which xournal is drawing
		//       does not match the size of erscribble, this gives some strange
		//       artifacts on the screen. Because xournal allows zooming, it will be
		//       very difficult to match the sizes of xournal with those of erscribble.
		
		// handle color, need to translate to the 4 colors that are used to
		// draw scribbles
		if ((color_rgba & 0xFF000000) == 0xFF000000) color = 0; // white
		else if ((color_rgba & 0xFF000000) > 0xA0000000) color = 1;
		else if ((color_rgba & 0xFF000000) > 0x40000000) color = 2;
		else color = 3; // black
		// TODO: colors do not seem to work (yet) for scribbles, all color values
		//       give the same result (dark grey)
		
		//printf ("scribble_driver_draw: size=%d, col=%d\n", size, (int)color);
		
		if (erscribble_driver_draw_now() || is_last_point)
		{
			if (is_last_point)
			{
				// add the last point 
				erscribble_driver_draw_record(&point, 
					(unsigned char)size,
					color,
					FALSE);
			}

			// draw it right now
			// only if not in protection mode
			if (g_protect_scribble == 0)
			{
				erscribble_driver_draw();
			}

			// reset
			erscribble_driver_draw_reset_context();
		}

		if (!is_last_point)
		{
			// record this point
			erscribble_driver_draw_record(&point,
				(unsigned char)size,
				color, 
				TRUE);
		}
	}
}

void scribble_force_update(void)
{
	if (ui.scribble_update_mode & SCRIBBLE_MODE_LABEL)
	{
		// updating a label, results in a full screen update of the display-driver
		// in sysd. (mis-using the 'layer-label' for this purpose.)
		gtk_label_set_text( GTK_LABEL(g_label), _("  Layer:  ") );
	}
	scibble_ll_update_display();
}

// This function can be used to force a full screen update, even if SCRIBBLE_MODE_DRIVER is used.
void scribble_force_update_now(void)
{
	// protect scribble updates
	scribble_protect();
	
	// use the label-trick to update the screen.
	gtk_label_set_text( GTK_LABEL(g_label), _("  Layer:  ") );
}

//==============================================================================
// Timer Implementation
//==============================================================================

static gboolean on_scribble_timeout(gpointer data)
{
	// unprotect
	g_protect_scribble = 0;
	// printf("unprot\n");
	
	return FALSE; // don't call again
}

// when this function is called, low level scribble calls to the driver will
// be queued for about 1 second. In most cases this prevents a full-screen update
// to interfere with a scribble-call. Doing both calls at the same time seems to
// lock up the DR...
void scribble_protect( void)
{
	// kill old timeout
	if (g_timeout > 0) g_source_remove(g_timeout);
	
	// protect (no scribbles will be send to the driver when this variable is set)
	g_protect_scribble = 1;
	
	// start new timeout (of 1 second, screen updates typically take less then 900ms)
	g_timeout = gdk_threads_add_timeout(1000, on_scribble_timeout, NULL);
	// printf("prot\n");
}


