/*
 * File Name: menu.c
 */

/*
 * This file is part of xournal.
 * based on the helloworld example of iRex
 *
 * hello_world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Mackx.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <glib/gi18n.h>

// system include files, between < >
#include <stdio.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "ipc.h"
#include "menu.h"

#include "xo-callbacks.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// menus for application, must be unique
static const char *MENU_MAIN                = "xournal_menu_main";
                                            
// menu groups, must be unique
static const char *GROUP_FILE               = "xournal_file";
static const char *GROUP_ACTIONS            = "xournal_actions";
static const char *GROUP_BROWSE             = "xournal_browse";
                                            
// menu items
//   group "File"                        
static const char *ITEM_FILE_PRINT          = "print";
static const char *ITEM_FILE_SAVEAS         = "saveas";
static const char *ITEM_FILE_SAVE           = "save";
static const char *ITEM_FILE_NEWPAGE        = "newpage";
static const char *ITEM_FILE_QUIT           = "quit";
                                
//   group "Actions"                          
static const char *ITEM_ACTION_NEW          = "new";
static const char *ITEM_ACTION_OPEN         = "open";
static const char *ITEM_ACTION_UNDO         = "undo";
static const char *ITEM_ACTION_REDO         = "redo";
static const char *ITEM_ACTION_FIVE         = "action5";

//   group "Browse"                          
static const char *ITEM_BROWSE_FIRST        = "first";
static const char *ITEM_BROWSE_PREV         = "previous";
static const char *ITEM_BROWSE_NEXT         = "next";
static const char *ITEM_BROWSE_LAST         = "last";
static const char *ITEM_BROWSE_FIVE         = "action5";

// item states
static const char *STATE_NORMAL             = "normal";
static const char *STATE_DISABLED           = "disabled";
//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------



//============================================================================
// Local Function Definitions
//============================================================================



//============================================================================
// Functions Implementation
//============================================================================

// initialise popup menu
void menu_init ( void ) 
{
    static gboolean firsttime = TRUE;
    const char      *group;

    // execute only once
    if ( !firsttime )
    {
        printf("function called twice");
        return;
    }
    firsttime = FALSE;

    // add groups
    group = GROUP_FILE;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_FILE_PRINT,        group, "sort_type"       );
    ipc_menu_add_item ( ITEM_FILE_SAVEAS,       group, "save"            );
    ipc_menu_add_item ( ITEM_FILE_SAVE,         group, "save"            );
    ipc_menu_add_item ( ITEM_FILE_NEWPAGE,      group, "paste"           );
    ipc_menu_add_item ( ITEM_FILE_QUIT,         group, "close"           );
	
    group = GROUP_ACTIONS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_ACTION_NEW,        group, "view_large"      );
    ipc_menu_add_item ( ITEM_ACTION_OPEN,       group, "open"            );
    ipc_menu_add_item ( ITEM_ACTION_UNDO,       group, "back"            );
    ipc_menu_add_item ( ITEM_ACTION_REDO,       group, "see_all"         );
    ipc_menu_add_item ( ITEM_ACTION_FIVE,       group, "info"            );

    group = GROUP_BROWSE;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_BROWSE_FIRST,      group, "/usr/local/share/xournal/pixmaps/first.png"        );
    ipc_menu_add_item ( ITEM_BROWSE_PREV,       group, "/usr/local/share/xournal/pixmaps/prev.png"        );
    ipc_menu_add_item ( ITEM_BROWSE_NEXT,       group, "/usr/local/share/xournal/pixmaps/next.png"        );
    ipc_menu_add_item ( ITEM_BROWSE_LAST,       group, "/usr/local/share/xournal/pixmaps/last.png"        );
    ipc_menu_add_item ( ITEM_BROWSE_FIVE,       group, "info"            );

    // add menus
    ipc_menu_add_menu( MENU_MAIN, GROUP_BROWSE, GROUP_ACTIONS, GROUP_FILE);

    // set menu texts
    menu_set_text();

	// not all menus are implemented, disable them
	group = GROUP_ACTIONS;
	ipc_menu_set_item_state (ITEM_ACTION_FIVE, group, STATE_DISABLED);
	group = GROUP_BROWSE;
	ipc_menu_set_item_state (ITEM_BROWSE_FIVE, group, STATE_DISABLED);
	
    // set menu context
    menu_show();    
}


// remove the proper popup menu
void menu_destroy (void)
{
    // remove the main menu
    ipc_remove_menu( MENU_MAIN );
}


// show the proper popup menu
void menu_show (void)
{
    // show the main menu
    ipc_menu_show_menu( MENU_MAIN );
}


// initialise popup menu names
// also call this function when the LOCALE has changed
void menu_set_text ( void ) 
{
    const char      *group;

    // set groups + items
    group = GROUP_FILE;
    ipc_menu_set_group_label( group,                     _("File")           );
    ipc_menu_set_item_label ( ITEM_FILE_PRINT,    group, _("Export to PDF")  );
    ipc_menu_set_item_label ( ITEM_FILE_SAVEAS,   group, _("Save As")        );
    ipc_menu_set_item_label ( ITEM_FILE_SAVE,     group, _("Save")           );
    ipc_menu_set_item_label ( ITEM_FILE_NEWPAGE,  group, _("New Page")       );
    ipc_menu_set_item_label ( ITEM_FILE_QUIT,     group, _("Close")          );
    //
    group = GROUP_ACTIONS;                                                     
    ipc_menu_set_group_label( group,                     _("Actions")        );
    ipc_menu_set_item_label ( ITEM_ACTION_NEW,    group, _("New")            );
    ipc_menu_set_item_label ( ITEM_ACTION_OPEN,   group, _("Open")           );
    ipc_menu_set_item_label ( ITEM_ACTION_UNDO,   group, _("Undo")           );
    ipc_menu_set_item_label ( ITEM_ACTION_REDO,   group, _("Redo")           );
    ipc_menu_set_item_label ( ITEM_ACTION_FIVE,   group, _("None")           );
    //
    group = GROUP_BROWSE;                                                     
    ipc_menu_set_group_label( group,                     _("Actions")        );
    ipc_menu_set_item_label ( ITEM_BROWSE_FIRST,  group, _("First")          );
    ipc_menu_set_item_label ( ITEM_BROWSE_PREV,   group, _("Previous")       );
    ipc_menu_set_item_label ( ITEM_BROWSE_NEXT,   group, _("Next")           );
    ipc_menu_set_item_label ( ITEM_BROWSE_LAST,   group, _("Last")           );
    ipc_menu_set_item_label ( ITEM_BROWSE_FIVE,   group, _("None")           );

    // set menus
    ipc_menu_set_menu_label( MENU_MAIN,  _("Xournal Menu") );
}


//----------------------------------------------------------------------------
// Callbacks from popupmenu
//----------------------------------------------------------------------------

// user has pressed a menu button
void menu_on_item_activated ( const gchar *item,
                              const gchar *group,
                              const gchar *menu,
                              const gchar *state )
{
    gboolean    ok = TRUE;
    char        *msg = NULL;

    if ( strcmp(group, GROUP_BROWSE) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_BROWSE_FIRST) == 0 )
            {
				on_viewFirstPage_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_BROWSE_PREV) == 0 )
            {
				on_viewPreviousPage_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_BROWSE_NEXT) == 0 )
            {
				on_viewNextPage_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_BROWSE_LAST) == 0 )
            {
				on_viewLastPage_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_BROWSE_FIVE) == 0 )
            {
            }
            else
            {
                printf("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            printf("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_ACTIONS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_ACTION_UNDO) == 0 )
            {
				on_editUndo_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_ACTION_REDO) == 0 )
            {
				on_editRedo_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_ACTION_NEW) == 0 )
            {
				on_fileNew_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_ACTION_OPEN) == 0 )
            {
				on_fileOpen_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_ACTION_FIVE) == 0 )
            {
            }
            else
            {
                printf("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            printf("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_FILE) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_FILE_PRINT) == 0 )
            {
				on_filePrintPDF_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_FILE_SAVEAS) == 0 )
            {
				on_fileSaveAs_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_FILE_SAVE) == 0 )
            {
				on_fileSave_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_FILE_NEWPAGE) == 0 )
            {
				on_journalNewPageEnd_activate(NULL, NULL);
            }
            else if ( strcmp(item, ITEM_FILE_QUIT) == 0 )
            {
				on_fileQuit_activate(NULL, NULL);
            }
            else
            {
                printf("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            printf("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else
    {
        printf("unexpected menu group [%s]", group);
        ok = FALSE;
    }
}
