//gcc -o test test.c `pkg-config gtk+-2.0 --libs --cflags`

#include <gtk/gtk.h>
/*
 * xepdmgrclient.c
 *
 * Implements communications with xepdmgr using X atoms/properties.
 *
 * History:
 *       4/09/2008 DRF Initial implementation.
 *       6/09/2008 DRF Make the comms with the server more reliable.
 *      25/04/2009 DRF Add EpdCancel().
 *	12/09/2009 DRF Add pwarren's modifications so it also compiles as C++.
 *
 * Authors: DRF Dario Rodriguez dario@softhome.net
 * (c) 2008 Dario Rodriguez
 * This program is licensed under the terms of the GNU LGPL v3 or greater.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <X11/Xlib.h>

#include "xepdmgrclient.h"

#define SHARED_ATOM_NAME_FOR_COMM "XEPDMGR_CMD"
#define SHARED_ATOM_NAME_FOR_ID "XEPDMGR_NEXTID"
#define MAXCMDSIZE 1024

typedef struct TypeEpd {
        Display *dpy;
        Window root;
        Atom atom;
        Atom atom_id;
} _sEpd;

/* Local functions protorypes */
static int EpdLocalSend(_sEpd *Epd, const char *Cmd);

/* Implementation of exported functions*/
sEpd *
EpdInit(char *Display)
{
        _sEpd *Epd;
        if((Epd=(_sEpd *)malloc(sizeof(_sEpd)))==NULL)
                return(NULL); /* Insufficient memory */
        memset(Epd,0,sizeof(_sEpd));
        if((Epd->dpy=XOpenDisplay(Display))==NULL) {
                EpdFini((sEpd *)Epd);
                return(NULL); /* Error connecting to display */
        }
        Epd->root=RootWindow(Epd->dpy,DefaultScreen(Epd->dpy));
        Epd->atom=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_COMM, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom, Epd->root, CurrentTime);
        Epd->atom_id=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_ID, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom_id, Epd->root, CurrentTime);
        return((sEpd *)Epd);
}

void
EpdFini(sEpd *EpdParam)
{
        _sEpd *Epd=(_sEpd *)EpdParam;
        if(Epd==NULL)
                return;
        if(Epd->dpy!=NULL) {
                XCloseDisplay(Epd->dpy);
                Epd->dpy=NULL;
        }
        free(Epd);
}

int
EpdFlush(sEpd *Epd)
{
        return(EpdLocalSend((_sEpd *)Epd,"Flush\n"));
}

int
EpdCancel(sEpd *Epd)
{
        return(EpdLocalSend((_sEpd *)Epd,"Cancel\n"));
}

int
EpdRefreshFull(sEpd *Epd)
{
        return(EpdLocalSend((_sEpd *)Epd,"RefreshFull\n"));
}

int
EpdRefreshTyping(sEpd *Epd)
{
        return(EpdLocalSend((_sEpd *)Epd,"RefreshTyping\n"));
}

int
EpdRefreshAuto(sEpd *Epd,int Enabled)
{
        return(EpdLocalSend((_sEpd *)Epd,(Enabled!=0)?"Auto(1)\n":"Auto(0)\n"));
}

/* Implementation of local functions */
static int
EpdLocalSend(_sEpd *Epd, const char *NewCmd)
{
        char *cmd_next_id,*cmd;
        Atom type;
        int format;
        int cur_id,next_id;
        unsigned long size, remain;
        char *text;
        if(Epd==NULL)
                return(-1);
        if(XGetWindowProperty(Epd->dpy, Epd->root,Epd->atom,0,MAXCMDSIZE,True,Epd->atom,
             &type,&format,&size,&remain,(unsigned char **)&cmd)!=Success || cmd==NULL)
                cmd=NULL;
        if(XGetWindowProperty(Epd->dpy, Epd->root,Epd->atom_id,0,MAXCMDSIZE,True,Epd->atom_id,
             &type,&format,&size,&remain,(unsigned char **)&cmd_next_id)!=Success || cmd_next_id==NULL)
             cmd_next_id=NULL;
        next_id=(cmd_next_id!=NULL && *cmd_next_id=='1')?1:0;
        if(cmd!=NULL && (cmd[0]=='0' || cmd[0]=='1' || cmd[0]=='?') && cmd[1]=='\n')
                cur_id=(*cmd=='?')?-1:*cmd-'0';
        else
                cur_id=-1; /* incorrect format or atom not read */
        if(cmd==NULL || (cmd_next_id!=NULL && cur_id!=next_id)) {
                /* server has already read this property; overwrite the property */
                if((text=(char *)malloc(2+strlen(NewCmd)+1))==NULL)
                        return(-1);
                text[0]=(cmd_next_id!=NULL)?next_id+'0':'?';
                text[1]='\n';
                strcpy(text+2,NewCmd);
        } else {
                /* Doesn't know if the server has read this yet; append to the property */
                if((text=(char *)malloc(strlen(cmd)+strlen(NewCmd)+1))==NULL)
                        return(-1);
                strcpy(text,cmd);
                strcat(text,NewCmd);
        }
        // printf("Sent: %s \"%s\"\n",(cur_id!=next_id)?"overwrite":"append",text);
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom, Epd->atom, 8,PropModeReplace, (unsigned char *) text, strlen(text)+1);
        XFlush(Epd->dpy);
        free(text);
        return(0);
}


sEpd *Epd=NULL;

void toggled_callback(GtkToggleButton *toggleBtn, gpointer data) {
    if (gtk_toggle_button_get_active(toggleBtn)) {
        gtk_button_set_label(GTK_BUTTON(toggleBtn), "Auto-refresh Enabled");
        EpdRefreshAuto(Epd,1);
    } else {
        gtk_button_set_label(GTK_BUTTON(toggleBtn), "Auto-refresh Disabled");
        EpdRefreshAuto(Epd,0);
    }
}

int main(int argc, char *argv[]) {

   if((Epd=EpdInit(NULL))==NULL)
      return(1); /*ERROR */


    GtkWidget *window;
    GtkWidget *toggleBtn;
    GtkWidget *quitBtn;
    GtkWidget *vbox;

    gtk_init(&argc, &argv);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    vbox=gtk_vbox_new(FALSE, 0);

    gtk_window_set_title(GTK_WINDOW(window), "GtkToggleButton");
    gtk_window_set_default_size(GTK_WINDOW(window), 768, 935);

    toggleBtn = gtk_toggle_button_new_with_label("Auto-refresh Enabled");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggleBtn), TRUE);



    gtk_container_add(GTK_CONTAINER(vbox), toggleBtn);

    g_signal_connect(GTK_OBJECT(toggleBtn), "toggled",
                     G_CALLBACK(toggled_callback), NULL);
    g_signal_connect(GTK_OBJECT(window), "destroy",
                     G_CALLBACK(gtk_main_quit), NULL);

    quitBtn = gtk_button_new_from_stock(GTK_STOCK_QUIT);
    g_signal_connect(G_OBJECT(quitBtn), "clicked", G_CALLBACK(gtk_main_quit),NULL);
    gtk_container_add(GTK_CONTAINER(vbox), quitBtn);


    gtk_container_add(GTK_CONTAINER(window), vbox);
    gtk_widget_show_all(window);

    EpdRefreshAuto(Epd,1);//enable auto-refresh as default

    gtk_main();

    EpdFini(Epd);
    return 0;
}
