/*
  Copyright (c) 2007 Scott Turner scotty1024@mac.com All Rights Reserved.

  This software is subject to, and may be distributed under, the
  GNU General Public License, Version 2. The license should have
  accompanied the software or you may obtain a copy of the license
  from the Free Software Foundation at http://www.fsf.org .

  This code is provided "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTY OF NON-INFRINGEMENT,
  OR ANY IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
  PURPOSE.
*/
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.image.ImageObserver;
import java.awt.MediaTracker;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import java.awt.image.ImageObserver;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;

import com.irex.sdk.DisplayUpdate;
import com.irex.sdk.ErIpcCmd;
import com.irex.sdk.ErIpcServer;
import com.irex.sdk.ErPageBar;
import com.irex.sdk.IpcServerCallBack;
import com.irex.sdk.ViewerSignal;
import com.irex.mobipocket.reader.Utils;

public class CBZViewer extends Frame implements IpcServerCallBack, KeyListener, MouseListener, WindowListener, ImageObserver {

    private ErIpcServer m_ersrv;
    private ErPageBar pagebar;
    private int IREX_READER_UA_ID = 110;

    protected ZipFile zFile = null;
    protected ArrayList zEntries = null;
    protected byte[] zBuffer = new byte[32768];
    protected int curPage = -1;
    protected Image screenImage = null;

    public static void main(String[] args)
	throws Exception
    {
	CBZViewer viewer = new CBZViewer(args[0]);
    }

    public CBZViewer(String aFileName)
	throws Exception
    {
	final long startTime = System.currentTimeMillis();
	System.out.println("url: " + aFileName);

	zFile = new ZipFile(aFileName);
	TreeSet sortNames = new TreeSet();
	for (Enumeration entries = zFile.entries(); entries.hasMoreElements() ;) {
	    sortNames.add(((ZipEntry)entries.nextElement()).getName());
	}

	zEntries = new ArrayList(zFile.size());
	for (Iterator entries = sortNames.iterator(); entries.hasNext(); ) {
	    zEntries.add(zFile.getEntry((String)entries.next()));
	}
	sortNames.clear();
	sortNames = null;

	// Set Frame size to iLiad application area
	setSize(768, 935);
 
	m_ersrv = ErIpcServer.getInstanceByChannelId(15, this);
	if (m_ersrv == null) {
	    System.out.print("Opening connection to ErIpcServer failed.");
	    throw new Exception("Couldn't connect to ErIpcServer.");
	}

	m_ersrv.startServer();

	pagebar = ErPageBar.getInstance();
	if (pagebar == null) {
	    System.out.println("pagebar client create error");
	    System.exit(1);
	} else {
	    pagebar.pbReset(IREX_READER_UA_ID);
	    pagebar.pbSetPageCount(IREX_READER_UA_ID, zEntries.size());
	    pagebar.pbSetCurrentPage(IREX_READER_UA_ID, 1);
	    pagebar.pbRedraw(IREX_READER_UA_ID);
	}

	// Setup listeners
	addMouseListener(this);
	addWindowListener(this);
	addKeyListener(this);

	// Make us the target for events
	requestFocus();

	// Make us visible
	setVisible(true);

	// Make page 1 visible
	setPage(1);

	// Report on startup time
	final long   t = System.currentTimeMillis() - startTime;
	final String d = "000" + t;
	System.out.println("CBZViewer initialized in " + (t / 1000L) + "."
			   + d.substring(d.length() - 3) + " seconds");
    }

    public void mouseClicked(MouseEvent e) { 
	System.out.println("Program closing.");
	System.exit(0);
    }
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void windowClosing(WindowEvent e) {
	System.out.println("Program closing.");
	System.exit(0);
    }
    public void windowDeactivated(WindowEvent e) {}
    public void windowActivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

    protected void setPage(int newPage) {
	if (newPage < 1) {
	    newPage = 1;
	} else if (newPage > zEntries.size()) {
	    newPage = zEntries.size();
	}

	if (curPage == newPage) {
	    return;
	}

	Utils.busyBlink();

	try {
	    InputStream is = zFile.getInputStream((ZipEntry)zEntries.get(newPage - 1));
	    ByteArrayOutputStream os = new ByteArrayOutputStream();
	    int bytes = 0;
	    do {
		bytes = is.read(zBuffer);
		if (bytes > 0) {
		    os.write(zBuffer, 0, bytes);
		}
	    } while (bytes > 0);
	    is.close();

	    screenImage = Toolkit.getDefaultToolkit().createImage(os.toByteArray());

	    MediaTracker mediaTracker = new MediaTracker(this);
	    mediaTracker.addImage(screenImage, 0);
	    try {
		mediaTracker.waitForID(0);
	    } catch (InterruptedException ie) {
		System.out.println(ie);
		System.exit(0);
	    }
	    os.close();

	    int width = screenImage.getWidth(this);
	    int height = screenImage.getHeight(this);
	    Image newImage = null;

	    if (height > width) {
		if (height > 963) {
		    newImage = screenImage.getScaledInstance( -1, 963, Image.SCALE_SMOOTH);
		} else if (width > 768) {
		    newImage = screenImage.getScaledInstance( 768, -1, Image.SCALE_SMOOTH);
		}
	    } else {
		if (width > 768) {
		    newImage = screenImage.getScaledInstance( 768, -1, Image.SCALE_SMOOTH);
		} else if (height > 963) {
		    newImage = screenImage.getScaledInstance( -1, 963, Image.SCALE_SMOOTH);
		}
	    }

	    if (newImage != null) {
		mediaTracker.addImage(newImage, 0);
		try {
		    mediaTracker.waitForID(0);
		} catch (InterruptedException ie) {
		    System.out.println(ie);
		    System.exit(0);
		}

		screenImage = newImage;
	    }

	    // Update page bar
	    curPage = newPage;
	    pagebar.pbSetCurrentPage(IREX_READER_UA_ID, newPage);
	    pagebar.pbRedraw(IREX_READER_UA_ID);

	    // Force Frame to re-draw itself
	    this.setVisible(false);
	    this.setVisible(true);
	} catch (Throwable t) {
	    t.printStackTrace();
	}
    }

    public void paint (Graphics g) {
	if (screenImage != null) {
	    int width = screenImage.getWidth(this);
	    int height = screenImage.getHeight(this);

	    // Draw image to Frame
	    g.drawImage(screenImage, (768 - width) / 2, (963 - height) / 2, this);

	    // Refresh iLiad display
	    DisplayUpdate.dispMainWinFullQ();
	    Utils.busyOff();
	}
    }

    public void messageReceived(DatagramPacket datagrampacket) {
        ErIpcCmd eripccmd = ViewerSignal.parseCommand(new String(datagrampacket.getData(), 0, datagrampacket.getLength()));
        if (eripccmd == null) {
            System.out.print("Invalid ErIpcCmd message.");
            return;
        }

        if (eripccmd.get_cmdcode() == 0) {
            String as[] = eripccmd.get_args();
            int newPage = 0;
	    try {
		newPage = Integer.parseInt(as[1]);
	    } catch (Throwable t) {
		System.out.println("Got invalid ErIpcCmd message argument.");
		return;
	    }

	    Utils.busyBlink();

	    setPage(newPage);
        } else {
	    System.out.println("Ignore ErIpcCmd message of code: " + eripccmd.get_cmdcode());
	}
    }

    public void keyPressed(KeyEvent keyevent) {
        //System.out.println("keyPressed: code= " + keyevent.getKeyCode());

        switch(keyevent.getKeyCode()) {
        case 116:
	    Utils.busyBlink();
            System.out.println("VK_F5 Quiting.");
	    System.exit(0);
            break;

        case 36:
            //System.out.println("VK_HOME Goto first page");
	    moveRelativePage(1);
            break;

        case 34:
            //System.out.println("VK_PAGE_DOWN Goto next page");
	    moveRelativePage(1);
            break;

        case 112:
            //System.out.println("VK_F1 Long jump forward.");
	    moveRelativePage(5);
            break;

        case 33:
            //System.out.println("VK_PAGE_UP Goto previous page");
	    moveRelativePage(-1);
            break;

        case 113:
            //System.out.println("VK_F2 Long jump backward.");
	    moveRelativePage(-5);
            break;

        default:
            break;
        }
        keyevent.consume();
    }

    public void keyReleased(KeyEvent keyevent) {
    }

    public void keyTyped(KeyEvent keyevent) {
    }

    protected void moveRelativePage(int delta) {
	setPage(curPage + delta);
    }

}
