/*
 * ScribbleMerger.java
 *
 * Created on 14 januari 2007, 0:12
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package iliadpdfscribblemerger;

import com.lowagie.text.pdf.BaseFont;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfWriter;
import com.lowagie.text.Rectangle;
import com.lowagie.text.Document;
import java.awt.Color;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.HashMap;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 *
 * @author Bert
 */
public class ScribbleMerger {
    //properties
    String strPdfSource;
    String strScribbleXmlPath;
    HashMap mapColors;
    //runtime data
    ArrayList arrScribblePages;
    HashMap mapScribblePages;
    

    /** Creates a new instance of ScribbleMerger */
    public ScribbleMerger(String strPdfSource, String strScribbleXmlPath) {
        this.strPdfSource = strPdfSource;
        this.strScribbleXmlPath = strScribbleXmlPath;
        
        arrScribblePages = new ArrayList();
        mapScribblePages = new HashMap();
        mapColors = null;
    }
    
    public boolean Merge(String strOutput, HashMap mapColors, boolean bScribblePagesOnly) {
        if (strOutput==null)
            return false;
        this.mapColors = mapColors;
        
        //append '.pdf' if needed
        if (!strOutput.toLowerCase().endsWith(".pdf"))
            strOutput = strOutput + ".pdf";
        
        //we use the source while creating the output, so overwriting isn't possible
        if (strOutput.equals(strPdfSource))
            return false;
        
        if ( !ParseXml() )
            return false;
        
        return MergeScribbles(strOutput, bScribblePagesOnly);
    }
    
    protected boolean ParseXml() {
        arrScribblePages.clear();
        mapScribblePages.clear();
        try {
            File file = new File(strScribbleXmlPath);
            if ( !file.exists() )
                return false;
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            org.w3c.dom.Document doc = db.parse(file);
            doc.getDocumentElement().normalize();
            
            NodeList nodePages = doc.getElementsByTagName("page");
            int nPageCount = nodePages.getLength();
            arrScribblePages.ensureCapacity(nPageCount);
            for (int nScribblePage = 0; nScribblePage < nPageCount; nScribblePage++) {
                Node nodePage = nodePages.item(nScribblePage);
                if (nodePage != null && nodePage.getNodeType() == Node.ELEMENT_NODE) {
                    Element elPage = (Element)nodePage;
                    ScribblePage sp = new ScribblePage(elPage);
                    arrScribblePages.add(sp);
                    mapScribblePages.put(new Integer(sp.nPageId), sp);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        
        return true;
    }
    
    protected boolean MergeScribbles(String strPdfDestination, boolean bScribblePagesOnly) {
        try {
            //1: create a reader for the original pdf file
            PdfReader reader = new PdfReader(strPdfSource);
            // get the total number of pages
            int nPageCount = reader.getNumberOfPages();
            // get the size of the first page
            Rectangle rSize = reader.getPageSize(1);
            float fWidth = rSize.width();
            float fHeight = rSize.height();
            
            if (bScribblePagesOnly) {
                //2: create a new pdf file & open it
                Document document = new Document(rSize, 0, 0, 0, 0);
                PdfWriter writer = PdfWriter.getInstance(document, new FileOutputStream(strPdfDestination));
                document.open();

                //3: merge the original pdf pages with the scribble
                PdfContentByte cb = writer.getDirectContent();
                for (int nPage=0; nPage<arrScribblePages.size(); nPage++) {
                    // get scribbles
                    ScribblePage spPage = (ScribblePage)arrScribblePages.get(nPage);
                    // create a new page
                    document.newPage();
                    // copy the original page to the new one
                    PdfImportedPage pageOrg = writer.getImportedPage(reader, spPage.nPageId);
                    cb.addTemplate(pageOrg, 0, 0);
                    // get the necessary dimensions
                    Rectangle rectCropBox = reader.getCropBox(spPage.nPageId);
                    // add the new content to the page
                    spPage.AddScribbles(cb, rectCropBox.top(), rectCropBox.left(), mapColors);
                    // add page number
                    BaseFont bf = BaseFont.createFont(BaseFont.HELVETICA, BaseFont.CP1252, BaseFont.NOT_EMBEDDED);
                    cb.beginText();
                    cb.setFontAndSize(bf, 11);
                    cb.showTextAligned(PdfContentByte.ALIGN_CENTER, spPage.nPageId + " / " + nPageCount, fWidth / 2, 10, 0);
                    cb.endText();
                }

                //4: close the document
                document.close();                
            } else {
                //2: Create a stamper that will copy the document to a new file
                PdfStamper stamp = new PdfStamper(reader, new FileOutputStream(strPdfDestination));

                //3: Add scribbles to each page
                PdfContentByte cbOver = null;
                for (int nPage=1; nPage<=reader.getNumberOfPages(); nPage++) {
                    ScribblePage spPage = (ScribblePage)mapScribblePages.get(new Integer(nPage));
                    //get content byte
                    cbOver = stamp.getOverContent(nPage);
                    // get the crop box
                    Rectangle rectCropBox = reader.getCropBox(nPage);
                    // add the scribbles to the page (if any)
                    if (spPage != null)
                        spPage.AddScribbles(cbOver, rectCropBox.top(), rectCropBox.left(), mapColors);
                }

                //4: closing PdfStamper will generate the new PDF file
                stamp.close();
            }
        }
        catch (Exception de) {
            de.printStackTrace();
            return false;
        }
        
        return true;
    }
        
    /////////////////////////////////////////////////////////////////////////////////
    // Static functions to allow direct merging
    /////////////////////////////////////////////////////////////////////////////////
    static boolean Merge(String strInputPath, String strOutputPath, char cMergeType, HashMap mapColors) {
        //check arguments
        File fileInput = new File(strInputPath);
        if (fileInput == null || !fileInput.exists() || !fileInput.isDirectory() || //it's a directory ...
                !new File(fileInput.getAbsoluteFile() + File.separator + "manifest.xml").exists() || //with a manifest.xml ...
                !new File(fileInput.getAbsoluteFile() + File.separator + "scribble.irx").exists()) { //and scribbles
            System.err.println("Invalid input path: " + strInputPath);
            return false;
        }

        //get input files
        String strSourceDir = fileInput.getAbsolutePath() + File.separator;
        String strPdfSource = getPdfFromIRexFolder(strSourceDir);
        if ( strPdfSource == null || !(new File(strPdfSource).exists()) ) {
            System.err.println("error: input pdf-file not found");
            return false;
        }
        String strXmlSource = strSourceDir +  "scribble.irx";
        if ( !(new File(strXmlSource).exists()) ) {
            System.err.println("error: no scribble file found");
            return false;
        }
        
        //create merger
        ScribbleMerger merger = new ScribbleMerger(strPdfSource, strXmlSource);
        
        boolean bMergeResult = false;
        if (cMergeType=='n' || cMergeType=='m' || cMergeType=='s') {
            File fileDestination = new File(strOutputPath);
            if (fileDestination == null || fileDestination.equals(fileInput)) {
                System.err.println("error: invalid output");
                return false;
            }
            String strDestination = fileDestination.getAbsolutePath();
            
            if (cMergeType=='m') {
                //strDestination is destination dir
                strDestination = strDestination + File.separator;
                //copy manifest
                if ( !fileDestination.mkdirs() ) {
                    System.err.println("error: creating output dir failed");
                    return false;
                }
                copyFile(strSourceDir +  "manifest.xml", strDestination + "manifest.xml");
                
                //merge
                String strOutputFile = strDestination + fileDestination.getName();
                bMergeResult = merger.Merge(strOutputFile, mapColors, false);
            } else {
                //strDestination is destination file
                bMergeResult = merger.Merge(strDestination, mapColors, (cMergeType=='s'));
            }
        } else if (cMergeType=='e') {
            //because the merging uses the original file, we create
            //a temporary pdf and replace the original one later
            
            try {
                //create temp file
                File tempFile = File.createTempFile("merge", ".pdf");
                
                //merge to temp file
                bMergeResult = merger.Merge(tempFile.getAbsolutePath(), mapColors, false);
                
                //remove scribbles
                new File(strSourceDir + "scribble.irx").delete();
                new File(strSourceDir + "scribble.xml").delete();
                
                //copy temp file over original one
                copyFile(tempFile.getAbsolutePath(), strPdfSource);
                
                //delete the temp file
                tempFile.delete();
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
        } else if (cMergeType=='r') {
            //because the merging uses the original file, we create
            //a temporary pdf and replace the dir with this file later
            
            try {
                //create temp file
                File tempFile = File.createTempFile("merge", ".pdf");
                
                //merge to temp file
                bMergeResult = merger.Merge(tempFile.getAbsolutePath(), mapColors, false);
                
                //remove sources
                deleteDirectory(fileInput);
                
                //copy temp file over original one
                copyFile(tempFile.getAbsolutePath(), fileInput.getAbsolutePath());
                
                //delete the temp file
                tempFile.delete();
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
        }
        
        return bMergeResult;
    }
    
    static protected boolean copyFile(String strSource, String strDest){
        try {
            FileChannel srcChannel = new FileInputStream(strSource).getChannel();
            FileChannel dstChannel = new FileOutputStream(strDest).getChannel();
            dstChannel.transferFrom(srcChannel, 0, srcChannel.size());
            srcChannel.close();
            dstChannel.close();
            return true;
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        return false;
    }
    
    static protected boolean deleteDirectory(File path) {
        if( path.exists() ) {
            File[] files = path.listFiles();
            for(int i=0; i<files.length; i++) {
                if(files[i].isDirectory()) {
                    deleteDirectory(files[i]);
                } else {
                    files[i].delete();
                }
            }
        }
        return( path.delete() );
    }
    
    static protected String getPdfFromIRexFolder(String strSourceDir) {
        //find scribble.irx, manifest.xml
        String strManifestSource = strSourceDir + "manifest.xml";
        String strXmlSource = strSourceDir + "scribble.irx";
        
        //parse manifest.xml for .pdf file
        String strPdfSource = null;
        try {
            File fileManifest = new File(strManifestSource);
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            org.w3c.dom.Document doc = db.parse(fileManifest);
            doc.getDocumentElement().normalize();
            
            NodeList nlstStartPage = doc.getElementsByTagName("startpage");
            if (nlstStartPage.getLength()!=1)
                return null;
            
            Node nodeStartPage = (Element)nlstStartPage.item(0);
            
            if (nodeStartPage.getFirstChild() != null)
                strPdfSource = nodeStartPage.getFirstChild().getNodeValue();
            
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        
        //check if pdf exists
        strPdfSource = strSourceDir + strPdfSource;
        if (!(new File(strPdfSource).exists()))
            return null;
        
        return strPdfSource;
    }
}
