// LRFFile.cs
//
// Copyright (C) 2004-2005  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using System.Text;
using System.IO;
using ICSharpCode.SharpZipLib.Zip.Compression;

namespace EditLrfMetaGUI
{
	/// <summary>
	/// Represents an LRF file.
	/// </summary>
	class LrfFile 
	{

		#region Constants

		/// <summary>
		/// Size of a WORD
		/// </summary>
		public const int WORD_SIZE = 2;

		/// <summary>
		/// Size of a DWORD
		/// </summary>
		public const int DWORD_SIZE = 4;

		/// <summary>
		/// Size of a QWORD
		/// </summary>
		public const int QWORD_SIZE = 8;

		/// <summary>
		/// Offset of length of uncompressed info block in LRF file
		/// </summary>
		public const int LRF_OFFSET_UNCOMXML_SIZE = 0x54;

		/// <summary>
		/// Offset of compressed info block data in LRF file
		/// </summary>
		public const int LRF_OFFSET_COMXML_DATA = LRF_OFFSET_UNCOMXML_SIZE + DWORD_SIZE;

		/// <summary>
		/// Offset of Header Signature
		/// </summary>
		public const int LRF_OFFSET_HEADER_SIGNATURE = 0x00;

		/// <summary>
		/// Offset of Header Version
		/// </summary>
		public const int LRF_OFFSET_HEADER_VERSION = 0x08;

		/// <summary>
		/// Offset of Header PseudoEncryption
		/// </summary>
		public const int LRF_OFFSET_HEADER_PSEUDOENCRYPTION = 0x0A;

		/// <summary>
		/// Offset of Header RootObjectID
		/// </summary>
		public const int LRF_OFFSET_HEADER_ROOTOBJECTID = 0x0C;

		/// <summary>
		/// Offset of Header NumObjects
		/// </summary>
		public const int LRF_OFFSET_HEADER_NUMOBJECTS = 0x10;

		/// <summary>
		/// Offset of Header ObjectIndexOffset
		/// </summary>
		public const int LRF_OFFSET_HEADER_OBJECTINDEXOFFSET = 0x18;

		/// <summary>
		/// Offset of Header Unknown1
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN1 = 0x20;

		/// <summary>
		/// Offset of Header Flags
		/// </summary>
		public const int LRF_OFFSET_HEADER_FLAGS = 0x24;

		/// <summary>
		/// Offset of Header Unknown2
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN2 = 0x25;

		/// <summary>
		/// Offset of Header Unknown3
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN3 = 0x26;

		/// <summary>
		/// Offset of Header Unknown4
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN4 = 0x28;

		/// <summary>
		/// Offset of Header Height
		/// </summary>
		public const int LRF_OFFSET_HEADER_HEIGHT = 0x2A;

		/// <summary>
		/// Offset of Header Width
		/// </summary>
		public const int LRF_OFFSET_HEADER_WIDTH = 0x2C;

		/// <summary>
		/// Offset of Header Unknown5
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN5 = 0x2E;

		/// <summary>
		/// Offset of Header Unknown6
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN6 = 0x2F;

		/// <summary>
		/// Offset of Header Unknown7
		/// </summary>
		public const int LRF_OFFSET_HEADER_UNKNOWN7 = 0x30;

		/// <summary>
		/// Offset of Header TOCObjectID
		/// </summary>
		public const int LRF_OFFSET_HEADER_TOCOBJECTID = 0x44;

		/// <summary>
		/// Offset of Header TOCObjectOffset
		/// </summary>
		public const int LRF_OFFSET_HEADER_TOCOBJECTOFFSET = 0x48;

		/// <summary>
		/// Offset of Header XMLCompSize
		/// </summary>
		public const int LRF_OFFSET_HEADER_COMXMLSIZE = 0x4C;

		/// <summary>
		/// Offset of Hearder Thumbnail Image Type
		/// </summary>
		public const int LRF_OFFSET_HEADER_THUMBIMAGETYPE = 0x4E;

		/// <summary>
		/// Offset of Header GIFSize
		/// </summary>
		public const int LRF_OFFSET_HEADER_GIFSIZE = 0x50;

		#endregion Constants

		#region Fields

		/// <summary>
		/// LRF Signature
		/// </summary>
		private byte[] _lrfSignature;

		/// <summary>
		/// Keeps track of how much the header has grown/shrunk.
		/// Without this the offsets would get messed up.
		/// </summary>
		private int _originalLRFObjectOffset;

		/// <summary>
		/// LRF Version
		/// </summary>
		private int _version;

		/// <summary>
		/// LRF PseudoEncryption
		/// </summary>
		private int _pseudoEncryption;

		/// <summary>
		/// LRF RootObjectID
		/// </summary>
		private int _rootObjectID;

		/// <summary>
		/// LRF NumObjects
		/// </summary>
		private Int64 _numObjects;

		/// <summary>
		/// LRF ObjectIndexOffset
		/// </summary>
		private Int64 _objectIndexOffset;

		/// <summary>
		/// LRF Unknown1
		/// </summary>
		private int _unknown1;

		/// <summary>
		/// LRF Flags
		/// </summary>
		private byte _flags;

		/// <summary>
		/// LRF Unknown2
		/// </summary>
		private byte _unknown2;

		/// <summary>
		/// LRF Unknown3
		/// </summary>
		private int _unknown3;

		/// <summary>
		/// LRF Unknown4
		/// </summary>
		private int _unknown4;

		/// <summary>
		/// LRF Height
		/// </summary>
		private int _height;

		/// <summary>
		/// LRF Width
		/// </summary>
		private int _width;

		/// <summary>
		/// LRF Unknown5
		/// </summary>
		private byte _unknown5;

		/// <summary>
		/// LRF Unknown6
		/// </summary>
		private byte _unknown6;

		/// <summary>
		/// LRF Unknown7
		/// </summary>
		private byte[] _unknown7;

		/// <summary>
		/// LRF TOCObjectID
		/// </summary>
		private int _tocObjectID;

		/// <summary>
		/// LRF TOCObjectOffset
		/// </summary>
		private int _tocObjectOffset;

		/// <summary>
		/// LRF XMLCompSize
		/// </summary>
		private int _xmlCompSize;

		/// <summary>
		/// LRF Thumbnail Image Type
		/// </summary>
		private int _thumbImageType;

		/// <summary>
		/// LRF GIFSize
		/// </summary>
		private int _gifSize;

		/// <summary>
		/// Path of LRF file
		/// </summary>
		private string _location;

		/// <summary>
		/// The LRF data objects (payload)
		/// </summary>
		private byte[] _lrfObjectData;

		/// <summary>
		/// The uncompressed UTF-16 info block
		/// </summary>
		private byte[] _xmlInfoBlock;

		/// <summary>
		/// The compressed UTF-16 info block
		/// </summary>
		private byte[] _compXmlInfoBlock;

		/// <summary>
		/// The GIF file data for the thumnail image
		/// </summary>
		private byte[] _gifData;

		/// <summary>
		/// The file size of the LRF file
		/// </summary>
		private long _fileSize;

		#endregion Fields

		/// <summary>
		/// Initializes a new instance of the <see cref="LrfFile"/> class, using data from the
		/// specified LRF file.
		/// </summary>
		/// <param name="filename">
		/// The name of the LRF file to read from.
		/// </param>
		public LrfFile(string filename) 
		{
			this._location = filename;
      
			this._fileSize = new FileInfo(filename).Length;

			/* Used to store the whole LRF file as bytes */
			byte[] lrfBytes = new Byte[this._fileSize];

			#region ReadEntireLRFFile
			/* Read the entire LRF file into lrfBytes */
			byte[] tempBuffer = new Byte[4096];
			FileStream fs = new FileStream(filename, FileMode.Open);
			int numBytesRead = 0;
			int curPosition = 0;
			while (0 != (numBytesRead = fs.Read(tempBuffer, 0, 4096))) 
			{
				Buffer.BlockCopy(tempBuffer, 0, lrfBytes, curPosition, numBytesRead);
				curPosition += numBytesRead;
			}
			fs.Close ();
			tempBuffer = null;
			#endregion

			/* Parse the entire header */
			this.parseHeader(lrfBytes);

			#region ReadCompressedInfoBlock
			/* Allocate the temporary memory required for the Compressed Info Block */
			int infoLength = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_COMXMLSIZE);
			byte[] bytesInfoData = new Byte[infoLength - DWORD_SIZE];

			/* Copy the Info block into the temporary buffer */
			Buffer.BlockCopy(lrfBytes, LRF_OFFSET_COMXML_DATA, bytesInfoData, 0, (infoLength - DWORD_SIZE));
			#endregion

			#region ReadGIFData
			/* Allocate the required memory for the GIF */      
			int gifLength = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_GIFSIZE);
			this._gifData = new Byte[gifLength];

			/* Copy the GIF data into bytesGIFData */
			Buffer.BlockCopy(lrfBytes, (LRF_OFFSET_UNCOMXML_SIZE + infoLength), this._gifData, 0, gifLength);
			#endregion

			#region ReadLRFObjectPayload
			/* Remember where we originally started */
			this._originalLRFObjectOffset = (LRF_OFFSET_UNCOMXML_SIZE + infoLength) + gifLength;

			/* Allocate the required memory for the data */
			int lrfDataOffset = (LRF_OFFSET_UNCOMXML_SIZE + infoLength) + gifLength;
			int lrfDataLength = (int)this._fileSize - lrfDataOffset;
			this._lrfObjectData = new Byte[lrfDataLength];

			/* Copy the LRF data into bytesLRFObjectData */
			Buffer.BlockCopy(lrfBytes, lrfDataOffset, this._lrfObjectData, 0, lrfDataLength);
			#endregion

			#region UncompressInfoBlock
			/* Allocate the required memory for the Uncompressed Info Block */
			int infoUncompressedLength = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_UNCOMXML_SIZE);
			this._xmlInfoBlock = new byte[infoUncompressedLength];

			/* Uncompress the info block */
			Inflater i = new Inflater();
			i.SetInput(bytesInfoData);
			i.Inflate( this._xmlInfoBlock );
			#endregion

			this.updateCompXmlInfoBlock();

		}

		/// <summary>
		/// The full path of the LRF file
		/// </summary>
		public string Location 
		{
			get { return this._location; }
			set { this._location = value; }
		}

		/// <summary>
		/// The GIF data for the LRF thumbnail image
		/// </summary>
		public byte[] GIFData 
		{
			get
			{
				return this._gifData;
			}
			set 
			{ 
				this._gifData = value;
				this._gifSize = this._gifData.Length;
	
				/* Only do this if we actually have a thumbnail image */
				if (this._gifSize > 0) 
				{
					/* Set the Image Type: 0x12 for PNG, 0x14 for GIF, 0x13 for BM and 0x10 for JPEG */
					if ((this._gifData[1] == 'P') && (this._gifData[2] == 'N') && (this._gifData[3] == 'G')) 
					{
						/* It's a PNG */
						this._thumbImageType = 0x12;
					} 
					else if ((this._gifData[0] == 'B') && (this._gifData[1] == 'M')) 
					{
						/* It's a BMP */
						this._thumbImageType = 0x13;
					} 
					else if ((this._gifData[6] == 'J') && (this._gifData[7] == 'F') && 
						(this._gifData[8] == 'I') && (this._gifData[9] == 'F')) 
					{
						/* It's a JFIF */
						this._thumbImageType = 0x11;
					} 
					else if ((this._gifData[0] == 'G') && (this._gifData[1] == 'I') && 
						(this._gifData[2] == 'F') && (this._gifData[3] == '8')) 
					{
						/* It's a GIF */
						this._thumbImageType = 0x14;
					} 
					else 
					{
						/* I give up, lets just call it a GIF and hope nobody notices */
						this._thumbImageType = 0x14;
					}
				}
			}
		}

		/// <summary>
		/// The file size of the LRF file
		/// </summary>
		public long FileSize 
		{
			get
			{
				return this._fileSize;
			}
		}

		/// <summary>
		/// The uncompressed UTF-16 info block
		/// </summary>
		public byte[] XMLInfoBlock 
		{
			get
			{
				return this._xmlInfoBlock;
			}
			set
			{
				this._xmlInfoBlock = value;
				this.updateCompXmlInfoBlock();
			}
		}

		/// <summary>
		/// Updates the compressed XMLInfo Block
		/// </summary>
		private void updateCompXmlInfoBlock() 
		{
			/* Allocate the required temporary memory for the Compressed Info Block */
			/* Yes, I know that this doesn't need to be 2, but I can't find the exact
			   size right. */
			byte[] tempCompXMLInfoBlock = new byte[this._xmlInfoBlock.Length * 2];

			/* Compress the info block into the temporary array */
			Deflater d = new Deflater();
			d.SetInput( this._xmlInfoBlock );
			d.Finish();
			/* Copy the size of the uncompressed block and the temporary array into the */
			/* permanent byte array */
			int length = d.Deflate(tempCompXMLInfoBlock);
			this._compXmlInfoBlock = new byte[length + DWORD_SIZE];
			byte[] bytesUncompressedXMLSize = BitConverter.GetBytes( this._xmlInfoBlock.Length );
			Buffer.BlockCopy(bytesUncompressedXMLSize, 0, this._compXmlInfoBlock, 0, DWORD_SIZE);
			Buffer.BlockCopy(tempCompXMLInfoBlock, 0, this._compXmlInfoBlock, DWORD_SIZE, length);
			this._xmlCompSize = this._compXmlInfoBlock.Length;
		}

		/// <summary>
		/// Walks the object tree, and moves all offsets by a certain value
		/// </summary>
		/// <parameter name="lrfData">
		/// The object data
		/// </parameter>
		/// <parameter name="numObjects">
		/// The number of objects
		/// </parameter>
		/// <parameter name="objectIndexOffset">
		/// The offset to the object index.
		/// </parameter>
		/// <parameter name="diff">
		/// The adjustment value for each offset
		/// </parameter>
		private void relocateOffsets(byte[] lrfData, Int64 numObjects, Int64 objectIndexOffset, int diff) 
		{
			for(int obNum=0; obNum<numObjects; obNum++) 
			{
				Int32 oldOffset = BitConverter.ToInt32(lrfData, (int)(objectIndexOffset + (16 * obNum) + 4));
				byte[] bytesNewOffset = BitConverter.GetBytes(oldOffset + diff);
				Buffer.BlockCopy(bytesNewOffset, 0, lrfData, (int)(objectIndexOffset + (16 * obNum) + 4), DWORD_SIZE);
			}
		}

		/// <summary>
		/// Get the entire LRF file as an array of bytes.
		/// </summary>
		/// <returns>
		/// LRF file as array of bytes.
		/// </returns>
		public byte[] GetBytes() 
		{
			byte[] bytesHeader = this.generateHeader();
			byte[] lrfData = new byte[bytesHeader.Length + this._compXmlInfoBlock.Length + this._gifData.Length + this._lrfObjectData.Length];

			/* Copy in the sections one after the other*/
			Buffer.BlockCopy(bytesHeader, 0, lrfData, 0, bytesHeader.Length);
			Buffer.BlockCopy(this._compXmlInfoBlock, 0, lrfData, bytesHeader.Length, this._compXmlInfoBlock.Length);
			Buffer.BlockCopy(this._gifData, 0, lrfData, bytesHeader.Length + this._compXmlInfoBlock.Length, this._gifData.Length);
			Buffer.BlockCopy( this._lrfObjectData, 0, lrfData, 
				bytesHeader.Length + this._compXmlInfoBlock.Length + this._gifData.Length, this._lrfObjectData.Length);

			int currentLRFObjectOffset = this._xmlCompSize + this._gifSize + bytesHeader.Length;
			int diff = currentLRFObjectOffset - this._originalLRFObjectOffset;
			this.relocateOffsets(lrfData, this._numObjects, BitConverter.ToInt64(bytesHeader, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET), diff);

			return lrfData;
		}

		/// <summary>
		/// Generate the header
		/// </summary>
		/// <returns>
		/// bytes array representing the LRF header.
		/// </returns>
		private byte[] generateHeader() 
		{
			byte[] header = new byte[84];

			/* Generate the signature */
			Buffer.BlockCopy(this._lrfSignature, 0, header, LRF_OFFSET_HEADER_SIGNATURE, 8);

			/* Generate the version */
			Byte[] bytesVersion = BitConverter.GetBytes(this._version);
			Buffer.BlockCopy(bytesVersion, 0, header, LRF_OFFSET_HEADER_VERSION, WORD_SIZE);

			/* Generate the PsuedoEncryption key */
			Byte[] bytesPseudoEncryption = BitConverter.GetBytes(this._pseudoEncryption);
			Buffer.BlockCopy(bytesPseudoEncryption, 0, header, LRF_OFFSET_HEADER_PSEUDOENCRYPTION, WORD_SIZE);

			/* Generate the RootObjectID */
			Byte[] bytesRootObjectID = BitConverter.GetBytes(this._rootObjectID);
			Buffer.BlockCopy(bytesRootObjectID, 0, header, LRF_OFFSET_HEADER_ROOTOBJECTID, DWORD_SIZE);

			/* Generate the NumberOfObjects */
			Byte[] bytesNumberOfObjects = BitConverter.GetBytes(this._numObjects);
			Buffer.BlockCopy(bytesNumberOfObjects, 0, header, LRF_OFFSET_HEADER_NUMOBJECTS, 8);

			/* Generate Unknown1 */
			Byte[] bytesUnknown1 = BitConverter.GetBytes(this._unknown1);
			Buffer.BlockCopy(bytesUnknown1, 0, header, LRF_OFFSET_HEADER_UNKNOWN1, DWORD_SIZE);

			/* Generate flags */
			header[LRF_OFFSET_HEADER_FLAGS] = this._flags;

			/* Generate Unknown2 */
			header[LRF_OFFSET_HEADER_UNKNOWN2] = this._unknown2;

			/* Generate Unknown3 */
			Byte[] bytesUnknown3 = BitConverter.GetBytes(this._unknown3);
			Buffer.BlockCopy(bytesUnknown3, 0, header, LRF_OFFSET_HEADER_UNKNOWN3, WORD_SIZE);

			/* Generate Unknown4 */
			Byte[] bytesUnknown4 = BitConverter.GetBytes(this._unknown4);
			Buffer.BlockCopy(bytesUnknown4, 0, header, LRF_OFFSET_HEADER_UNKNOWN4, WORD_SIZE);

			/* Generate Height */
			Byte[] bytesHeight = BitConverter.GetBytes(this._height);
			Buffer.BlockCopy(bytesHeight, 0, header, LRF_OFFSET_HEADER_HEIGHT, WORD_SIZE);

			/* Generate Width */
			Byte[] bytesWidth = BitConverter.GetBytes(this._width);
			Buffer.BlockCopy(bytesWidth, 0, header, LRF_OFFSET_HEADER_WIDTH, WORD_SIZE);

			/* Generate Unknown5 */
			header[LRF_OFFSET_HEADER_UNKNOWN5] = this._unknown5;

			/* Generate Unknown6 */
			header[LRF_OFFSET_HEADER_UNKNOWN6] = this._unknown6;

			/* Generate Unknown7 */
			Buffer.BlockCopy(this._unknown7, 0, header, LRF_OFFSET_HEADER_UNKNOWN7, 20);

			/* Generate the TOCObjectID */
			Byte[] bytesTocObjectID = BitConverter.GetBytes(this._tocObjectID);
			Buffer.BlockCopy(bytesTocObjectID, 0, header, LRF_OFFSET_HEADER_TOCOBJECTID, DWORD_SIZE);

			/* Generate the xmlCompSize */
			Byte[] bytesXMLCompSize = BitConverter.GetBytes(this._xmlCompSize);
			Buffer.BlockCopy(bytesXMLCompSize, 0, header, LRF_OFFSET_HEADER_COMXMLSIZE, WORD_SIZE);

			/* Generate Thumbnail Image Type */
			Byte[] bytesThumbImageType = BitConverter.GetBytes(this._thumbImageType);
			Buffer.BlockCopy(bytesThumbImageType, 0, header, LRF_OFFSET_HEADER_THUMBIMAGETYPE, WORD_SIZE);

			/* Generate the gifSize */
			Byte[] bytesGIFSize = BitConverter.GetBytes(this._gifSize);
			Buffer.BlockCopy(bytesGIFSize, 0, header, LRF_OFFSET_HEADER_GIFSIZE, DWORD_SIZE);

			int currentLRFObjectOffset = this._xmlCompSize + this._gifSize + header.Length;
			int diff = currentLRFObjectOffset - this._originalLRFObjectOffset;

			/* Generate the ObjectIndexOffset */
			this._objectIndexOffset = this._objectIndexOffset + diff;
			Byte[] bytesObjectIndexOffset = BitConverter.GetBytes(this._objectIndexOffset);
			Buffer.BlockCopy(bytesObjectIndexOffset, 0, header, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET, 8);
      
			/* Generate the tocObjectOffset */
			this._tocObjectOffset = this._tocObjectOffset + diff;
			Byte[] bytesTocObjectOffset = BitConverter.GetBytes(this._tocObjectOffset);
			Buffer.BlockCopy(bytesTocObjectOffset, 0, header, LRF_OFFSET_HEADER_TOCOBJECTOFFSET, DWORD_SIZE);

			return header;
		}

		/// <summary>
		/// Parse the header from an LRF byte array
		/// </summary>
		private void parseHeader(byte[] lrfBytes) 
		{
			/* Extract the signature */
			this._lrfSignature = new Byte[8];
			Buffer.BlockCopy(lrfBytes, LRF_OFFSET_HEADER_SIGNATURE, this._lrfSignature, 0, 8);

			/* Extract the version */
			this._version = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_VERSION);

			/* Extract the PsuedoEncryption key */
			this._pseudoEncryption = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_PSEUDOENCRYPTION);

			/* Extract the RootObjectID */
			this._rootObjectID = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_ROOTOBJECTID);

			/* Extract the NumberOfObjects */
			this._numObjects = BitConverter.ToInt64(lrfBytes, LRF_OFFSET_HEADER_NUMOBJECTS);

			/* Extract the ObjectIndexOffset */
			this._objectIndexOffset = BitConverter.ToInt64(lrfBytes, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET);

			/* Extract Unknown1 */
			this._unknown1 = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN1);

			/* Extract flags */
			this._flags = lrfBytes[LRF_OFFSET_HEADER_FLAGS];

			/* Extract Unknown2 */
			this._unknown2 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN2];

			/* Extract Unknown3 */
			this._unknown3 = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN3);

			/* Extract Unknown4 */
			this._unknown4 = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN4);

			/* Extract Height */
			this._height = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_HEIGHT);

			/* Extract Width */
			this._width = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_WIDTH);

			/* Extract Unknown5 */
			this._unknown5 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN5];

			/* Extract Unknown6 */
			this._unknown6 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN6];

			/* Extract Unknown7 */
			this._unknown7 = new Byte[20];
			Buffer.BlockCopy(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN7, this._unknown7, 0, 20);

			/* Extract the TOCObjectID */
			this._tocObjectID = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_TOCOBJECTID);

			/* Extract the tocObjectOffset */
			this._tocObjectOffset = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_TOCOBJECTOFFSET);

			/* Extract the xmlCompSize */
			this._xmlCompSize = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_COMXMLSIZE);

			/* Extract Thumbnail Image Type */
			this._thumbImageType = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_THUMBIMAGETYPE);

			/* Extract the gifSize */
			this._gifSize = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_GIFSIZE);
		}
	}
}
