/*
 * File Name: scribbles.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "scribbles.h"

// system include files

// local include files
#include "log.h"
#include "math.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// the maximum number of entries in the list
#define SIZE_OF_LIST 100

// margin for no move
#define NO_MOVE 2

// number for slope that is very big

#define BIG_SLOPE 1000.0

//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

// type to store point information in the list
typedef struct _list_entry
{
	int  x;
	int  y;
	int  length; // length of segment (this and previous)
	char direction; // direction of the segment
} list_entry, *p_list_entry;

// The list
static list_entry list_data[SIZE_OF_LIST];

// first unused position in list_data
static int list_position = 0;

// The direction of the stroke
static char list_direction[SIZE_OF_LIST];
static char list_direction_condensed[SIZE_OF_LIST];

typedef struct _char_entry {
	char character; // the character to match for
	char *match;	// the directional pattern
	char *length;   // length of the pattern
} char_entry, *p_char_entry;

// This array contains the directional 'pattern' to match for, add patterns
// if more characters or alternatives for digits are needed
// The length from A-H, each representing 1/8 of the total length of the pattern
static char_entry number_data[] = {
	 { '1', "2", 				"H"}
	,{ '2', "96321236",			"ABAADAAD"}
	,{ '3', "963214632147",		"ABABABBABABA"}
	,{ '4', "1236",				"EAAD"}
	,{ '5', "123632147",		"ABABABABA"}
	,{ '6', "412369874",		"ADBABABAB"}
	,{ '7', "631",				"EAF"}
	,{ '8', "1236987896321",	"AAAAAABAAAAAB"}
	,{ '9', "4123698247",		"BABAAABDBA"}
	//,{ '0', "6248"}
};

#if 0
static char_entry number_data[] = {
	 { '1', "2"}
	,{ '2', "9316"}
	,{ '3', "3131"}
	,{ '4', "16"}
	,{ '5', "231"}
	,{ '6', "167"}
	,{ '7', "61"}
	,{ '8', "16761"}  // maybe not the most logical for an 8 ...
	,{ '9', "761"}    // this also does not start were you would normally start
	,{ '9', "43824"}  // alternative 9
	//,{ '0', "6248"}
};
#endif

// this list contains the neighbours for every direction
// first is direction, others are neightbours
static char *valid_matches[9] = {
	"142",
	"213",
	"326",
	"471",
	"5",
	"639",
	"748",
	"879",
	"986"
};

//==============================================================================
// Local Function Definitions
//==============================================================================

static void calc_direction();
static void condense();
char match();

//==============================================================================
// Functions Implementation
//==============================================================================


// reset the internal list of positions and start at first position
void scribble_start(int posx, int posy)
{
    //LOGPRINTF("(%d,%d)",posx, posy);

	// reset position
	list_position = 0;
	scribble_add(posx, posy);
}

// add a position to the list
void scribble_add(int posx, int posy)
{
    //LOGPRINTF("(%d,%d)",posx, posy);

	// check if room in list
	if (list_position < SIZE_OF_LIST)
	{
		list_data[list_position].x = posx;
		list_data[list_position].y = posy;
		list_position++;
	}
}

// add a position to the list
void scribble_end(int posx, int posy)
{
    //LOGPRINTF("(%d,%d)",posx, posy);

	// check if room in list
	if (list_position < SIZE_OF_LIST)
	{
		list_data[list_position].x = posx;
		list_data[list_position].y = posy;
		list_position++;
	}
}

// convert list to number
int scribble_get_number()
{
	char number;
	
    //LOGPRINTF("entry");

	// first convert the segments to a string of directions
	calc_direction();
	
	// condense the stream of directions
	condense();
	
	// match the directions with the numbers
	number = match();
	LOGPRINTF("number:%d", (number-'0'));

	return number;
}

void scribble_draw_last(GdkDrawable *drawable, GdkGC *gc)
{
	int i;
	int x0, y0;
	
	// initialise with position 0
	x0 = list_data[0].x;
	y0 = list_data[0].y;
	for (i=0; i<list_position; i++)
	{
		gdk_draw_line (drawable, gc, x0, y0, list_data[i].x, list_data[i].y);
		
		// store previous
		x0 = list_data[i].x;
		y0 = list_data[i].y;
	}
}
//==============================================================================
// Local Function Implementation
//==============================================================================

// Determine what the direction is for the gathered segments
// direction is as follows: 789
//                          456
//                          123
// so like the keypad on your keyboard
static void calc_direction()
{
	int i;				// index in positions
	int j;              // index in directions
	char dir;
	int deltax, deltay; // delta in x and y direction of segment
	int length2;        // square of length
	int length;
	int sum_length;     // sum of all segments length
	int avg_len;        // average length
	float slope;        // slope of the segment
	
    //LOGPRINTF("%d entries", list_position);

	list_direction[0] = '\0';
	
	// TODO: there are a lot of short segments in the list, it would be an idea to
	//       group segments together as long as their do not deviate too much and calc
	//       the direction afterwards.
	//       This should get ride of the '5's
	
	// First calc length of each segment and the total sum of the length
	// the length of some segments are very high compared to others, so the
	// length should be used to compensate
	// NOTE: the behaviour that sometimes very long segments are returned by the
	//       motion_notify events also leads to wrong detections!
	list_data[0].length = 0;
	sum_length = 0;
	for (i=1; i<list_position; i++)
	{
		double dlen;
		
		// calculate the delta
		deltax = list_data[i].x - list_data[i-1].x; // right is positive
		deltay = list_data[i-1].y - list_data[i].y; // up is positive
		
		// length (we only need measure for length, square of length is easy to calc)
		length2 = deltax*deltax + deltay*deltay;
		dlen = sqrt((double)length2);
		
		list_data[i].length = (int)dlen;
		sum_length += list_data[i].length;
	}
	
	// calc average length, using the actual list_positions can give incorrect
	// results when there is a very large segment and few others, so set fixed
	// average when few positions are available
	if (list_position > 5)
	{
		avg_len = sum_length/list_position;
		if (avg_len == 0)
		{
			LOGPRINTF("was 0");
			avg_len = 1;
		}
	}
	else
	{
		avg_len = 4;
	}
	
	// determine direction
	j = 0;
	for (i=1; i<list_position; i++)
	{
		// calculate the delta
		deltax = list_data[i].x - list_data[i-1].x; // right is positive
		deltay = list_data[i-1].y - list_data[i].y; // up is positive
		
		//LOGPRINTF("pos%d: %d %d - %d %d = %d %d", i, list_data[i-1].x, list_data[i-1].y, list_data[i].x, list_data[i].y, deltax, deltay);
		// length (we only need measure for length, square of length is easy to calc)
		length2 = deltax*deltax + deltay*deltay;
		
		// slope (x/y)
		if (deltay != 0)
		{
			slope = (float)deltax/(float)deltay;
		}
		else
		{
			slope = BIG_SLOPE;
		}
		//LOGPRINTF("len%d: %d %f", i, list_data[i].length, slope);
		
		// 5: no move
		if (length2 < NO_MOVE) { dir = '5'; }
		// 1: down left
		else if ((deltay < 0) && (slope< 3.0 && slope> 0.33)) { dir = '1'; }
		// 2: down
		else if ((deltay < 0) && (slope<= 0.33 && slope>=-0.33)) { dir = '2'; }
		// 3: down right
		else if ((deltay < 0) && (slope<-0.33 && slope>-3.0)) { dir = '3'; }
		// 4: left
		else if ((deltax < 0) && (slope>= 3.0 || slope<=-3.0)) { dir = '4'; }
		// 6: right
		else if ((deltax > 0) && (slope>= 3.0 || slope<=-3.0)) { dir = '6'; }
		// 7: up left
		else if ((deltay > 0) && (slope<-0.33 && slope>-3.0)) { dir = '7'; }
		// 8: up
		else if ((deltay > 0) && (slope<= 0.33 && slope>=-0.33)) { dir = '8'; }
		// 9: up right
		else if ((deltay > 0) && (slope> 0.33 && slope< 3.0)) { dir = '9'; }
		// else?
		else { dir = '5'; }

		list_data[i].direction = dir;
		
		//LOGPRINTF("dir:%c (avg:%d)", dir, avg_len);
		// store
		length = list_data[i].length;
		while (length > 0)
		{
			list_direction[j] = dir;
			if (j < (SIZE_OF_LIST-1)) j++;
			
			// compensate for average length
			length -= avg_len;
		}
		// end string
		list_direction[j] = '\0';
	}
	// we could sum up all segments with the same direction to remove small
	// ones. (and extend them again)
}

// limit the number of directions, to ease the final compare.
static void condense()
{
	int i, j;
	
	// Use a simple condense function only remove duplicates
	
	//LOGPRINTF("entry:%s", list_direction);

	// phase 0: remove begin and end entries to make algo more stable
	// put a '5' in the entries, so they will be removed
	if (strlen(list_direction) >= 1)
	{
		list_direction[0] = '5';
		list_direction[strlen(list_direction)-1] = '5';
	}
	list_direction_condensed[0] = '\0';
	
	// phase I: remove 'no movement' --------------------------------------
	j = 0;
	for (i=0; i<strlen(list_direction); i++)
	{
		// skip no movement
		if (list_direction[i] != '5')
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	//LOGPRINTF("condensedI:%s",list_direction_condensed);
	
#if 0
	// phase II: remove single entries
	//           not sure if this is the correct step, I wanted to remove some 'false'
	//           small segments. This code will remove one segments of each series
	//           A better alternative is probably to sum all segment positions before
	//           converting them to a single char and then remove short segments
	strcpy(list_direction, list_direction_condensed);
	j = 0;
	for (i=0; i<strlen(list_direction); i++)
	{
		// only add if same as next
		if (list_direction[i] == list_direction[i+1])
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
		// or if it matched the previous
		else if ( (i>0) && (strchr(valid_matches[list_direction[i]-'1'], list_direction[i-1] ) != NULL) )
		{
			// add previous!!
			list_direction_condensed[j] = list_direction[i-1];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	LOGPRINTF("condensedII:%s",list_direction_condensed);
#endif

#if 0
	// phase III: remove doubles
	// TODO: check what happens when doubles are not removed, will the matches
	//       improve?
	strcpy(list_direction, list_direction_condensed);
	// copy first value
	list_direction_condensed[0] = list_direction[0];
	list_direction_condensed[1] = '\0';
	j = 1;
	for (i=1; i<strlen(list_direction); i++)
	{
		// only add if different from previous
		if (list_direction_condensed[j-1] != list_direction[i])
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	LOGPRINTF("condensedIII:%s",list_direction_condensed);
#endif
}

// calc how many segments are to be expected based on the length
void calc_max_len(char *in, int *max)
{
	int in_size = strlen(in);
	int i;
	
	for (i=0; i<8; i++)
	{
		max[i] = (in_size*(i+1))/8;
		if (max[i] == 0)
		{
			max[i] = 1;
		}
		//LOGPRINTF("%d=%d", i, max[i]);
	}
}

char match()
{
	int entries;
	int i;
	int best = -1;       // index to best scoring entry
	int best_score = -1000;// best score so far
	int score;           // current score
	int pos_list=0;      // position in the direction-list
	char *s;             // current string to match for
	char *l;             // current length to match for
	int pos_match=0;     // position in the match string
	int number = -1;     // -1 no valid value found
	int found = -1;      // number of chars matched on one segment
	int max_len[8];      // calculated max len according to length-array of match
	int count=0;         // number of matches to one char
	int matched = 0;     // number of matched chars in s
	
	//LOGPRINTF("matching:%s", list_direction_condensed);

	// calc change of match for every number
	entries = sizeof(number_data)/sizeof(*number_data);
	
	// check every entry in the number_data-array, the best candidate wins...
	// I experimented a little with the scoring of good and bad directions,
	// at this moment all 'matches' have equal weight and a mismatch double
	// weight of a match.
	// There is a lot of room here for experiments ...
	for (i=0; i<entries; i++)
	{
		// character to match for
		pos_list = 0;
		s = number_data[i].match;
		l = number_data[i].length;
		calc_max_len(list_direction_condensed, max_len);
		score = 0;
		count = 0;
		//LOGPRINTF("match %s against:%s [%s] (%c)", list_direction_condensed, s, l, number_data[i].character);

		// first find first matching string from the input
		found = 0;
		while ((!found) && (list_direction_condensed[pos_list] != '\0'))
		{
			if (list_direction_condensed[pos_list] == s[0])
			{
				found = 1;
			}
			else if (strchr(valid_matches[s[0]-'1'], list_direction_condensed[pos_list] ) != NULL)
			{
				found = 1;
			}
			else
			{
				score -= 10;
				//LOGPRINTF("match: %c %c find first score:%d", s[0], list_direction_condensed[pos_list], score);
			}
			pos_list++;
		}
		
		// match as many characters from input string
		pos_match = 0;
		while ((pos_match < strlen(s)) && (list_direction_condensed[pos_list] != '\0'))
		{
			found = 0;
			while ((found >= 0) && (list_direction_condensed[pos_list] != '\0'))
			{
				// check if equal to current
				if (list_direction_condensed[pos_list] == s[pos_match])
				{
					count++;
					if (count <= max_len[l[pos_match]-'A'])
					{
						// only count if max length is not exceeded
						score += 30;
					}
					else if (count > (2 * max_len[l[pos_match]-'A']))
					{
						// too many -> penalty
						score -= 10;
					}
					//LOGPRINTF("match: %c %c current score:%d", s[pos_match], list_direction_condensed[pos_list], score);
					pos_list++;
					found++;
					if (count == 1) matched++;
				}
				
				// if exact match of next go to next
				else if (s[pos_match+1] && (list_direction_condensed[pos_list] == s[pos_match+1]))
				{
					// it is matching with next, quit internal while on condensed list
					// and continue with same character on next direction in s.
					if (found == 0)
					{
						// no character matched, penalty depends on length of skipped part
						score -= (10 * (l[pos_match]-'A'));
					    //LOGPRINTF("match: %c %c next skip score:%d", s[pos_match], list_direction_condensed[pos_list], score);
					}
					found = -1;
				}

				// check if near current
				else if (strchr(valid_matches[s[pos_match]-'1'], list_direction_condensed[pos_list] ) != NULL)
				{
					count++;
					if (count <= max_len[l[pos_match]-'A'])
					{
						// only count if max length is not exceeded
						score += 10;
					}
					//LOGPRINTF("match: %c %c near current score:%d", s[pos_match], list_direction_condensed[pos_list], score);
					pos_list++;
					found++;
					if (count == 1) matched++;
				}
				
				// does not match with current direction, break out of while
				else
				{
					if (found == 0)
					{
						// no character matched, penalty depends on length of skipped part
						score -= (50 * (l[pos_match]-'A'));
					    //LOGPRINTF("match: %c %c skip score:%d", s[pos_match], list_direction_condensed[pos_list], score);
					}
					found = -1;
				}
			}
			pos_match++;
			count=0;
		}
		
		// now compensate score for directions that have not yet been matched
		if (s[pos_match] != '\0')
		{
			pos_match++;
			while (s[pos_match] != '\0')
			{
				score -= (50 * (max_len[l[pos_match]-'A']-1));
				//OGPRINTF("match: %c missing score:%d", s[pos_match], score);
				pos_match++;
			}
		}
		
		// and for the input segments that have not yet been matched
		{
			int penalty = 10;
			while (list_direction_condensed[pos_list] != '\0')
			{
				score -= penalty;
				penalty += 2; // increase penalty when there is more input left
				//LOGPRINTF("match: %c too much input score:%d", list_direction_condensed[pos_list], score);
				pos_list++;
			}
		}
		
		// adjust score for length, lower score if matched string was very long
		// not sure what the effect is ...
		//score = score - ((int)strlen(s) * 5);
		
		// check how it scored
		if (score > best_score)
		{
			//LOGPRINTF("better %d", score);
			best_score = score;
			best = i;
		}
	}
	
	// Only one can win
	if (best >= 0)
	{
		number = number_data[best].character;
		//LOGPRINTF("best:%d=%c", best, number);
	}
	
	return number;
}
