/*
 * File Name: mxModel.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxModel.h"
#include "generator.h"
#include "solver.h"

// system include files
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

// local include files
#include "log.h"
#include "main.h"
#include "i18n.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// At this moment only one model can be active at any time
#define MXMODEL_MAX_MODELS 1

// needs to be big enough for largest game 13x17 has 285 entries!
#define MAX_LINE 300

#define FILE_HEADER "MX"
#define FILE_GAME_SUDOKU "SU"
#define FILE_GAME_KAKURO "KA"
//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

// info on the available (active) models
static int nr_models = 0;
static mxmodel_game_data game_data_array[MXMODEL_MAX_MODELS];

// static data for all supported game types
// kakuro games have extra cell on each side to display the 'sum-numbers'
static mxmodel_game supported_game_types[] = 
{
	  // type              h   v  empty   id        game_type
	 {mxmodel_sudoku3x3,    9,  9,  81,    "3x3   ", FILE_GAME_SUDOKU}
	,{mxmodel_sudoku4x4,   16, 16, 256,    "4x4   ", FILE_GAME_SUDOKU}
	,{mxmodel_kakuro5x5,    7,  7,   0,    "5x5   ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro6x6,    8,  8,   0,    "6x6   ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro8x8,   10, 10,   0,    "8x8   ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro10x10, 12, 12,   0,    "10x10 ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro12x12, 14, 14,   0,    "12x12 ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro15x15, 17, 17,   0,    "15x15 ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro11x17, 13, 19,   0,    "11x17 ", FILE_GAME_KAKURO}
	,{mxmodel_kakuro13x17, 15, 19,   0,    "13x17 ", FILE_GAME_KAKURO}
};

//==============================================================================
// Local Function Definitions
//==============================================================================

static int mxmodel_file_load_sudoku_old(int model, char *filename, int no_warning);
static void calc_ornaments(int model);
static void release_ornaments(int model);
static void count_empty_squares(int model);
static int get_count_must_values(int model);

//==============================================================================
// Functions Implementation
//==============================================================================

// find 'static' game type data
static p_mxmodel_game find_model_data(mxmodel_type type)
{
	p_mxmodel_game data = NULL;
	int i=0;
	
	while (i<sizeof(supported_game_types)/sizeof(*supported_game_types))
	{
		if (supported_game_types[i].type == type)
		{
			data = &supported_game_types[i];
			break; // out of while
		}
		i++;
	}
	
	return data;
}

// find 'static' game info from name
static p_mxmodel_game find_model_data_from_name(char *game, char *model)
{
	p_mxmodel_game data = NULL;
	int i=0;
	
	while (i<sizeof(supported_game_types)/sizeof(*supported_game_types))
	{
		if ((strncmp(game, supported_game_types[i].game_type, 2) == 0) &&
			(strncmp(model, supported_game_types[i].id, 6) == 0))
		{
			data = &supported_game_types[i];
			break; // out of while
		}
		i++;
	}
	
	return data;
}

// create a new model for a game, returns instance to model
int mxmodel_create(mxmodel_type type)
{
	int model = -1;
	p_mxmodel_game_data game_data;
	p_mxmodel_game      data;
	
	LOGPRINTF("entry: type:%d", type);

	// check if more models can be created
	if (nr_models < MXMODEL_MAX_MODELS)
	{
		data = find_model_data(type);
		if (data != NULL)
		{
			p_mxmodel_cell cells;
			int i, row, col;

			model = nr_models;
			LOGPRINTF("New model:%d", model);
			nr_models++;
			// get pointer to game data
			game_data = &game_data_array[model];
			// copy static data to game structure
			memcpy(&game_data->game, data, sizeof(*data));
			
			// reserve memory for cells of the game
			cells = malloc(sizeof(mxmodel_cell) * 
						   game_data->game.horizontal *
						   game_data->game.vertical);
			game_data->cells = cells;
			
			// init cells
			i=0;
			for (row = 0; row < game_data->game.vertical; ++row)
			{
				for (col = 0; col < game_data->game.horizontal; ++col) 
				{
					cells[i].flags = MXMODEL_CELL_READONLY; //no edit without game-data ...
					cells[i].must_value = -1; //unknown
					cells[i].value = -1; // not assigned
					if (mxmodel_is_kakuro(model) == 1)
					{
						if ((row != 0) && (row != (game_data->game.vertical-1)) &&
							(col != 0) && (col != (game_data->game.horizontal-1)))
						{
							// kakuro do not change border cells
							cells[i].flags |= MXMODEL_CELL_MODIFYABLE;
						}
					}
					else
					{
						// sudoku all cells modifyable
						cells[i].flags |= MXMODEL_CELL_MODIFYABLE;
					}
					i++;
				}
			}
			
			// calc the ornaments
			calc_ornaments(model);
		}
	}
	
	return model;
}

// delete data of the model (release all resources)
int mxmodel_delete(int model)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// release memory
	release_ornaments(model);
	free(game_data_array[model].cells);
	
	// update admin
	game_data_array[model].game.type = mxmodel_unknown;
	nr_models--;
	
	return 0;
}

// check if model is a sudoku
int mxmodel_is_sudoku(int model)
{
	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	if (strcmp(game_data_array[model].game.game_type, FILE_GAME_SUDOKU) == 0)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

// check if model is a kakuro
int mxmodel_is_kakuro(int model)
{
	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}

	if (strcmp(game_data_array[model].game.game_type, FILE_GAME_KAKURO) == 0)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

// is there a generator available
int mxmodel_can_generate(int model)
{
	int can_generate = 0;
	
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// and set if generation is possible
	if (mxmodel_is_sudoku(model) == 1)
	{
	    can_generate = 1;
	}
	else if (mxmodel_is_kakuro(model) == 1)
	{
	    can_generate = 0;
	}
	else
	{
		// unknown!?
	    can_generate = 0;
	}
	
	return can_generate;
}

// generate a new game for the instance with the indicated difficulty level
int mxmodel_generate(int model, int level)
{
	LOGPRINTF("entry: model:%d level:%d", model, level);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// and generate the random game data for the model if supported
	if (mxmodel_can_generate(model) == 1)
	{
		p_mxmodel_game data;
		// get pointer to 'static' game data
		data = find_model_data(game_data_array[model].game.type);
		if (data)
		{
			// reset empty squares
			game_data_array[model].game.empty_squares = data->empty_squares;
		}
		// generate data for soduku (see generate.c)
		mxmodel_gen_soduku_random_game(&game_data_array[model]);
		// and solve it ... (see solve.c)
		mxmodel_get_solution(&game_data_array[model]);
		
		// supported
		return 1;
	}
	
	// not supported
	return 0;
}

// verifies the game and returns number of incorrect answers
int mxmodel_verify(int model)
{
	p_mxmodel_game_data game_data;
	int nr_incorrect = 0;
	int i;
	
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}

	game_data = &game_data_array[model];

	// generic checker over all cells
	for (i=0; i<game_data->game.horizontal * game_data->game.vertical; i++)
	{
		// only handle cells that are not read-only
		if ((game_data->cells[i].flags & MXMODEL_CELL_READONLY) == 0)
		{
			// if not correct, increase count
			if (game_data->cells[i].value != game_data->cells[i].must_value)
			{
				nr_incorrect++;
			}
		}
	}
	
	return nr_incorrect;
}

// check if the must values for a model are complete
int mxmodel_check_complete(int model)
{
	int ok = 0;
	int solutions;
	
	// first calc solution, with the current given values
	solutions = mxmodel_solve(&game_data_array[model]);
	
	// count the cells that need to be filled (must_values should be up-to-date,
	// so after the solve-action!)
	count_empty_squares(model);
	
	// for kakuro we need to calc the ornaments
	calc_ornaments(model);

	// now verify correctness
	if (mxmodel_is_sudoku(model))
	{
		if (solutions > 0)
		{
			ok = 1;
		}
	}
	else
	{
		// kakuro is always OK I guess?
		ok = 1;
	}
	
	return ok;
}

// get info on the game (into supplied structure)
int mxmodel_get_info(int model, p_mxmodel_game game)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// copy game data
	game->type       = game_data_array[model].game.type;
	game->horizontal = game_data_array[model].game.horizontal;
	game->vertical   = game_data_array[model].game.vertical;
	
	return 0;
}

// get cell info (into supplied structure)
int mxmodel_get_cell_info(int model, int cell_nr, p_mxmodel_cell cell)
{
	p_mxmodel_cell game_cell;
	
	//LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	game_cell = &game_data_array[model].cells[cell_nr];
	cell->flags      = game_cell->flags;
	cell->must_value = game_cell->must_value;
	cell->value      = game_cell->value;

	//LOGPRINTF("leave: flags: %d must:%d value:%d", game_cell->flags, game_cell->must_value, game_cell->value);

	return 1;
}

// can change call (0=no)
int mxmodel_can_change_cell(int model, int cell_nr)
{
	p_mxmodel_cell game_cell;
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	game_cell = &game_data_array[model].cells[cell_nr];
	if ((game_cell->flags & MXMODEL_CELL_READONLY) != 0)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

// set cell value
int mxmodel_set_cell_value(int model, int cell_nr, int value)
{
	p_mxmodel_cell game_cell;
	
	//LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// get cell
	game_cell = &game_data_array[model].cells[cell_nr];

	if ((game_cell->flags & MXMODEL_CELL_READONLY) != 0)
	{
		// read only
		return 0;
	}
	else
	{
		// update empty_squares-count
		// was not filled and now filled
		if ((value != -1) && (game_cell->value == -1))
		{
			// cell filled
			game_data_array[model].game.empty_squares--;
		}
		
		// was filled and now not filled
		if ((value == -1) && (game_cell->value != -1))
		{
			// cell cleared
			game_data_array[model].game.empty_squares++;
		}
		
		// store value in model
		game_cell->value = value;
			
		// return the number of empty_squares
		return game_data_array[model].game.empty_squares;
	}
}

// set cell must_value
int mxmodel_set_cell_must_value(int model, int cell_nr, int value)
{
	p_mxmodel_cell game_cell;
	
	//LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// get cell
	game_cell = &game_data_array[model].cells[cell_nr];

	if ((game_cell->flags & MXMODEL_CELL_MODIFYABLE) == 0)
	{
		// not modifyable
		return 0;
	}
	else
	{
		// store value in model
		game_cell->must_value = value;
	}
	
	return 1;
}

static void count_empty_squares(int model)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	int i = 0;
	int count = 0;

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return; // FAILED!
	}

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			//LOGPRINTF("%d %d", game_cell->value, game_cell->must_value);
			if ((game_cell->value == -1) && (game_cell->must_value != -1))
			{
				count++;
			}
			i++;
		}
	}
	
	game_data_array[model].game.empty_squares = count;
	LOGPRINTF("count=%d", count);
}

static int get_count_must_values(int model)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	int i = 0;
	int count = 0;

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED!
	}

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			//LOGPRINTF("%d %d", game_cell->value, game_cell->must_value);
			if (game_cell->must_value != -1)
			{
				count++;
			}
			i++;
		}
	}
	
	LOGPRINTF("count=%d", count);
	return count;
}

// return the number of empty_squares
int mxmodel_get_empty_cells(int model)
{
	//LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}

	return game_data_array[model].game.empty_squares;
}

static mxmodel_type read_header(FILE *game_file)
{
	mxmodel_type game_type = mxmodel_unknown;
	char header[11] = "           "; // make empty

	LOGPRINTF("entry");

	fread(header, 1, 10, game_file);
	
	// Check game-file header
	if (strncmp(&header[0], FILE_HEADER, 2) == 0)
	{
		char strType[3];
		char strModel[7];
		p_mxmodel_game data;
		
		strncpy(strType, &header[2], 2);
		strType[2] = '\0';
		strncpy(strModel, &header[4], 6);
		strModel[6] = '\0';
		
		// find statis data for game that matches with the type and model from the file
		data = find_model_data_from_name (strType, strModel);
		if (data != NULL)
		{
			game_type = data->type;
		}
	}
	
	return game_type;
}

static void write_header(FILE *fp, p_mxmodel_game_data game_data)
{
	p_mxmodel_game data;

	// format
	fwrite(FILE_HEADER, 1, 2, fp);
	
	// game type
	data = find_model_data(game_data->game.type);
	if (data)
	{
		fwrite(data->game_type, 1, 2, fp);
		fwrite(data->id,        1, 6, fp);
	}
	else
	{
		fwrite("UNKNOWN!", 1, 8, fp);
	}
}

// get type of a game file
mxmodel_type mxmodel_file_get_type(char *filename)
{
	FILE *game_file;
	mxmodel_type game_type = mxmodel_unknown;
	
	LOGPRINTF("file: %s", filename);
	if( (game_file = fopen(filename, "r")) != NULL) 
	{
		game_type = read_header(game_file);
		
		fclose(game_file);
	}
	return game_type;
}

static void skip_over_comma(FILE *fp)
{
	char c=' ';
	
	while ((!feof(fp)) && (c != ','))
	{
		c = fgetc(fp);
		//LOGPRINTF("skip:0x%02x", (int)c);
	}
}

static void skip_nr_bytes(FILE *fp, int n)
{
	int i=0;
	char c;
	
	while ((!feof(fp)) && (i<n))
	{
		c = fgetc(fp);
		i++;
	}
}

static int read_line(FILE *fp, char *line, int max_len)
{
	int n;
	
	line[0] = '\0';
	
	// skip to comma
	skip_over_comma(fp);
	
	// first read size
	fscanf(fp, "%d,", &n);
	
	if (n <= max_len)
	{
		int r;
		
		// read len chars
		r = fread(line, 1, n, fp);
		if (r != n)
		{
			LOGPRINTF("Error in file, expected %d, got %d", n, r);
		}
	}
	else
	{
		skip_nr_bytes(fp, n);
		LOGPRINTF("Error to much data in file, max line = %d", max_len);
	}
	
	line[n] = '\0';
	//LOGPRINTF("len=%d line:[%s]", n, line);
	
	return n;
}

static int skip_line(FILE *fp)
{
	int n;
	
	// skip to comma
	skip_over_comma(fp);
	
	// first read size
	fscanf(fp, "%d,", &n);
	
	skip_nr_bytes(fp, n);
	
	return n;
}

static int char_to_num(char c)
{
	int num = -1;
	
	if (c == ' ')
	{
		num = -1;
	}
	else if (c <= '9')
	{
		num = c - '0';
	}
	else
	{
		num = 10 + c - 'A';
	}
	return num;
}

static char num_to_char(int v)
{
	char c = ' ';
	
	if (v < 0)
	{
		c = ' ';
	}
	else if ((v >= 0) && (v < 10))
	{
		c = (char)('0' + v);
	}
	else
	{
		c = (char)('A' + v - 10);
	}
	return c;
}

static void load_must_data(int model, char *line)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	int value;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			// Solution file, protect all cells that do not have a solution!
			game_cell = &game_data_array[model].cells[i];
			if (line[i] != ' ')
			{
				value = char_to_num(line[i]);
				game_cell->flags &= ~MXMODEL_CELL_READONLY;
			}
			else
			{
				value = -1;
				game_cell->flags |= MXMODEL_CELL_READONLY;
			}
			//LOGPRINTF("[%d,%d]=%d", col, row, value);
			i++;
			game_cell->must_value = value;
		}
	}
}

static void save_must_data(int model, FILE *fp)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	char c;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			i++;
			if (game_cell->must_value >= 0)
			{
				c = num_to_char(game_cell->must_value);
			}
			else
			{
				c = ' ';
			}
			fputc(c, fp);
		}
	}
}

static void load_entered_data(int model, char *line)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	int value;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			if (line[i] != ' ')
			{
				value = char_to_num(line[i]);
				game_cell->value = value;
			}
			else
			{
				game_cell->value = -1; // unknown
			}
			i++;
		}
	}
}

static void save_entered_data(int model, FILE *fp)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	char c;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			i++;
			if (game_cell->value >= 0)
			{
				c = num_to_char(game_cell->value);
			}
			else
			{
				c = ' ';
			}
			fputc(c, fp);
		}
	}
}

static void load_initial_data(int model, char *line)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	int value;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			//LOGPRINTF("Initial[%d] = %c", i, line[i]);
			if (line[i] != ' ')
			{
				value = char_to_num(line[i]);
				game_cell->value = value;
				game_cell->must_value = value;
				game_cell->flags |= MXMODEL_CELL_READONLY;
			}
			else
			{
				// make empty writable cell
				game_cell->value = -1;
				game_cell->must_value = -1;
				game_cell->flags &= ~MXMODEL_CELL_READONLY;
			}
			//LOGPRINTF("Initial[%d] = %d %d", i, game_cell->must_value, game_cell->value);
			i++;
		}
	}
}

static void save_initial_data(int model, FILE *fp)
{
	int row, col;
	p_mxmodel_cell game_cell; // the cell to update
	char c;
	int i = 0;

	LOGPRINTF("game: %dx%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);

	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[i];
			i++;
			if ((game_cell->flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY)
			{
				c = num_to_char(game_cell->must_value);
			}
			else
			{
				c = ' ';
			}
			//LOGPRINTF("[%d] = %c", (i-1), c);
			fputc(c, fp);
		}
	}
}

int mxmodel_solve(p_mxmodel_game_data game_data)
{
	int solved = 0;
	LOGPRINTF("entry");

	switch (game_data->game.type)
	{
		case mxmodel_sudoku3x3:
		case mxmodel_sudoku4x4:
			solved = mxmodel_get_solution(game_data);
		break;
		default:
			// there is no solver for this game type
			// solution should be provided in the game file
		break;
	}
	return solved;
}

// load game from file
int mxmodel_file_load(int model, char *filename, int no_warning)
{
	mxmodel_type game_type = mxmodel_unknown;
	FILE *game_file;
	char mode;
	char line[MAX_LINE];
	int len;     // the calculated length of a 'line'
	int size;    // the size of line as it is read
	int level;   // level of initial game 

	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	game_type = mxmodel_file_get_type(filename);
	LOGPRINTF("type:%d", (int)game_type);
	
	// check if game to be loaded matches with model
	if (game_type != game_data_array[model].game.type)
	{
		if (game_type == mxmodel_unknown)
		{
			// try the old type file
			return mxmodel_file_load_sudoku_old(model, filename, no_warning);
		}
		LOGPRINTF("ERROR: incorrect model used!");
		return -1;
	}
	
	// now read the file
	if( (game_file = fopen(filename, "r")) == NULL) 
	{
		if ((no_warning & 1) == 0)
		{
			char *message;
			message = g_strdup_printf(_("Unable to open %s!"), filename);
			show_popup_msg("?", message);
			g_free(message);
		}
		return 0;
	}
	
	// file is opened, skip over header (correctness is already checked)
	game_type = read_header(game_file);
	
	// determine the length of a 'line'
	len = game_data_array[model].game.vertical * game_data_array[model].game.horizontal; 
	
	while(!feof(game_file))
	{
		mode = fgetc(game_file);
		//LOGPRINTF("reading:0x%02x", mode);
		while ((mode == '\n') || (mode == '\r') || (mode == ' '))
		{
			mode = fgetc(game_file);
			//LOGPRINTF("reading:0x%02x", mode);
		}
		
		switch(mode)
		{
		// solution S,size,<solution> solution=[1-9]* | [0-F]*
		case 'S' :
			size = read_line(game_file, line, MAX_LINE);
			if (size != len)
			{
				// length does not match, incorrect file
				fclose(game_file);
				LOGPRINTF("\nError in S: %d\n", size);
				return -2;
			}
			// load the data
			load_must_data(model, line);
		break;
				
		// hints H,size,<hints> hints=NUM[,NUM]*
		case 'H' :
			// not handled
			skip_line(game_file);
		break;
				
		// initial value Ix,size,<solution> solution=[1-9]* | [0-9A-F]* |
		case 'I' : 
			// next char is level '0'-'9' '0'=level not defined
			level = fgetc(game_file);
			size = read_line(game_file, line, MAX_LINE);
			if (size != len)
			{
				// length does not match, incorrect file
				fclose(game_file);
				LOGPRINTF("\nError in I: %d\n", size);
				return -2;
			}
			// load the initial values
			load_initial_data(model, line);
		break;
				
		// entered values E,size,<solution> solution=[1-9]* | [0-F]*
		case 'E' :
			size = read_line(game_file, line, MAX_LINE);
			if (size != len)
			{
				// length does not match, incorrect file
				fclose(game_file);
				LOGPRINTF("\nError in E: %d\n", size);
				return -2;
			}
			// load entered data
			load_entered_data(model, line);
		break;
				
		default:
			// not handled
			skip_line(game_file);
		break;
		}
	}

	// and solve it ... to update the must_values (in case they were not loaded
	mxmodel_solve(&game_data_array[model]);
	
	// count the cells that need to be filled (must_values should be up-to-date,
	// so after the solve-action!)
	count_empty_squares(model);
	
	// for kakuro we need to calc the ornaments
	calc_ornaments(model);
	
	fclose(game_file);
	
	return 1;	
}

// load sudoku game from file (old format)
static int mxmodel_file_load_sudoku_old(int model, char *filename, int no_warning)
{
	FILE *game_file;
	char number[2];         // sudoku square value as string
	int row, col;
	int value;              // value of the cell
	int flags;              // flags for the cell
	p_mxmodel_cell game_cell; // the cell to update
	int index = 0;            // index in cell-array
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	if( (game_file = fopen(filename, "r")) == NULL) 
	{
		if ((no_warning & 1) == 0)
		{
			char *message;
			message = g_strdup_printf(_("Unable to open %s!"), filename);
			show_popup_msg("?", message);
			g_free(message);
		}
		return 0;
	}

	// NUMBER will be treated as a string to place within sudoku square entry
	number[1] = '\0';

	// reset empty cells
	game_data_array[model].game.empty_squares = 
		game_data_array[model].game.vertical *
		game_data_array[model].game.horizontal;
	
	// This code assumes the same format for the files a gtk-sudoku
	// TODO: Create a new file format, preferably with several difficulty
	//       levels in the same file.
	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[index++];
	
			// check for 'unexpected' EOF
			if ( (*number = fgetc(game_file)) == EOF ) 
			{
				if (game_data_array[model].game.horizontal != 9)
				{
					show_popup_msg("!!", _("Your file's too short\nIt's probably a 9x9 sudoku."));
				}
				else
				{
					show_popup_msg("!!", _("Your file's missing some values."));
				}
				if (fclose(game_file) != 0)
				{
					g_print("  * Unable to close %s.\n", filename);
				}
				return 0;
			}
			
			flags = 0;
			switch (*number) 
			{
				// 9x9 sudoku
				case '1': case '2': case '3':
				case '4': case '5': case '6':
				case '7': case '8': case '9':
				    value = *number - '0'; // '1' = 1, etc
					break;
				// extra symbols 16x16 sudoku
				case '0': case 'A': case 'B':
				case 'C': case 'D': case 'E':
				case 'F':
					if (game_data_array[model].game.horizontal != 16) 
					{
						show_popup_msg("!!", _("Incorrect file format"));
						return FALSE;
					}
					value = (*number == '0') ? 0 : 10 + *number - 'A'; // '0' = 0, 'A' = 10, etc
					break;
				// 9x9 sudoku, unchangable values
				case 'L': case 'M': case 'N':
				case 'O': case 'P': case 'Q':
				case 'R': case 'S': case 'T':
					value = *number - 'L' + 1;
					flags = MXMODEL_CELL_READONLY;
					break;
				case 'K':
				case 'U': case 'V': case 'W':
				case 'X': case 'Y': case 'Z':
					if (game_data_array[model].game.horizontal != 16) 
					{
						show_popup_msg("!!", _("Incorrect file format"));
						return FALSE;
					}
					value = (*number == 'K') ? 0 : *number - 'U' + 10;  // 'K' = 0, 'U' = 10, etc
					flags = MXMODEL_CELL_READONLY;
					break;
				// treat all other input as a blank square
				default:
				    value = -1;
					break;
			} // switch
			
			// store values
			//LOGPRINTF("read: [%d] = %d (flags:%d)", index-1, value, flags);
			game_cell->flags = flags;
			game_cell->value = value;
			if (flags & MXMODEL_CELL_READONLY)
			{
				// the value was a given value
				game_cell->must_value = value;
			}
			else
			{
				game_cell->must_value = -1;
			}
			if (value != -1)
			{
				// cell is filled
				game_data_array[model].game.empty_squares--;

			}
		}
	}
	
	// and solve it ... to update the must_values
	mxmodel_get_solution(&game_data_array[model]);

	if (fclose(game_file) != 0)
	{
		g_print("  * Unable to close %s.\n", filename);
	}
	
	return 1;
}

// save game to file
int mxmodel_file_save(int model, char *filename)
{
	FILE *fp=NULL;
	int len;
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return -1; // FAILED
	}
	
	// now read the file
	if( (fp = fopen(filename, "w")) == NULL) 
	{
		char *message;
		message = g_strdup_printf(_("Unable to open %s!"), filename);
		show_popup_msg("?", message);
		g_free(message);
		return 0;
	}
	
	// determine the length of a 'line'
	len = game_data_array[model].game.vertical * game_data_array[model].game.horizontal;
	
	// write header
	write_header(fp, &game_data_array[model]);
	fprintf(fp, "\n");
	
	// write data
	// Solution (for kakuro)
	if (mxmodel_is_kakuro(model))
	{
		fprintf(fp, "S,%d,", len);
		save_must_data(model, fp);
		fprintf(fp, "\n");
	}
	// Initial (for sudoku)
	if (mxmodel_is_sudoku(model))
	{
		fprintf(fp, "I0,%d,", len);
		save_initial_data(model, fp);
		fprintf(fp, "\n");
	}
	// Entered values (always)
	{
		fprintf(fp, "E,%d,", len);
		save_entered_data(model, fp);
		fprintf(fp, "\n");
	}
	
	fclose(fp);
	
	return 1;
}

// get the ornaments for the game
char **mxmodel_get_ornaments(int model)
{
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return NULL; // FAILED
	}
	
	return game_data_array[model].pp_ornaments;
}
//==============================================================================
// Local Function Implementation
//==============================================================================

static void calc_ornaments(int model)
{
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return; // FAILED
	}
	
	// release previous ornaments
	release_ornaments(model);
	
	// calc the ornaments
	if (mxmodel_is_sudoku(model) == 1 )
	{
		int len; // len of a single block of the sudoku
		int row, col;
		char **pp=NULL;
		int i=0;
		
		// memory for pointers
		len = (int)(sqrt((double)game_data_array[model].game.horizontal)+0.5);
		pp = malloc(game_data_array[model].game.vertical * game_data_array[model].game.horizontal * 4);
		game_data_array[model].pp_ornaments = pp;
		
		LOGPRINTF("Allocating new ornaments (%dx%d) for Sudoku-%d", game_data_array[model].game.horizontal, game_data_array[model].game.vertical, len);
		// loop over all cells, reserve memory and fill
		
		for (row = 0; row < game_data_array[model].game.vertical; ++row)
		{
			for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
			{
				pp[i] = malloc(10);
				// top
				if (row == 0) { pp[i][0] = '4'; }
				else if ((row % len) == 0) { pp[i][0] = '2'; }
				else { pp[i][0] = '1'; }
				
				// right
				if (col == (len*len -1)) { pp[i][1] = '4'; }
				else if ((col % len) == (len-1)) { pp[i][1] = '2'; }
				else { pp[i][1] = '1'; }
				
				// bottom
				if (row == (len*len -1)) { pp[i][2] = '4'; }
				else if ((row % len) == (len-1)) { pp[i][2] = '2'; }
				else { pp[i][2] = '1'; }
				
				// left
				if (col == 0) { pp[i][3] = '4'; }
				else if ((col % len) == (len)) { pp[i][3] = '2'; }
				else { pp[i][3] = '1'; }
				
				pp[i][4] = '\0';
				i++;
			}
		}
	}
	else if (mxmodel_is_kakuro(model) == 1)
	{
		int row, col;
		char **pp=NULL;
		int i=0, j=0, sum=0, s;
		int grid_only = 0;
		
		// memory for pointers
		pp = malloc(game_data_array[model].game.vertical * game_data_array[model].game.horizontal * 4);
		//LOGPRINTF("Malloc Array:0x%08x", (int)pp);
		game_data_array[model].pp_ornaments = pp;
		
		// check if all must values are empty, in which case we draw the complete
		// grid
		if (get_count_must_values(model) == 0)
		{
			grid_only = 1;
			LOGPRINTF("Grid Only Mode");
		}
		
		LOGPRINTF("Allocating new ornaments (%dx%d) for Kakuro", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);
		// loop over all cells, reserve memory and fill
		for (row = 0; row < game_data_array[model].game.vertical; ++row)
		{
			for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
			{
				pp[i] = malloc(10);
				//LOGPRINTF("Malloc entry[%d]:0x%08x", i, (int)pp[i]);
				if ((grid_only == 1) && (game_data_array[model].cells[i].flags & MXMODEL_CELL_MODIFYABLE))
				{
					// normal cell in grid mode
					strcpy(pp[i], "1111AAAA");
				}
				else if (game_data_array[model].cells[i].must_value < 0)
				{
					// invisible cell
					strcpy(pp[i], "1111BBBB");
				}
				else
				{
					// normal cell
					strcpy(pp[i], "1111AAAA");
				}
				//LOGPRINTF("[%d]=%s", i, pp[i]);
				i++;
			}
		}
		
		if (grid_only == 1)
		{
			// no counts will be shown
			return;
		}
		// calc the sums to be shown horizonal
		i=0;j=-1;
		for (row = 0; row < game_data_array[model].game.vertical; ++row)
		{
			for (col = 0; col < (game_data_array[model].game.horizontal-1); ++col) 
			{
				i = row*game_data_array[model].game.horizontal + col;
			
				//LOGPRINTF("[%d(%d)]=%d-%d", i, j, game_data_array[model].cells[i].must_value, game_data_array[model].cells[i+1].must_value);
				if (j == -1) // still searching
				{
					if ((game_data_array[model].cells[i].must_value == -1) && 
						(game_data_array[model].cells[i+1].must_value != -1))
					{
						j=i; // found candidate
						sum = 0;
					}
				}
				else
				{
					if ((game_data_array[model].cells[i].must_value != -1) && 
						(game_data_array[model].cells[i+1].must_value == -1))
					{
						// count last one
						sum += game_data_array[model].cells[i].must_value;
						// change right type of left cell
						game_data_array[model].pp_ornaments[j][5] = (char)('C' + sum);
						// change left type of right cell
						game_data_array[model].pp_ornaments[i+1][7] = (char)('C' + sum);
						// reset 'search'
						j = -1;
						//LOGPRINTF("H[%d] = %d", i, sum);
					}
					else
					{
						sum += game_data_array[model].cells[i].must_value;
					}
				}
			}
		}
		// calc the sums to be shown vertically
		i=0;j=-1;
		s = game_data_array[model].game.horizontal; // step for next cell
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			for (row = 0; row < (game_data_array[model].game.vertical-1); ++row)
			{
				i = row*game_data_array[model].game.horizontal + col;
				//LOGPRINTF("[%d(%d)]=%d-%d", i, j, game_data_array[model].cells[i].must_value, game_data_array[model].cells[i+s].must_value);
				
				if (j == -1) // still searching
				{
					if ((game_data_array[model].cells[i].must_value == -1) && 
						(game_data_array[model].cells[i+s].must_value != -1))
					{
						j=i; // found candidate
						sum = 0;
					}
				}
				else
				{
					if ((game_data_array[model].cells[i].must_value != -1) && 
						(game_data_array[model].cells[i+s].must_value == -1))
					{
						// count last one
						sum += game_data_array[model].cells[i].must_value;
						// change bottom type of top cell
						game_data_array[model].pp_ornaments[j][6] = (char)('C' + sum);
						// change top type of bottom cell
						game_data_array[model].pp_ornaments[i+s][4] = (char)('C' + sum);
						// reset 'search'
						j = -1;
						//LOGPRINTF("V[%d] = %d", i, sum);
					}
					else
					{
						sum += game_data_array[model].cells[i].must_value;
					}
				}
			}
		}
	}
	else
	{
		// unknown !?
	}
	
}

static void release_ornaments(int model)
{
	int row, col;
	int i=0;

	LOGPRINTF("entry : 0x%x", (int)game_data_array[model].pp_ornaments);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		LOGPRINTF("Incorrect model:%d", model);
		return; // FAILED
	}
	
	// loop over all cells, free memory
	i=0;
	if (game_data_array[model].pp_ornaments)
	{
		LOGPRINTF("Releasing old ornaments (%dx%d)", game_data_array[model].game.horizontal, game_data_array[model].game.vertical);
		for (row = 0; row < game_data_array[model].game.vertical; ++row)
		{
			for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
			{
				if (game_data_array[model].pp_ornaments[i])
				{
					free(game_data_array[model].pp_ornaments[i]);
					//LOGPRINTF("Free entry[%d]:0x%08x", i, (int)game_data_array[model].pp_ornaments[i])
				}
				i++;
			}
		}
		free(game_data_array[model].pp_ornaments);
		//LOGPRINTF("Free Array:0x%08x", (int)game_data_array[model].pp_ornaments);
		game_data_array[model].pp_ornaments = NULL;
	}
}
