/*
 * File Name: main.c
 */

/*
 * This file is part of mxSudoku.
 * based on the helloworld example of iRex
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Mackx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <glib.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "do.h"
#include "mxModel.h"
#include "mxControl.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";

FILE *fplog = NULL;

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

//============================================================================
// Local Function Definitions
//============================================================================

static void     main_set_text           ( void );

static gboolean on_startup_complete     (gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

// set locale (language) (called from ipc.c)
void main_set_locale (const char *locale)
{
    LOGPRINTF("entry: locale [%s]", locale);

    g_setenv("LANG", locale, TRUE);
    setlocale(LC_ALL, "");

    main_set_text();
    menu_set_text();
}


// set screen texts
static void main_set_text (void)
{
    LOGPRINTF("entry");
	// will be called when the local is updated, so redraw all
	// widgets that have a label.
	
	// TODO
}


// terminate application
void main_quit (void)
{
    WARNPRINTF("entry");
    
	mxcontrol_quit_game();
	
    menu_destroy();
    
    if (g_main_window)
    {
        gtk_widget_destroy(g_main_window);
        g_main_window = NULL;
    }    
    
    if (gtk_main_level() > 0)
    {
        WARNPRINTF("quit main loop");
        gtk_main_quit();
    }
    else
    {
        WARNPRINTF("no main loop to quit, exit directly");
        _exit(0);
    }
}


// terminate signal
static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


// report startup completed
static gboolean on_startup_complete (gpointer data)
{
    LOGPRINTF("entry");

	// tell sysd that we are alive
    ipc_sys_startup_complete();

    return FALSE; // remove timer source
}

// main function
int main (int argc, char *argv[])
{
    GtkWidget        *window;
    struct sigaction on_term;
	int i=1;
	int demo_mode = 0;

	// The log file have been changed to allow logging to be captured on the DR
	// the first line gives logging on stderr, the 2nd line logs in a file.
	// please select the one you need.
	fplog = NULL;

	while (i < argc)
	{
		// should all logs go to stderr? (note this leaves to unacceptable delays on the real device!)
		if (strcmp(argv[i], "err") == 0)
		{
			fplog = stderr;
		}
		// should all logging go to a (fixed) log file?
		if (strcmp(argv[i], "log") == 0)
		{
			LOGINIT("/media/mmcblk0p1/Programs/mxlog.txt");
		}
		// should scribble test mode be enabled?
		if (strcmp(argv[i], "scribble") == 0)
		{
			demo_mode = MXCTRL_DEMO_MODE_SCRIBBLE;
		}
		
		i++;
	}	
    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

    // init domain for translations
    setlocale(LC_ALL, "");
	//LOGPRINTF("old locale:%s", bindtextdomain( GETTEXT_PACKAGE, NULL ));
	bindtextdomain( GETTEXT_PACKAGE, "/media/mmcblk0p1/Programs/_mxSudoku/locale" );
	//LOGPRINTF("new locale:%s", bindtextdomain( GETTEXT_PACKAGE, NULL ));
    textdomain(GETTEXT_PACKAGE);
    
    // init gtk
    gtk_init(&argc, &argv);

	// init modules
    ipc_set_services();
    menu_init();

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // create the top level window
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), PACKAGE " " VERSION);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    g_main_window = window;

	// make sure that everything, window and label, are visible 
    gtk_widget_show(window);

    // tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);
    
    // create a new (empty) game board let control decide which game
	mxcontrol_create_game(window, mxmodel_unknown);
	mxcontrol_set_demo_mode(demo_mode);

    // run the main loop
    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    // clean up modules

    return 0;
}


// run error dialog
void run_error_dialog (const gchar *msg)
{
    GtkWidget   *dialog = NULL;

    ERRORPRINTF("entry: msg [%s]", msg);

    dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     msg );

    gtk_dialog_run( GTK_DIALOG(dialog) );
    gtk_widget_destroy( dialog );
}

void show_popup_msg(const char *title,
	                const char *message)
{
	GtkWidget *msg;
	GtkWidget *hbox;
	GtkWidget *label;

	msg = gtk_dialog_new_with_buttons(
			title,
			GTK_WINDOW(g_main_window),
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR, 
			GTK_STOCK_OK, GTK_RESPONSE_NONE, NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(msg)->vbox), hbox);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 8);
	gtk_widget_show(hbox);

	label = gtk_label_new(message);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	g_signal_connect(G_OBJECT(msg), "response", 
			G_CALLBACK(gtk_widget_destroy), NULL);
	gtk_widget_show(msg);
}

void do_close(void)
{
    LOGPRINTF("entry");

	// 
	main_quit();
}

void main_about_game(void)
{
	GtkWidget *about_dialog;
	GtkWidget *vbox;
	GdkPixbuf *pix;
	GError *error = NULL;
	GtkWidget *logo;
	PangoFontDescription *font;
	GtkWidget *program_version;
	GtkWidget *author_label;
	GtkWidget *info;
	
	// TODO: make something that looks better in code ...

	about_dialog = gtk_dialog_new_with_buttons(
			_("About mxSudoku"), GTK_WINDOW(g_main_window),
			GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL,
			GTK_STOCK_CLOSE, GTK_RESPONSE_NONE,
			NULL);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_box_pack_start(
			GTK_BOX(GTK_DIALOG(about_dialog)->vbox), vbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
	gtk_widget_show(vbox);

	pix = gdk_pixbuf_new_from_file("/media/mmcblk0p1/Programs/_mxSudoku/icons/sudoku_new.png", &error);
	if(error == NULL) {         // no error loading image
		logo = gtk_image_new_from_pixbuf(pix);
		gtk_box_pack_start(GTK_BOX(vbox), logo, TRUE, TRUE, 0);
		gtk_widget_show(logo);
	}

	font = pango_font_description_new();
	pango_font_description_set_family(font, "Romeodn");
	pango_font_description_set_size(font, 13 * PANGO_SCALE);
	pango_font_description_set_weight(font, PANGO_WEIGHT_BOLD);

	program_version = gtk_label_new(PACKAGE_STRING);
	gtk_widget_modify_font(program_version, font);
	gtk_box_pack_start(GTK_BOX(vbox), program_version, TRUE, TRUE, 0);
	gtk_widget_show(program_version);

	font = pango_font_description_new();
	pango_font_description_set_family(font, "Romeodn");
	pango_font_description_set_size(font, 9 * PANGO_SCALE);
	pango_font_description_set_weight(font, PANGO_WEIGHT_BOLD);

	info = gtk_label_new( 
		_("First select a number on the lower keypad, then select the\n"
		"position in the sudoku/kakuro that should get the samevalue.\n"
		"Use \'C\' to clear values\n"
		"    \'V\' to verify current values\n"
		"    \'?\' to cheat...\n")
		);
	gtk_widget_modify_font(program_version, font);
	gtk_box_pack_start(GTK_BOX(vbox), info, TRUE, TRUE, 0);
	gtk_widget_show(info);
	
	info = gtk_label_new( 
        _("Sudoku:\n"
		"Fill in the blank squares so that each row, each column and each\n"
        "3-by-3 block contain all of the digits 1 thru 9.\n")
		);
	gtk_widget_modify_font(program_version, font);
	gtk_box_pack_start(GTK_BOX(vbox), info, TRUE, TRUE, 0);
	gtk_widget_show(info);

	info = gtk_label_new( 
        _("Kakuro:\n"
		"Kakuro puzzles are like a cross between a crossword and a Sudoku puzzle.\n"
		"Instead of letters, each block contains the digits 1 through 9. The\n"
		"same digit will never repeat within a word. If you add the digits in\n"
		"a word, the sum will be the number shown in the clue. Clues are shown\n"
		"on the left and right sides of \"across\" words, and on the top and\n"
		"bottom sides of \"down\" words.\n"
		)
		);
	gtk_widget_modify_font(program_version, font);
	gtk_box_pack_start(GTK_BOX(vbox), info, TRUE, TRUE, 0);
	gtk_widget_show(info);

	pango_font_description_set_family(font, "Andale Mono");
	pango_font_description_set_size(font, 7 * PANGO_SCALE);
	pango_font_description_set_weight(font, PANGO_WEIGHT_NORMAL);

	author_label = gtk_label_new("2009 Mackx");
	gtk_widget_modify_font(author_label, font);
	gtk_box_pack_start(GTK_BOX(vbox), author_label, TRUE, TRUE, 0);
	gtk_widget_show(author_label);

	// translation
	author_label = gtk_label_new(_("Translation: 2009"));
	gtk_widget_modify_font(author_label, font);
	gtk_box_pack_start(GTK_BOX(vbox), author_label, TRUE, TRUE, 0);
	gtk_widget_show(author_label);

	gtk_dialog_run(GTK_DIALOG(about_dialog));
	gtk_widget_destroy(about_dialog);
}


